/**
 * 
 */
package code.simulator;

import java.io.ByteArrayInputStream;
import java.io.ByteArrayOutputStream;
import java.io.IOException;
import java.io.ObjectInput;
import java.io.ObjectInputStream;
import java.io.ObjectOutput;
import java.io.ObjectOutputStream;
import java.lang.reflect.AccessibleObject;
import java.lang.reflect.Field;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.HashMap;
import java.util.Hashtable;
import java.util.LinkedList;
import java.util.Map;
import java.util.TreeSet;
import java.util.Map.Entry;

import code.AcceptStamp;
import code.CounterVV;
import code.Env;
import code.NoSuchEntryException;
import code.ObjId;
import code.VV;
import code.NodeId;
import code.AcceptVV;
import code.VVIterator;
import code.branchDetecting.BranchID;
import code.security.SangminConfig;
import code.security.ahs.DependencyVV;
import code.serialization.IrisInputStream;
import code.serialization.IrisOutputStream;
import code.serialization.IrisSerializable;
import code.serialization.SerializationHelper;
import code.simulator.agreement.Tuple;
import code.simulator.log.Log;

/**
 * @author princem
 *
 */
public class HashedVV extends AcceptVV implements Obj2Bytes, IrisSerializable{

  private LinkedList<Entry<NodeId, Hash>> hashes;
  private HashMap<NodeId, Entry<NodeId, Hash>> map;
  transient final boolean initialized; 
  transient Hash hash = null;
  
  /**
   * 
   */
  public HashedVV(){
    map = new HashMap<NodeId, Entry<NodeId, Hash>>();
    hashes = new LinkedList<Entry<NodeId, Hash>>();
    initialized = true;
  }

  /**
   * @param toCopyVV
   */
  public HashedVV(VV toCopyVV, LinkedList<Entry<NodeId, Hash>> hashes){
    super(toCopyVV);
    initialized = true;
    // TODO Auto-generated constructor stub
    assert toCopyVV.getSize() == hashes.size();
    map = new HashMap<NodeId, Entry<NodeId, Hash>>();
    this.hashes = new LinkedList<Entry<NodeId, Hash>>(); 
    this.hashes.addAll(hashes);
    for(Entry<NodeId, Hash> e: this.hashes){
      map.put(e.getKey(), e);
    }   
  }

  
  /**
   * @param toCopyVV
   */
  public HashedVV(VV toCopyVV, HashMap<NodeId, Hash> hashes){
    super(toCopyVV);
    initialized = true;
    TreeSet<Hash> sortedTreeSet = new TreeSet<Hash>(hashes.values());
    HashMap<Hash, NodeId> invHashes = new HashMap<Hash, NodeId>();
    for(Entry<NodeId, Hash> e: hashes.entrySet()){
      invHashes.put(e.getValue(), e.getKey());
    }
    assert toCopyVV.getSize() == hashes.size();
    assert hashes.keySet().containsAll(toCopyVV.getNodes());
    map = new HashMap<NodeId, Entry<NodeId, Hash>>();
    this.hashes = new LinkedList<Entry<NodeId, Hash>>(); 
    for(Hash h: sortedTreeSet){
      Entry<NodeId, Hash> e1 = new Tuple<NodeId, Hash>(invHashes.get(h), h);
      this.hashes.add(e1);
      map.put(invHashes.get(h), e1);
    }    
    if(!SangminConfig.useHash){
      hash = Hash.NullHash;
    }else{
      hash = new Hash(this);
    }
  }
  
  public HashedVV createMatchingHashedVV(HashMap<NodeId, Hash> hashMap){
    assert this.size() == hashMap.size();
    assert initialized; 
    LinkedList<Entry<NodeId, Hash>> newHashes = new LinkedList<Entry<NodeId, Hash>>();
    boolean diff = false;
    for(Entry<NodeId, Hash> e:hashes){
      assert hashMap.containsKey(e.getKey());
      if(e.getValue().equals(hashMap.get(e.getKey()))){
        newHashes.add(e);
      }else{
        diff = true;
        newHashes.add(new Tuple<NodeId, Hash>(e.getKey(), hashMap.get(e.getKey())));
      }
    }
    HashedVV ret = new HashedVV(this, newHashes);
    if(!diff){
      assert Log.getHash(this).equals(Log.getHash(ret)):Log.getHash(this)  + "\n" + Log.getHash(ret) + "\n"+ this.hashes + "\n" + ret.hashes + "\n"+Arrays.toString(obj2Bytes()) + "\n"+Arrays.toString(ret.obj2Bytes());;
    }
    return ret;
  }
  
  /**
   * Return the time stamp stored at the token
   **/ 
  public final Hash getHashByIteratorToken(Object iterToken)
  {
   // assert initialized: this.toLongString(); 
    NodeId nodeId = null;
    Hash hash = null;

    assert(iterToken instanceof NodeId);
    nodeId = (NodeId)iterToken;
    assert map.containsKey(nodeId): nodeId + " hashes: " + hashes;
    hash = this.map.get(nodeId).getValue();
    assert(hash != null);
    return hash;
  }

  /**
   * Return the time stamp entry in this VV for the given node
   **/ 
  public final Hash getHashByServer(NodeId nodeId)
  throws NoSuchEntryException
  {
    assert initialized; 
    if(!map.containsKey(nodeId) ){
      throw(new NoSuchEntryException());
    }
    Hash hash = map.get(nodeId).getValue();

    return hash;
  }

  /**
   * create a new AcceptVV by replacing all occurences of N1 by N2 
   * @param bm HashMap<N1, N2>
   * @return
   */
  @Override
  public HashedVV applyBranchMap(HashMap<NodeId, NodeId> bm){
    assert initialized; 
    LinkedList<Entry<NodeId, Hash>> newHashes = new LinkedList<Entry<NodeId, Hash>>();
    CounterVV cvv = new CounterVV();
    try{
      for(Entry<NodeId, Hash> e: hashes){
        NodeId n = e.getKey();
        if(bm.containsKey(n)){
          if(!cvv.containsNodeId(bm.get(n)) || (cvv.getStampByServer(bm.get(n)) < this.getStampByIteratorToken(n))){
            cvv.addMaxAS(new AcceptStamp(this.getStampByIteratorToken(n), bm.get(n)));
            newHashes.add(new Tuple<NodeId, Hash>(bm.get(n), e.getValue()));
          }
        }else{
          if(!cvv.containsNodeId(n) || (cvv.getStampByServer(n) < this.getStampByIteratorToken(n))){
            cvv.addMaxAS(new AcceptStamp(this.getStampByIteratorToken(n), n));
            newHashes.add(e);
          }
        }
      }
    }catch(NoSuchEntryException e){
      // TODO Auto-generated catch block
      e.printStackTrace();
      assert false;
    }
    return new HashedVV(cvv, newHashes);
  }

  public HashedVV applyPOMRemap(POMRemap remap){
    assert initialized; 
    LinkedList<Entry<NodeId, Hash>> newHashes = new LinkedList<Entry<NodeId, Hash>>();
    CounterVV cvv = new CounterVV();
    for(Entry<NodeId, Hash> e: hashes){
      NodeId n = e.getKey();
      if(n.equals(remap.getOrigNodeId()) && super.getStampByIteratorToken(n) >= remap.getTs()){
        assert !cvv.containsNodeId(remap.getNewNodeId());
        cvv.addMaxAS(new AcceptStamp(this.getStampByIteratorToken(n), remap.getNewNodeId()));
        newHashes.add(new Tuple<NodeId, Hash>(remap.getNewNodeId(), e.getValue()));
      }else{
        assert !cvv.containsNodeId(n);
        cvv.addMaxAS(new AcceptStamp(this.getStampByIteratorToken(n), n));
        newHashes.add(e);
      }
    }
    HashedVV ret = new HashedVV(cvv, newHashes);
    assert Arrays.equals(this.obj2Bytes(), ret.obj2Bytes()):"\n"+ this.hashes + "\n" + ret.hashes + "\n"+Arrays.toString(obj2Bytes()) + "\n"+Arrays.toString(ret.obj2Bytes());
    assert Log.getHash(this).equals(Log.getHash(ret));
    return ret;
  }

  
 
  public int compareTo(HashedVV hdvv){
    assert initialized; 
    int ret = super.compareTo(hdvv);
    if(ret != 0){
      return ret;
    }else{
      int i = 0;
      for(Entry<NodeId, Hash> h: hashes){
        ret = ((Tuple)h).compareTo(hdvv.hashes.get(i));
        if(ret != 0){
          return ret;
        }
      }
      return 0;
    }
  }


  public void writeExternal(ObjectOutput out)  throws IOException{
    writeExternal(out, false);
  }

  /** 
   *  Serialization -- optimization to reduce the pikles in the serialization  
   *                   and improve the performance -- removed from  
   *                   TaggedOutputStream to here 
   **/ 
  public void writeExternal(ObjectOutput out, boolean optimize)
  throws IOException{ 
    if(Node.enableSecurity){
      out.writeBoolean(optimize);
    }
    if(Node.enableSecurity && optimize){
      out.write(this.hash.getHashVal());
    }
    out.writeShort(this.getSize());
    try{
      for(Entry<NodeId, Hash> e: hashes) {
        if(SangminConfig.forkjoin && SangminConfig.securityLevel > SangminConfig.SIGNEDVV){
          ((BranchID)e.getKey()).writeExternal(out);
//          out.writeObject(e.getKey());
        } else {
          out.writeLong(e.getKey().getIDint());
        }
        out.writeLong(this.getStampByServer(e.getKey()));
        if(Node.enableSecurity && !optimize){
          out.write(e.getValue().getHashVal());
        }
      }
    }catch(NoSuchEntryException e){
      e.printStackTrace();
      assert false;
    }
  }

//  public void readExternal(ObjectInput in)throws IOException,
//  ClassNotFoundException{
//    this.readExternal(in, false);
//  }

  /** 
   *  Serialization --  optimization to reduce the pikles in the serialization  
   *                   and improve the performance --originated from  
   *                   TaggedInputStream 
   **/ 
  public void readExternal(ObjectInput in)
  throws IOException,
  ClassNotFoundException{
    HashMap<NodeId, Entry<NodeId, Hash>> _map = new HashMap<NodeId, Entry<NodeId, Hash>>();
    LinkedList<Entry<NodeId, Hash>> _hashes = new LinkedList<Entry<NodeId, Hash>>();
    boolean _initialized = true;
    boolean optimized = false;
    Hash _hash = null;
    if(Node.enableSecurity){
       optimized = in.readBoolean();
      _initialized = !optimized;
      if(optimized){
        byte[] _buf  = new byte[Hash.Length];
        in.readFully(_buf);
        _hash = new Hash(_buf, false);
      }
    }
    
    int avvSize = in.readShort();

    //AcceptStamp[] vvTime = new AcceptStamp[avvSize];
    Hashtable mystamps = new Hashtable();
    for(int i=0; i<avvSize; i++){
      NodeId nodeId;
      if(SangminConfig.forkjoin && SangminConfig.securityLevel > SangminConfig.SIGNEDVV){
        nodeId = new BranchID(-1);
        ((BranchID)nodeId).readExternal(in);
//        nodeId = (BranchID)in.readObject();
      } else {
        nodeId = new BranchID(in.readLong());
      }
      mystamps.put(nodeId,  new Long(in.readLong()));

      Hash hash = null;
      if(Node.enableSecurity && _initialized){
        byte [] hashVal  = new byte[Hash.Length];
        in.readFully(hashVal);
        hash = new Hash(hashVal, false);
      }else{
        hash = Hash.NullHash;
      }
      Tuple<NodeId, Hash> t= new Tuple<NodeId, Hash>(nodeId, hash);
      _hashes.add(t);
      _map.put(t.getKey(), t);
    }
    
    Field[] f = new Field[5];      
    try{

      f[0] = this.getClass().getDeclaredField("hashes");
      f[1] = AcceptVV.class.getDeclaredField("stamps");
      f[2] = this.getClass().getDeclaredField("map");
      f[3] = this.getClass().getDeclaredField("initialized");
      f[4] = this.getClass().getDeclaredField("hash");

    }catch(NoSuchFieldException ne){
      ne.printStackTrace();
      assert false:ne;
    }
    try{
      AccessibleObject.setAccessible(f, true);
    } catch (SecurityException se){
      assert false;
    }
    try{
      f[0].set(this, _hashes);
      f[1].set(this, mystamps);
      f[2].set(this, _map);
      f[3].setBoolean(this, _initialized);
      f[4].set(this, _hash);

    }catch(IllegalArgumentException ie){
      ie.printStackTrace();
      assert false;
    }catch(IllegalAccessException iae){
      assert false;
    }

    try{
      AccessibleObject.setAccessible(f, false);
    } catch (SecurityException se){
      assert false;
    }
    this.hash = this.getHash(); // if hash is already set, this instruction acts as a no-op, otherwise hash is computed
    assert hash != null;
  }

  /** 
   *  corrupts the hashedVV on which this method is invoked by replacing its hashes
   **/ 
  public boolean initialize(HashMap<NodeId, Hash> hashes2){
    assert !initialized; 
    HashMap<NodeId, Entry<NodeId, Hash>> _map = new HashMap<NodeId, Entry<NodeId, Hash>>();
    LinkedList<Entry<NodeId, Hash>> _hashes = new LinkedList<Entry<NodeId, Hash>>();

    boolean _initialized = true;
    assert(Node.enableSecurity);
    assert hash != null;

    assert this.size() == hashes2.size();
    for(Entry<NodeId, Hash> entry: this.hashes){
      if(!hashes2.containsKey(entry.getKey())){
        System.err.println(entry.getKey() + "is not present in local state..revernting to normal sync");
        return false;
      }
      Hash hash = hashes2.get(entry.getKey());
      Tuple<NodeId, Hash> t= new Tuple<NodeId, Hash>(entry.getKey(), hash);
      _hashes.add(t);
      _map.put(t.getKey(), t);
    }
    
    Field[] f = new Field[5];      
    try{

      f[0] = this.getClass().getDeclaredField("hashes");
      f[1] = AcceptVV.class.getDeclaredField("stamps");
      f[2] = this.getClass().getDeclaredField("map");
      f[3] = this.getClass().getDeclaredField("initialized");
      f[4] = this.getClass().getDeclaredField("hash");

    }catch(NoSuchFieldException ne){
      ne.printStackTrace();
      assert false:ne;
    }
    try{
      AccessibleObject.setAccessible(f, true);
    } catch (SecurityException se){
      assert false;
    }
    try{
      f[0].set(this, _hashes);
      f[1].set(this, super.stamps);
      f[2].set(this, _map);
      f[3].set(this, _initialized);
      f[4].set(this, hash);

    }catch(IllegalArgumentException ie){
      ie.printStackTrace();
      assert false;
    }catch(IllegalAccessException iae){
      assert false;
    }

    try{
      AccessibleObject.setAccessible(f, false);
    } catch (SecurityException se){
      assert false;
    }
    
    if(new Hash(this).equals(hash)){
      return true;
    }else{
      return false;
    }
  }

  

  public byte[] obj2Bytes(){
    assert initialized; 
    try{
      SerializationHelper serHelper = new SerializationHelper();
      serHelper.writeHDVV(this);
      serHelper.close();
      return serHelper.toBytes();
    }catch(Exception e){
      e.printStackTrace();
    }
    assert false;
    return null;
  }

  public static void testExt(){
    DependencyVV dvv = new DependencyVV();
    dvv.put(new BranchID(5), 3);
    String data = "data";
    byte[] bytes = new byte[Hash.Length];
    for(int i=0; i<Hash.Length; i++){
      bytes[i] = 'a';
    }

    Hash hash = new Hash(bytes, false);
    HashMap<NodeId, Hash> hashes = new HashMap<NodeId, Hash>();
    hashes.put(new BranchID(5), hash);
    HashedVV hdvv = new HashedVV(dvv, hashes);

    ByteArrayOutputStream bs;
    ObjectOutputStream oos;


    try{
      bs = new ByteArrayOutputStream();
      oos = new ObjectOutputStream(bs);
      System.out.println("Serialise it");
      hdvv.writeExternal(oos);

      oos.flush();
      //bs.toByteArray();
      System.out.println("read it" + bs.size());
      HashedVV hdvv1 = null;
      ObjectInputStream ois = new ObjectInputStream(new ByteArrayInputStream(bs.toByteArray()));
      try{
        hdvv1 = new HashedVV();
        hdvv1.readExternal(ois); 
      }catch(ClassNotFoundException e){
        e.printStackTrace();
      }

      if(hdvv.equals(hdvv1)){
        System.out.println("Success");
      } else {
        System.out.println("Fail : \n" + hdvv.toLongString() + "\n" + hdvv1.toLongString());

      }
      
      bs = new ByteArrayOutputStream();
      oos = new ObjectOutputStream(bs);
      hdvv.writeExternal(oos, true);
      oos.flush();
      System.out.println("Serialise it"+bs.size());
      ois = new ObjectInputStream(new ByteArrayInputStream(bs.toByteArray()));
      try{
        hdvv1 = new HashedVV();
        hdvv1.readExternal(ois); 
      }catch(ClassNotFoundException e){
        e.printStackTrace();
      }

      assert(hdvv.equals(hdvv1)):hdvv + " " + hdvv.getHashes() + "\n" + hdvv1 + " " + hdvv1.getHashes();
      System.out.println(hdvv1.isInitialized());
      assert !hdvv1.isInitialized():hdvv1.isInitialized();
      assert hdvv.isInitialized();
      HashMap<NodeId, Hash> map = new HashMap<NodeId, Hash>();
      for(Entry<NodeId, Hash> entry: hdvv.hashes){
        map.put(entry.getKey(), entry.getValue()); 
      }
      hdvv1.initialize(map);
      assert hdvv1.isInitialized();
      assert(hdvv.equals(hdvv1));

    }catch(IOException e){
      e.printStackTrace();
    }
  }
  
  public static void testStr(){
    DependencyVV dvv = new DependencyVV();
    dvv.put(new BranchID(5), 3);
    byte[] bytes = new byte[Hash.Length];
    for(int i=0; i<Hash.Length; i++){
      bytes[i] = 'a';
    }

    Hash hash = new Hash(bytes, false);
    HashMap<NodeId, Hash> hashes = new HashMap<NodeId, Hash>();
    hashes.put(new BranchID(5), hash);
    HashedVV hdvv = new HashedVV(dvv, hashes);

    ByteArrayOutputStream bs;
    IrisOutputStream oos;


    try{
      bs = new ByteArrayOutputStream();
      oos = new IrisOutputStream(bs);
      System.out.println("Serialise it ");
      hdvv.writeToStream(oos, false);

      oos.flush();
      //bs.toByteArray();
      System.out.println("read it" + bs.size() + "\n" + Arrays.toString(bs.toByteArray()));
      HashedVV hdvv1 = null;
      IrisInputStream ois = new IrisInputStream(new ByteArrayInputStream(bs.toByteArray()));
      hdvv1 = new HashedVV();
      hdvv1.readFromStream(ois, false); 


      if(hdvv.equals(hdvv1)){
        System.out.println("Success");
      } else {
        System.out.println("Fail : \n" + hdvv.toLongString() + "\n" + hdvv1.toLongString());

      }

      bs = new ByteArrayOutputStream();
      oos = new IrisOutputStream(bs);
      hdvv.writeToStream(oos, true);
      oos.flush();
      System.out.println("Serialise it"+bs.size());
      ois = new IrisInputStream(new ByteArrayInputStream(bs.toByteArray()));
      hdvv1 = new HashedVV();
      hdvv1.readFromStream(ois, true); 


      assert(hdvv.equals(hdvv1));
      System.out.println(hdvv1.isInitialized());
      assert !hdvv1.isInitialized():hdvv1.isInitialized();
      assert hdvv.isInitialized();
      HashMap<NodeId, Hash> map = new HashMap<NodeId, Hash>();
      for(Entry<NodeId, Hash> entry: hdvv.hashes){
        map.put(entry.getKey(), entry.getValue()); 
      }
      hdvv1.initialize(map);
      assert hdvv1.isInitialized();
      assert(hdvv.equals(hdvv1));

    }catch(IOException e){
      e.printStackTrace();
    }
  }

  public static void main(String[] args) {

    assert Node.enableSecurity;
    testStr();
    testExt();

  }


  public Collection<Entry<NodeId, Hash>> getHashes(){
    return hashes;
  }

  

  @Override
  public int hashCode(){
    final int prime = 31;
    int result = super.hashCode();
    result = prime * result + ((hashes == null) ? 0 : hashes.hashCode());
    return result;
  }

  @Override
  public boolean equals(Object obj){
    if(this == obj){
      return true;
    }
    if(!super.equals(obj)){
      return false;
    }
    if(!(obj instanceof HashedVV)){
      return false;
    }
    HashedVV other = (HashedVV) obj;
    if(hashes == null){
      if(other.hashes != null){
        return false;
      }
    }else if(!this.getHash().equals(other.getHash())){
      return false;
    }
    return true;
  }
  

  public String toLongString(){
    
    String s = super.toString();
    
    s+= ", Hashes = [";
    for(Entry<NodeId, Hash> e : hashes){
      s += "( " + e.getKey().toString() + ", ";
      s += e.getValue().toLongString() + ") ";
    }
    s+= " ]";
    
    for(Entry<NodeId, Entry<NodeId,Hash>> e :map.entrySet()){
      s += "[ " + e.getKey().toString() + ",";
      s += " [ " +  e.getValue().getKey().toString() +", "; 
      s += e.getValue().getValue().toLongString() + " ] ] ";
    }
    
    
    return s;
  }

  /**
   * @return the initialized
   */
  public boolean isInitialized(){
    return initialized;
  }

  public Hash getHash(){
    if(hash == null){
      hash = new Hash(this);
    }
    return hash;
  }
  public void readFromStream(IrisInputStream in, boolean optimize)
  throws IOException{
    readFromStream(in, optimize, 0);
  }
  
  public void readFromStream(IrisInputStream in, boolean optimize, long base)
      throws IOException{
    HashMap<NodeId, Entry<NodeId, Hash>> _map = new HashMap<NodeId, Entry<NodeId, Hash>>();
    LinkedList<Entry<NodeId, Hash>> _hashes = new LinkedList<Entry<NodeId, Hash>>();
    boolean _initialized = true;
    boolean optimized = false;
    if(SangminConfig.sanityCheck){
      byte b = in.readByte();
      assert b == (byte)0xAB: b + " " + (byte)0xAB;
    }
    Hash _hash = null;
    if(Node.enableSecurity){
       optimized = optimize;//in.readBoolean();
      _initialized = !optimized;
      if(optimized){
        byte[] _buf  = new byte[Hash.Length];
        in.readFully(_buf);
        _hash = new Hash(_buf, false);
      }
    }
    
    int avvSize = in.readShort();

    //AcceptStamp[] vvTime = new AcceptStamp[avvSize];
    Hashtable mystamps = new Hashtable();
    for(int i=0; i<avvSize; i++){
      NodeId nodeId;
      if(SangminConfig.forkjoin && SangminConfig.securityLevel > SangminConfig.SIGNEDVV){
        nodeId = new BranchID(-1);
        ((BranchID)nodeId).readFromStream(in, optimized);
//        nodeId = (BranchID)in.readObject();
      } else {
        nodeId = new BranchID(in.readLong());
      }
      long ts = in.readLong();
      ts = ((base!=0)?(base - ts):ts);
      mystamps.put(nodeId, ts);

      Hash hash = null;
      if(Node.enableSecurity && _initialized){
        byte [] hashVal  = new byte[Hash.Length];
        in.readFully(hashVal);
        hash = new Hash(hashVal, false);
      }else{
        hash = Hash.NullHash;
      }
      Tuple<NodeId, Hash> t= new Tuple<NodeId, Hash>(nodeId, hash);
      _hashes.add(t);
      _map.put(t.getKey(), t);
    }
    
    this.setFields(_map, _hashes, _hash, mystamps, _initialized);
    
    this.hash = this.getHash(); // if hash is already set, this instruction acts as a no-op, otherwise hash is computed
    assert hash != null;
    if(SangminConfig.sanityCheck){
      byte b = in.readByte();
      assert b == (byte)0xAC: b + " " + (byte)0xAC;
    }
  }

  private void setFields(HashMap<NodeId, Entry<NodeId, Hash>> _map, LinkedList<Entry<NodeId, Hash>> _hashes, Hash _hash, Hashtable mystamps, boolean _initialized){
    Field[] f = new Field[5];      
    try{

      f[0] = this.getClass().getDeclaredField("hashes");
      f[1] = AcceptVV.class.getDeclaredField("stamps");
      f[2] = this.getClass().getDeclaredField("map");
      f[3] = this.getClass().getDeclaredField("initialized");
      f[4] = this.getClass().getDeclaredField("hash");

    }catch(NoSuchFieldException ne){
      ne.printStackTrace();
      assert false:ne;
    }
    try{
      AccessibleObject.setAccessible(f, true);
    } catch (SecurityException se){
      assert false;
    }
    try{
      f[0].set(this, _hashes);
      f[1].set(this, mystamps);
      f[2].set(this, _map);
      f[3].setBoolean(this, _initialized);
      f[4].set(this, _hash);

    }catch(IllegalArgumentException ie){
      ie.printStackTrace();
      assert false;
    }catch(IllegalAccessException iae){
      assert false;
    }

    try{
      AccessibleObject.setAccessible(f, false);
    } catch (SecurityException se){
      assert false;
    }
    
  }

  public void writeToStream(IrisOutputStream out, boolean optimize) throws IOException {
    writeToStream(out, optimize, 0);
  }

  public void writeToStream(IrisOutputStream out, boolean optimize, long base)
      throws IOException{
    if(SangminConfig.sanityCheck){
      out.writeByte(0xAB);
    }
    if(Node.enableSecurity){
//      out.writeBoolean(optimize);
    }
    if(Node.enableSecurity && optimize){
      out.write(this.hash.getHashVal());
    }
    out.writeShort(this.getSize());
    try{
      for(Entry<NodeId, Hash> e: hashes) {
        if(SangminConfig.forkjoin && SangminConfig.securityLevel > SangminConfig.SIGNEDVV){
          ((BranchID)e.getKey()).writeToStream(out, optimize);
//          out.writeObject(e.getKey());
        } else {
          out.writeLong(e.getKey().getIDint());
        }
        long ts = this.getStampByServer(e.getKey());
        ts = ((base!=0)?(base - ts):ts);
        out.writeLong(ts);
        if(Node.enableSecurity && !optimize){
          out.write(e.getValue().getHashVal());
        }
      }
    }catch(NoSuchEntryException e){
      e.printStackTrace();
      assert false;
    }    
    if(SangminConfig.sanityCheck){
      out.writeByte(0xAC);
    }
  }

 
}
