package code.simulator;

import java.io.ByteArrayOutputStream;
import java.io.IOException;
import java.io.ObjectInputStream;
import java.io.ObjectOutputStream;
import java.net.*;

import code.branchDetecting.*;
import code.security.SangminConfig;
import code.serialization.IrisInputStream;
import code.serialization.IrisObjectInputStream;
import code.serialization.IrisObjectOutputStream;
import code.serialization.IrisOutputStream;
import code.simulator.agreement.Tuple;
import code.AcceptVV;
import code.Config;
import code.CounterVV;
import code.Env;
import code.NodeId;
import code.VV;
import code.security.SangminConfig;
import code.security.SangminConfig.DebugLevel;


import java.util.Arrays;
import java.util.HashMap;
import java.util.LinkedList;

public class IncomingConnectionHandler{

  // receiver here means receiver of the syncPacket--hence this node
  // as an additional step in the protocol, the receiver should send its id to the sender
  // hashmap mapping receiver to mapping from sender to socket
  HashMap<Long, Socket> socketMap; 
  HashMap<Long, Tuple<IrisInputStream, IrisOutputStream>> streamMap; 
  public static final int MAGIC = 0x0BCDEF55;
  
  HashMap<Long, CounterVV> lastSentVV;

  private final IrisNode myNode;
  
  public IncomingConnectionHandler(IrisNode myNode){
    this.myNode = myNode;
    lastSentVV = new HashMap<Long, CounterVV>();
    socketMap = new HashMap<Long, Socket>();
    streamMap = new HashMap<Long, Tuple<IrisInputStream, IrisOutputStream>>();
  }
  
  synchronized private Socket createSocket(long sender) throws UnknownHostException, IOException{
    String dns = Config.getDNS(new NodeId(sender));
    int portInval = Config.getPortInval(new NodeId(sender));
    Socket s = new Socket(dns, portInval);
    s.setSendBufferSize(1024*1024);
    s.setReceiveBufferSize(1024*1024*10);
//    IrisObjectOutputStream iro = new IrisObjectOutputStream(s.getOutputStream());
    IrisOutputStream iro = new IrisOutputStream(s.getOutputStream());
    //sending the id
    System.out.println("receiver " + myNode.getBranchID().getIDint() + " sending init-command to " + sender);
    iro.writeInt(MAGIC);
    iro.writeLong(myNode.getBranchID().getIDint());
    iro.flush();
//    iro.reset();
    return s;
  }
  
  synchronized private Socket getSocketForSyncPacket(long sender) throws UnknownHostException, IOException{
    // sender : node that sends a syncPacket

    if(!socketMap.containsKey(sender)){
      Socket s = createSocket(sender);
      socketMap.put(sender, s);
      assert s.isConnected();
      assert !s.isClosed();
      return s;
    } else {
      if(socketMap.containsKey(sender)){
        Socket s = socketMap.get(sender);
        if(s.isConnected()){
          return s;        
        } else {
          assert false: "Attempts to connect to sender " + sender + " on port " + Config.getPortInval(new NodeId(sender)) + "failed";
        }
      }      
      Socket s = createSocket(sender);
      assert s.isConnected();
      assert !s.isClosed();
      socketMap.put(sender, s);
      return s;                  
    }

  }
  
  synchronized private Tuple<IrisInputStream, IrisOutputStream> getStreamsForSyncPacket(long sender) throws UnknownHostException, IOException{
    // sender : node that sends a syncPacket

    if(!streamMap.containsKey(sender)){
      Socket s = createSocket(sender);
      socketMap.put(sender, s);
      Tuple<IrisInputStream, IrisOutputStream> streamTuple = 
        new Tuple<IrisInputStream, IrisOutputStream>(new IrisInputStream(s.getInputStream(), false), new IrisOutputStream(s.getOutputStream()));
      streamMap.put(sender, streamTuple);
      if(!s.isConnected()){
        System.err.println( "Attempts to connect to sender " + sender + " on port " + Config.getPortInval(new NodeId(sender)) + "failed");
        throw new IOException("couldn't connect to sender " + sender);
      }
      if(s.isClosed()){
        System.err.println("Attempts to connect to sender " + sender + " on port " + Config.getPortInval(new NodeId(sender)) + "failed");
        throw new IOException("couldn't connect to sender " + sender);
      }
      return streamTuple;
    } else {
      assert(streamMap.containsKey(sender));
      return streamMap.get(sender);

    }

  }
  
  public void shutdown(){

    for( Socket s : socketMap.values()){
      try{
        s.close();
      }catch(IOException e){
        e.printStackTrace();
      }
    }
  }


  //shutdown = true;
    //    if (sock != null){
    //      try{
    //        sock.close();// this call will make any thread currently blocked in accept()
    //        // throw a SocketException
    //      }catch(IOException ioe){
    //        System.err.println(" InvalStreamReceiver shutting down ignore Exception: " 
    //            +ioe.toString());
    //      }
    //      sock = null;
    //    }
  
  
  public synchronized VV getLastSentVV(long sender){
    if(!lastSentVV.containsKey(sender)){
      lastSentVV.put(sender, new CounterVV());
    }
    return lastSentVV.get(sender);
  }
  
  public synchronized VV advanceLastSentVV(long sender, VV vv){
    if(!lastSentVV.containsKey(sender)){
      lastSentVV.put(sender, new CounterVV());
    }
    lastSentVV.get(sender).addMaxVV(vv);
    return lastSentVV.get(sender);
  }
  
  public synchronized SyncStatus connect(BranchID sender, SyncRequest sr, boolean optimized) throws Exception{
    // start connection to n
    // send the request 
    // and notify me when the response arrive
    try{
	if(SangminConfig.useByteArraysForIO){
	    return connectOld(sender, sr, optimized);
	}
      assert !SangminConfig.useByteArraysForIO;
      Tuple<IrisInputStream, IrisOutputStream> streamTuple = getStreamsForSyncPacket(sender.getIDint());
      IrisInputStream ois = streamTuple.getKey();
      IrisOutputStream ios = streamTuple.getValue();
      long startSend = ios.getBytes();
      long startRecv = ois.getBytesRead();
      VV lastVV = this.getLastSentVV(sender.getIDint());
      CounterVV newLastVV = new CounterVV(sr.getStartVV());
      //      Env.printDebug(myNode.getBranchID().getIDint() + " newLastVV " + newLastVV + " lastVV " + lastVV);
      if(lastVV != null && optimized){
        sr.setIncrementalVV(lastVV);
      }else{
        assert !sr.isIncrementalVV();
      }
//      sr.writeToStream(ios, false);
      this.sendPkt(ios, sr);
      ios.flush();
      
      Env.logWrite(myNode.getBranchID().getIDint() + " sent " + (ios.getBytes()-startSend) + " as syncRequest to " + sender.getIDint());
      //System.out.println(myNode.getBranchID().getIDint() + " sent " + (ios.getBytes()-startSend) + " as syncRequest to " + sender.getIDint());
//      System.out.println("wrote sync req" + Arrays.toString(bs.toByteArray()));
      if(SangminConfig.debugLevel==DebugLevel.Verbose)System.out.println(myNode.getBranchID().getIDint() + "wrote sync req " + sr);
      ios.flush();
      SyncPacket pkt = readPkt(ois);
      
      while(pkt.getType() == SyncPacket.NoCachedState){
	  Env.logEvent("received NoCachedState message from " + sender.getIDint());
	  long lastB =ios.getBytes(); 
	  sr.resetIncrementalVV(lastVV);
	  sr.writeToStream(ios, false);
	  ios.flush();
	  Env.logWrite(myNode.getBranchID().getIDint() + " sent " + (ios.getBytes()-lastB) + 
		       " as syncRequest to " + sender.getIDint());
	  pkt = readPkt(ois);
	  assert pkt.getType() != SyncPacket.NoCachedState;
      }
      pkt.getBodySize();
      long sentBytes = ios.getBytes()-startSend;
      long receivedBytes = ois.getBytesRead()-startRecv;
      long bodySize = pkt.getBodySize();
      if(SangminConfig.debugLevel==DebugLevel.Verbose)Env.printDebug(myNode.getBranchID().getIDint() + "received syncpkt " + pkt);

      Env.logEvent(System.currentTimeMillis() + " GOSSIP myId: " + myNode.getBranchID() + " senderId: " + sender.getIDint() +
          " sentBytes: " + sentBytes + " receivedBytes: " + receivedBytes + " bodyBytes: " + bodySize + " metadataSize: " + (receivedBytes-bodySize));
      
      this.advanceLastSentVV(sender.getIDint(), newLastVV);
      
      Tuple<SyncStatus, AcceptVV> res=  myNode.applySyncPacketAcceptVV(pkt);
      if(SangminConfig.debugLevel==DebugLevel.Verbose)Env.printDebug(myNode.getBranchID().getIDint() + "applied syncpkt " + res);

      if(res.getKey().status == SyncStatus.SyncSuccessful){
        newLastVV.advanceTimestamps(res.getValue());
        this.advanceLastSentVV(sender.getIDint(), newLastVV);
        res.getKey().setBodies(pkt.getBodies());
      }else{
        this.lastSentVV.remove(sender.getIDint()); // clear the cache
      }
      return res.getKey();
    }catch(UnknownHostException e){
      Socket sock = socketMap.get(sender);    
      if(sock != null && !sock.isClosed()){
        try{
          sock.close();
        }catch(Exception e1){
          System.out.println(System.currentTimeMillis() + " received exception " + e1);
        }
      this.socketMap.remove(sender.getIDint());
      this.streamMap.remove(sender.getIDint());
      }
    }catch(IOException e){
      Socket sock = socketMap.get(sender);    
      if(sock != null && !sock.isClosed()){
        try{
          sock.close();
        }catch(Exception e1){
          System.out.println(System.currentTimeMillis() + " received exception " + e1);
        }
        this.socketMap.remove(sender.getIDint());

        this.streamMap.remove(sender.getIDint());
      }
    }
    
    return new SyncStatus(SyncStatus.NetworkError);
  }
  
  public synchronized SyncStatus connectOld(BranchID sender, SyncRequest sr, boolean optimized) throws Exception{
    // start connection to n
    // send the request 
    // and notify me when the response arrive
    Socket sock = null;
    try{
      sock = getSocketForSyncPacket(sender.getIDint());
      assert sock.isConnected();
      
      IrisOutputStream ios;
      IrisInputStream ois = new IrisInputStream(sock.getInputStream());

      ByteArrayOutputStream bs = null;
      if(SangminConfig.useByteArraysForIO){
        bs = new ByteArrayOutputStream();
        ios = new IrisOutputStream(bs);
      }else{
        ios = new IrisOutputStream(sock.getOutputStream());
      }
      VV lastVV = this.getLastSentVV(sender.getIDint());
      CounterVV newLastVV = new CounterVV(sr.getStartVV());
      if(lastVV != null){
        sr.setIncrementalVV(lastVV);
      }else{
        assert !sr.isIncrementalVV();
      }
//      sr.writeToStream(ios, false);
      this.sendPkt(ios, sr);
      ios.flush();
      
      Env.logWrite(myNode.getBranchID().getIDint() + " sent " + ios.getBytes() + " as syncRequest to " + sender.getIDint());
      System.out.println(myNode.getBranchID().getIDint() + " sent " + ios.getBytes() + " as syncRequest to " + sender.getIDint());
//      System.out.println("wrote sync req" + Arrays.toString(bs.toByteArray()));
//      System.out.println("wrote sync req" + sr);
      if(SangminConfig.useByteArraysForIO){
        sock.getOutputStream().write(bs.toByteArray());
      }
      ios.flush();
      sock.getOutputStream().flush();

      SyncPacket pkt = readPkt(ois);
      System.out.println("receved resposnse :" + pkt);
      
      while(pkt.getType() == SyncPacket.NoCachedState){
        long lastB =ios.getBytes(); 
        sr.resetIncrementalVV(lastVV);
        sr.writeToStream(ios, false);
        ios.flush();
        Env.logWrite(myNode.getBranchID().getIDint() + " sent " + (ios.getBytes()-lastB) + 
            " as syncRequest to " + sender.getIDint());
        pkt = readPkt(ois);
        assert pkt.getType() != SyncPacket.NoCachedState;
      }
      long sentBytes = ios.getBytes();
      long receivedBytes = ois.getBytesRead();
      
      Env.logEvent(System.currentTimeMillis() + " GOSSIP myId: " + myNode.getBranchID() + " senderId: " + sender.getIDint() +
          " sentBytes: " + sentBytes + " receivedBytes: " + receivedBytes);
      
      this.advanceLastSentVV(sender.getIDint(), newLastVV);
      
      Tuple<SyncStatus, AcceptVV> res=  myNode.applySyncPacketAcceptVV(pkt);
      if(res.getKey().status == SyncStatus.SyncSuccessful){
        newLastVV.advanceTimestamps(res.getValue());
        this.advanceLastSentVV(sender.getIDint(), newLastVV);
        res.getKey().setBodies(pkt.getBodies());
      }else{
        this.lastSentVV.remove(sender.getIDint()); // clear the cache
      }
      return res.getKey();
    }catch(UnknownHostException e){
      e.printStackTrace();
      if(sock != null && !sock.isClosed()){
        try{
          sock.close();
        }catch(Exception e1){
          System.out.println(System.currentTimeMillis() + " received exception " + e1);
          e1.printStackTrace();
        }
      this.socketMap.remove(sender.getIDint());
      }
    }catch(IOException e){
      e.printStackTrace();
      if(sock != null && !sock.isClosed()){
        try{
          sock.close();
        }catch(Exception e1){
          System.out.println(System.currentTimeMillis() + " received exception " + e1);
          e1.printStackTrace();
        }
        this.socketMap.remove(sender.getIDint());
      }
    }
    
    return new SyncStatus(SyncStatus.NetworkError);
  }
  
  private void sendPkt(IrisOutputStream ios, SyncRequest sr) throws IOException{
    if(SangminConfig.useByteArraysForIO){
      ByteArrayOutputStream bs = new ByteArrayOutputStream(500);
      IrisOutputStream is = new IrisOutputStream(bs);
      sr.writeToStream(is, false);
      is.flush();
      byte[] buf= bs.toByteArray();
      assert buf.length == is.getBytes();
      ios.writeShort(buf.length);
      ios.write(buf);
    }
    else{
      sr.writeToStream(ios, false);
    }
  }
  
  private SyncPacket readPkt(IrisInputStream ois) throws IOException{
    SyncPacket pkt = new SyncPacket();
    pkt.readFromStream(ois, false);
    if(pkt.isHasBodies()){
      pkt.readBodyFromStream(ois, false);
    }
//    Env.logWrite(myNode.getBranchID().getIDint() + " received " + pkt);
    return pkt;
  }
  
//  public SyncStatus connectExt() throws Exception{
//    // start connection to n
//    // send the request 
//    // and notify me when the response arrive
////    String dns = Config.getDNS(sender);
////    int portInval = Config.getPortInval(sender);
//    try{
////      System.out.println("tryign to connect from " + myNode.getBranchID() + " to " + dns + " at port " + portInval + " of node " + sender + " w/ oid " + sr.getObjId());
//      
//      
//      sock = getSocketForSyncPacket((int)sender.getIDint(), (int) myNode.getBranchID().getIDint());
//      assert sock.isConnected();
//      
//      ByteArrayOutputStream bos = new ByteArrayOutputStream();
//      ObjectOutputStream ios = new IrisObjectOutputStream(bos);
//      sr.writeExternal(ios);
//      ios.flush();
//      Env.logWrite(this.myNode.getBranchID().getIDint() + " sent " + bos.size() + " as syncRequest to " + sender.getIDint());
//
//      sock.getOutputStream().write(bos.toByteArray());
//      sock.getOutputStream().flush();
//      ios.reset();
//      ios.close();
//      bos.reset();
//      bos.close();
//            
////      System.out.println("Sent a syncRequest to " + sender + " with " + sr.getStartVV() + " oid " + sr.getObjId());
//      try{
//
//        IrisObjectInputStream ois = new IrisObjectInputStream(sock.getInputStream());
//        
//        SyncPacket pkt = new SyncPacket();
////        pkt.readExternal(ois);
//        pkt.readExternalCorePacket(ois);
//        pkt.readExternalBodies(ois);
////        
////        System.out.println("Received SyncPacket from " + sender + " my id: " +myNode.getBranchID() + 
////            "\n " + pkt);
//
//        return myNode.applySyncPacket(pkt);
//      }catch(ClassNotFoundException e){
//        e.printStackTrace();
//      }
//      
//      
//    }catch(UnknownHostException e){
//      e.printStackTrace();
//    }catch(IOException e){
//      e.printStackTrace();
//    }
//    
//    return new SyncStatus(SyncStatus.NetworkError);
//  }
}
