
package code.simulator;

import java.io.ByteArrayInputStream;
import java.io.ByteArrayOutputStream;
import java.io.IOException;
import java.io.Serializable;
import java.security.PrivateKey;
import java.security.PublicKey;
import java.security.Signature;
import java.util.Arrays;
import java.util.Random;
import java.security.*;

import code.Env;
import code.AcceptStamp;
import code.Config;
import code.NodeId;
import code.ObjId;
import code.ObjInvalTarget;
import code.ResultStats;
import code.SummaryHash;
import code.security.Crypto;
import code.security.DataHash;
import code.security.SangminConfig;
import code.security.SecurePreciseInv;
import code.security.ahs.DependencyVV;
import code.serialization.IrisInputStream;
import code.serialization.IrisOutputStream;
import code.serialization.SerializationHelper;


public class IrisDataObject extends IrisObject{

  final private byte[] bytes;
  private Hash hash;
  static byte[] buffer = new byte[Hash.Length];
  static final public int otherSigMetadataLen = 2 /*signer*/+ 2 /*signature length*/ + 
                                                Crypto.KeyLen/8;
  static byte[] signature = null;
  static{

    for(int i = 0; i < Hash.Length; i++){
      buffer[i] = (byte)i;
    } 
  }

  public IrisDataObject(byte[] bytes){
    this.bytes = bytes;    
    hash = getHash();
  }

  public IrisDataObject(byte[] bytes, Hash hash){
    this.bytes = bytes;
//    assert Arrays.equals(this.getHash().getHashVal(), hash.getHashVal());
    this.hash = hash;
  }

  public static IrisDataObject createIrisDataObject(byte[] bytes, long signer) throws IOException{
    long s = System.currentTimeMillis();
    IrisDataObject ido;
    if(SangminConfig.hackedSignature || SangminConfig.hackedHash){
      ByteArrayOutputStream bs = new ByteArrayOutputStream();
      IrisOutputStream io = new IrisOutputStream(bs);
      io.write(bytes);            
      byte[] hash = null;
      try{
        MessageDigest md = MessageDigest.getInstance(Hash.ALGORITHM);
        hash = md.digest(bytes);
        if(SangminConfig.hackedHash && !SangminConfig.hackedSignature){
          io.write(hash);
        }
        if(SangminConfig.hackedSignature){
          io.unoptimizedWriteShort((short)signer);
          byte[] buf = sign(hash, signer);
          io.unoptimizedWriteShort(buf.length); //signature
          if(buf.length == Crypto.KeyLen/8){
            io.write(buf);
          }else{
            byte[] paddedBuf = new byte[Crypto.KeyLen/8];
            System.arraycopy(buf, 0, paddedBuf, 0, buf.length);
            io.write(paddedBuf);
          }
        }   
      } catch (Exception e){
        System.out.println(e.toString());
        e.printStackTrace();
        System.exit(1);
        assert false;
      }
      io.flush();
      ido = new IrisDataObject(bs.toByteArray());
    }else{
      ido = new IrisDataObject(bytes);
    }

    long e = System.currentTimeMillis();
    Env.logWrite("creation took " + (e-s));
    return ido;
    
  }
  
  /**
   * create an IrisDataObject using paddedBytes; do all the work (signature and hashes) on bytes
   * @param bytes
   * @param paddedBytes
   * @param signer
   * @return
   * @throws IOException
   */
  public static IrisDataObject createIrisDataObject(byte[] bytes, byte[] paddedBytes, long signer) throws IOException{
    long s = System.currentTimeMillis();
    IrisDataObject ido;
    if(SangminConfig.hackedSignature || SangminConfig.hackedHash){
      ByteArrayOutputStream bs = new ByteArrayOutputStream();
      IrisOutputStream io = new IrisOutputStream(bs);
      byte[] hash = null;
      try{
        MessageDigest md = MessageDigest.getInstance(Hash.ALGORITHM);
        hash = md.digest(bytes);
        if(SangminConfig.hackedHash && !SangminConfig.hackedSignature){
          io.write(hash);
        }
        if(SangminConfig.hackedSignature){
          io.unoptimizedWriteShort((short)signer);
          byte[] buf = sign(hash, signer);
          io.unoptimizedWriteShort(buf.length); //signature
          if(buf.length == Crypto.KeyLen/8){
            io.write(buf);
          }else{
            byte[] paddedBuf = new byte[Crypto.KeyLen/8];
            System.arraycopy(buf, 0, paddedBuf, 0, buf.length);
            io.write(paddedBuf);
          }
        }   
      } catch (Exception e){
        System.out.println(e.toString());
        e.printStackTrace();
        System.exit(1);
        assert false;
      }
      io.flush();
      ido = new IrisDataObject(paddedBytes);
    }else{
      ido = new IrisDataObject(bytes);
    }

    long e = System.currentTimeMillis();
    Env.logWrite("creation took " + (e-s));
    return ido;
    
  }

  /**
   * verify if the padded data bytes have all the relevant security fields. Only relevant when 
   * hacked security features are used.
   * @return
   * @throws IOException
   */
  public boolean verify() throws IOException{
    return verify(bytes);
  }
  
  public boolean verifyold() throws IOException{
    long s = System.currentTimeMillis();
    boolean ret = true;
    if(SangminConfig.hackedHash || SangminConfig.hackedSignature){
      ByteArrayInputStream bs = new ByteArrayInputStream(this.bytes);
      IrisInputStream iis = new IrisInputStream(bs);
      try{
        if(SangminConfig.hackedHash && !SangminConfig.hackedSignature){
          byte[] remBytes = new byte[bytes.length-Hash.Length];
          iis.readFully(remBytes);
          byte[] hashBytes = new byte[Hash.Length];
          iis.readFully(hashBytes);
          MessageDigest md = MessageDigest.getInstance(Hash.ALGORITHM);
          byte[] computedHash = md.digest(remBytes);
          if(Arrays.equals(hashBytes, computedHash)){
            ret = true;
          }else{
	      System.out.println("FAIL\n" + Arrays.toString(hashBytes) + "\n computed hash " + Arrays.toString(computedHash));
            assert false: "this shouldn't occur: attachedHash" + 
            Arrays.toString(hashBytes) + "\n computed hash " + Arrays.toString(computedHash);
            ret = false; 
          }
        }
        
        if(SangminConfig.hackedSignature){
          byte[] remBytes = new byte[bytes.length-this.otherSigMetadataLen];
          iis.readFully(remBytes);
          long signer = iis.unoptimizedReadShort();
          short len = iis.unoptimizedReadShort();
          byte[] signature = new byte[len];
          if(len == Crypto.KeyLen/8){
            iis.readFully(signature);
          }else{
            byte[] paddedSig = new byte[Crypto.KeyLen/8];
            iis.readFully(paddedSig);
            System.arraycopy(paddedSig, 0, signature, 0, len);
          }
          MessageDigest md = MessageDigest.getInstance(Hash.ALGORITHM);
          byte[] hashBytes = md.digest(remBytes);
          ret = this.verifySig(hashBytes, signature, signer);
        }

      } catch (Exception e){
        System.out.println(e.toString());
        e.printStackTrace();
        System.exit(1);
        assert false;
      }
    }
    long e = System.currentTimeMillis();
    Env.logWrite("verification took " + (e-s));
    return ret;
  }
    
    public static boolean verify(byte[] paddedBytes) throws IOException{
      long s = System.currentTimeMillis();
      boolean ret = true;
      if(SangminConfig.hackedHash || SangminConfig.hackedSignature){
        try{
          if(SangminConfig.hackedHash && !SangminConfig.hackedSignature){
            int len = paddedBytes.length-Hash.Length;
            byte[] hashBytes = new byte[Hash.Length];
            System.arraycopy(paddedBytes, len, hashBytes, 0, Hash.Length);
            MessageDigest md = MessageDigest.getInstance(Hash.ALGORITHM);
            md.update(paddedBytes, 0, (paddedBytes.length-Hash.Length));
            byte[] computedHash = md.digest(); 
            if(Arrays.equals(hashBytes, computedHash)){
              ret = true;
            }else{
                System.out.println("FAIL\n" + Arrays.toString(hashBytes) + "\n computed hash " + Arrays.toString(computedHash));
              assert false: "this shouldn't occur: attachedHash" + 
              Arrays.toString(hashBytes) + "\n computed hash " + Arrays.toString(computedHash);
              ret = false; 
            }
          }
          
          if(SangminConfig.hackedSignature){
            int datalen = paddedBytes.length-otherSigMetadataLen;
            byte[] remainingMetaData = new byte[otherSigMetadataLen];
            System.arraycopy(paddedBytes, datalen, remainingMetaData, 0, otherSigMetadataLen);
            ByteArrayInputStream bs = new ByteArrayInputStream(remainingMetaData);
            IrisInputStream iis = new IrisInputStream(bs);
            long signer = iis.unoptimizedReadShort();
            short len = iis.unoptimizedReadShort();
            byte[] signature = new byte[len];
            if(len == Crypto.KeyLen/8){
              iis.readFully(signature);
            }else{
              byte[] paddedSig = new byte[Crypto.KeyLen/8];
              iis.readFully(paddedSig);
              System.arraycopy(paddedSig, 0, signature, 0, len);
            }
            MessageDigest md = MessageDigest.getInstance(Hash.ALGORITHM);
            md.update(paddedBytes, 0, datalen);
            byte[] hashBytes = md.digest(); 
            ret = verifySig(hashBytes, signature, signer);
          }

        } catch (Exception e){
          System.out.println(e.toString());
          e.printStackTrace();
          System.exit(1);
          assert false;
        }
      }
      long e = System.currentTimeMillis();
      Env.logWrite("verification took " + (e-s));
      return ret;
  }

  public IrisDataObject(Object o){
    this.bytes = SerializationHelper.serialize(o);
    hash = new Hash(bytes, true);
    //hash = null;
  }


  public static byte[] sign(byte[] buf, long signer){
    PrivateKey privKey = Config.privateKeys.get(signer);

    //  if(SangminConfig.disableSignatureVerfication){
    //  byte[] b =new byte[1];
    //  b[0] = (byte)0;
    //  return b;
    //  }else{


    try{
      //Signature sig = Signature.getInstance(privKey.getAlgorithm());
      //      long start = System.currentTimeMillis();
      Signature sig = Crypto.getSignature();    
      sig.initSign(privKey);           
      sig.update(buf);
      byte[] ret = sig.sign();
      Crypto.returnSignature(sig);
      //      long end = System.currentTimeMillis();
      //      System.out.println(" signing took " + (end-start));

      return ret;
    }catch(Exception e){
      System.out.println(e.toString());
      System.out.println(e.getStackTrace());
      System.exit(-1);
    }
    return null;
  }

  //public boolean verify(PublicKey pubKey){
  public static boolean verifySig(byte[]buf, byte[] signature, long signer){
    PublicKey pubKey = Config.publicKeys.get(signer);
    try{
      //Signature sig = Signature.getInstance(pubKey.getAlgorithm());
      Signature sig = Crypto.getSignature();    
      sig.initVerify(pubKey);

      //      SerializationHelper sh = this.obj2BytesExcludingSignature();
      //      sh.close();
      sig.update(buf);
      assert signature != null;
      boolean ret = sig.verify(signature);
      Crypto.returnSignature(sig);
      return ret;
    }catch(Exception e){
      System.out.println(e.toString());
      System.out.println(e.getStackTrace());
      System.exit(-1);
    }

    return false;

  }

  //  public static byte[] sign(byte[] buf, long signer){
  //    PrivateKey privKey = Config.privateKeys.get(signer);
  //
  //    //  if(SangminConfig.disableSignatureVerfication){
  //    //  byte[] b =new byte[1];
  //    //  b[0] = (byte)0;
  //    //  return b;
  //    //  }else{
  //    try{
  //      //Signature sig = Signature.getInstance(privKey.getAlgorithm());
  //      Signature sig = Crypto.getSignature();    
  //      sig.initSign(privKey);           
  //      sig.update(buf);
  //      byte[] ret = sig.sign();
  //      Crypto.returnSignature(sig);
  //      return ret;
  //    }catch(Exception e){
  //      System.out.println(e.toString());
  //      System.out.println(e.getStackTrace());
  //      System.exit(-1);
  //    }
  //    return null;
  //  }
  //
  //  //public boolean verify(PublicKey pubKey){
  //  public static boolean verifySig(byte[]buf, byte[] signature, long signer){
  //    PublicKey pubKey = Config.publicKeys.get(signer);
  //    try{
  //      //Signature sig = Signature.getInstance(pubKey.getAlgorithm());
  //      Signature sig = Crypto.getSignature();    
  //      sig.initVerify(pubKey);
  //
  //      //      SerializationHelper sh = this.obj2BytesExcludingSignature();
  //      //      sh.close();
  //      sig.update(buf);
  //      assert signature != null;
  //      boolean ret = sig.verify(signature);
  //      Crypto.returnSignature(sig);
  //      return ret;
  //    }catch(Exception e){
  //      System.out.println(e.toString());
  //      System.out.println(e.getStackTrace());
  //      System.exit(-1);
  //    }
  //
  //    return false;
  //
  //  }


  @Override
  public Hash getHash(){
    if(hash == null){
      if(SangminConfig.useHash){
        hash = new Hash(bytes, true);
      }else{
        byte[] small = new byte[Hash.Length];
        for(int i = 0; i < Hash.Length; i++){
          small[i] = Hash.NullHash.getHashVal()[i];
        }
        small[0] = this.bytes[0];
        small[1] = bytes[1];
        small[2] = bytes[2];
        hash = new Hash(small, false);
      }
    }
    return hash;
  }

  @Override
  public int hashCode(){
    //assert hash != null;
    final int prime = 31;
    int result = 1;
    result = prime * result + ((hash == null) ? 0 : hash.hashCode());
    return result;
  }

  @Override
  public boolean equals(Object obj){
    //assert hash != null;
    if(this == obj){
      return true;
    }
    if(obj == null){
      return false;
    }
    if(!(obj instanceof IrisDataObject)){
      return false;
    }
    IrisDataObject other = (IrisDataObject) obj;
    if(hash == null){
      if(other.hash != null){
        return false;
      }
    }else if(!hash.equals(other.hash)){
      return false;
    }
    return true;
  }

  @Override
  public String toString(){
    return "IrisDataObject [hash=" + hash +"]";
  }

  public Object getObject(){
    return SerializationHelper.deserilize(bytes);
  }

  public byte[] getUnsafeBytes(){
    return bytes;
  }

  public static void main(String[] args) throws IOException{
    int datasize = 3;
    try{
      Config.readKeys();
    } catch (Exception e){
      System.out.println("exception in reading keys");
      e.printStackTrace();
      System.exit(-1);
    }
    System.out.println("Key reading completed.");
    PrivateKey privKey = (PrivateKey)Config.privateKeys.get(new Long(1));
    PublicKey pubKey = (PublicKey)Config.publicKeys.get(new Long(1));
    byte[] buffer = new byte[datasize];
    for(int i = 0; i < datasize; i++){
      buffer[i] = (byte)i;
    }
    Hash h = new Hash(buffer, true);
    System.out.println(h + "\n" + Arrays.toString(buffer) + "\n" + Arrays.toString(sign(h.getHashVal(), 1)));
    SangminConfig.hackedHash =true;
    for(int i = 0; i < 2; i++){
      SangminConfig.hackedSignature = (i%2)!=0;
      System.out.println("hackedSignature " + SangminConfig.hackedSignature);
    long t = System.currentTimeMillis();
    IrisDataObject ido = null;
    for(int j = 0; j < 1000; j++){
      ido = IrisDataObject.createIrisDataObject(buffer, 1);
    }
    long e = System.currentTimeMillis();
    //    assert(ido.verify());
    //    long f = System.currentTimeMillis();
    System.out.println("creating object took." + (e-t) + "\n");

    t = System.currentTimeMillis();

    for(int j = 0; j < 1000; j++){
      ido.verify();
    }
    e = System.currentTimeMillis();
    assert(ido.verify()): "\n" + Arrays.toString(ido.getUnsafeBytes());
    
    IrisDataObject test = new IrisDataObject(new String("test").getBytes());
    assert test.verify(ido.getUnsafeBytes());
    //    long f = System.currentTimeMillis();
    System.out.println("verification completed." + (e-t) + "\n");

    }
  }


}
