/**
 * 
 */
package code.simulator;

import java.io.ByteArrayOutputStream;
import java.io.ObjectOutputStream;
import java.util.*;

import code.Config;
import code.CounterVV;
import code.AcceptStamp;
import code.AcceptVV;
import code.Env;
import code.LatencyWatcher;
import code.NodeId;
import code.ObjId;
import code.SubscriptionSet;
import code.VVIterator;
import code.lasr.db.DbTransaction;
import code.lasr.db.TxnFailedException;
import code.security.SangminConfig;
import code.security.ahs.DependencyVV;
import code.serialization.IrisObjectOutputStream;
import code.serialization.IrisOutputStream;
import code.serialization.SerializationHelper;
import code.simulator.agreement.AgreementParams;
import code.simulator.agreement.Tuple;
import code.simulator.branchManager.SimpleTree;
import code.simulator.branchManager.SimpleTreeNode;
import code.simulator.checkpoint.Checkpoint;
import code.simulator.log.Log;
import code.simulator.log.LogStatus;
import code.simulator.notification.Notification;
import code.simulator.notification.NotificationQueue;
import code.simulator.notification.NotificationThread;
import code.security.SangminConfig.DebugLevel;
import code.PreciseInv;
import code.branchDetecting.*;
import code.simulator.protocolFilters.AgreementGCProtocol;
import code.simulator.protocolFilters.BWSyncFilter;
import code.simulator.protocolFilters.DiscardedWriteFilter;
import code.simulator.protocolFilters.EvictionCertificate;
import code.simulator.protocolFilters.GCCertificate;
import code.simulator.protocolFilters.GCProtocol;
import code.simulator.protocolFilters.POMFilter;
import code.simulator.protocolFilters.SyncFilter;
import code.simulator.store.*;

/**
 * @author princem
 * @desc An IrisNode knows about 
 * (1) a faulty node list -with a corresponding POM,
 *      - direct communication with these nodes is not accepted
 *       
 * (2) an evicted node list- with a corresponding removal certificate, 
 *      - these nodes are excluded from dvv generated in future
 *      - subsequent checkpoints place the certificate in place of writes and various branches corresponding to this node
 *     
 * (3) a checkpoint and corresponding opaque certificate that it transfers along with the checkpoint
 * 
 * (4) a list of filters (for sync/updates/writes/conflicts/new branch (when a branch is observed for the first time)): writes imply write to data store, update means an update was received) 
 * and associated callbacks that are called when these events occur
 *      - sync filter returns a boolean indicating whether the sync should be accepted or not
 *      - update filter notifies the application when an update to specifies namespace occurs 
 *        PS: can be invoked multiple times for the same update when writes are rolledback and rolled forward
 *      - writes: same as update except that it is invoked on modification to the object space
 *      - conflict: notifies the application on conflicts in the specified name space and allows the application to return some value of object
 *      - new branch: notifies when a new branch found after phase 1 of sync (pre-sync exchange) and allows application to decide whether to accept the sync or not
 *      
 * (5) object state, that appears in form of serializable objects, for various objIds
 */
public class IrisNode extends Node{

  //  public boolean syncDone = false;
  public static boolean printStats = false;

  public static final boolean bestEffort = false;

  public static boolean measureTime = false;

  protected final Listener listener;

  private HashMap<SubscriptionSet, LinkedList<NamespaceWatch>> watches;

  private NotificationQueue queue;
  private NotificationThread notificationThread;
  /**
   * determines whether for simulation runs (local), we use sync2 based on the SyncPacket or simple sync
   */
  public static final boolean useSync2 = SangminConfig.useSync2;
  public static int numNodes = 1000; 
  public static String globalCertificatesString = "/iris/certificates/";
  public static HashMap<Long, ObjId> certificatesFiles;
  public static boolean useAgreementCheckpoint = true;
  private IncomingConnectionHandler ich;

  public static final long timeout = 6*1000; // 6 seconds
  LinkedList<SyncFilter> syncFilters;
  //  LinkedList<NamespaceFilter> namespaceFilters;
  LinkedList<POMFilter> pomFilters;
  LinkedList<DiscardedWriteFilter> discardedWriteFilters;

  LinkedList<Certificate> epochAdvancingCertificates;

  private final AgreementGCProtocol gcProtocol;
  private final AgreementParams params;
  long storageCost = 0; //storage cost in bytes
  long logSize = 0;

  /**
   * @param branchId
   */
  public IrisNode(BranchID branchId){
    this(branchId, branchId);
  }

  public IrisNode(BranchID branchId, BranchID ConfigIdx){
    super(branchId);

    syncFilters = new LinkedList<SyncFilter>();
    //    namespaceFilters = new LinkedList<NamespaceFilter>();
    discardedWriteFilters = new LinkedList<DiscardedWriteFilter>();
    pomFilters = new LinkedList<POMFilter>();
    watches = new HashMap<SubscriptionSet, LinkedList<NamespaceWatch>>();
    epochAdvancingCertificates = new LinkedList<Certificate>();
    certificatesFiles = new HashMap<Long, ObjId>();
    queue = new NotificationQueue();
    //    System.out.println(this + " has queue "  +queue.hashCode());
    notificationThread = new NotificationThread(queue);

    int numN = Config.getNumLogicalNodes() > 0 ? Config.getNumLogicalNodes():Config.getNumNodes();
    this.numNodes = numN;
    int numF = (numN-1)/5;
    LinkedList<Long> allNodes = new LinkedList<Long>();
    Enumeration<NodeId> enumerator = Config.getKnownNodeIds();
    while(enumerator.hasMoreElements()){
      NodeId n = enumerator.nextElement();
      allNodes.add(n.getIDint());
    }
    params = new AgreementParams(numN, numF, branchId.getIDint(), allNodes);
    store.setIgnoreSet(SubscriptionSet.makeSubscriptionSet(AgreementGCProtocol.gcProposalDir+"*"));
    if(IrisNode.useAgreementCheckpoint){
      gcProtocol = new AgreementGCProtocol(params, this.getEpoch(), timeout, this);
      this.addNamespaceWatch(gcProtocol, SubscriptionSet.makeSubscriptionSet(AgreementGCProtocol.gcProposalDir+"*"));
    }else{
      gcProtocol = null;
    }

    for(long i = 0; i < IrisNode.numNodes; i++){
      certificatesFiles.put(i, new ObjId(globalCertificatesString+i));
    }

    if(!SangminConfig.useSimulator){
      listener = new Listener(this, ConfigIdx);
      ich = new IncomingConnectionHandler(this);
      listener.start();
    }else{
      listener = null;
    }
  }

  synchronized public boolean isEvicted(Long node){
    return this.evictedNodes.contains(node);
  }

  public IrisNode(BranchID branchId, String localConfigFile){
    this(branchId);
    Config.readConfig(localConfigFile);
  }

  //PRINCEM--OLD INTERFACE
  //  synchronized public boolean removeBody(ObjId o, AcceptStamp as){
  //    return store.removeBody(o, as);
  //  }
  //  
  //  synchronized public boolean addBody(ObjId o, IrisDataObject body){
  //    return store.addBody(o, body);
  //  }
  //  
  //  synchronized public IrisObject getBody(ObjId o, Hash hash){
  //    return store.getBody(o, hash);
  //  }

  public void removeBody(Hash hash){
    DbTransaction txn = null;
    if(SangminConfig.usePersistentStore){
      txn = pe.getDB().newTransaction();
    }
    if(bodyStore.containsBody(txn, hash)){
      bodyStore.removeBody(txn, hash);
    }
    if(SangminConfig.usePersistentStore){

      try{
        txn.commit();
      }catch(TxnFailedException e){
        System.err.println("checkpoint txn failed");
        e.printStackTrace();
      }
    }
  }

  public void addBody(IrisDataObject body){
    DbTransaction txn = null; 
    if(SangminConfig.usePersistentStore){
      txn = pe.getDB().newTransaction();
    }
    bodyStore.addBody(txn, body);
    if(SangminConfig.usePersistentStore){

      try{
        txn.commit();
      }catch(TxnFailedException e){
        System.err.println("checkpoint txn failed");
        e.printStackTrace();
      }
    }
  }

  public IrisDataObject getBody(Hash hash){
    DbTransaction txn = null; 
    if(SangminConfig.usePersistentStore){
      txn = pe.getDB().newTransaction();
    }
    IrisDataObject object = bodyStore.getBody(txn, hash);
    if(SangminConfig.usePersistentStore){

      try{
        txn.commit();
      }catch(TxnFailedException e){
        System.err.println("checkpoint txn failed");
        e.printStackTrace();
      }
    }
    return object;
  }

  public boolean containsBody(Hash hash){
    DbTransaction txn = null;
    if(SangminConfig.usePersistentStore){

      txn = pe.getDB().newTransaction();
    }
    boolean ret = bodyStore.containsBody(txn, hash);
    if(SangminConfig.usePersistentStore){

      try{
        txn.commit();
      }catch(TxnFailedException e){
        System.err.println("checkpoint txn failed");
        e.printStackTrace();
      }
    }
    return ret;
  }

  synchronized public void addSyncFilter(SyncFilter sf){
    if(sf instanceof GCProtocol && this.useAgreementCheckpoint){
      return;
    }
    syncFilters.add(sf);
  }

  synchronized public void addDiscardeWriteFilter(DiscardedWriteFilter dwf){
    discardedWriteFilters.add(dwf);
  }

  synchronized public void addNamespaceWatch(NamespaceWatch nw, SubscriptionSet ss){
    if(!watches.containsKey(ss)){
      watches.put(ss, new LinkedList<NamespaceWatch>());
    }
    this.watches.get(ss).add(nw);
  }

  synchronized public void removeNamespaceWatch(NamespaceWatch nw, SubscriptionSet ss){
    this.watches.get(nw).remove(nw);
  }

  synchronized public void addPOMFilter(POMFilter pf){
    pomFilters.add(pf);
  }

  synchronized public void removeSyncFilter(SyncFilter sf){
    syncFilters.remove(sf);
  }

  synchronized public HashedVV getMaxVV(Collection<HashedVV> vvs) throws Exception{
    CounterVV cvv = new CounterVV();
    for(HashedVV vv: vvs){
      cvv.addMaxVV(this.remapHashedDependencyVV(vv));
    }
    return this.getHashedDependencyVV(logs.getMinimalVV(cvv));
  }

  protected void addBodies(SyncPacket sp, SyncRequest sr){
    if(!SangminConfig.separateBodies || (sr.getBodyReq() == SyncRequest.NoBodies && sr.getObjId() == null)){
      return;
    }
    LinkedList<IrisDataObject> bodies = new LinkedList<IrisDataObject>();
    if(sp.getType() == SyncPacket.SenderSameEpoch){
      AcceptVV commitCut = null;
      if((sr.getBodyReq()&1) == SyncRequest.UncommittedLastBodies){
        commitCut = getCommitCut(); // send only the recent uncommitted bodies
      }else{
        assert sr.getBodyReq() == SyncRequest.NoBodies;
        commitCut = this.getCurrentVV();
      }

      Iterator<SimPreciseInv> iter = sp.getWrites().iterator();
      DbTransaction txn = null;
      if(SangminConfig.usePersistentStore){
        txn = pe.getDB().newTransaction();
      }
      while(iter.hasNext()){
        SimPreciseInv spi = iter.next();
        if(!commitCut.includes(spi.getAcceptStamp())){ 
          if(store.getBody(spi.getObjId(), spi.getData().getHash()) != null || 
              (sr.getBodyReq()&1) == SyncRequest.UncommittedLastBodies){
            IrisDataObject body = bodyStore.getBody(txn, spi.getData().getHash());
            if(body != null){
              bodies.add(body);
            }
            
          }
        }
      }
      
      //add bodies for the objId
      if(sr.getObjId() != null){
        LinkedList<IrisObject> objBodies = this.read(sr.getObjId());
        for(IrisObject io: objBodies){
          assert io instanceof IrisDataObject;
          if(!bodies.contains(io)){
            bodies.add((IrisDataObject)io);
          }
        }
      }
      if(SangminConfig.usePersistentStore){
        try{
          txn.commit();
        }catch(TxnFailedException e){
          System.err.println("checkpoint txn failed");
          e.printStackTrace();
        }
      }

    }else if(sp.getType() == SyncPacket.SenderHigherEpoch){
      //      GCCertificate gcCert = (GCCertificate)sp.getCertificates().getFirst();
      //      Checkpoint chkPt = gcCert.getCheckpoint();
      //      Iterator<ObjId> iter = chkPt.objectStore.iterator();
      //      DbTransaction txn = null; 
      //      if(SangminConfig.usePersistentStore){
      //        pe.getDB().newTransaction();
      //      }
      //
      //      //check objectStore
      //      while(iter.hasNext()){
      //        ObjId o = iter.next();
      //        if((sr.getBodyReq()&2) == SyncRequest.CheckpointBodies || o.equals(sr.getObjId())){
      //          Collection<StoreEntry> ste = chkPt.objectStore.get(o);
      //          for(StoreEntry se: ste){
      //            Hash hash = se.getData().getHash();
      //            if(bodyStore.containsBody(txn, hash)){
      //              bodies.add(bodyStore.getBody(txn, hash));
      //            }
      //          }
      //        }
      //      }
      //
      //      //check unverifiablewrites
      //      for(SimPreciseInv spi: chkPt.unverifiableWrites){
      //        if((sr.getBodyReq()&2) == SyncRequest.CheckpointBodies || spi.getObjId().equals(sr.getBodyReq())){
      //          if(store.getBody(spi.getObjId(), spi.getData().getHash()) != null && 
      //              bodyStore.containsBody(txn, spi.getData().getHash())){
      //            //          iter.remove();
      //            bodies.add(bodyStore.getBody(txn, spi.getData().getHash()));
      //          }
      //        }
      //      }
      //      if(SangminConfig.usePersistentStore){
      //
      //
      //        try{
      //          txn.commit();
      //        }catch(TxnFailedException e){
      //          System.err.println("checkpoint txn failed");
      //          e.printStackTrace();
      //        }
      //      }
    }
    sp.setBodies(bodies);
    if(sr.getObjId() != null && SangminConfig.separateBodies && sp.getBodies().size() == 0){
	if(SangminConfig.debugLevel==DebugLevel.Verbose)
	    System.out.println("SP: " + sp + "\nSR: " +sr);
     /*     assert false: "Currently sync with objId is only invoked on a read failure and therefore these \n" +
          		"conditions should never be true---the fact that fast read failed implies that the sender must have \n" +
     		"newer or older updates; older updates are only possible when we failover; newer updates \n" +
     		"imply that these conditions should never be true";  */
    }
  }

  public synchronized Collection<SimPreciseInv> commitCutAdvanced(AcceptVV oldCommitCut, AcceptVV newCommitCut){
    TreeSet<SimPreciseInv> invals = this.findNewWritesIncludingLastWrite(oldCommitCut, newCommitCut);
    Iterator<SimPreciseInv> iter = invals.iterator();
    while(iter.hasNext()){
      SimPreciseInv spi = iter.next();
      if(oldCommitCut.includes(spi.getAcceptStamp())){
        iter.remove();
      }
    }
    return invals;
  }

  public synchronized AcceptVV getCommitCut(){
    //query commit filter (if there is one)..otherwise return ()
    return new AcceptVV();
  }

  synchronized public Tuple<SyncPacket, AcceptVV> getSyncPacketAcceptVV(SyncRequest sr){
    return new Tuple<SyncPacket, AcceptVV> (this.getSyncPacket(sr), this.getCurrentVV());
  }

  //princem: Two types of sync requests possible now--sync with bodies requested only for uncommitted objects
  // and syncs with bodies requested for all objects
  public SyncPacket getSyncPacket(SyncRequest sr){
    SyncPacket sp  = null;
    // the function ignores that 
    synchronized(this){ // slighly messy code to avoid locking during IO
      if(sr.isOptimized() || sr.getEpoch() == this.getEpoch()){

        if(sr.getObjId() != null){
          System.out.println("sync with objId");
        }
        //apply POM
        if(!sr.isOptimized()){
          this.notifyNewForks(sr.getPoms());
        }
        //      if(faultyNodes.contains(sr.getSender()) || evictedNodes.contains(sr.getSender())){
        //        return new SyncPacket(SyncPacket.SyncDenied, this.getEpoch(), this.getBranchID());
        //      }
        sp = new SyncPacket(SyncPacket.SenderSameEpoch, this.getEpoch(), this.getBranchID(), sr.isOptimized());
        HashSet<ProofOfMisbehavior> myPOMs = new HashSet<ProofOfMisbehavior>();
        //TreeSet<SimPreciseInv> lastWrites = this.findNewWritesIncludingLastWrite(sr.startVV, sr.startVV);
        //TreeSet<SimPreciseInv> lastWrites = this.logs.findNewWritesIncludingLastWrite(lastVV);
        if(!sr.isOptimized()){
          AcceptVV lastVV = sr.getStartVV().cloneMinVVOmit(this.getCurrentVV());
          TreeSet<SimPreciseInv> lastWrites = this.logs.findLastWritesIncludingLastWrite(lastVV);
	  //  if(SangminConfig.debugLevel==DebugLevel.Verbose)System.out.println("sr" + sr +"\ncvv:"+this.getCurrentVV()+"\nlastWrites:"+lastWrites);
          sp.setLastWrites(lastWrites);
          myPOMs.addAll(this.pomList);
          myPOMs.removeAll(sr.getPoms());
        }

        TreeSet<SimPreciseInv> newWrites = this.findNewWrites(sr.getStartVV());
        sp.setWritesAndPOM(newWrites, myPOMs);

        if(this.omission_fault){
          sp.omitWrites();
        }

      }else if(sr.getEpoch() < this.getEpoch()){
        sp = new SyncPacket(SyncPacket.SenderHigherEpoch, this.getEpoch(), this.getBranchID(), sr.isOptimized());
        LinkedList<Certificate> certificates = this.getNewerEpochCertificate(sr.getEpoch());
        if(this.useAgreementCheckpoint){
          GCCertificate cert = null;
          for(Certificate c: certificates){
            if(cert == null || (cert.getEpoch() < c.getEpoch())){
              cert = (GCCertificate)c;
            }
          }
          try{
            Checkpoint chkPt = this.generateNewCheckpoint(this.getMinimalVV(this.remapHashedDependencyVV(cert.getOmitVV())), cert.getEpoch(), this.getMinimalVV(this.remapHashedDependencyVV(cert.getEndVV())));
            certificates.clear();
            certificates.add(new GCCertificate(chkPt));
          }catch(Exception e){
            e.printStackTrace();
            assert false;
          }
        }
        sp.setCertificates(certificates);
      }else{
        return new SyncPacket(SyncPacket.SenderLowerEpoch, this.getEpoch(), this.getBranchID(), sr.isOptimized());
      }
    }
    this.addBodies(sp, sr);
    return sp;
  }
  
  public SyncStatus pseudoSync(SimPreciseInv spi, IrisDataObject ido) throws Exception{
    return pseudoSync( spi,  ido, false);
  }
  
  public SyncStatus pseudoSync(SimPreciseInv spi, IrisDataObject ido, boolean track) throws Exception{
    SyncPacket sp = generatePseudoSyncPkt(spi);
    LinkedList<IrisDataObject> bodies = new LinkedList<IrisDataObject>();
    if(ido != null){
      bodies.add(ido);
    }
    sp.setBodies(bodies);
    // benign race condition here---some one may concurrently add 
    // a node to evictedNode but it will be detected later
    // checking it here eliminates the lock acquistion most of the time
    if(evictedNodes.contains(spi.getNodeId().getIDint()) || faultyNodes.contains(spi.getNodeId().getIDint())){
      return new SyncStatus(SyncStatus.SenderFaulty);
    }
    else{
      if(this.getEpoch() != spi.getEpoch()){
        return new SyncStatus(SyncStatus.EpochChanged);
      }
      long start1 = System.nanoTime();
      SyncStatus ss = this.applySyncPacket(sp, track);
      long end1 = System.nanoTime();
      if(SangminConfig.fineGrainedTracking)System.out.println(track + " applySyncPacket took " + ((double)(end1-start1))/1000000);
      return ss;
    }

  }

  public static SyncPacket generatePseudoSyncPkt(SimPreciseInv spi){
    SyncPacket sp = new SyncPacket(SyncPacket.SenderSameEpoch, spi.getEpoch(), (BranchID)spi.getNodeId(), true);
    TreeSet<SimPreciseInv> writes = new TreeSet<SimPreciseInv>();
    writes.add(spi);
    HashSet<ProofOfMisbehavior> poms = new HashSet<ProofOfMisbehavior>();
    sp.setWritesAndPOM(writes, poms);
    sp.setLastWrites(new TreeSet<SimPreciseInv>());
    return sp;
  }

  /**
   * apply the packet and return the new VV
   * @param pkt
   * @return
   * @throws Exception
   */
  public Tuple<SyncStatus, AcceptVV> applySyncPacketAcceptVV(SyncPacket pkt) throws Exception{
    return new Tuple<SyncStatus, AcceptVV>(this.applySyncPacket(pkt), this.getCurrentVV());
  }

  public SyncStatus applySyncPacket(SyncPacket pkt) throws Exception{
    return this.applySyncPacket(pkt, false);
  }
  
  public SyncStatus applySyncPacket(SyncPacket pkt, boolean track) throws Exception{
    try{
      /*
      if(pkt.getBodies() != null){
        for(IrisDataObject ido: pkt.getBodies()){
	    if(SangminConfig.hackedSignature){
            boolean res = ido.verify();
	    }
        }
      }
       */
      //System.out.println("applying SyncPacket :" + pkt );
      //
      // DURABLE LOGGING CODE
      if(SangminConfig.usePersistentStore && (!pkt.isOptimized() || pkt.getWrites().size() > 0 || pkt.isHasBodies())){
        long start = System.nanoTime();
        DbTransaction txn = pe.getDB().newTransaction();
        //      if(SangminConfig.usePersistentStore){

        // the second condition (now commented out) prevents logging of 
        // information on reads at 
        // clients in Depot
        if(pkt.getBodies() != null 
            //            && SangminConfig.BodyConfig == SyncRequest.UncommittedLastBodies
        ){
          for(IrisDataObject ido: pkt.getBodies()){
            assert ido != null;
            this.bodyStore.addBody(txn, ido);
          }
        }

        if(!pkt.isOptimized() || pkt.getWrites().size() > 0){
          logPacket(txn, pkt);
        }

        try{
          txn.commit();
        }catch(TxnFailedException e){
          System.err.println("checkpoint txn failed");
          e.printStackTrace();
        }
        long end = System.nanoTime();
        if(SangminConfig.fineGrainedTracking)System.out.println(track + " disk write on sync took: " + (((double)(end-start))/1000000));
      }else{
        if(pkt.getBodies() != null){
          for(IrisDataObject ido: pkt.getBodies()){
            this.bodyStore.addBody(null, ido);
          }
        }
      }
      //DURABILITY CODE ENDS
      synchronized(this){
        //int start_logsize = getLogSize();
        //AcceptVV start_cvv = getCurrentVV();
        if(pkt.getType() == SyncPacket.SenderHigherEpoch){
          if(pkt.getEpoch() <= this.getEpoch()){
            System.err.println("CHANGE IN EPOCH DUE TO ASYNCHRONY SINCE LAST REQ WAS SENT..\n" + pkt + "\n"+this.getEpoch() +"\nRETURNING" );
            return new SyncStatus(SyncStatus.EpochChanged);
          }
          //        if(shouldFail) assert pkt.getEpoch() > this.getEpoch();
          if(this.applyCertificates(pkt.getCertificates())){
            return new SyncStatus(SyncStatus.ReceivedNewCertificate);
          }else{
            return new SyncStatus(SyncStatus.CertFilterReject);
          }
        }else if(pkt.getType() == SyncPacket.SenderSameEpoch){
          if(!pkt.isOptimized()){

            if(pkt.getEpoch() != this.getEpoch()){
              System.err.println("CHANGE IN EPOCH DUE TO ASYNCHRONY SINCE LAST REQ WAS SENT..\n" + pkt + "\n"+this.getEpoch() +"\nRETURNING" );
              return new SyncStatus(SyncStatus.EpochChanged);
            }

            if(shouldFail) assert pkt.getEpoch() == this.getEpoch();
            this.notifyNewForks(pkt.getPoms());
          }
          //for( SimPreciseInv spi : pkt.getWrites()){
          //  if(spi.getObjId().getPath().startsWith(AgreementGCProtocol.gcProposalDir)){
          //    System.out.println("Receiving " + spi);
          //  }
          //}  
          // first try and sync the lastWrites
          LinkedList<SimPreciseInv> lastWritesSet = new LinkedList<SimPreciseInv>();
          if(SangminConfig.useSimulator){
            for(SimPreciseInv spi: pkt.getLastWrites()){

              // we are simply creating clones to ensur that different nodes have different copies of writes so that 
              // they don't overwrite each other's changes 
              SimPreciseInv sspi = spi.cloneWithNewNodeId(spi.getNodeId());
              sspi.getFlags().clear();
              lastWritesSet.add(sspi);
            }
          }else{
            lastWritesSet.addAll(pkt.getLastWrites());
          }

          //       now the regular writes
          if(SangminConfig.useSimulator){
            //        spiSet = new TreeSet<SimPreciseInv>();
            for(SimPreciseInv spi: pkt.getWrites()){

              // we are simply creating clones to ensur that different nodes have different copies of writes so that 
              // they don't overwrite each other's changes 
              SimPreciseInv sspi = spi.cloneWithNewNodeId(spi.getNodeId());
              sspi.getFlags().clear();
              lastWritesSet.add(sspi);
            }
          }else{
            lastWritesSet.addAll(pkt.getWrites());
          }

          // then the remaining writes
          BranchID b = null;
          if(!pkt.isOptimized()){
            b = pkt.getSender();
          }
          long start1 = System.nanoTime();
          SyncStatus ss = this.logSync(b, lastWritesSet);
          long end1 = System.nanoTime();
          if(SangminConfig.fineGrainedTracking)System.out.println(track + " logSync took " + ((double)(end1-start1))/1000000);
          if(ss.status == SyncStatus.SyncSuccessful){
            int cur_logsize = getLogSize();
            //if(start_logsize + pkt.getWrites().size() != cur_logsize){
            //  System.out.println("WRONG SYNC : syncpkt = " + pkt + "\nstart CVV = " + start_cvv + "\nCVV = " + getCurrentVV() + "\nstart log size = " + start_logsize+ "\ncur log size = " + cur_logsize);
            //}
            System.out.println("sync with " + pkt.getWrites().size() + " finished with status " + ss + " , total log size = " + cur_logsize);
          } else {
	      System.out.println("sync with " + pkt.getWrites().size() + " finished with status " + ss + "prevDVV: " + this.getCurrentVV() + "sync packet" + pkt);
          }
          dumpLogs();
          return ss;
        }else if(pkt.getType() == SyncPacket.SyncDenied){
          return new SyncStatus(SyncStatus.SenderFaulty);
        }else{
          if(pkt.getEpoch() >= this.getEpoch()){
            System.err.println("CHANGE IN EPOCH DUE TO ASYNCHRONY SINCE LAST REQ WAS SENT..\n" + pkt + "\n"+this.getEpoch() +"\nRETURNING" );
            return new SyncStatus(SyncStatus.EpochChanged);
          }

          //        if(shouldFail) assert pkt.getEpoch() < this.getEpoch();
          // do nothing
          return new SyncStatus(SyncStatus.SenderOldEpoch);
        }
      }
    }finally{
      this.printStorageCost("sync", null);

    }

  }

  public void dumpLogs(){
    return;
    /*
    Log l = this.getLogTest();
    int count = 0;
    int s = 0;
    for(NodeId n : l.keySet()){
      s = l.get(n).size();
      count += s;
      System.out.println(System.currentTimeMillis() + " log size for node " + n  +  " : " + s);
    }
    System.out.println(System.currentTimeMillis() + " total log size for node " + this.getBranchID()  +  " : " + count);
    */
  }

  public int getLogSize(){
    Log l = this.getLogTest();
    int count = 0;
    int s = 0;
    for(NodeId n : l.keySet()){
      s = l.get(n).size();
      count += s;
    }
    return count;
  }
  
  public SyncStatus sync2(BranchID sender, AcceptVV avv, boolean optimized) throws Exception{
    return sync2(sender, avv, null, optimized);
  }

  public SyncStatus sync2(BranchID sender, AcceptVV avv, ObjId o, boolean optimized) throws Exception{
    SyncRequest sr; 
    synchronized(this){
      if(evictedNodes.contains(sender.getIDint()) || faultyNodes.contains(sender.getIDint())){
        return new SyncStatus(SyncStatus.SenderFaulty);
      }
      //      sr = new SyncRequest(avv, this.getEpoch(), this.pomList, this.getBranchID().getIDint(), SangminConfig.BodyConfig, o);
      sr = new SyncRequest(avv, this.getEpoch(), this.pomList, this.getBranchID().getIDint(),
          Config.getBodyConfig(new NodeId(this.getBranchID().getIDint())), o, optimized);
      //      System.out.println("BODY REQUESTED : " + (Config.getBodyConfig(new NodeId(this.getBranchID().getIDint()))==SangminConfig.BodyConfig));
    }
    SyncStatus status = ich.connect(sender, sr, optimized);
    return status;
  }

  synchronized public TreeSet<Long> getFaultyNodes(){
    return this.faultyNodes;
  }

  @Override
  public void close(){
    if(listener != null){
      listener.shutdown();
    }
    if(ich != null){
      ich.shutdown();
    }
    super.close();
  }

  /**
   * does the modification required in the local state on receiving a new certificate
   * @param c
   * @throws Exception 
   */
  synchronized public boolean applyNewLocalCertificate(Certificate c) throws Exception{
    if(this.addCertificate(c)){
      for(SyncFilter sf: syncFilters){
        sf.notifyEpochSync(this, c);
      }
      return true;
    }else{
      return false;
    }
  }


  /**
   * does the modification required in the local state on receiving a new certificate
   * @param c
   */
  synchronized public boolean addCertificate(Certificate c) throws Exception{
    if(getEpoch() > c.getEpoch()){
      if(shouldFail) assert false:" stale certificate";
      return false;
    }else if (getEpoch() == c.getEpoch()){
      return false;
    }
    if(shouldFail) assert c.getEpoch() == (this.getEpoch()+1);
    if(!bestEffort){
      // ensure that we are not missing any certificate since last checkpoint
      this.epochAdvancingCertificates.add(c);
    }
    if(c instanceof GCCertificate){
      GCCertificate gcCert = (GCCertificate)c;
      this.applyCheckpoint(gcCert.getCheckpoint());
    }else if(c instanceof EvictionCertificate){
      applyEvictionCertificate((EvictionCertificate)c);
      logs.addEvictedNodes(((EvictionCertificate) c).getNodeId());
    }else{
      if(shouldFail) assert false;
      return false;
    }
    if(!bestEffort){
      //      this.write(IrisNode.certificatesFiles.get(this.myBranchId.getIDint()), new IrisDataObject(new LinkedList<Certificate>(this.epochAdvancingCertificates)));
    }
    return true;
  }

  /**
   * apply eviction certificate and advance 
   * @param ec
   */
  synchronized protected void applyEvictionCertificate(EvictionCertificate ec){
    //if(myBranchId.equals(ec.nodeId)){
    if(ec.getNodeId().equals(myBranchId)){
      // I'm a faulty node!!
      System.out.println("Not applying Eviction Certificate for myself-myID: "
          + myBranchId + ", EC ID : " + ec.getNodeId());
      return;
    }
    System.out.println("Eviction Certificate Applied at " + myBranchId );
    this.addEvictedNode(ec.getNodeId());
    // trigger the eviction on local state
    HashMap<BranchID,LinkedList<PreciseInv>> unacceptableWrites = new HashMap<BranchID,LinkedList<PreciseInv>>();

    NodeId bid = ec.getNodeId();
    if(ec.getNodeId() instanceof BranchID){
      if(shouldFail) assert bid.equals(((BranchID)bid).getBaseId());
    }

    VVIterator vvi = getCurrentVV().getIterator();
    while(vvi.hasMoreElements()){
      BranchID n = (BranchID)vvi.getNext();
      if(n.contains(ec.getNodeId())){
        unacceptableWrites.put(n, new LinkedList<PreciseInv>(logs.get(n)));
        if(ec.getForkedWrites().containsKey(n)){
          unacceptableWrites.get(n).removeAll(ec.getForkedWrites().get(n));
        }
        if(unacceptableWrites.get(n).size() == 0){
          unacceptableWrites.remove(n);
        }
      }
    }

    LinkedList<SimPreciseInv> discardedWrites = this.rollbackState(unacceptableWrites);
    System.out.println("Discarding from Eviction Certificate at " + myBranchId + ": " + discardedWrites.size());

    if(shouldFail) assert getEpoch() == (ec.getEpoch() - 1);
    this.epochCount = ec.getEpoch();
  }

  /**
   * prune the state to not include these writes and any writes that depend on any of these writes
   * @param unacceptableWrites
   * @param initialState defines the starting state 
   * @return discarded writes
   * 
   */
  synchronized public LinkedList<SimPreciseInv> rollbackState(HashMap<BranchID,LinkedList<PreciseInv>> unacceptableWrites){
    if(lastCheckpoint != null){
      return rollbackState(unacceptableWrites, lastCheckpoint.objectStore, lastCheckpoint.omitVV, lastCheckpoint.lastWrite);
    }else{
      return rollbackState(unacceptableWrites, new Store(), new AcceptVV(), new HashMap<NodeId, SimPreciseInv>());
    }
  }


  //  /***
  //   * create DVV for a new precise update based on the current status of log
  //   * Exclude evicted faulty nodes from the DVV
  //   * @return
  //   */
  //  @Override
  //  protected DependencyVV generateDVV(){
  //    DependencyVV dvv = super.generateDVV();
  //    LinkedList<NodeId> droppedNodes = new LinkedList<NodeId>();
  //    for(VVIterator vvi = dvv.getIterator(); vvi.hasMoreElements();){
  //      NodeId n = vvi.getNext();
  //      for(NodeId b: evictedNodes){
  //        if(((BranchID)n).contains(b)){
  //          droppedNodes.add(n);
  //        }
  //      }
  //    }
  //    for(NodeId b: droppedNodes){
  //      dvv.dropNode(b);
  //    }
  //    return dvv;
  //  }

  /**
   * Returns all the writes that have been accepted by this node that were created by the fautly node
   * @param n
   * @return
   */
  synchronized public HashMap<BranchID, TreeSet<PreciseInv>> getAcceptedForkedWrites(NodeId n){
    HashMap<BranchID, TreeSet<PreciseInv>> hm  = new HashMap<BranchID, TreeSet<PreciseInv>>();
    for(NodeId b: logs.keySet()){
      BranchID bid = (BranchID)b;
      if(bid.contains(n)){
        hm.put(bid, new TreeSet<PreciseInv>(logs.get(bid)));
      }
    }
    return hm;
  }

  @Override
  public SimPreciseInv write(ObjId o, IrisDataObject data){
    /*      synchronized(this){
    if(evictedNodes.contains(this.getBranchID().getIDint())){
	System.err.println("trying to write at evicted node" + this);
	return null;
	}
     }*/
    // hack to offload work before acquiring lock
    data.getHash();
    if(shouldFail) assert data!= null;
    if(shouldFail) assert o!= null;
    if(shouldFail) assert o.getPath() != null;
    long starttime=0,endtime=0;
    if(measureTime){
      starttime = System.currentTimeMillis();
    }
    SimPreciseInv spi = super.write(o, data);
    if(measureTime){
      endtime = System.currentTimeMillis();
      LatencyWatcher.put("WRITE", endtime -starttime);
    }

    if(measureTime){
      starttime = System.currentTimeMillis();
    }
    printStorageCost("update", spi);
    synchronized(this){
      TreeSet<StoreEntry> tse = store.get(o);
      if(tse == null){
        this.notifyWatches(o, new StoreEntry(spi));
      }else{
        for(StoreEntry se: tse){
          if(se.getAcceptStamp().eq(spi.getAcceptStamp())){
            this.notifyWatches(o, se);
          }
        }
      }
    }
    if(measureTime){
      endtime = System.currentTimeMillis();
      LatencyWatcher.put("NotifyWatches", endtime -starttime);
    }
    //if(spi.getObjId().getPath().startsWith(AgreementGCProtocol.gcProposalDir)){
    //  System.out.println("Writing " + spi);
    //}    
    return spi;
  }

  @Override 
  synchronized protected Node createNewForkedNode(BranchID sbid) throws Exception{
    IrisNode n = new IrisNode(sbid);
    n.sync(this);
    n.lamportClock = this.lamportClock;
    n.lastUsedCVV = lastUsedCVV;
    return n;
  }


  synchronized protected LinkedList<Certificate> getNewerEpochCertificate(int epoch){
    LinkedList<Certificate> newCert = new LinkedList<Certificate>();
    if(this.getEpoch() > epoch){
      for(Certificate c: this.epochAdvancingCertificates){
        if(c.getEpoch() > epoch){
          newCert.add(c);
        }
      }
    }
    return newCert;
  }

  private boolean applyCertificates(LinkedList<Certificate> certs) throws Exception{
    boolean status = true;
    for(Certificate c: certs){
      if(c.verify() && this.addCertificate(c)){
        for(SyncFilter sf: syncFilters){
          sf.notifyEpochSync(this, c);
        }
      }else{
        status = false;
      }
    }
    return status;
  }

  @Override 
  synchronized protected boolean epochSync(Node n) throws Exception{
    boolean sync = true;
    if(n.getEpoch() > getEpoch()){
      LinkedList<Certificate> certs = ((IrisNode)n).getNewerEpochCertificate(this.getEpoch());
      applyCertificates(certs);
    }else if(n.getEpoch() < getEpoch()){
      sync = false;
    }
    return sync;

  }


  @Override
  synchronized protected SyncStatus logSync(BranchID n, Collection<SimPreciseInv> spiSet) throws Exception{
    if(n != null && (evictedNodes.contains(n.getIDint()) || faultyNodes.contains(n.getIDint()))){
      return new SyncStatus(SyncStatus.SenderFaulty);
    }
    boolean sync = true;
    boolean stateModified = false;
    byte status = SyncStatus.SyncSuccessful;
    TreeSet<Long> nodes = new TreeSet<Long>();
    //ensure that we have all the certificates that people refer in the files..otherwise abort sync
    for(SimPreciseInv spi: spiSet){
      if(spi.getObjId().getPath().startsWith(this.globalCertificatesString)){
        //LinkedList<Certificate> certs = (LinkedList<Certificate>)spi.data;
        IrisObject irisObj = spi.getData();
        assert irisObj instanceof IrisDataObject;
        LinkedList<Certificate> certs = (LinkedList<Certificate>)((IrisDataObject)irisObj).getObject();
        for(Certificate c: certs){
          if(this.getEpoch() < c.getEpoch()){
            sync = false;
            status = SyncStatus.CertFilterReject;
          }
        }
      }
      nodes.add(spi.getAcceptStamp().getNodeId().getIDint());
    }

    HashedVV oldVV = this.getHashedDependencyVV(logs.getMinimalVV(this.getCurrentVV()));
    //    HashMap<Long, TreeSet<BranchID>> bk = logs.snapshotBranchTreeMap(nodes);
    //      Store oldStore = store.clone();

    SyncStatus ss;
    if(sync){
      //System.out.println("Starting store txn " + store.getStoreId());
      store.startTxn();
      long start = System.nanoTime();
      ss = applyAll(spiSet);
      stateModified = true;
      long end = System.nanoTime();
      if(SangminConfig.fineGrainedTracking)System.out.println("applyAll took " + ((double)(end-start))/1000000);
      sync = ss.status == SyncStatus.SyncSuccessful;
      if(sync){
        AcceptVV mappedOldVV = logs.getClosure(this.remapHashedDependencyVV(oldVV, true));
        applyToStore(mappedOldVV);
        long end1 = System.nanoTime();
        if(SangminConfig.fineGrainedTracking)System.out.println("applyToStore took " + ((double)(end1-end))/1000000);
      }
    }else{
      ss = new SyncStatus(status);
    }


    if(sync){
      Collection<Tuple<ObjId, TreeSet<StoreEntry>>> addedEntries = store.getAddedEntries();
      for(SyncFilter sf: syncFilters){
        if(!sf.notifySync(this, getCurrentVV(), 
            n, 
            new TreeSet<SimPreciseInv>(spiSet), addedEntries 
        )){
          sync = false;
          status = SyncStatus.SyncFilterReject;
          ss = new SyncStatus(status);
          break;
        }
      }
    }

    if(sync){
    //System.out.println("ending store txn " + store.getStoreId());

      store.endTxn(true);
      //        AcceptVV mappedOldVV = logs.getClosure(this.remapHashedDependencyVV(oldVV, true));
      //        applyToStore(mappedOldVV);
    }
    else if(stateModified){
      //System.out.println("ending store txn " + store.getStoreId());

      this.logs.resetState(oldVV);//, bk);
      Env.printDebug("Sync failed. " + ss + "\noldVV" + oldVV + "\ncurVV"+this.getCurrentVV());

      store.endTxn(false);
    }
    for(SyncFilter sf: syncFilters){
      sf.syncOver(this, sync);
    }

    return ss;

  }



  /**
   * apply to store all invals newer than mappedOldVV in the log
   */
  @Override
  protected void applyToStore(AcceptVV mappedOldVV){
    Collection<SimPreciseInv> invals = this.findNewWrites(mappedOldVV);
    for(SimPreciseInv spi: invals){
      long start = System.nanoTime();
      StoreEntry se = storeApply(spi);
      long end1 = System.nanoTime();
      this.notifyWatches(spi.getObjId(), se);
      long end = System.nanoTime();
      if(SangminConfig.fineGrainedTracking)System.out.println(" notifyWatches took " + ((double)(end-end1))/1000000);
      if(SangminConfig.fineGrainedTracking)System.out.println(" storeApply took " + ((double)(end1-start))/1000000);
    }

  }

  synchronized protected void notifyWatches(ObjId o, StoreEntry se){
    for(SubscriptionSet subS:watches.keySet()){
      if(!subS.getIntersection(SubscriptionSet.makeSubscriptionSet(o.getPath())).isEmpty()){
        for(NamespaceWatch nw: watches.get(subS)){
          queue.enQueue(new Notification(nw, se, o, this.getEpoch()));
        }
      }
    }
  }

  @Override
  synchronized public boolean notifyNewForks(Collection<ProofOfMisbehavior> newPOM){
    boolean ret = false;
    for(ProofOfMisbehavior pom: newPOM){
      if(!this.pomList.contains(pom)){
        NodeId bid = null;
        if(pom instanceof ConcurrentPOM){
          bid = ((ConcurrentPOM)pom).getFaultyBranch().getNodeId();
        }else{
          bid = ((BranchID)pom.getParent().getNodeId()).getBaseId();
        }
        if(bid.getIDint() != this.getBranchID().getIDint() || pom instanceof ConcurrentPOM){
          // update log
          POMRemap pr = new POMRemap();
          logs.applyPOM(pom, pr);
          if(pr.isInit()){
            store.applyPOMRemap(pr);
          }
          pomList.add(pom);
          ret = true;
          // add entry
          if(!evictedNodes.contains(bid.getIDint())){
            // add POMs and notify handlers
            this.addFaultyNode(bid, pom);
          }

          //notify filters
          for(POMFilter pf: pomFilters){
            pf.receivePOM(this, pom);
            if((pr.isInit())){
              try{
                pf.receivePOMRemap(this, pr);
              }catch(Exception e){
                e.printStackTrace();
                assert false;
              }
            }
          }
        }
      }
    }

    return ret;
  }

  @Override
  synchronized public boolean handleFailure(SimPreciseInv spi, LogStatus status){
    boolean ret = false;
    if(status.getVerificationStatus() == LogStatus.CompatibilityFail && 
        spi.getDVV() instanceof HashedVV && 
        ((HashedVV)spi.getDVV()).isInitialized()){
	System.out.println("handleFailure called " + spi);
      // create and process POM
      ProofOfMisbehavior pom = logs.getProofOfMisbehavior(spi);
      LinkedList<ProofOfMisbehavior> ll = new LinkedList<ProofOfMisbehavior>();
      ll.add(pom);
      if(spi.getNodeId().getIDint() != this.getBranchID().getIDint()){
        // so that we don't generate POMs against ourself :)
        return this.notifyNewForks(ll);
      }
    }else if(status.getVerificationStatus() == LogStatus.MissingPOMForForkedChildBranch){
      // find parent branch
      LinkedList<BranchID> parents = logs.getParents(spi);
      assert parents.size() == 1;
      BranchID parent = parents.getFirst();
      for(ProofOfMisbehavior pom: this.pomList){
        if((!(pom instanceof ConcurrentPOM) && 
            pom.getParent().getAcceptStamp().getNodeId().getIDint() == parent.getIDint() && !logs.containsPOM(pom)) 
            //            || ((SecureSimPreciseInv)pom.child1).generateMyHash().equals(spi.generateMyHash())
        ){
          POMRemap pr = new POMRemap();
          ret |= logs.applyPOM(pom, pr);
          if(pr.isInit()){
            store.applyPOMRemap(pr);
            for(POMFilter pf: this.pomFilters){
              try{		
                pf.receivePOMRemap(this, pr);
              }catch(Exception e){
                e.printStackTrace();
                assert false;
              }

            }
          }
        }
      }
    }
    return ret;
  }


  synchronized public SyncStatus sync(Node n, AcceptVV avv, boolean optimize) throws Exception{
    return sync(n, avv, null, optimize);
  }

  synchronized public SyncStatus sync(Node n, AcceptVV avv, ObjId o, boolean optimize) throws Exception{
    if(evictedNodes.contains(n.getBranchID().getIDint()) || faultyNodes.contains(n.getBranchID().getIDint())){
      return new SyncStatus(SyncStatus.SenderFaulty);
    }
    else{
      SyncStatus ss;
      if(useSync2){
        SyncRequest sr = new SyncRequest(avv, this.getEpoch(), this.pomList, this.getBranchID().getIDint(), SangminConfig.BodyConfig, o, optimize);
        SyncPacket sp = ((IrisNode)n).getSyncPacket(sr);
        //SyncStatus ss1 = ((IrisNode)n).applySyncPacket(sp);
        ss = this.applySyncPacket(sp);
        //        if(ss1.status != SyncStatus.SyncSuccessful){
        //          n.logs.compare(this.logs, ss1.as);
        //          assert false;
        //          
        //        }
        if(ss.status == SyncStatus.InclusionFailed && !optimize){
          System.out.println("Inclusion FAILED");
          n.logs.compare(this.logs, ss.as);
        }
      }else{
        ss =  super.sync(n);
        printStorageCost("sync", null);
      }

      return ss;
    }

  }

  @Override
  synchronized public SyncStatus sync(Node n) throws Exception{
    return sync(n, this.getCurrentVV(), false);
  }

  public SyncStatus exponentialBackoffSync(Node sender) throws Exception{
    return exponentialBackoffSync(sender, null);
  }

  /**
   * repeat a series of syncs 
   * @param sender
   * @param initialVV
   * @return
   * @throws Exception 
   */
  public SyncStatus exponentialBackoffSync(Node sender, ObjId o) throws Exception{
    // keep syncing until we are successful or the result is not InclusionFailed
    SyncStatus ss= null;
    AcceptVV curVV = this.getCurrentVV();
    SyncStatus prevStatus = null;
    int backoffCount = 2;
    int numTries = 0;
    if(SangminConfig.tryOptimizedSync){
      ss = this.sync(sender, curVV, o, true);
    }
    if(ss == null || ss.status != SyncStatus.SyncSuccessful){
      ss = this.sync(sender, curVV, o, false);
    }
    while(ss.status == SyncStatus.InclusionFailed 
//        || ss.status == SyncStatus.FIFOFailed 
//        || ss.status == SyncStatus.DVVInclusionFailed
        ){
      if(shouldFail) assert ss.status == SyncStatus.InclusionFailed ;
//      || ss.status == SyncStatus.FIFOFailed 
//      || ss.status == SyncStatus.DVVInclusionFailed;
      if(prevStatus != null && prevStatus.as.getNodeId().equals(ss.as.getNodeId())){
        backoffCount *= 2;
      }else{
        backoffCount = 2;
      }
      long newClock = ss.as.getLocalClock()-backoffCount;
      if(newClock < 0){
        if(numTries >0){
          sender.logs.compare(this.logs, ss.as);
          assert false;
          return ss;
        }
        System.out.println("newClock negative " +newClock);
        newClock = -1;
        numTries++;
      }

      //assert backoffCount >= 0;
      System.out.println("sync to node " + sender.getBranchID() + " failed with " + curVV + " ss: " + ss);
      // emulate binary search?
      CounterVV cvv = new CounterVV(curVV);
      cvv.addMinAS(new AcceptStamp(newClock, ss.as.getNodeId()));
      curVV = new AcceptVV(cvv);
      System.out.println("trying with new acceptVV " + curVV);
      prevStatus = ss;
      ss = this.sync(sender, curVV, o, false);

    }
    return ss;
  }

  public SyncStatus exponentialBackoffSync(BranchID sender, AcceptVV startVV) throws Exception{
    return exponentialBackoffSync(sender, startVV, null);
  }
  /**
   * repeat a series of syncs 
   * @param sender
   * @param initialVV
   * @return
   */
  public SyncStatus exponentialBackoffSync(BranchID sender, AcceptVV startVV, ObjId o) throws Exception{
    // keep syncing until we are successful or the result is not InclusionFailed
    SyncStatus ss = null;
    AcceptVV curVV = startVV;
    SyncStatus prevStatus = null;
    int backoffCount = 2;

    int numTries = 0;

    if(SangminConfig.tryOptimizedSync){
      ss = this.sync2(sender, curVV, o, true);
    }
    if(ss == null || ss.status != SyncStatus.SyncSuccessful){
      Env.logWrite("OPTIMIZED SYNC FAILED WITH STATUS" +ss);
      ss = this.sync2(sender, curVV, o, false);
    }
    while(ss.status == SyncStatus.InclusionFailed 
//        || ss.status == SyncStatus.FIFOFailed 
//        || ss.status == SyncStatus.DVVInclusionFailed
        ){
      if(shouldFail) assert ss.status == SyncStatus.InclusionFailed ;
//      || ss.status == SyncStatus.FIFOFailed 
//      || ss.status == SyncStatus.DVVInclusionFailed;
      if(prevStatus != null && prevStatus.as.getNodeId().equals(ss.as.getNodeId())){
        backoffCount *= 2;
      }else{
        backoffCount = 2;
      }
      long newClock = ss.as.getLocalClock()-backoffCount;
      if(newClock < 0){
        if(numTries >0){
          return prevStatus;
//          assert false; to deal with faults
        }
        System.out.println("newClock negative " +newClock);
        newClock = -1;
        numTries++;
      }
      System.out.println("sync to node " + sender + " failed with " + curVV + " ss: " + ss);
      // emulate binary search?
      CounterVV cvv = new CounterVV(curVV);
      cvv.addMinAS(new AcceptStamp(newClock, ss.as.getNodeId()));
      cvv.dropNegatives();
      curVV = new AcceptVV(cvv);
      System.out.println("trying with new acceptVV " + curVV);
      prevStatus = ss;
      ss = this.sync2(sender, curVV, o, false);
    }

    //    System.out.println("SS:" + ss);
    return ss;
  }



  @Override
  public void garbageCollect(AcceptVV omitVV){
    try{
      if(IrisNode.useAgreementCheckpoint){
        HashedVV hvv = null;
        synchronized(this){
          if(omitVV instanceof HashedVV){
            hvv = this.getMinimalVV(this.remapHashedDependencyVV((HashedVV)omitVV));
          }else{
            assert logs.getCurrentVV().includes(omitVV);
            hvv = getHashedDependencyVV(omitVV);
            hvv = this.getMinimalVV(hvv);
          }
        }
        gcProtocol.startGC(hvv);
      }else{
        synchronized(this){
          for(SyncFilter sf: this.syncFilters){
            if(sf instanceof GCProtocol){
              ((GCProtocol)sf).startGC(this, omitVV, this.getEpoch()+1);
            }
          }
        }
      }
    }catch(Exception e){
      // TODO Auto-generated catch block
      e.printStackTrace();
      assert false;
    }
  }



  @Override
  synchronized public LinkedList<SimPreciseInv> applyCheckpoint(Checkpoint checkpoint) throws Exception{
    try{
      LinkedList<SimPreciseInv> discardedWrites = super.applyCheckpoint(checkpoint);
      for(SubscriptionSet ss: this.watches.keySet()){
        for(NamespaceWatch nw: this.watches.get(ss)){
          queue.enQueue(new Notification(nw, checkpoint.getEpoch()));
        }
      }

      Iterator<ObjId> iter = this.store.iterator();
      while(iter.hasNext()){
        ObjId o = iter.next();
        for(StoreEntry se: this.store.get(o))
          this.notifyWatches(o, se);
      }
      if(IrisNode.bestEffort){
        epochCount = 0; // reset the epochCount that was increment
      }
      long size = 0;
      for(SimPreciseInv spi: discardedWrites){
        size += BWSyncFilter.timestampSize + (spi.dvv.getSize()>0?spi.dvv.getSize()-1:0)*BWSyncFilter.timestampSize + BWSyncFilter.numNodeSize;
      }
      if(printStats)System.out.println("DISCARD: nodeid " + this.myBranchId + " "+ discardedWrites.size()  + " storageCost " + size);
      handleDiscardedWrites(discardedWrites);
      return discardedWrites;
    }catch(Exception e){
      System.out.println("checkpoint application failed");
      throw e;
    }
  }

  synchronized public void handleDiscardedWrites(LinkedList<SimPreciseInv> discardedWrites){
    // re-instate any certificates that were discarded
    // notify all discarded write filters

    for(SimPreciseInv spi:discardedWrites){
      if(spi.getObjId().equals(IrisNode.certificatesFiles.get(this.myBranchId.getIDint()))){
        this.write(IrisNode.certificatesFiles.get(this.myBranchId.getIDint()), new IrisDataObject(new LinkedList<Certificate>(this.epochAdvancingCertificates)));
      }
    }
    for(DiscardedWriteFilter df: discardedWriteFilters){
      df.notifyDiscarded(this, discardedWrites);
    }
  }

  synchronized public void printStorageCost(String event, Object o){
    if(printStats){
      if(event.equals("sync") || event.equals("gc")){
        storageCost = 0;
        logSize = 0;
        for(NodeId bid: logs.keySet()){
          logSize += logs.get(bid).size();
          storageCost += BWSyncFilter.branchIDSize + BWSyncFilter.hashSize;
          for(PreciseInv pi: logs.get(bid)){
            SimPreciseInv spi = (SimPreciseInv)pi;
            storageCost += BWSyncFilter.timestampSize + (spi.dvv.getSize()>0?spi.dvv.getSize()-1:0)*BWSyncFilter.timestampSize + BWSyncFilter.numNodeSize;
          }
        }
      }else{
        logSize++;
        SimPreciseInv spi = (SimPreciseInv)o;
        storageCost += BWSyncFilter.timestampSize + (spi.dvv.getSize()>0?spi.dvv.getSize()-1:0)*BWSyncFilter.timestampSize + BWSyncFilter.numNodeSize;
      }

      System.out.println(System.currentTimeMillis() + " STORAGE: nodeid " + this.myBranchId + " " + storageCost + " logSize: " + logSize);
      //      
      //      int numFaultyWrites = 0;
      //      for(NodeId n: logs.keySet()){
      //        if(n.getIDint() == 9){
      //          numFaultyWrites += logs.get(n).size();
      //        }
      //      }
      //      System.out.println(System.currentTimeMillis() + " FaultyWrites: " + numFaultyWrites);
      //      
    }
  }

  public AgreementGCProtocol getGCProtocol(){
    assert this.useAgreementCheckpoint;
    return gcProtocol;
  }

  synchronized public AcceptVV getCurrentHashedVV(){
    // TODO Auto-generated method stub
    try{
      return this.getHashedDependencyVV(this.getCurrentVV());
    }catch(Exception e){
      // TODO Auto-generated catch block
      e.printStackTrace();
      assert false;
    }
    return null;
  }


}
