package code.simulator;

import java.io.BufferedInputStream;
import java.io.ByteArrayInputStream;
import java.io.ByteArrayOutputStream;
import java.io.IOException;
import java.io.ObjectInputStream;
import java.io.ObjectOutputStream;
import java.net.ServerSocket;
import java.net.Socket;
import java.util.Arrays;
import java.util.HashMap;
import java.util.Iterator;
import java.util.LinkedList;

import code.AcceptVV;
import code.Config;
import code.CounterVV;
import code.Env;
import code.NodeId;
import code.branchDetecting.BranchID;
import code.security.SangminConfig;
import code.serialization.IrisInputStream;
import code.serialization.IrisObjectInputStream;
import code.serialization.IrisObjectOutputStream;
import code.serialization.IrisOutputStream;
import code.simulator.agreement.Tuple;


/**
 * this class receives syncPackets from the network. 
 * it receives one packet and then disconnects
 * @author princem
 *
 */
public class Listener extends Thread{
  private static boolean dbg = false;
  private IrisNode node;
  private boolean shutdown;
  private ServerSocket ss = null;

  public LinkedList<ListenerWorker> workerMap;
  //private BranchID configIdx; // index for config file
  private NodeId configIdx; // index for config file
  /** 
   *  Constructor 
   **/ 
  public Listener(IrisNode n){    
    this.shutdown = false;
    this.setName("Listener thread for " + n);
    this.node = n;
    this.configIdx = new NodeId(node.myBranchId.getIDint());
    setDaemon(true);
    workerMap = new LinkedList<ListenerWorker>();
  }

  public Listener(IrisNode n, BranchID configIdx){
    this(n);
    this.configIdx = new NodeId(configIdx.getIDint());
  }


  public void shutdown(){
    shutdown = true;
    if (ss != null){
      try{
        ss.close();// this call will make any thread currently blocked in accept()
        // throw a SocketException
        synchronized(workerMap){
          Iterator<ListenerWorker> iter = workerMap.iterator();
          while(iter.hasNext()){
            ListenerWorker wt = iter.next();
            wt.shutdown();
            iter.remove();
          }
        }
      }catch(IOException ioe){
        System.err.println(" InvalStreamReceiver shutting down ignore Exception: " 
            +ioe.toString());
      }
      ss = null;
    }
  }
  /** 
   *  Main thread method 
   **/ 
  public void run(){
    Socket connection = null;

    if( dbg ){
      Env.dprintln(dbg, "InvalStreamReceiver::run on port " 
          + Config.getPortInval(configIdx));
    }
    System.out.println(node.getBranchID().getIDint() + " running listener " +  " at port " + Config.getPortInval(configIdx));

    try{


      try{
        ss = new ServerSocket(Config.getPortInval(configIdx));
      } catch(IOException f){
        Env.sprinterrln("InvalReceiver cannot open server socket on port " 
            + Config.getPortInval(configIdx) + ": " + f);
        f.printStackTrace();
        //        System.exit(1);
        return;
      }
      System.out.println("IrisNode Server socket is created");



      while(!shutdown){
        try{
          connection = ss.accept(); // Should get a HeartbeatSocket

          connection.setTcpNoDelay(true);
          connection.setSendBufferSize(1024*1024*10);

          this.workerMap.add(new ListenerWorker(connection, node));
          workerMap.getLast().start();

        }catch(IOException e){ 
          if(!shutdown){
            e.printStackTrace();
            Env.warn("InvalStreamReceiver thread exception" + e +
                " dropping this request and continuing : my node id " + configIdx);
          }
        }
      }
      if(dbg){
        System.out.println("================Inval Stream receiver stops =============");
      }
    }
    finally{
      if(ss != null){
        try{
          if( dbg ){
            Env.dprintln(dbg, "InvalStreamReceiver::close port " 
                + Config.getPortInval(configIdx));
          }

          ss.close();
        }
        catch(IOException sdf){
        }
        ss = null;
      }else{
        for(ListenerWorker wt: workerMap){
          wt.shutdown();
        }
      }
    }
  }

}

class ListenerWorker extends Thread{

  Socket connection;
  IrisNode node;
  private boolean shutdown;
  AcceptVV cachedVV; // stores the cached copy of the last known vv of this node
  long sender;


  ListenerWorker(Socket s, IrisNode node){
    connection = s;
    this.node = node;
    shutdown = false;
    this.setDaemon(true);
    this.setName("Iris Lintener Worker for node " + node.getBranchID());
    cachedVV = null;
  }


  public void shutdown(){
    shutdown = true;
    if (connection != null){
      try{
        connection.close();// this call will make any thread currently blocked in accept()
        // throw a SocketException
      }catch(IOException ioe){
        System.err.println(" InvalStreamReceiver shutting down ignore Exception: " 
            +ioe.toString());
      }
      connection = null;
    }
  }

  private void writePkt(IrisOutputStream loos, SyncPacket pkt) throws IOException{
    long orig = loos.getBytes();
    pkt.writeToStream(loos, false);
    loos.flush();
    long size = loos.getBytes()-orig;
    if(pkt.isHasBodies()){
      pkt.writeBodyToStream(loos, false);
    }
    loos.flush();
    long bodySize = loos.getBytes()-size-orig;
    int numWrites = 0;
    if(pkt.getType() == SyncPacket.SenderSameEpoch || pkt.getType() == SyncPacket.SenderHigherEpoch ){
      numWrites =  pkt.getWrites().size();
    }
    Env.logWrite(node.getBranchID().getIDint() + " sent " + size + " as syncPacket-metadata to " + sender + " numWrites " + numWrites
       );

    Env.logWrite(node.getBranchID().getIDint() + " sent " + bodySize + " as syncPacket-bodies to " + sender);
    
  }
  
  private SyncRequest readPkt(IrisInputStream ois) throws IOException{
    assert connection.isConnected();
    SyncRequest sr = new SyncRequest();
    if(SangminConfig.useByteArraysForIO){
      short len = ois.readShort();
      byte[] buf = new byte[len];
      ois.readFully(buf);
      ByteArrayInputStream bis = new ByteArrayInputStream(buf);
      ois = new IrisInputStream(bis);
    }
    sr.readFromStream(ois, false);
    return sr;
  }
  
  public void run(){
    IrisInputStream ois = null;
    IrisOutputStream loos = null;
    sender = -1;
    try{
      ois = new IrisInputStream(connection.getInputStream());
      loos = new IrisOutputStream(connection.getOutputStream());
      int magic = ois.readInt();
      if(magic == IncomingConnectionHandler.MAGIC){
        sender = ois.readLong();
        System.out.println("sender " + node.getBranchID().getIDint()  + " received init-command from " + sender);
        this.setName("Listener-sender" + node.getBranchID().getIDint() +"-receiver-" + sender);
      }else{
        throw new IOException("invalid input on socket " + magic);
      }

    }catch(IOException e){ 
      if(!shutdown){
        e.printStackTrace();
        Env.warn("InvalStreamReceiver thread exception" + e +
            " dropping this request and continuing : my node id " + node.getBranchID());
        if(!connection.isClosed())
          try{
            connection.close();
          }catch(IOException e1){
            System.out.println(System.currentTimeMillis() + " received exception " + e1);
            e1.printStackTrace();
          }
        this.shutdown = true;
        return;
      }
    }     
    while(!shutdown){
      try{

        assert connection.isConnected();
        
        SyncRequest sr = new SyncRequest();

        assert connection.isConnected();

//        ois = new IrisInputStream(connection.getInputStream());
//        sr.readFromStream(ois, false);
        sr = this.readPkt(ois);

        Env.logWrite(sender + " sent " + sr);
	//        Env.printDebug(sender + " sent " + sr);
        
        if(sr.isIncrementalVV()){
	    //	    Env.printDebug("received request for incrementalVV" + sr);
          if(cachedVV == null){
            SyncPacket pkt = new SyncPacket(SyncPacket.NoCachedState, -1, new BranchID(-1), true);
            writePkt(loos, pkt);
	    continue;
          }else{
            sr.resetIncrementalVV(cachedVV);
	    //	    Env.printDebug("reset incrementalVV; FullVV " + sr);
          }
        }
        
        //advance cachedVV
        if(cachedVV == null){
          cachedVV = sr.getStartVV();
        }else{
          cachedVV = cachedVV.cloneMaxVV(sr.getStartVV());
        }
        Tuple<SyncPacket, AcceptVV> pkt = this.node.getSyncPacketAcceptVV(sr);

	//	Env.printDebug("sending syncpacket  " + pkt.getKey());
        writePkt(loos, pkt.getKey());
        loos.flush();
        cachedVV = cachedVV.cloneMaxVV(pkt.getValue());
	//	Env.printDebug("updated cachedVV  " + cachedVV);
      }catch(IOException e){ 
        if(!shutdown){
          System.out.println(System.currentTimeMillis() + " received exception " + e);
          Env.warn("InvalStreamReceiver thread exception" + e +
              " dropping this request and continuing : my node id " + node.getBranchID());
          try{
            if(!connection.isClosed())connection.close();
          }catch(IOException e1){
            System.out.println(System.currentTimeMillis() + " received exception " + e1);
          }
          this.shutdown = true;
          return;
        }
      }      

    } // End of while(true)

  }


}
