package code.simulator;

import java.io.FileNotFoundException;
import java.io.IOException;
import java.util.*;
import java.util.Map.Entry;

import com.sleepycat.je.DatabaseException;
import java.io.ByteArrayOutputStream;

import code.serialization.IrisOutputStream;
import code.*;
import code.branchDetecting.*;
import code.lasr.db.Database;
import code.lasr.db.DbException;
import code.lasr.db.DbTransaction;
import code.lasr.db.TxnFailedException;
import code.security.*;
import code.simulator.agreement.Tuple;
import code.simulator.checkpoint.Checkpoint;
import code.simulator.log.*;
import code.simulator.persistentLog.PersistentStore;
import code.simulator.protocolFilters.AgreementGCProtocol;
import code.simulator.store.BodyStore;
import code.simulator.store.BodyStoreInterface;
import code.simulator.store.PersistentBodyStore;
import code.simulator.store.PersistentEnv;
import code.simulator.store.Store;
import code.simulator.store.StoreInterface;
import code.simulator.store.StoreEntry;

public class Node {

  public  static boolean enableSecurity = (SangminConfig.securityLevel > SangminConfig.SIGNATURE);
  public static boolean useSignature = SangminConfig.securityLevel >SangminConfig.NONE;
  
  public final static boolean useIncrementalDVV = true;
  public static boolean useNewCheckpoint = true;

  protected Checkpoint lastCheckpoint = null;

  protected BranchID myBranchId;

  protected AcceptVV lastUsedCVV;
  protected CounterVV indirectlyCoveredVV;

  protected long lamportClock;

  protected boolean forked = false;
  protected boolean omission_fault = false;

  public static boolean dropInconsistentWrites = false;
  protected int epochCount;

  protected Log logs;

  public static final boolean shouldFail = enableSecurity;

  public PersistentStore pe;

  HashSet<ProofOfMisbehavior> pomList;
  TreeSet<Long> faultyNodes;
  TreeSet<Long> evictedNodes;

  protected StoreInterface store;
  protected BodyStoreInterface bodyStore;

  public Node(BranchID branchId){
    this.myBranchId = branchId;
    logs = new Log();
    store = new Store();
    lastUsedCVV = new AcceptVV();
    indirectlyCoveredVV = new CounterVV();
    lamportClock = -1;
    epochCount = 0;
    evictedNodes = new TreeSet<Long>();
    faultyNodes = new TreeSet<Long>();
    pomList = new HashSet<ProofOfMisbehavior>();


    if(!SangminConfig.usePersistentStore){
      pe = null;
      bodyStore = new BodyStore();
    }else{
      try{
        String dbPath = SangminConfig.configFile+this.getBranchID().getIDint();//Config.getLocalStore(branchId);
        System.out.println("dbPath for node " + this.getBranchID() + " is " + dbPath);
        pe = new PersistentStore(dbPath, SangminConfig.BDBCacheSize, true, SangminConfig.syncToDisk); 
        bodyStore = new PersistentBodyStore(pe);
      }
      catch(DbException e){
        e.printStackTrace();
        if(pe != null){
          try{
            pe.close();
          }catch(Exception ee){
            System.err.println("Exception during database close:");
            ee.printStackTrace();
          }
        }
        System.exit(1);
      }catch(IOException e){
        e.printStackTrace();
        if(pe != null){
          try{
            pe.close();
          }catch(Exception ee){
            System.err.println("Exception during database close:");
            ee.printStackTrace();
          }
        }
        System.exit(1);
      }
    }
  }

  public void close(){
    synchronized(this){
      if(pe != null){
        pe.close();
      }
    }
  }

  synchronized public void setLamportClockTest(long lc){
    lamportClock = lc;
  }

  synchronized public AcceptVV getCurrentVV(){
    return logs.getCurrentVV();
  }

  synchronized public BranchID getBranchID(){
    return myBranchId;
  }

  synchronized public int getEpoch(){
    return epochCount;
  }

  synchronized protected NodeLog getLog(NodeId bid){
    if(!logs.containsKey(bid)){
      logs.put(bid, new NodeLog()); 
    }
    return logs.get(bid);
  }

  /**
   * This function must only be used for testing
   * @return
   */
  synchronized public Log getLogTest(){
    return logs;
  }

  /**
   * This function must only be used for testing
   * @return
   */
  synchronized public StoreInterface getStoreTest(){
    return store;
  }

  synchronized public StoreInterface getStore(){
    return store.clone();
  }

  synchronized public boolean canWrite(){
    return !forked;
  }

  synchronized public boolean isOmissionFaulty(){
    return omission_fault;
  }

  synchronized public void setOmissionFault(){
    this.omission_fault = true;
  }

  public SimPreciseInv write(ObjId objId, IrisDataObject data){
    return write(objId, data, false);
  }

  private synchronized SimPreciseInv write(ObjId objId, IrisObject data, boolean delete){
    if(evictedNodes.contains(this.getBranchID().getIDint())){
      System.err.println("trying to write at evicted node" + this);
      return null;
    }
    if(shouldFail) assert !forked;
    lamportClock++;
    AcceptStamp as = new AcceptStamp(lamportClock, myBranchId);
    AcceptVV dvv = null;
    try{
      dvv = logs.getMinimalVV(generateDVV());
    }catch(Exception e){
      e.printStackTrace();
      assert false;
    }
    SimPreciseInv spi;
   

    IrisObject origdata = data;
    if(!delete && SangminConfig.separateBodies && !objId.getPath().startsWith(AgreementGCProtocol.gcProposalDir)){
	assert data instanceof IrisDataObject;
	data = new IrisHashObject(data.getHash());
    }

    if(this.enableSecurity){
      HashMap<NodeId, SimPreciseInv> invals = logs.getInvals(dvv);
      HashedVV vv = new HashedVV(dvv, logs.getHashMap(invals));
      assert data != null;
      if(!delete){
        spi = new SecureSimPreciseInv(objId, as, vv, data, this.getEpoch(), this.store.getObjectHashes(objId));
      }else{
        spi = new SecureSimPreciseInv(objId, as, vv, this.getEpoch(), this.store.getObjectHashes(objId));
      }
    }else{
      if(!delete){
        spi = new SimPreciseInv(objId, as, dvv, data, this.store.getObjectHashes(objId), false);
      }else{
        spi = new SimPreciseInv(objId, as, dvv, Store.deleteObject, this.store.getObjectHashes(objId), true);
      }
    }
    LogStatus ls = apply(spi);
    assert ls.getApplyStatus() == LogStatus.ApplySuccessful: ls  + " myID" + this.getBranchID();
//    if(!delete && SangminConfig.separateBodies && !objId.getPath().startsWith(AgreementGCProtocol.gcProposalDir)){
//      assert data instanceof IrisDataObject;
//      if(SangminConfig.usePersistentStore){
//        try{
//          txn.commit();
//        }catch(TxnFailedException e){
//          System.err.println("add txn failed");
//          e.printStackTrace();
//          return null;
//        }
//        
//      }else{
//        bodyStore.addBody(null, (IrisDataObject)data); 
//      }
//      data = new IrisHashObject(data.getHash());
//    }
    long start = System.nanoTime();
    DbTransaction txn = null;
    if(SangminConfig.usePersistentStore){
      //	  long start = System.currentTimeMillis();
      txn = pe.getDB().newTransaction();
    }else{
      txn = null;
    }
    if(!delete && SangminConfig.separateBodies && !objId.getPath().startsWith(AgreementGCProtocol.gcProposalDir)){
      assert origdata instanceof IrisDataObject;
      bodyStore.addBody(txn, (IrisDataObject)origdata); 
    }
    if(SangminConfig.usePersistentStore){
      try{
        this.logPacket(txn, IrisNode.generatePseudoSyncPkt(spi));
        txn.commit();
      }catch(TxnFailedException e){
        System.err.println("add txn failed");
        e.printStackTrace();
        return null;
      }catch(IOException e1){
        e1.printStackTrace();
        txn.abort();
        return null;
      }
      long end = System.nanoTime();
      if(SangminConfig.fineGrainedTracking)System.out.println("disk write on write took: " + (((double)(end-start))/1000000));
    }
    return spi;
  }

  public void logPacket(DbTransaction txn, SyncPacket pkt) throws IOException{
    ByteArrayOutputStream bs;
    IrisOutputStream oos;

    bs = new ByteArrayOutputStream();
    oos = new IrisOutputStream(bs);
    pkt.writeToStream(oos, false);
    oos.flush();
    byte[] corePkt = bs.toByteArray();
    if(SangminConfig.fineGrainedTracking)System.out.println("writing syncpacket to disk " + corePkt.length);
    pe.append(txn, corePkt);
  }

  /**
   * read an immutabled body from the store
   * on failure, returns null
   * @param hashes
   * @return
   */
  public LinkedList<IrisObject> immutableRead(LinkedList<IrisObject> hashes){
    assert(SangminConfig.separateBodies);
    DbTransaction txn = null;
    if(SangminConfig.usePersistentStore){
      txn = pe.getDB().newTransaction();
    }
    LinkedList<IrisObject> newReads = new LinkedList<IrisObject>();
    for(IrisObject io: hashes){
      IrisDataObject ido = bodyStore.getBody(txn, io.getHash());
      if(ido != null){
        newReads.add(ido);
      }else{
        newReads = null;
        break;
      }
    }
    if(SangminConfig.usePersistentStore){
      try{
        txn.commit();
      }catch(TxnFailedException e){
        System.err.println("read txn failed");
        e.printStackTrace();
        return null;
      }
    }
    return newReads;
  }

  public SimPreciseInv delete(ObjId objId){
    return write(objId, Store.deleteObject, true);
  }

  synchronized public LinkedList<IrisObject> read(ObjId objId){
    LinkedList<IrisObject> reads = store.getData(objId);
    if(SangminConfig.separateBodies){
      DbTransaction txn = null;
      if(SangminConfig.usePersistentStore){
        txn = pe.getDB().newTransaction();
      }
      //      long start = System.currentTimeMillis();


      LinkedList<IrisObject> newReads = new LinkedList<IrisObject>();
      for(IrisObject io: reads){
        IrisDataObject ido = bodyStore.getBody(txn, io.getHash());
        if(ido != null){
          newReads.add(ido);
        }else{
          newReads.add(io);
        }
      }
      if(SangminConfig.usePersistentStore){

        try{
          txn.commit();
        }catch(TxnFailedException e){
          System.err.println("read txn failed");
          e.printStackTrace();
          return null;
        }

      }
      //      long end = System.currentTimeMillis();
      //      System.out.println(" reading  " + objId + " disk read took " + (end-start));
      return newReads;
    }
    return reads;
  }

  /**
   * read the metadata (hash) for the specified key
   * @param objId
   * @return
   */
  synchronized public LinkedList<IrisObject> efficientRead(ObjId objId){
    return store.getData(objId);
  }

  /**
   * read a range of keys
   * @param objIdStart
   * @param objIdEnd
   * @return
   */
  synchronized public LinkedList<Entry<ObjId, LinkedList<Object>>> read(ObjId objIdStart, ObjId objIdEnd){
    Map<ObjId, TreeSet<StoreEntry>> subStore = store.subMap(objIdStart, true, objIdEnd, true);
    LinkedList<Entry<ObjId, LinkedList<Object>>> readVals = new LinkedList<Entry<ObjId, LinkedList<Object>>>();
    for(ObjId o: subStore.keySet()){
      readVals.add(new Tuple(o, store.getData(o)));
    }
    return readVals;
  }

  /**
   * Read the raw-metadata for the specified key (storeEntry)
   * @param objId
   * @return
   */
  synchronized public LinkedList<StoreEntry> efficientRawRead(ObjId objId){
    return store.getRawData(objId);
  }


  /**
   * read the raw metadata augmented with bodies wherever possible based on local information
   * @param objId
   * @return
   */
  synchronized public LinkedList<StoreEntry> rawRead(ObjId objId){
    LinkedList<StoreEntry> reads = store.getRawData(objId);
    if(SangminConfig.separateBodies){
      LinkedList<StoreEntry> newReads = new LinkedList<StoreEntry>();
      
      DbTransaction txn = null;
      if(SangminConfig.usePersistentStore){
        txn = pe.getDB().newTransaction();
      }
      for(StoreEntry io: reads){
        IrisDataObject ido = bodyStore.getBody(txn, io.getData().getHash());
        if(ido != null){
          newReads.add(io.addBody(ido));
        }else{
          newReads.add(io);
        }
      }
      if(SangminConfig.usePersistentStore){

        try{
          txn.commit();
        }catch(TxnFailedException e){
          System.err.println("read txn failed");
          e.printStackTrace();
          System.exit(1);
          return null;
        }
      }
      return newReads;
    }
    return reads;

  }

  /**
   * ensure that all the keys have associated bodies
   */
  synchronized public void testRead(){
    Iterator<ObjId> iter = store.iterator();
    if(!SangminConfig.separateBodies){
      return;
    }
    DbTransaction txn = null;
    if(SangminConfig.usePersistentStore){

      txn = pe.getDB().newTransaction();
    }
    while(iter.hasNext()){
      ObjId objId = iter.next();
      LinkedList<StoreEntry> reads = store.getRawData(objId);
      for(StoreEntry io: reads){
        assert bodyStore.containsBody(txn, io.getData().getHash());
      }
    }
    if(SangminConfig.usePersistentStore){

      try{
        txn.commit();
      }catch(TxnFailedException e){
        System.err.println("read txn failed");
        e.printStackTrace();
      }
    }
  }

  synchronized protected void storeInitialize(StoreInterface objectStore ){
    store.clear();
    Iterator<ObjId> iter = objectStore.iterator();
    while(iter.hasNext()){
      ObjId o = iter.next();
      store.put(o, objectStore.get(o));
    }
  }

  /***
   * create DVV for a new precise update based on the current status of log
   * 
   * @return
   */
  synchronized protected AcceptVV generateDVV(){
      //    if(!enableSecurity){
      //      return new AcceptVV();
      //    }
    if(this.useIncrementalDVV){
      VVIterator vvi = getCurrentVV().getIterator();
      while(vvi.hasMoreElements()){
        NodeId n = vvi.getNext();
        if(shouldFail) assert n instanceof BranchID: n;
      }
      // can instead use this.indirectlyCoveredVV
      AcceptVV myCVV = new AcceptVV(getCurrentVV()).dropNegatives();
      assert myCVV.includes(lastUsedCVV): myCVV +"\n" + lastUsedCVV;
      AcceptVV usefulVV = myCVV.getDiff(lastUsedCVV).dropNegatives();
      return usefulVV;
    }else{
      return getCurrentVV();
    }
  }

  /**
   * store is a collection of store to objId
   * Whenever we receive a new write, we compare it with existing writes (in lamport order) and then 
   * overwrite any superseding value
   * @param spi
   */
  synchronized protected StoreEntry storeApply(SimPreciseInv spi){
    if(spi.getFlags().contains(Flags.InconsistentWrite) && Node.dropInconsistentWrites){
      return null;
    }
    
    /* if(spi.getData() instanceof IrisDataObject && SangminConfig.separateBodies && 
        store.getIgnoreSet().getIntersection(SubscriptionSet.makeSubscriptionSet(spi.getObjId().getPath())).isEmpty()){
	
	DbTransaction txn = null;
	if(SangminConfig.usePersistentStore){
	    
	    txn = pe.getDB().newTransaction();
	}
	bodyStore.addBody(txn, (IrisDataObject)spi.getData());
	if(SangminConfig.usePersistentStore){
	    
	    try{
          txn.commit();
	    }catch(TxnFailedException e){
		System.err.println("checkpoint txn failed");
		e.printStackTrace();
	    }
	}
	spi.replaceIrisObject(new IrisHashObject(spi.getData().getHash()));
	}*/
    StoreEntry se = store.apply(spi);
    OracleNotifier.notifyWrite(spi, myBranchId, this.omission_fault);
    //    return store.apply(spi, logs);
    return se;
  }

  /**
   * store is a collection of store to objId
   * Whenever we receive a new write, we compare it with existing writes (in lamport order) and then 
   * overwrite any superseding value
   * @param spi
   */
  synchronized protected StoreEntry storeApply(StoreInterface store, SimPreciseInv spi){
    if(spi.getFlags().contains(Flags.InconsistentWrite) && Node.dropInconsistentWrites){
      return null;
    }
    //    return store.apply(spi, logs);
    return store.apply(spi);
  }

  synchronized protected void resetState(){

    lastUsedCVV = new AcceptVV();
    this.indirectlyCoveredVV = new CounterVV();
    //lamportClock = -1;
  }

  synchronized protected LogStatus apply(SimPreciseInv spi){
      long start = System.nanoTime();
      LogStatus status = logApply(spi, Log.ApplyIfAllPass);
      long end = System.nanoTime();
      if(SangminConfig.fineGrainedTracking)System.out.println(" logApply took " + ((double)(end-start))/1000000);

      if(status.getApplyStatus() == LogStatus.ApplySuccessful){
	  storeApply(spi);
	  long end1 = System.nanoTime();
	  if(SangminConfig.fineGrainedTracking)System.out.println(" storeApply took " + ((double)(end1-end))/1000000);

      }
    return status;
  }

  synchronized private LogStatus logApply(SimPreciseInv spi, byte applyCondition){
    LogStatus status = null ;
    try{
      //  verify = verify & enableSecurity;
      if((applyCondition == Log.Apply) || SangminConfig.enableInconsistencyNotification || (getCurrentVV().includes(spi.dvv))){
        // sanity check
        if(lastCheckpoint != null){
          VVIterator vvi = spi.dvv.getIterator();
          boolean included = false; // is any componenet of the dvv present in the omitVV to make it unverifiable
          while(vvi.hasMoreElements()){
            NodeId n = vvi.getNext();
            if(getOmitVV().containsNodeId(n) && (spi.dvv.getStampByIteratorToken(n) < getOmitVV().getStampByIteratorToken(n))){
              included = true;

            }
          }
          //PRINCEM: Don't think this check will work anymore as unverifiable and unacceptable writes may 
          //be received now--issue inconsistency notificaitons
          if(included && !lastCheckpoint.isUnverifiableAcceptable(spi) && !SangminConfig.enableInconsistencyNotification){
            if(shouldFail) assert false: spi.dvv + " " + getOmitVV();

          }
        }else if (applyCondition == Log.Apply){
          if(shouldFail) assert applyCondition == Log.Apply;
        }
        // if the update is new and
        // if the update is local, then also update the lastUsedCVV
        // simple policy, set lastUsedCVV to CVV
        // complex policy: track from which nodes have we received *new* writes
        // and remove components that have been covered by other node's writes (except for local component)
        //    this.indirectlyCoveredVV.addMaxAS(spi.getAcceptStamp());
        //    long ts = spi.getAcceptStamp().getLocalClock();
        //    for(VVIterator vvi = spi.dvv.getIterator(); vvi.hasMoreElements(); ){
        //    NodeId n = vvi.getNext();
        //    if(indirectlyCoveredVV.containsNodeId(n) && 
        //    (indirectlyCoveredVV.getStampByIteratorToken(n) <= spi.dvv.getStampByIteratorToken(n)) &&
        //    !n.equals(myBranchId)){
        //    indirectlyCoveredVV.dropNode(n);
        //    if(shouldFail) assert ts > spi.dvv.getStampByIteratorToken(n);
        //    }
        //    }
        // simple policy

        status = logs.apply(spi, applyCondition);

        if(status.getApplyStatus() == LogStatus.ApplySuccessful || status.getApplyStatus() == LogStatus.PseudoApplySuccessful){
          lamportClock = (lamportClock<spi.getAcceptStamp().getLocalClock()?spi.getAcceptStamp().getLocalClock():lamportClock);
          if(spi.getNodeId().equals(myBranchId) && !getCurrentVV().includes(spi.getAcceptStamp())){
            lastUsedCVV = lastUsedCVV.cloneMaxVV(new AcceptVV(spi.dvv));
          }
          VVIterator vvi = getCurrentVV().getIterator();
          while(vvi.hasMoreElements()){
            NodeId n = vvi.getNext();
            if(shouldFail) assert n instanceof BranchID: n;
          }

        }


      }else{
        // if(shouldFail)if(shouldFail) assert false: spi + " "  + this.getCurrentVV() + " verify: "+ (applyCondition != Log.Apply) + " getOmitVV() " + (lastCheckpoint != null? getOmitVV(): "null");
        status = LogStatus.makeLogStatusFailedVerificationStatus(LogStatus.DVVInclusionFail);
      }

      boolean retry = handleFailure(spi, status);
      if(retry){
        status = this.logApply(spi, applyCondition);
      }

      return status;
    }finally{
      //System.out.println("status " +status + " cvv " + this.getCurrentVV());
    }
  }

  /**
   * 
   * @param spi
   * @param status
   * @return values indicates whether the inval should be reapplied or not?
   */
  synchronized public boolean handleFailure(SimPreciseInv spi, LogStatus status){
    return false;
  }

  synchronized public HashedVV getOmitVV(){
    return logs.getOmitVV();
  }

  synchronized public void notifyAdvanceEpoch(Node n, int curEpoch){
    if(curEpoch < this.epochCount){
      System.out.println("My epoch " + epochCount + " smaller than current epoch " + curEpoch + " at node " + n.getBranchID());
      // can initiate a sync
    }
  }

  synchronized public String toString(){
    return "Node:" + this.myBranchId + " CVV:" + this.getCurrentVV();
  }


  synchronized public boolean notifyNewForks(Collection<ProofOfMisbehavior> newPOM) throws Exception{
    return false;
    // do nothing
  }




  synchronized public void addFaultyNode(NodeId n, ProofOfMisbehavior pom){
    NodeId n1;
    if(n instanceof BranchID){
      n1 = ((BranchID)n).getBaseId();
    }else{
      n1= n;
    }
    if(shouldFail) assert !evictedNodes.contains(n1);

    if(!faultyNodes.contains(n1.getIDint())){
      faultyNodes.add(n1.getIDint());
    }

  }

  ///-------------------------NEEDS CLEANUP-----------------------------
  /**
   * find writes newer than vv
   * @param vv
   * @return
   */
  protected TreeSet<SimPreciseInv> findNewWrites(AcceptVV vv){
    //then a regular sync
      long start = System.nanoTime();
    TreeSet<SimPreciseInv> spiSet = new TreeSet<SimPreciseInv>();
    AcceptVV diff = getCurrentVV().getDiff(vv);
    VVIterator vvi = diff.getIterator();
    while(vvi.hasMoreElements()){
      NodeId nid = vvi.getNext();
      if(shouldFail) assert nid instanceof BranchID: nid;

      long ts = -1; // time stamp of the last write that receiver have
      if(vv.containsNodeId(nid)){
        try{
          ts = vv.getStampByServer(nid);
        }catch(NoSuchEntryException e){
          if(shouldFail) assert false;
          // TODO Auto-generated catch block
          e.printStackTrace();
        }
      }
      if(diff.getStampByIteratorToken(nid) != -1){
        NodeLog list = getLog((BranchID)nid);
        for(PreciseInv pi:list){
          SimPreciseInv spi = (SimPreciseInv)pi;
          if(spi.getAcceptStamp().getLocalClock() > ts){
            spiSet.add(spi);
          }
        }
      }
    }
    long end = System.nanoTime();
    if(SangminConfig.fineGrainedTracking)System.out.println("findNewWrites on write took: " + (((double)(end-start))/1000000));
    return spiSet;
  }


  protected TreeSet<SimPreciseInv> findNewWritesIncludingLastWrite(AcceptVV vv){
    return findNewWritesIncludingLastWrite(vv, getCurrentVV());
  }


  /**
   * find writes newer than vv ensuring that at least one write is included
   * @param vv
   * @return
   */
  protected TreeSet<SimPreciseInv> findNewWritesIncludingLastWrite(AcceptVV vv, AcceptVV endVV){
    //then a regular sync
    TreeSet<SimPreciseInv> spiSet = new TreeSet<SimPreciseInv>();
    AcceptVV cvv = endVV;
    VVIterator vvi = cvv.getIterator();
    while(vvi.hasMoreElements()){
      NodeId nid = vvi.getNext();
      if(shouldFail) assert nid instanceof BranchID: nid;

      long ts = -1; // time stamp of the last write that receiver have
      if(vv.containsNodeId(nid)){
        try{
          ts = vv.getStampByServer(nid);
        }catch(NoSuchEntryException e){
          if(shouldFail) assert false;
          e.printStackTrace();
        }
      }
      long endTS = cvv.getStampByIteratorToken(nid);
      if(endTS != -1){
        NodeLog list = getLog((BranchID)nid);
        if(endTS < ts){
          ts = endTS;
        }
        for(PreciseInv pi:list){
          SimPreciseInv spi = (SimPreciseInv)pi;
          long localClock = spi.getAcceptStamp().getLocalClock();
          if(localClock >= ts && localClock <= endTS) {
            spiSet.add(spi);
          }
        }
      }
      //
    }

    //    vvi = cvv.getIterator();
    //    while(vvi.hasMoreElements()){
    //      NodeId nid  = vvi.getNext();
    //      if(!diff.containsNodeId(nid)){
    //        // ensure that forking problem with same ts are caught
    //        // if my ts is same as the sender or if i am behind
    //        // put the lastWrite I have 
    //        
    //          NodeLog list = getLog((BranchID)nid);
    //          if(shouldFail) assert list.size() > 0;
    //          spiSet.add((SimPreciseInv)list.get(list.size()-1));
    //        }
    //        
    //      
    //    }

    return spiSet;
  }
  ///-------------------------NEEDS CLEANUP ENDS-----------------------------

  synchronized protected SyncStatus logSync(BranchID n, Collection<SimPreciseInv> spiSet) throws Exception{
    //    if(shouldFail) assert n.epochCount == epochCount;
    //    if(shouldFail) assert (n.lastCheckpoint == null)?lastCheckpoint==null:lastCheckpoint.equals(n.lastCheckpoint);

    AcceptVV oldVV = logs.getMinimalVV(this.getCurrentVV());
    HashMap<NodeId, Hash> hashes = logs.getHashMap(oldVV);

    SyncStatus ls = applyAll(spiSet);
    AcceptVV mappedOldVV = logs.getClosure(logs.remap(new HashedVV(oldVV, hashes), true));

    applyToStore(mappedOldVV);
    return ls;
  }

  /**
   * apply to store all invals newer than mappedOldVV in the log
   */
  protected void applyToStore(AcceptVV mappedOldVV){
    Collection<SimPreciseInv> invals = this.findNewWrites(mappedOldVV);
    for(SimPreciseInv spi: invals){
      storeApply(spi);
    }
  }

  synchronized protected SyncStatus applyAll(Collection<SimPreciseInv> spiSet){
    LogStatus status; 
    if(!myBranchId.isForked()){
      //System.out.println("Not forked");
      for(SimPreciseInv spi:spiSet){

        if(SimPreciseInv.useHopCount){
          spi.hopCount++;
        }

        if(Node.enableSecurity){
          //          if(shouldFail) assert getCurrentVV().includes(spi.dvv): "CVV: " + getCurrentVV() + " spi " + spi;
        }
        long stime =0,etime=0;
        if(IrisNode.measureTime){
          stime = System.currentTimeMillis();
        }    
        if(spi.getEpoch() > this.getEpoch()){
          SyncStatus ss = new SyncStatus(LogStatus.ImproperEpochNumber, spi.getAcceptStamp());
          return ss;
        }
        if(!spi.verifySig()){
          SyncStatus ss = new SyncStatus(LogStatus.SignatureVerificationFail, spi.getAcceptStamp());
          return ss;
        }
        status = this.logApply(spi, Log.ApplyIfAllPass);
        if(status.getApplyStatus() == LogStatus.PseudoApplySuccessful){
          System.out.println("Duplicate write is applied to log : " + spi);
        }
        if(status.getApplyStatus() == LogStatus.ApplyFailed){
          SyncStatus ss = SyncStatus.makeSyncStatusFromLogStatus(status, spi.getAcceptStamp());
          //System.out.println(ss);
          return ss;
        }
        
        if(IrisNode.measureTime){
          etime = System.currentTimeMillis();
          LatencyWatcher.put("APPLYINV", etime-stime);
        }
        assert this.getCurrentVV().includes(spi.getAcceptStamp()): this.getCurrentVV() + "  spi " + spi;
        //System.out.println("applied " + spi);
      }
      //System.out.println("Applied succeeded");
    }else{
      //System.out.println("Forked");
      for(SimPreciseInv spi:spiSet){
        // skip a write if it is from a concurrent branch or if it depends on a write from a concurrent branch (checked 
        // by ensuring that its DVV is satisfied 
        if(!getBranchID().isConcurrentForkedBranch(spi.getAcceptStamp().getNodeId()) && 
            getCurrentVV().includes(spi.dvv)){
          status = this.logApply(spi, Log.ApplyIfAllPass);
          if(status.getApplyStatus() == LogStatus.ApplyFailed){
            SyncStatus ss = SyncStatus.makeSyncStatusFromLogStatus(status, spi.getAcceptStamp());
            return ss;
          }
        }else{
          //System.out.println("skipping applying writes from my forked branch" + spi);
          SyncStatus ss = new SyncStatus(SyncStatus.WriteFromConcurrentBranch, spi.getAcceptStamp());
          return ss;
        }
      }
    }
    return new SyncStatus(SyncStatus.SyncSuccessful);
  }

  protected Node createNewForkedNode(BranchID sbid) throws Exception{
    Node n = new Node(sbid);
    n.sync(this);
    n.lastUsedCVV = lastUsedCVV;
    return n;
  }

  synchronized public LinkedList<Node> fork(int branches) throws Exception{
    if(shouldFail) assert !forked;
    if(shouldFail) assert branches >= 2;
    forked = true;
    LinkedList<Node> childBranches = new LinkedList<Node>();
    for(int i = 0; i < branches; i++){
      BranchID b;
      if(SangminConfig.useSimulatorId){
        if(shouldFail) assert myBranchId instanceof SimBranchID;
        b = new SimBranchID((SimBranchID)myBranchId, i);
      }else{
        b= myBranchId;
      }
      Node n = createNewForkedNode(b);
      childBranches.add(n);
    }
    return childBranches;
  }

  synchronized public void garbageCollect(AcceptVV omitVV) {
    if(this.useNewCheckpoint){
      try{
        newGarbageCollect(omitVV);
      }catch(Exception e){
        // TODO Auto-generated catch block
        e.printStackTrace();
        assert false;
      }
    }else{
      Checkpoint chkPt = generateCheckpoint(omitVV, epochCount+1);
      System.out.println("GC is created at node " + this.myBranchId + " , cp : " + chkPt);
      try{
        this.applyCheckpoint(chkPt);
      }catch(Exception e){
        // TODO Auto-generated catch block
        e.printStackTrace();
        assert false;
      }
    }
  }

  public synchronized HashedVV getHashedDependencyVV(VV vv) throws Exception{
    return new HashedVV(vv, logs.getHashMap(vv));
  }

  synchronized public HashedVV remapHashedDependencyVV(HashedVV vv) throws Exception{
    return remapHashedDependencyVV(vv, true);
  }

  synchronized public HashedVV remapHashedDependencyVV(HashedVV vv, boolean strict) throws Exception{
    AcceptVV remappedVV = logs.remap(vv, strict);
    return this.getHashedDependencyVV(remappedVV);
  }

  synchronized public HashedVV getClosure(HashedVV vv) throws Exception{
    AcceptVV closure = logs.getClosure(vv);
    return new HashedVV(closure, logs.getHashMap(closure));
  }

  synchronized public HashedVV getMinimalVV(HashedVV vv) throws Exception{
    AcceptVV minVV = logs.getMinimalVV(vv);
    return new HashedVV(minVV, logs.getHashMap(minVV));
  }

  synchronized public void newGarbageCollect(AcceptVV omitVV) throws Exception{
    Checkpoint chkPt = generateNewCheckpoint(this.getHashedDependencyVV(omitVV), epochCount+1);
    System.out.println("GC is created at node " + this.myBranchId + " , cp : " + chkPt);
    this.applyNewCheckpoint(chkPt);
  }


  public synchronized Checkpoint generateNewCheckpoint(HashedVV omitVV, int newEpochCount) throws Exception{
    return this.generateNewCheckpoint(omitVV, newEpochCount, this.getHashedDependencyVV(this.getCurrentVV()));
  }

  private TreeSet<Long> getKnownFaultyNodes(VV vv){
    TreeSet<Long> nodeSet = new TreeSet<Long>();
    TreeSet<Long> faultyNodeSet = new TreeSet<Long>();
    VVIterator vvi = vv.getIterator();
    while(vvi.hasMoreElements()){
      Long n = vvi.getNext().getIDint();
      if(nodeSet.contains(n)){
        faultyNodeSet.add(n);
      }else{
        nodeSet.add(n);
      }
    }
    return faultyNodeSet;

  }

  public synchronized Checkpoint generateNewCheckpoint(HashedVV omitVV, int newEpochCount, HashedVV hcvv) throws Exception{
    if(newEpochCount < this.epochCount){
      throw new Exception("invalid epoch count" + newEpochCount + " current epoch" + epochCount);
    }
    if(this.getBranchID().getIDint() == 0){
      System.out.println();
    }
    HashedVV newOmitVV = remapHashedDependencyVV(omitVV);
    HashedVV closureOmitVV = this.getClosure(newOmitVV);
    HashedVV cvv = this.getClosure(this.remapHashedDependencyVV(hcvv));
    //    assert closureOmitVV.equals(this.getHashedDependencyVV(logs.getDeepClosure(closureOmitVV).cloneMinVVOmit(cvv))): "\n" + closureOmitVV + "\n" + logs.getDeepClosure(closureOmitVV).cloneMinVVOmit(cvv) + "\n" + omitVV;
    TreeSet<SimPreciseInv> unverifiableWrites = new TreeSet<SimPreciseInv>();
    CounterVV endVV = new CounterVV(closureOmitVV);

    TreeMap<AcceptStamp, TreeSet<Byte>> flagMap = new TreeMap<AcceptStamp, TreeSet<Byte>>();
    HashMap<NodeId, SimPreciseInv> lastGCWrites = new HashMap<NodeId, SimPreciseInv>();
    AcceptVV prevOmitVV = new AcceptVV();
    if(lastCheckpoint != null){
      prevOmitVV = this.remapHashedDependencyVV((HashedVV)lastCheckpoint.omitVV);
    }

    // all the checkpoints should be totally ordered
    if(shouldFail) assert (lastCheckpoint == null) || (closureOmitVV.includes(getOmitVV()));
    // the following condition will be true for correct nodes in real setup but not necessarilty true for the faulty nodes
    // however, in those cases, search should work
    if(shouldFail) assert getCurrentVV().includes(closureOmitVV);

    TreeSet<Long> faultyNodes = this.getKnownFaultyNodes(newOmitVV);
    TreeSet<BranchID> illegalBranches = logs.findIllegalBranches(newOmitVV.getNodes(), new LinkedList<NodeId>());
    Tuple<AcceptVV, LinkedList<SimPreciseInv>> extendedOmitVVAndMap = this.getExtendedOmitVV(newOmitVV, logs.getBaseVV(omitVV), faultyNodes,illegalBranches);

    HashMap<NodeId, NodeId> branchMap = logs.getAcceptableBranchMap(newOmitVV);

    LinkedList<SimPreciseInv> newBranchInvals = extendedOmitVVAndMap.getValue();
    Iterator<SimPreciseInv> iter = newBranchInvals.iterator();
    while(iter.hasNext()){
      SimPreciseInv sspi = iter.next();
      if(!cvv.includes(sspi.getAcceptStamp())){
        iter.remove();
      }
    }
    branchMap.putAll(this.generateHashMap(newBranchInvals));
    AcceptVV extendedOmitVV = extendedOmitVVAndMap.getKey().cloneMaxVV(closureOmitVV).cloneMinVVOmit(cvv);

    TreeSet<SimPreciseInv> acceptableBranches = new TreeSet<SimPreciseInv>();

    CounterVV faultyOmitVV = new CounterVV(extendedOmitVVAndMap.getKey().cloneMinVVOmit(cvv));
    CounterVV faultyEndVV = new CounterVV();//extendedOmitVVAndMap.getKey().cloneMinVVOmit(cvv));
    CounterVV newBranchVV = new CounterVV();
    for(SimPreciseInv sspi: newBranchInvals){
      faultyEndVV.addMaxAS(sspi.getAcceptStamp());
      faultyOmitVV.addMaxAS(sspi.getAcceptStamp());//because we'll include these in acceptableBranches so we don't want to include them in unverifiableWrites
      newBranchVV.addMaxAS(sspi.getAcceptStamp());
    }

    AcceptVV curVV = this.getCurrentVV().cloneMinVVOmit(cvv);
    VVIterator vvi = curVV.getIterator();
    while(vvi.hasMoreElements()){
      NodeId n = vvi.getNext();
      assert n != null;
      if(faultyNodes.contains(n.getIDint())){
        if(!closureOmitVV.containsNodeId(n)){
          faultyEndVV.addMaxAS(new AcceptStamp(curVV.getStampByIteratorToken(n), n));
          assert logs.get(n).size()>0;
          faultyOmitVV.addMaxAS(logs.get(n).get(0).getAcceptStamp());
          newBranchVV.addMaxAS(logs.get(n).get(0).getAcceptStamp());
        }else{
          if(closureOmitVV.getStampByServer(n) < curVV.getStampByServer(n)){
            faultyEndVV.addMaxAS(new AcceptStamp(curVV.getStampByIteratorToken(n), n));
          }
          faultyOmitVV.addMaxAS(new AcceptStamp(closureOmitVV.getStampByServer(n), n));
          if(newOmitVV.containsNodeId(n)){ // so that only the last branches are added..not the previous ones
            newBranchVV.addMaxAS(new AcceptStamp(newOmitVV.getStampByServer(n), n));
          }
        }
      }
    }

    // set acceptable branches
    TreeSet<NodeId> minNodeId = logs.getMinimalRoots(newBranchVV.getNodes());
    vvi = newBranchVV.getIterator();
    while(vvi.hasMoreElements()){
      NodeId n = vvi.getNext();
      if(minNodeId.contains(n) && !closureOmitVV.containsNodeId(n)){
        Entry<Boolean, Integer> search = logs.get(n).binarySearch(newBranchVV.getStampByIteratorToken(n));
        assert search.getKey();
        SimPreciseInv spi  = (SimPreciseInv)logs.get(n).get(search.getValue());
        spi = spi.cloneWithNewNodeId(spi.getNodeId());
        acceptableBranches.add(spi);
        flagMap.put(spi.getAcceptStamp(), (TreeSet<Byte>)((SimPreciseInv)spi).getFlags().clone());
        flagMap.get(spi.getAcceptStamp()).add(Flags.InconsistentWrite);
        spi.getFlags().clear();
      }
    }

    endVV.addMaxVV(faultyEndVV);
    extendedOmitVV= extendedOmitVV.cloneMinVV(new AcceptVV(faultyOmitVV));

    TreeSet<SimPreciseInv> invals = new TreeSet<SimPreciseInv>();
    // start from the last checkpoint and collect a sorted list of all writes. 
    // Then apply these writes in the lamport clock order.
    // This gives us the objectStore
    // Continue going until you have covered all the writes
    VVIterator vvii = cvv.getIterator();
    while(vvii.hasMoreElements()){
      BranchID n = (BranchID)vvii.getNext();
      long ts;
      if(extendedOmitVV.containsNodeId(n)){
        ts = extendedOmitVV.getStampByIteratorToken(n);
      }else{
        ts = -1;
      }
      NodeLog log = this.getLog(n);
      for(PreciseInv pi:log){
        if(!cvv.includes(pi.getAcceptStamp())){
          //          assert false:"\n"+cvv + "\n" + pi + "\n"+this.getCurrentVV();
          break;
        }else{
          SimPreciseInv spi = (SimPreciseInv)pi;
          if(ts < spi.getAcceptStamp().getLocalClock()){
            // if for any dvv component, dvv is older than omitVV, then add the write
            VVIterator vvii1 = spi.dvv.getIterator();
            while(vvii1.hasMoreElements()){
              BranchID n1 = (BranchID)vvii1.getNext();
              if(extendedOmitVV.containsNodeId(n1) && (extendedOmitVV.getStampByIteratorToken(n1) > spi.dvv.getStampByIteratorToken(n1))){
                endVV.advanceTimestamps(spi.getAcceptStamp());
                break;
              }
            }
          }else{
            if(ts == spi.getAcceptStamp().getLocalClock() && newOmitVV.containsNodeId(n)){
              lastGCWrites.put(n, ((SimPreciseInv)spi).cloneWithNewNodeId(spi.getNodeId()));
              ((SimPreciseInv)lastGCWrites.get(n)).getFlags().clear();
            }
            if(closureOmitVV.includes(spi.getAcceptStamp()) && !prevOmitVV.includes(spi.getAcceptStamp())){
              invals.add(spi);
            }
          }
        }
      }

      endVV = new CounterVV(logs.getDeepClosure(endVV).cloneMinVVOmit(cvv));

      for(PreciseInv pi: log){
        SimPreciseInv spi = (SimPreciseInv)pi;
        if(!extendedOmitVV.includes(spi.getAcceptStamp()) && endVV.includes(spi.getAcceptStamp())){
          SimPreciseInv sspi = ((SimPreciseInv)spi).cloneWithNewNodeId(spi.getNodeId());
          sspi.getFlags().clear();
          unverifiableWrites.add(sspi);
          if(!sspi.getFlags().isEmpty() || (illegalBranches.contains(sspi.getNodeId()) && !faultyNodes.contains(sspi.getNodeId().getIDint()))){
            flagMap.put(pi.getAcceptStamp(), (TreeSet)((SimPreciseInv)pi).getFlags().clone());
            if(illegalBranches.contains(sspi.getNodeId()) && !faultyNodes.contains(sspi.getNodeId().getIDint())){
              flagMap.get(pi.getAcceptStamp()).add(Flags.InconsistentWrite);
            }
          }
        }
      }

      //      if(shouldFail) assert !(newOmitVV.containsNodeId(n) ^ lastGCWrites.containsKey(n)): newOmitVV + " lastGCWrites " + lastGCWrites + " n " + n;
    }

    StoreInterface objectStore = null;

    //copy store if lastCheckpoint is not null
    if(this.lastCheckpoint != null){
      objectStore = this.lastCheckpoint.objectStore.clone();
    }else{
      objectStore = new Store();
      objectStore.setIgnoreSet(store.getIgnoreSet());
    }

    //sanity check code
    //    CounterVV testClosureOmitVV = new CounterVV(closureOmitVV);
    //    for(AcceptStamp as: closureOmitVV.getAllStamps()){
    //      testClosureOmitVV.addMaxVV(logs.getDeepClosure(as));
    //    }
    //    assert testClosureOmitVV.equals(closureOmitVV):"\n"+testClosureOmitVV+"\n"+closureOmitVV;
    //    
    //    System.out.println("invals:" + invals);
    for(SimPreciseInv spi: invals){
      this.storeApply(objectStore, spi);
    }
    //    System.out.println("Checkpoint creation succeeded");
    assert new Hash(newOmitVV).equals(new Hash(omitVV)):"\n"+Arrays.toString(newOmitVV.obj2Bytes()) +"\n" + Arrays.toString(omitVV.obj2Bytes());
    Checkpoint chkPt = new Checkpoint(newOmitVV, (Store)objectStore, unverifiableWrites, lastGCWrites, flagMap, newEpochCount, this.getHashedDependencyVV(logs.getMinimalVV(endVV)), acceptableBranches);
    return chkPt.applyBranchMap(branchMap);
  }


  private HashMap<NodeId, NodeId> generateHashMap(LinkedList<SimPreciseInv> invals){
    HashMap<NodeId, NodeId> branchMap = new HashMap<NodeId, NodeId>();

    for(SimPreciseInv sspi: invals){
      branchMap.put(sspi.getNodeId(), new BranchID(sspi.getNodeId().getIDint(), sspi.getAcceptStamp().getLocalClock(), sspi.generateMyHash()));
    }
    return branchMap;
  }

  /**
   * extends omitVV to include renamed sibling branches (or illegal branches) when exactly one sibling branch was included in the original omitVV. Note that faulty nodes are not included because 
   * they have more than one sibling branches in the original omitVV.
   * @param newOmitVV
   * @return
   */
  private Tuple<AcceptVV, LinkedList<SimPreciseInv>> getExtendedOmitVV(AcceptVV newOmitVV, HashMap<Long, Long> nodeIdTS, TreeSet<Long> faultyNodes, TreeSet<BranchID> illegalBranches){
    CounterVV cvv = new CounterVV();
    LinkedList<SimPreciseInv> invals = new LinkedList<SimPreciseInv>();
    for(NodeId n: illegalBranches){
      if(!faultyNodes.contains(n.getIDint())){
        assert nodeIdTS.containsKey(n.getIDint());
        assert !cvv.containsNodeId(n);
        long ts = nodeIdTS.get(n.getIDint());
        NodeLog nl = logs.get(n);
        Entry<Boolean, Integer> search = nl.binarySearch(ts);
        int omitVVIndex = search.getValue();
        int branchMapIndex = search.getValue();
        SimPreciseInv sspi = null;
        if(search.getKey()){
          omitVVIndex--; // to get the previous element
        }else{
          branchMapIndex++;
        }
        if(omitVVIndex >= 0 && omitVVIndex < nl.size()){ // index+1 is a valid location
          sspi = (SimPreciseInv)nl.get(omitVVIndex);
          cvv.setStampByNodeId(n, sspi.getAcceptStamp().getLocalClock());
        }
        if(branchMapIndex >= 0 && branchMapIndex < nl.size()){ // index+1 is a valid location

          //          if((index+1) < nl.size()){
          //            // the next element will determine the branchID
          //            sspi = (SecureSimPreciseInv)nl.get(index+1);
          sspi = (SimPreciseInv)nl.get(branchMapIndex);
          assert sspi.getAcceptStamp().getLocalClock() >= ts;
          invals.add(sspi);
          //          }
        }
      }
    }

    return new Tuple<AcceptVV,LinkedList<SimPreciseInv>>(new AcceptVV(cvv), invals);
  }

  /**
   * add the flags to the writes in the local state and 
   * return these writes after modifying them according to the branches present in the checkpoint 
   * @param chkPt
   * @param flagMap 
   * @return
   */
  private Tuple<TreeSet<SimPreciseInv>, Collection<ProofOfMisbehavior>> addInconsistencyFlags(Checkpoint chkPt) throws Exception{

    AcceptVV mappedOmitVV = this.remapHashedDependencyVV((HashedVV)chkPt.omitVV, false);

    TreeSet<Hash> acceptableHashes = new TreeSet<Hash>();

    for(NodeId n:chkPt.lastWrite.keySet()){
      acceptableHashes.add(chkPt.lastWrite.get(n).generateMyHash());
    }
    for(SimPreciseInv spi: chkPt.unverifiableWrites){
      acceptableHashes.add(spi.generateMyHash());
    }

    for(SimPreciseInv spi: chkPt.acceptableBranches){
      acceptableHashes.add(spi.generateMyHash());
    }

    // this omitVV contains only the components of chkPt.omitVV that can be locally mapped successfully
    AcceptVV omitVV = logs.getClosure(mappedOmitVV);

    // this branchMap contains mapping only based on components that can be successfully mapped
    HashMap<NodeId, NodeId> branchMap = logs.getAcceptableBranchMap(mappedOmitVV);

    // this omitVV contains all the entries that are locally present mapped and the remaining entries copied from the 
    // chkPt.omitVV so that the illegal branches can be correctly computed
    // the completeOmitVV contains a null branch for branches that can't be mapped
    HashSet<NodeId> unmappedBranches = new HashSet<NodeId>();
    AcceptVV completeOmitVV = chkPt.omitVV.applyBranchMap(branchMap);
    VVIterator vvii = completeOmitVV.getIterator();
    while(vvii.hasMoreElements()){
      NodeId n = vvii.getNext();
      if(!omitVV.containsNodeId(n)){
        unmappedBranches.add(n);
      }
    }
    TreeSet<Long> faultyNodes = this.getKnownFaultyNodes(chkPt.omitVV);
    TreeSet<BranchID> illegalBranches = logs.findIllegalBranches(mappedOmitVV.getNodes(), unmappedBranches);
    Tuple<AcceptVV, LinkedList<SimPreciseInv>> extendeOmitVVAndMap = this.getExtendedOmitVV(mappedOmitVV, logs.getBaseVV(chkPt.omitVV), faultyNodes, illegalBranches);

    branchMap.putAll(this.generateHashMap(extendeOmitVVAndMap.getValue()));

    // generate new POMS and apply them to log
    LinkedList<ProofOfMisbehavior> newPOMs = new LinkedList<ProofOfMisbehavior>();
    for(SimPreciseInv sspi:extendeOmitVVAndMap.getValue()){
      //find matching nodeId in omitVV
      VVIterator vvi = chkPt.omitVV.getIterator();
      while(vvi.hasMoreElements()){
        NodeId n = vvi.getNext();
        if(n.getIDint() == sspi.getNodeId().getIDint()){
          SimPreciseInv concurSSPI = (SimPreciseInv)chkPt.lastWrite.get(n);
          ConcurrentPOM cpom = new ConcurrentPOM(sspi, concurSSPI);
          newPOMs.add(cpom);
          //          logs.applyPOM(cpom, new POMRemap());
        }
      }
    }
    AcceptVV extendedOmitVV = extendeOmitVVAndMap.getKey().cloneMaxVV(mappedOmitVV).cloneMaxVV(omitVV);

    TreeSet<SimPreciseInv> writes = new TreeSet<SimPreciseInv>();

    VVIterator vvi = getCurrentVV().getIterator();
    while(vvi.hasMoreElements()){
      BranchID n = (BranchID)vvi.getNext();

      long ts;
      if(extendedOmitVV.containsNodeId(n)){
        ts = extendedOmitVV.getStampByIteratorToken(n);
      }else{
        ts = -1;
      }
      NodeLog log = this.getLog(n);
      for(PreciseInv pi:log){
        SimPreciseInv spi = (SimPreciseInv)pi;
        // the <=  condition ensures that hte new components added due to the extendedOmiTVV can be added while the !acceptableHashes.contains ensures 
        // that components already present are not duplicated
        if(ts < spi.getAcceptStamp().getLocalClock() && !acceptableHashes.contains(spi.generateMyHash()) && 
            !omitVV.includes(spi.getAcceptStamp()) && !faultyNodes.contains(spi.getNodeId().getIDint())){
          // if for any dvv component, dvv is older than omitVV, then add the write
          SimPreciseInv newSPI = spi.applyBranchMap(branchMap);
          newSPI.getFlags().clear();
          if(illegalBranches.contains(spi.getNodeId())){
            newSPI.getFlags().add(Flags.InconsistentWrite);
          }
          writes.add(newSPI);
        }
      }
    }

    return new Tuple<TreeSet<SimPreciseInv>, Collection<ProofOfMisbehavior>>(writes, newPOMs);
  }

  synchronized protected LinkedList<SimPreciseInv> applyNewCheckpoint(Checkpoint checkpoint) throws Exception{
    // if omitVV is present in CVV then check if I must the same history (i.e. same checkpoint)- no need to check as a quorum is verifying it
    // otherwise, for any nodes for which my ts is > than checkpoint.omitVV[node] ensure that we have the same hashes
    // if conflict, ensure that POM present for all such nodes, otherwise create one
    // ensure that the checkpoint is attested by n-f nodes


    // Any post-eviction checkpoint should omit the writes by the faulty node and indicate an eviction certificate in the 
    // certificates directory.The eviction certificate carries the last writes from different branches
    // Lazily, the eviction certificate is revoked
    // Rather than keeping this functionality distributed, a trusted node can be assigned this duty of integrating nodes 
    // that were disconnected forever.

    Checkpoint backupChkPt = this.lastCheckpoint;
    Log backupLog = this.logs;
    StoreInterface backupStore = this.store;
    AcceptVV backupLastUsedVV = this.lastUsedCVV;


    try{
      this.lastCheckpoint = checkpoint;
      LinkedList<SimPreciseInv> discardedWrites = new LinkedList<SimPreciseInv>();

      Tuple<TreeSet<SimPreciseInv>, Collection<ProofOfMisbehavior>> inconsInfo = this.addInconsistencyFlags(checkpoint); 
      this.notifyNewForks(inconsInfo.getValue());
      TreeSet<SimPreciseInv> extraInvals = inconsInfo.getKey();

      this.resetState();

      logs = new Log();
      store = new Store();
      store.setIgnoreSet(backupStore.getIgnoreSet());
      storeInitialize(checkpoint.objectStore);

      for(NodeId bid: checkpoint.lastWrite.keySet()){
        LogStatus ls = logApply(checkpoint.lastWrite.get(bid), Log.Apply);
        assert ls.getApplyStatus() == LogStatus.ApplySuccessful:ls;
      }

      for(SimPreciseInv spi: checkpoint.acceptableBranches){
        //        assert spi.getFlags().isEmpty();
        spi.getFlags().clear();
        if(checkpoint.flagMap.containsKey(spi.getAcceptStamp())){
          spi.flags.addAll(checkpoint.flagMap.get(spi.getAcceptStamp()));
        }
        LogStatus ls = logApply(spi, Log.Apply);
        assert ls.getApplyStatus() == LogStatus.ApplySuccessful:ls;
        storeApply(spi);
      }

      if(shouldFail) assert getCurrentVV().includes(checkpoint.omitVV): "\n" + this.getCurrentVV() +"\n" + checkpoint.omitVV;
      logs.applyCheckpoint(checkpoint);

      for(SimPreciseInv spi:checkpoint.unverifiableWrites){
        spi.flags.clear();
        if(checkpoint.flagMap.containsKey(spi.getAcceptStamp())){
          spi.flags.addAll(checkpoint.flagMap.get(spi.getAcceptStamp()));
        }
        LogStatus ls = logApply(spi, Log.ApplyIfFIFOPass);
        assert ls.getApplyStatus() == LogStatus.ApplySuccessful:ls + "\n" + spi + "\n" + logs;
        storeApply(spi);
      }



      // notify about evicted nodes
      try{
        HashSet<Long> nodes = new HashSet<Long>();
        VVIterator vvi = checkpoint.omitVV.getIterator();
        while(vvi.hasMoreElements()){
          NodeId n = vvi.getNext();
          if(nodes.contains(n.getIDint())){
            evictedNodes.add(n.getIDint());
            logs.addEvictedNodes(n);
          }
          nodes.add(n.getIDint());
        }
      }catch(Exception e){
        // TODO Auto-generated catch block
        e.printStackTrace();
        assert false;
      }

      // add extra invals
      for(SimPreciseInv spi: extraInvals){
        // if marked inconsistent then don't apply, otherwise apply
        if(spi.getFlags().isEmpty() || SangminConfig.enableInconsistencyNotification){
          LogStatus ls = this.logApply(spi, Log.ApplyIfAllPass);
          assert ls.getApplyStatus() == LogStatus.ApplySuccessful:ls + "\n" + spi + "\n" + logs;
          this.storeApply(spi);
        }
      }

      if(shouldFail) assert epochCount < checkpoint.epochCount;
      this.epochCount = (this.epochCount > checkpoint.epochCount?epochCount:checkpoint.epochCount);

      // special case to ensure that if we have accepted multiple branches for out state then we treat ourself as evicted so that we can't create any more writes
      for(ProofOfMisbehavior pom: inconsInfo.getValue()){
        ConcurrentPOM cpom = (ConcurrentPOM)pom;
        if(cpom.getFaultyBranch().getNodeId().getIDint() == this.getBranchID().getIDint()){
          this.addEvictedNode(this.getBranchID());
        }
      }

      cleanup();

      return discardedWrites;
    }catch(Exception e){
      this.lastCheckpoint = backupChkPt;
      this.logs = backupLog;
      this.store = backupStore;
      this.lastUsedCVV = backupLastUsedVV;
      throw e;
    }

  }

  /**
   * cleanup the unreferenced bodies
   */
  public void cleanup(){
    //move bodies from previous store to the new store
    if(SangminConfig.separateBodies){
      Iterator<ObjId> iter = store.iterator();
      TreeSet<Hash> referencedBodies = new TreeSet<Hash>();
      while(iter.hasNext()){
        ObjId o = iter.next();
        TreeSet<StoreEntry> ste = store.get(o);
        //PRINCEM: THE FOLLOWING CODE HCANGED SINCE WE SEPARATED BODIES FROM STORE ENTRIES
        // THE EARLIER CODE IS STILL AROUND IN COMMENTED FORMAT
        for(StoreEntry se: ste){
          //PRINCEM earlier code to move bodies over
          //*************
          //for(ObjId o: backupStore.keySet()){
          //for(StoreEntry se: backupStore.get(o)){
          //        if(se.getData() instanceof IrisDataObject){
          //          store.addBody(o, (IrisDataObject)se.getData());
          //        }
          //}

          //}
          //****************
          //PRINCEM: new code
          referencedBodies.add(se.getData().getHash());
        }
      }
      DbTransaction txn = null;
      if(SangminConfig.usePersistentStore){

        txn = pe.getDB().newTransaction();
      }
      bodyStore.retainHashes(txn, referencedBodies);
      if(SangminConfig.usePersistentStore){

        try{
          txn.commit();
        }catch(TxnFailedException e){
          System.err.println("checkpoint txn failed");
          e.printStackTrace();
        }
      }
    }
  }

  /**
   * verify if the given checkpoint is well-formed: i.e. if considering my local state, the checkpoint
   * @param chkPt
   * @return
   */
  synchronized public boolean verifyCheckpoint(Checkpoint chkPt) throws Exception{
    // we need to check the following conditions:
    // 1) All the unverifiable writes are present in my local state
    // 2) endVV* computed as the closure of (the max of the AcceptStamps of the unverifiable writes and omitVV) matches the 
    // endVV of the checkpoint
    // 3) as per my local log, all the writes between omitVV and the endVV* are included in the unverifiable writes

    AcceptVV omitVV = logs.remap((HashedVV)chkPt.omitVV, true);
    CounterVV endVV = new CounterVV(omitVV);
    TreeSet<Hash> hashes = new TreeSet<Hash>();
    for(SimPreciseInv spi: chkPt.unverifiableWrites){
      BranchID b = logs.getBranchID(spi);
      endVV.addMaxAS(new AcceptStamp(spi.getAcceptStamp().getLocalClock(), b));
      hashes.add(((SimPreciseInv)spi).generateMyHash());
    }

    AcceptVV endVVClosure = logs.getDeepClosure(endVV);

    Checkpoint chkPt1 = this.generateNewCheckpoint((HashedVV)chkPt.omitVV, chkPt.epochCount, this.getHashedDependencyVV(endVVClosure));
    if(!chkPt1.omitVV.equals(chkPt.omitVV)){
      return false;
    }else if(!chkPt1.objectStore.equals(chkPt.objectStore)){
      return false;
    }else if(!chkPt1.lastWrite.equals(chkPt.lastWrite)){
      return false;
    }else if(chkPt1.unverifiableWrites.size() != chkPt.unverifiableWrites.size()){
      return false;
    }else{
      for(SimPreciseInv spi: chkPt1.unverifiableWrites){
        if(!hashes.contains(((SimPreciseInv)spi).generateMyHash())){
          return false;
        }
      }
      return true;
    }
  }

  synchronized public AcceptVV getNextLargestVV(AcceptVV omitVV){
    if(omitVV instanceof HashedVV){
      try{
        omitVV = this.remapHashedDependencyVV((HashedVV)omitVV);
      }catch(Exception e){
        // TODO Auto-generated catch block
        e.printStackTrace();
        assert false;
      }
    }
    CounterVV ret = new CounterVV(omitVV);
    VVIterator vvi = omitVV.getIterator();
    while(vvi.hasMoreElements()){
      BranchID bid = (BranchID)vvi.getNext();
      if(shouldFail) assert logs.containsKey(bid);
      for(PreciseInv pi: logs.get(bid)){
        SimPreciseInv spi = (SimPreciseInv)pi;
        if(spi.getAcceptStamp().getLocalClock() >
        omitVV.getStampByIteratorToken(bid)){
          ret.advanceTimestamps(spi.getAcceptStamp());
          break;
        }
      }
    }
    if(omitVV instanceof HashedVV){
      try{
        return this.getHashedDependencyVV(ret);
      }catch(Exception e){
        // TODO Auto-generated catch block
        e.printStackTrace();
        assert false;
      } 
      return null;
    }else{
      return new AcceptVV(ret);
    }
  }

  synchronized public HashSet<ProofOfMisbehavior> getPOMMap(){
    return pomList;
  }

  public boolean isFaulty(Long l){
    // TODO Auto-generated method stub
    return this.faultyNodes.contains(l);
  }

  synchronized public void addEvictedNode(NodeId n){
    evictedNodes.add(n.getIDint());
    faultyNodes.remove(n.getIDint());


    //    LinkedList<ProofOfMisbehavior> removePOMs = new LinkedList<ProofOfMisbehavior>();
    //    for(BranchID bid: pomList.keySet()){
    //      if(bid.contains(n1)){
    //        removeBranches.add(bid);
    //      }
    //    }
    //    
    //    for(BranchID bid: removeBranches){
    //      pomList.remove(bid);
    //    }
  }
  ///-------------------OLD CHECKPOINT CODE----------------------------------------------------------------------------------------------------------------------------
  ///-------------------OLD CHECKPOINT CODE----------------------------------------------------------------------------------------------------------------------------  ///-------------------OLD CHECKPOINT CODE----------------------------------------------------------------------------------------------------------------------------


  /**
   * prune the state to not include these writes and any writes that depend on any of these writes
   * all writes older than omitVV must also be removed
   * @param unacceptableWrites
   * @param omitVV 
   * @param initialState defines the starting state 
   * @return discarded writes
   * 
   */
  synchronized protected LinkedList<SimPreciseInv> rollbackState(HashMap<BranchID,LinkedList<PreciseInv>> unacceptableWrites, 
      StoreInterface _initialState, 
      AcceptVV omitVV, HashMap<NodeId, SimPreciseInv> lastWrite){
    // start at the last given state and reapply all the writes to the store/log except for the ones that are declared unacceptable or the ones that depend on unacceptable writes
    HashSet<AcceptStamp> unacceptableWritesClosure = new HashSet<AcceptStamp>();

    LinkedList<SimPreciseInv> discardedWrites = new LinkedList<SimPreciseInv>();

    boolean verify = omitVV.equals(new AcceptVV()); // if omitVV is same as 0 then everything should be verifiable

    for(BranchID bid: unacceptableWrites.keySet()){
      for(PreciseInv spi: unacceptableWrites.get(bid)){
        unacceptableWritesClosure.add(spi.getAcceptStamp());
      }
    }

    TreeSet<PreciseInv> sortedLog = new TreeSet<PreciseInv>();
    for(NodeId bid: logs.keySet()){
      sortedLog.addAll(logs.get(bid));
    } 

    // reset state
    this.resetState();

    logs = new Log();
    store = new Store();

    // apply the checkpoint

    //copy initial state
    storeInitialize(_initialState);
    indirectlyCoveredVV = new CounterVV(omitVV);

    //  // also add the unverifiable writes included in the checkpoint to the set of writes that need to be applied
    for(NodeId bid: lastWrite.keySet()){
      logApply(lastWrite.get(bid), Log.Apply);
    }
    if(shouldFail) assert getCurrentVV().includes(omitVV);

    // set the lamport clock to the max value
    VVIterator vvi = omitVV.getIterator();
    while(vvi.hasMoreElements()){
      NodeId n = vvi.getNext();
      lamportClock = ((lamportClock < omitVV.getStampByIteratorToken(n))?omitVV.getStampByIteratorToken(n):lamportClock);
    }

    // now advance the state by applying any admissible writes that I have and that were not included in the checkpoint
    for(PreciseInv pi: sortedLog){
      SimPreciseInv spi = (SimPreciseInv)pi;
      if(!omitVV.includes(spi.getAcceptStamp())){
        boolean valid = true;
        //      if(!omitVV.includes(spi.getAcceptStamp()) && !unacceptableWritesClosure.contains(spi.getAcceptStamp())){
        if(!unacceptableWritesClosure.contains(spi.getAcceptStamp())){
          AcceptVV av = new AcceptVV(spi.dvv);
          for(AcceptStamp as: av.getAllStamps()){
            if(unacceptableWritesClosure.contains(as)){
              valid = false;
              discardedWrites.add(spi);
              break;
            }
          }
        }else{
          if(!omitVV.includes(spi.getAcceptStamp())){
            discardedWrites.add(spi);
          }
          valid = false;
        }

        // if valid write
        if(valid){
          // all of these writes were already verifed
          this.logApply(spi, (verify?Log.ApplyIfAllPass:Log.Apply));
          //          this.logApply(spi, (verify?Log.ApplyIfAllPass:Log.ApplyIfFIFOPass));

          this.storeApply(spi);

        }else{
          unacceptableWritesClosure.add(spi.getAcceptStamp());
        }
      }
    }

    return discardedWrites;
  }



  synchronized public Checkpoint generateCheckpoint(AcceptVV newOmitVV, int newEpochCount){

    if(this.useNewCheckpoint){
      try{
        return this.generateNewCheckpoint(this.getHashedDependencyVV(newOmitVV), newEpochCount);
      }catch(Exception e){
        // TODO Auto-generated catch block
        e.printStackTrace();
        assert false;
      }
    }
    StoreInterface objectStore = new Store();
    TreeSet<SimPreciseInv> unverifiableWrites = new TreeSet<SimPreciseInv>();
    HashMap<NodeId, SimPreciseInv> lastGCWrites = new HashMap<NodeId, SimPreciseInv>();
    TreeMap<AcceptStamp, TreeSet<Byte>> flagMap = new TreeMap<AcceptStamp, TreeSet<Byte>>();

    CounterVV endVV = new CounterVV(newOmitVV);

    // all the checkpoints should be totally ordered
    if(shouldFail) assert (lastCheckpoint == null) || (newOmitVV.includes(getOmitVV()));
    // the following condition will be true for correct nodes in real setup but not necessarilty true for the faulty nodes
    // however, in those cases, search should work
    if(shouldFail) assert getCurrentVV().includes(newOmitVV);

    //copy store if lastCheckpoint is not null
    if(this.lastCheckpoint != null){
      Iterator<ObjId> iter = lastCheckpoint.objectStore.iterator();
      while(iter.hasNext()){
        ObjId o = iter.next();
        objectStore.put(o, lastCheckpoint.objectStore.get(o));
      }
    }

    // start from the last checkpoint and collect a sorted list of all writes. 
    // Then apply these writes in the lamport clock order.
    // This gives us the objectStore
    // Continue going until you have covered all the writes
    VVIterator vvi = getCurrentVV().getIterator();
    while(vvi.hasMoreElements()){
      BranchID n = (BranchID)vvi.getNext();
      long ts;
      if(newOmitVV.containsNodeId(n)){
        ts = newOmitVV.getStampByIteratorToken(n);
      }else{
        ts = -1;
      }
      NodeLog log = this.getLog(n);
      //      Vector<SimPreciseInv> unverifiableWriteLog = new Vector<SimPreciseInv>();
      for(PreciseInv pi:log){
        SimPreciseInv spi = (SimPreciseInv)pi;
        if(ts < spi.getAcceptStamp().getLocalClock()){
          // if for any dvv component, dvv is older than omitVV, then add the write
          VVIterator vvii = spi.dvv.getIterator();
          while(vvii.hasMoreElements()){
            BranchID n1 = (BranchID)vvii.getNext();
            if(newOmitVV.containsNodeId(n1) && (newOmitVV.getStampByIteratorToken(n1) > spi.dvv.getStampByIteratorToken(n1))){
              endVV.advanceTimestamps(spi.getAcceptStamp());
              break;
            }
          }
        }else{
          if(ts == spi.getAcceptStamp().getLocalClock()){
            lastGCWrites.put(n, spi);
            flagMap.put(pi.getAcceptStamp(), (TreeSet)((SimPreciseInv)pi).getFlags().clone());
          }
          objectStore.apply(spi);
        }
      }
      for(PreciseInv pi: log){
        SimPreciseInv spi = (SimPreciseInv)pi;
        if(!newOmitVV.includes(spi.getAcceptStamp()) && endVV.includes(spi.getAcceptStamp())){
          unverifiableWrites.add(spi);
          flagMap.put(pi.getAcceptStamp(), (TreeSet<Byte>)((SimPreciseInv)pi).getFlags().clone());
        }
      }

      if(shouldFail) assert !(newOmitVV.containsNodeId(n) ^ lastGCWrites.containsKey(n)): newOmitVV + " lastGCWrites " + lastGCWrites + " n " + n;
    }
    TreeSet<SimPreciseInv> acceptableBranches = new TreeSet<SimPreciseInv>();

    return new Checkpoint(newOmitVV, (Store)objectStore, unverifiableWrites, lastGCWrites, flagMap, newEpochCount, new AcceptVV(endVV), acceptableBranches);
  }

  synchronized protected LinkedList<SimPreciseInv> applyCheckpoint(Checkpoint checkpoint) throws Exception{
    // if omitVV is present in CVV then check if I must the same history (i.e. same checkpoint)- no need to check as a quorum is verifying it
    // otherwise, for any nodes for which my ts is > than checkpoint.omitVV[node] ensure that we have the same hashes
    // if conflict, ensure that POM present for all such nodes, otherwise create one
    // ensure that the checkpoint is attested by n-f nodes


    // Any post-eviction checkpoint should omit the writes by the faulty node and indicate an eviction certificate in the 
    // certificates directory.The eviction certificate carries the last writes from different branches
    // Lazily, the eviction certificate is revoked
    // Rather than keeping this functionality distributed, a trusted node can be assigned this duty of integrating nodes 
    // that were disconnected forever.

    if(this.useNewCheckpoint){
      return this.applyNewCheckpoint(checkpoint);
    }
    this.lastCheckpoint = checkpoint;
    LinkedList<SimPreciseInv> discardedWrites = new LinkedList<SimPreciseInv>();


    if(checkpoint.endVV.includes(getCurrentVV())){
      this.resetState();

      store = new Store();
      logs = new Log();

      storeInitialize(checkpoint.objectStore);

      for(NodeId bid: checkpoint.lastWrite.keySet()){
        logApply(checkpoint.lastWrite.get(bid), Log.Apply);
      }

      if(shouldFail) assert getCurrentVV().includes(checkpoint.omitVV);

      for(SimPreciseInv spi:checkpoint.unverifiableWrites){
        spi.flags.clear();
        if(checkpoint.flagMap.containsKey(spi.getAcceptStamp())){
          spi.flags.addAll(checkpoint.flagMap.get(spi.getAcceptStamp()));
        }
        logApply(spi, Log.ApplyIfFIFOPass);
        storeApply(spi);
      }
    }else{
      // find out the set of writes (not older than omitVV) that have become unverifiable due to this checkpoint and yet 
      // are not included in the set of unverifiable writes for the checkpoint
      // next, rollback state after applying the checkpoint to remove the effect of such writes
      HashMap<BranchID,LinkedList<PreciseInv>> unacceptableWrites = new HashMap<BranchID,LinkedList<PreciseInv>>();
      VVIterator vvi = getCurrentVV().getIterator();
      while(vvi.hasMoreElements()){
        BranchID n = (BranchID)vvi.getNext();
        long ts;
        if(checkpoint.omitVV.containsNodeId(n)){
          ts = checkpoint.omitVV.getStampByIteratorToken(n);
        }else{
          ts = -1;
        }

        NodeLog log = this.getLog(n);
        LinkedList<PreciseInv> unverifiableWriteLog = new LinkedList<PreciseInv>();
        for(PreciseInv pi:log){
          SimPreciseInv spi = (SimPreciseInv)pi;
          if(ts <= spi.getAcceptStamp().getLocalClock()){
            // if for any dvv componenet, dvv is older than omitVV, then add the write unless its present in the acceptableUnverifiableWrites
            VVIterator vvii = spi.dvv.getIterator();
            while(vvii.hasMoreElements()){
              BranchID n1 = (BranchID)vvii.getNext();
              if(checkpoint.omitVV.containsNodeId(n1) && 
                  (checkpoint.omitVV.getStampByIteratorToken(n1) > spi.dvv.getStampByIteratorToken(n1)) && 
                  !checkpoint.unverifiableWrites.contains(spi)){
                unverifiableWriteLog.add(spi);
                break;
              }
            }
          }
        }
        if(unverifiableWriteLog.size() != 0){
          unacceptableWrites.put(n, unverifiableWriteLog);
        }
      }

      discardedWrites = rollbackState(unacceptableWrites, checkpoint.objectStore, checkpoint.omitVV, checkpoint.lastWrite);


      //    for(BranchID bid: checkpoint.lastWrite.keySet()){
      //    logApply(checkpoint.lastWrite.get(bid), false);
      //    }

      discardedWrites.removeAll(checkpoint.lastWrite.values());

      // apply unverifiable writes in a checkpoint
      TreeSet<SimPreciseInv> writes = new TreeSet<SimPreciseInv>();
      writes.addAll(checkpoint.unverifiableWrites);
      discardedWrites.removeAll(checkpoint.unverifiableWrites);
      for(SimPreciseInv spi: writes){
        if(!getCurrentVV().includes(spi.getAcceptStamp())){
          this.logApply(spi, Log.ApplyIfFIFOPass);
          this.storeApply(spi);
        }
      }
      if(!discardedWrites.isEmpty()){
        System.out.println(discardedWrites);
      }

    }
    if(shouldFail) assert epochCount < checkpoint.epochCount;
    this.epochCount = (this.epochCount > checkpoint.epochCount?epochCount:checkpoint.epochCount);

    logs.applyCheckpoint(checkpoint);

    return discardedWrites;


  }

  //-------------------------------------------------SIMULATOR SYNC CODE----------------------------------------------------------------

  /**
   * parent->child mapping
   * @param vv
   * @return
   */
  public HashMap<BranchID, TreeSet<BranchID>> findForks(AcceptVV vv){
    if(shouldFail) assert SangminConfig.useSimulatorId;

    HashMap<BranchID, TreeSet<BranchID>> branches = new HashMap<BranchID, TreeSet<BranchID>>();
    VVIterator vvi = vv.getIterator();
    while(vvi.hasMoreElements()){
      SimBranchID bid = (SimBranchID)vvi.getNext();
      while(bid.parent != null){
        if(branches.containsKey(bid.parent)){
          if(!branches.get(bid.parent).contains(bid)){
            branches.get(bid.parent).add(bid);
          }
        }else{
          branches.put(bid.parent, new TreeSet<BranchID>());
          branches.get(bid.parent).add(bid);
        }
        bid = bid.parent;
      }
    }
    return branches;
  }

  /**
   * notifies n of the new forks that it didn't know and this of the new forks that it didn't know
   * @param n
   * @return
   */
  synchronized protected void branchSync(Node n){
    if(SangminConfig.useSimulatorId){
      HashMap<BranchID, TreeSet<BranchID>> myBK = findForks(getCurrentVV());
      HashMap<BranchID, TreeSet<BranchID>> senderBK = findForks(n.getCurrentVV());
      HashMap<BranchID, TreeSet<BranchID>> mergedBK = new HashMap<BranchID, TreeSet<BranchID>>();

      HashMap<BranchID, ProofOfMisbehavior> senderNewPOM = new HashMap<BranchID, ProofOfMisbehavior>();
      HashMap<BranchID, ProofOfMisbehavior> myNewPOM = new HashMap<BranchID, ProofOfMisbehavior>();

      //    merge
      for(BranchID bid: myBK.keySet()){
        mergedBK.put(bid, new TreeSet<BranchID>());
        mergedBK.get(bid).addAll(myBK.get(bid));
      }

      for(BranchID bid: senderBK.keySet()){
        if(!mergedBK.containsKey(bid)){
          mergedBK.put(bid, new TreeSet<BranchID>());
        }
        mergedBK.get(bid).addAll(senderBK.get(bid));
      }


      //    detect forks
      for(BranchID bid: mergedBK.keySet()){
        if(mergedBK.get(bid).size() > 1){
          //        fork found
          if(!myBK.containsKey(bid) || myBK.get(bid).size() <= 1){
            myNewPOM.put(bid, new ProofOfMisbehavior(DummySimPreciseInv.getSecureSimPreciseInv(new AcceptStamp(-1, bid))));
          }
          if(!senderBK.containsKey(bid) || senderBK.get(bid).size() <= 1){
            senderNewPOM.put(bid, new ProofOfMisbehavior(DummySimPreciseInv.getSecureSimPreciseInv(new AcceptStamp(-1, bid))));
          }
        }
      }

      try{
        if(senderNewPOM.size() > 0){
          n.notifyNewForks(senderNewPOM.values());
        }
        if(myNewPOM.size() > 0){
          notifyNewForks(myNewPOM.values());
        }
      }catch(Exception e){
        e.printStackTrace();
        assert false;
      }
    }

  }

  /**
   * sync from Node n to update local state
   * @param n
   */
  synchronized public SyncStatus sync(Node n) throws Exception{
    if(epochSync(n)){
      if(SangminConfig.useSimulatorId){
        n.branchSync(this);
      }
      if(this.getCurrentVV().includes(n.getCurrentVV())){
        return new SyncStatus(SyncStatus.SyncSuccessful);
      }else{
        // find new stuff

        Collection<SimPreciseInv> syncWrites = n.findNewWrites(this.getCurrentVV());
        return logSync(n.getBranchID(), 
            syncWrites);
      }
    }else{
      n.notifyAdvanceEpoch(this, epochCount);
      return new SyncStatus(SyncStatus.SenderOldEpoch);
    }
  }

  /**
   * returns true if the sender and receiver are in the same epoch, possibly after the sync
   * false otherwise
   * @param n
   * @return
   */
  synchronized protected boolean epochSync(Node n) throws Exception{
    if(n.epochCount == epochCount){
      if(shouldFail) assert (n.lastCheckpoint == null?(lastCheckpoint == null):lastCheckpoint.equals(n.lastCheckpoint));
      return true;
    }
    else if(n.epochCount > epochCount){
      // first do a checkpoint sync
      if(shouldFail) assert (n.lastCheckpoint != null);

      //apply the last checkpoint from n and advance epoch
      this.applyCheckpoint(n.lastCheckpoint);
      return true;
    }else{
      return false;
    }
  }

}
