package code.simulator;

import java.io.ByteArrayInputStream;
import java.io.ByteArrayOutputStream;
import java.io.Externalizable;
import java.io.IOException;
import java.io.InputStream;
import java.io.ObjectInput;
import java.io.ObjectInputStream;
import java.io.ObjectOutput;
import java.io.ObjectOutputStream;
import java.io.OutputStream;
import java.io.Serializable;
import java.lang.reflect.AccessibleObject;
import java.lang.reflect.Field;

import java.util.Arrays;
import java.util.HashMap;
import java.util.Hashtable;
import java.util.Iterator;
import java.util.LinkedList;
import java.util.Random;
import java.util.TreeSet;
import java.util.HashSet;

import code.AcceptStamp;
import code.Config;
import code.Env;
import code.NodeId;
import code.ObjId;
import code.ObjInvalTarget;
import code.PreciseInv;
import code.branchDetecting.BranchID;
import code.security.SangminConfig;
import code.security.ahs.DependencyVV;
import code.serialization.IrisInputStream;
import code.serialization.IrisObjectInputStream;
import code.serialization.IrisObjectOutputStream;
import code.serialization.IrisOutputStream;
import code.serialization.IrisSerializable;
import code.serialization.SerializationHelper;
import code.simulator.log.Log;
import code.simulator.protocolFilters.AgreementGCProtocol;

/**
 * a sync packet indicates one the following conditions
 * 1) Sender is in newer epoch and therefore, he is including newer certificates
 * 2) sender is in older epoch and therefore he is aborting
 * 3) List of invals and POMs
 * 
 * PS: last writes are separated from the regular writes to ensure that the backoff strategy eventually succeeds. Without this separation, 
 * it is possible that the sync never succeeds with some node. For instance if nodes A and B are trying to sync and they are forked by F.
 * Now it is possible that B first checks a write from A that depends on a fork write from F that B thinks it has. So, the inclusion check 
 * will fail. However, backoff won't help as never will it be the case that compatibility check fails. By separating the last writes we ensure 
 * that A and B will first notice the fork and then start accepting writes.
 * 
 * @author princem
 *
 */
public class SyncPacket implements Externalizable, IrisSerializable{

  public final static byte SenderLowerEpoch = 0;
  public final static byte SenderHigherEpoch = 1;
  public final static byte SenderSameEpoch = 2;
  public final static byte SyncDenied = 3;
  public static final byte NoCachedState = 4;

  //either null (case 0), list of certificates (case 1), or set of invals (case 3) and a set of POMs
  private TreeSet<SimPreciseInv> writes;

  private TreeSet<SimPreciseInv> lastWrites;

  private HashSet<ProofOfMisbehavior> poms;
  private LinkedList<Certificate> certificates;

  private LinkedList<IrisDataObject> bodies; 
  private boolean hasBodies = false;

  private byte pktType;
  private int epoch;
  private BranchID sender;
  private boolean optimized;

  public SyncPacket(){
    // for serialization
    this.pktType = -1;
    this.epoch = -1;
    this.sender = null;
    bodies = null;
    optimized = false;
  }

  public SyncPacket(byte pktType, int epoch, BranchID sender, boolean optimized){
    assert pktType == SenderLowerEpoch || pktType == SenderHigherEpoch || pktType == SenderSameEpoch || pktType == SyncDenied || pktType == NoCachedState;
    this.pktType = pktType;
    this.epoch = epoch;
    this.sender = sender;
    bodies = null;
    this.optimized = optimized;
  }

  public TreeSet<SimPreciseInv> getWrites(){
    assert pktType == SenderSameEpoch:pktType;
    return writes;
  }
  
  public void setWritesAndPOM(TreeSet<SimPreciseInv> writes, HashSet<ProofOfMisbehavior> poms){
    assert pktType == SenderSameEpoch;
    this.writes = writes;
    this.poms = poms;

  }

  public void omitWrites(){
    if(writes.isEmpty())
      return;

    Random r = new Random(1);
    TreeSet<SimPreciseInv> omitset = new TreeSet<SimPreciseInv>();
    TreeSet<SimPreciseInv> replaceset = new TreeSet<SimPreciseInv>();
    Iterator<SimPreciseInv> iter = writes.iterator();
    //int nwrites = writes.size();
    Hashtable<NodeId, SimPreciseInv> lastwrites = new Hashtable<NodeId, SimPreciseInv>();
    while(iter.hasNext()){
      SecureSimPreciseInv spi = (SecureSimPreciseInv)iter.next();
      lastwrites.put(spi.getNodeId(), spi);

      // To implement the real worst case, we modify the data of the writes
      // and set the embargoed bit to distinguish the modified writes when it is applied on an oracle
      SecureSimPreciseInv clone = new SecureSimPreciseInv(spi.getObjId(), spi.getAcceptStamp(), spi.dvv,
          new IrisDataObject(spi.data.getHash()), spi.signature, true, spi.flags, spi.objectHashes, spi.epoch);
      lastwrites.put(spi.getNodeId(), clone);
    }

    writes = new TreeSet<SimPreciseInv>();
    writes.addAll(lastwrites.values());

    //    int nwrites2 = writes.size();
    //    if(nwrites > nwrites2){
    //      System.out.println("Dropping "  + (nwrites-nwrites2) + " writes");
    //    }

    //    while(iter.hasNext()){
    //      SecureSimPreciseInv spi = (SecureSimPreciseInv)iter.next();
    //      if(r.nextFloat()<0.5){
    //        omitset.add(spi);
    //        SecureSimPreciseInv clone = new SecureSimPreciseInv(spi.getObjId(), spi.getAcceptStamp(), spi.dvv,
    //            spi.databytes, spi.hash, spi.signature, true);
    //        clone.databytes[0] = (byte) (clone.databytes[0] ^ 0xff);
    //        replaceset.add(clone);
    //      }
    //    }
    //    if(writes.removeAll(omitset)){
    //      System.out.println("Omitting " + omitset.size() +" writes");
    //    }
    //    writes.addAll(replaceset);
  }

  public BranchID getSender(){
    assert !optimized;
    return sender;
  }

  public int getEpoch(){
    assert !optimized;
    return epoch;
  }

  public byte getType(){
    return pktType;
  }

  public HashSet<ProofOfMisbehavior> getPoms(){
    if(optimized){
      return new HashSet<ProofOfMisbehavior>();
    }
    assert pktType == SenderSameEpoch;
    return poms;
  }

  public LinkedList<Certificate> getCertificates(){
    if(optimized){
      return new LinkedList<Certificate>();
    }
    assert pktType == SenderHigherEpoch;
    return certificates;
  }
  public void setCertificates(LinkedList<Certificate> certificates){
    assert pktType == SenderHigherEpoch;
    this.certificates = certificates;
  }

  public String toString(){
    String ret = "epoch : " + epoch +"\npktType : "+pktType + "\n";
    if(this.lastWrites != null){
      ret += "LastWrites : " + lastWrites.toString() + "\n";
    } 
    if(writes != null){
      ret += "Writes : " + writes.toString() + "\n";
    } 
    if(poms != null){
      ret += "POMs"+poms.toString() +"\n";
    }
    if(certificates != null){
      ret += "Certificates : " + certificates.toString() + "\n";
    }
    if(bodies != null){
      ret += "bodies : " + bodies.toString() + "\n";
    }
    return ret;
  }

  public boolean equals(Object obj){

    if(!(obj instanceof SyncPacket)){
      return false;
    }
    SyncPacket spkt = (SyncPacket)obj;
    if(!this.sender.equals(spkt.sender)) return false;
    if(this.epoch!=spkt.epoch) return false;
    if(this.pktType != spkt.pktType) return false;

    if(this.certificates==null && spkt.certificates!=null) return false;
    if(spkt.certificates==null && this.certificates!=null) return false;
    if(certificates!= null && !certificates.equals(spkt.certificates)) return false;

    if(this.poms==null && spkt.poms!=null) return false;
    if(spkt.poms==null && this.poms!=null) return false;
    if(poms !=null && !poms.equals(spkt.poms)) return false;

    if(this.writes==null && spkt.writes!=null) return false;
    if(spkt.writes==null && this.writes!=null) return false;
    if(writes != null && !writes.equals(spkt.writes)) return false;

    return true;
  }

  public TreeSet<SimPreciseInv> getLastWrites(){
    if(optimized){
      return new TreeSet<SimPreciseInv>();
    }
    return lastWrites;
  }

  /**
   * 
   * @param lastWrites
   */
  public void setLastWrites(TreeSet<SimPreciseInv> lastWrites){
    assert pktType == SyncPacket.SenderSameEpoch;
    this.lastWrites = lastWrites;
  }

  /**
   * @return the bodies
   */
  public LinkedList<IrisDataObject> getBodies(){
    return bodies;
  }

  /**
   * @param bodies the bodies to set
   */
  public void setBodies(LinkedList<IrisDataObject> bodies){
    assert pktType == SyncPacket.SenderSameEpoch || pktType == SyncPacket.SenderHigherEpoch;
    this.bodies = bodies;
    this.hasBodies = true;
  }

  public void readExternal(ObjectInput in) throws IOException,
  ClassNotFoundException{
    this.readExternalCorePacket(in);
    this.readExternalBodies(in);
  }

  public void writeExternal(ObjectOutput out) throws IOException{
    this.writeExternalCorePacket(out);
    this.writeExternalBodies(out);  
  }

  public void readExternalCorePacket(ObjectInput in) throws IOException,
  ClassNotFoundException{
    pktType = SyncPacket.SenderSameEpoch;
    epoch = -1;
    sender = null;
    lastWrites = null;
    certificates = null;
    bodies = null;
    poms = null;
    writes = null;
    boolean optimize = in.readBoolean();

    if(!optimize){
      pktType = in.readByte();
      epoch = in.readInt();
      sender = new BranchID(-1); 
      sender.readExternal(in);
    }

    switch(pktType)
    {
    
    case SyncPacket.SenderHigherEpoch:
    {
      // write certificates
      short s = in.readShort();
      certificates = new LinkedList<Certificate>();
      for(int i = 0; i < s; i++){
        certificates.add((Certificate)in.readObject());
      }

      // write bodies
//      s = in.readShort();
//      if(s != 0){
//        _bodies = new LinkedList<IrisDataObject>();
//        for(int i = 0; i < s; i++){
//          short size = in.readShort();
//          byte[] buf = new byte[size];
//          in.readFully(buf);
//          _bodies.add(new IrisDataObject(buf));
//        }
//      }
      break;
    }
    case SyncPacket.SenderLowerEpoch:
      break;
    case SyncPacket.SenderSameEpoch:
    {
      short s;
      if(!optimize){
        //poms
        s = in.readShort();
        poms = new HashSet<ProofOfMisbehavior>();
        for(int i = 0; i < s; i++){
          poms.add((ProofOfMisbehavior)in.readObject());
        }

        // lastWrites
        s = in.readShort();
        lastWrites = new TreeSet<SimPreciseInv>();
        for(int i = 0; i < s; i++){
          SimPreciseInv sspi;
          if(IrisNode.enableSecurity){
            sspi = new SecureSimPreciseInv();
          }else{
            sspi = new SimPreciseInv();
          }
          sspi.readExternal(in);
          lastWrites.add(sspi);

        }     
      }
      //writes
      s = in.readShort();
      writes = new TreeSet<SimPreciseInv>();
      for(int i = 0; i < s; i++){
        SimPreciseInv sspi;
        if(IrisNode.enableSecurity){
          sspi = new SecureSimPreciseInv();
        }else{
          sspi = new SimPreciseInv();
        }
        sspi.readExternal(in);
        writes.add(sspi);

      }     

      break;
    }
    case SyncPacket.SyncDenied:
      break;
    default:
      assert false;
    }

  }

  public void readExternalBodies(ObjectInput in) throws IOException,
  ClassNotFoundException{
    if(pktType == SyncPacket.SenderSameEpoch || pktType == SyncPacket.SenderHigherEpoch){
      // read bodies
      short  s = in.readShort();
      if(s != 0){
        assert bodies == null;
        LinkedList<IrisDataObject>    _bodies = new LinkedList<IrisDataObject>();
        for(int i = 0; i < s; i++){
          int size = in.readInt();
          byte[] buf = new byte[size];
          in.readFully(buf);
          _bodies.add(new IrisDataObject(buf));
        }
        this.bodies = _bodies;
      }
    }
  }

  public void writeExternalBodies(ObjectOutput out) throws IOException{
    // write bodies
    
    if(pktType == SyncPacket.SenderSameEpoch || pktType == SyncPacket.SenderHigherEpoch){
      if(bodies != null){

        out.writeShort(bodies.size());
        for(IrisDataObject ido: bodies){
          byte[] buf = ido.getUnsafeBytes();
          out.writeInt(buf.length);
          out.write(buf);
        }
      }else{
        out.writeShort(0);
      }
    }
    
  }

  public void writeExternalCorePacket(ObjectOutput out) throws IOException{
    out.writeBoolean(optimized);
    if(!optimized){
      out.writeByte(this.pktType);
      out.writeInt(epoch);
      sender.writeExternal(out);
    }else{
      assert pktType == SyncPacket.SenderSameEpoch;
    }
    switch(pktType)
    {
    case SyncPacket.SenderHigherEpoch:
      // write certificates
      out.writeInt(certificates.size());
      for(Certificate c: certificates){
        out.writeObject(c);
      }


      break;
    case SyncPacket.SenderLowerEpoch:
      break;
    case SyncPacket.SenderSameEpoch:
      if(!optimized){
        //poms
        out.writeShort(poms.size());
        for(ProofOfMisbehavior pom: poms){
          out.writeObject(pom);
        }

        // lastWrites
        out.writeShort(lastWrites.size());
        for(SimPreciseInv spi: lastWrites){
          spi.writeExternal(out);
        }
      }

      //writes
      out.writeShort(writes.size());
      for(SimPreciseInv spi: writes){
        spi.writeExternal(out);
      }

      break;
    case SyncPacket.SyncDenied:
      break;
    default:
      assert false;
    }

  }
  
  public static void testStr() throws IOException{
    DependencyVV dvv = new DependencyVV();
    dvv.put(new BranchID(5), 3);
    String data = "data";
    byte[] bytes = new byte[Hash.Length];
    for(int i=0; i<Hash.Length; i++){
      bytes[i] = 'a';
    }

    Hash hash = new Hash(bytes, false);
    HashMap<NodeId, Hash> hashes = new HashMap<NodeId, Hash>();
    hashes.put(new BranchID(5), hash);
    HashedVV hdvv = new HashedVV(dvv, hashes);

    TreeSet<Hash> objHashes = new TreeSet<Hash>();
    objHashes.add(Hash.NullHash);
    objHashes.add(hash);

    IrisObject irisData = new IrisDataObject(data.getBytes());
    SecureSimPreciseInv sspi = new SecureSimPreciseInv(new ObjId(AgreementGCProtocol.gcProposalDir+"test"), new AcceptStamp(10,new BranchID(10)), hdvv, irisData , 0, objHashes);

    // test same epoch packet
    SyncPacket sp = new SyncPacket(SyncPacket.SenderSameEpoch, 0, new BranchID(6), false);
    TreeSet<SimPreciseInv> lastWrites = new TreeSet<SimPreciseInv>();
    lastWrites.add(sspi);
    sp.setLastWrites(lastWrites);

    HashSet<ProofOfMisbehavior> poms = new HashSet<ProofOfMisbehavior>();
    poms.add(new ProofOfMisbehavior(sspi, sspi, sspi));

    TreeSet<SimPreciseInv> writes = new TreeSet<SimPreciseInv>();
    writes.add(sspi);

    sp.setWritesAndPOM(writes, poms);
    byte[] body = new byte[1000*1000*10];
    for(int i = 0; i < body.length; i++){
      body[i] = (byte)(i%128);
    }
    IrisDataObject ido = new IrisDataObject("d".getBytes());
    LinkedList<IrisDataObject> bodies = new LinkedList<IrisDataObject>();
    bodies.add(ido);
    sp.setBodies(bodies);

    ByteArrayOutputStream bs;
    IrisOutputStream oos;

    bs = new ByteArrayOutputStream();
    oos = new IrisOutputStream(bs);
    System.out.println("Serialise it");
    //        oos.writeObject(sp);
    //    sp.writeExternal(oos);
    sp.writeToStream(oos, false);
    sp.writeBodyToStream(oos, false);

    oos.flush();
    System.out.println(sp);
    //bs.toByteArray();
    System.out.println("read it " + bs.toByteArray().length);
    System.out.println("read it " + Arrays.toString(bs.toByteArray()));
    SyncPacket sp2 = null;
    IrisInputStream ois = new IrisInputStream(new ByteArrayInputStream(bs.toByteArray()));
    sp2 = new SyncPacket();
    sp2.readFromStream(ois, false);
    sp2.readBodyFromStream(ois, false);

    if(sp.equals(sp2)){
      System.out.println("Success");
    } else {
      System.out.println("Fail : \n" + sspi + "\n" + sp2);
      assert false;
    }

    SyncPacket sp1 = new SyncPacket(SyncPacket.SenderSameEpoch, 0, new BranchID(6), true);
    sp1.setLastWrites(lastWrites);
    sp1.setWritesAndPOM(writes, poms);
    sp1.setBodies(bodies);
    bs = new ByteArrayOutputStream();
    oos = new IrisOutputStream(bs);
    System.out.println("Serialise it");
    //        oos.writeObject(sp);
    //    sp.writeExternal(oos);
    sp1.writeToStream(oos, true);
    sp1.writeBodyToStream(oos, true);

    oos.flush();
    //bs.toByteArray();
    System.out.println("read it " + bs.toByteArray().length);
    ois = new IrisInputStream(new ByteArrayInputStream(bs.toByteArray()));
    sp2 = new SyncPacket();
    sp2.readFromStream(ois, true);
    sp2.readBodyFromStream(ois, true);


    if(sp1.getWrites().size() != sp2.getWrites().size() || (!sp1.bodies.equals(sp2.bodies))){
      System.out.println("Fail : \n" + sp1.getWrites() + "\n" + sp2.getWrites());
      System.out.println("Fail : \n" + sp1.bodies + "\n" + sp2.bodies);
      assert false;
    } else {
      System.out.println("Success");
    }
    
//    sp2 = new SyncPacket();
//    sp2.readFromStream(ois, false);
//    sp2.readBodyFromStream(ois, false);
//System.out.println(sp2);
//    if(sp.equals(sp2)){
//      System.out.println("Success");
//    } else {
//      System.out.println("Fail : \n" + sspi + "\n" + sp2);
//      assert false;
//    }

    SyncPacket sp3 = new SyncPacket(SyncPacket.NoCachedState, 0, new BranchID(6), true);
    bs = new ByteArrayOutputStream();
    oos = new IrisOutputStream(bs);
    System.out.println("Serialise it");
    //        oos.writeObject(sp);
    //    sp.writeExternal(oos);
    sp3.writeToStream(oos, true);
    sp3.writeBodyToStream(oos, true);

    oos.flush();
    //bs.toByteArray();
    System.out.println("read it " + bs.toByteArray().length);
    ois = new IrisInputStream(new ByteArrayInputStream(bs.toByteArray()));
    sp2 = new SyncPacket();
    sp2.readFromStream(ois, true);
    sp2.readBodyFromStream(ois, true);

    if(sp3.getType() != sp2.getType()){
      assert false;
    } else {
      System.out.println("Success");
    }
  }
  
  public static void testExt() throws IOException{
    DependencyVV dvv = new DependencyVV();
    dvv.put(new BranchID(5), 3);
    String data = "data";
    byte[] bytes = new byte[Hash.Length];
    for(int i=0; i<Hash.Length; i++){
      bytes[i] = 'a';
    }

    Hash hash = new Hash(bytes, false);
    HashMap<NodeId, Hash> hashes = new HashMap<NodeId, Hash>();
    hashes.put(new BranchID(5), hash);
    HashedVV hdvv = new HashedVV(dvv, hashes);

    TreeSet<Hash> objHashes = new TreeSet<Hash>();
    objHashes.add(Hash.NullHash);
    objHashes.add(hash);

    IrisObject irisData = new IrisDataObject(data.getBytes());
    SecureSimPreciseInv sspi = new SecureSimPreciseInv(new ObjId(AgreementGCProtocol.gcProposalDir+"test"), new AcceptStamp(10,new BranchID(10)), hdvv, irisData , 0, objHashes);

    // test same epoch packet
    SyncPacket sp = new SyncPacket(SyncPacket.SenderSameEpoch, 0, new BranchID(6), false);
    TreeSet<SimPreciseInv> lastWrites = new TreeSet<SimPreciseInv>();
    lastWrites.add(sspi);
    sp.setLastWrites(lastWrites);

    HashSet<ProofOfMisbehavior> poms = new HashSet<ProofOfMisbehavior>();
    poms.add(new ProofOfMisbehavior(sspi, sspi, sspi));

    TreeSet<SimPreciseInv> writes = new TreeSet<SimPreciseInv>();
    writes.add(sspi);

    sp.setWritesAndPOM(writes, poms);
    byte[] body = new byte[1000*1000*10];
    for(int i = 0; i < body.length; i++){
      body[i] = (byte)(i%128);
    }
    IrisDataObject ido = new IrisDataObject(body);
    LinkedList<IrisDataObject> bodies = new LinkedList<IrisDataObject>();
    bodies.add(ido);
    sp.setBodies(bodies);

    ByteArrayOutputStream bs;
    ObjectOutputStream oos;

    bs = new ByteArrayOutputStream();
    oos = new ObjectOutputStream(bs);
    System.out.println("Serialise it");
    //        oos.writeObject(sp);
    //    sp.writeExternal(oos);
    sp.writeExternalCorePacket(oos);
    sp.writeExternalBodies(oos);

    oos.flush();
    //bs.toByteArray();
    System.out.println("read it " + bs.toByteArray().length);
    SyncPacket sp2 = null;
    ObjectInputStream ois = new ObjectInputStream(new ByteArrayInputStream(bs.toByteArray()));
    try{
      sp2 = new SyncPacket();
      sp2.readExternalCorePacket(ois);
      sp2.readExternalBodies(ois);
      //      sp2 = (SyncPacket)ois.readObject();
    }catch(ClassNotFoundException e){
      e.printStackTrace();
    }

    if(sp.equals(sp2)){
      System.out.println("Success");
    } else {
      System.out.println("Fail : \n" + sspi + "\n" + sp2);
    }

    SyncPacket sp1 = new SyncPacket(SyncPacket.SenderSameEpoch, 0, new BranchID(6), true);
    sp1.setLastWrites(lastWrites);
    sp1.setWritesAndPOM(writes, poms);
    sp1.setBodies(bodies);
    bs = new ByteArrayOutputStream();
    oos = new ObjectOutputStream(bs);
    System.out.println("Serialise it");
    //        oos.writeObject(sp);
    //    sp.writeExternal(oos);
    sp1.writeExternalCorePacket(oos);
    sp1.writeExternalBodies(oos);

    oos.flush();
    //bs.toByteArray();
    System.out.println("read it " + bs.toByteArray().length);
    ois = new ObjectInputStream(new ByteArrayInputStream(bs.toByteArray()));
    try{
      sp2 = new SyncPacket();
      sp2.readExternal(ois);
      //      sp2 = (SyncPacket)ois.readObject();
    }catch(ClassNotFoundException e){
      e.printStackTrace();
    }

    if(sp1.getWrites().size() != sp2.getWrites().size() || (!sp1.bodies.equals(sp2.bodies))){
      System.out.println("Fail : \n" + sp1.getWrites() + "\n" + sp2.getWrites());
      System.out.println("Fail : \n" + sp1.bodies + "\n" + sp2.bodies);
    } else {
      System.out.println("Success");
    }
    
    
  }

  public static void main(String[] args) throws IOException{

    Node.useSignature = true;
    Node.enableSecurity = true;
    
    Config.readKeys();
//    testExt();
    testStr();
  }

  public boolean isOptimized(){
    return optimized;
  }

  public void readFromStream(IrisInputStream is, boolean optim) throws IOException{
    epoch = -1;
    sender = null;
    lastWrites = null;
    this.certificates = null;
    bodies = null;
    poms = null;
    writes = null;
    if(SangminConfig.sanityCheck){
      byte b = is.readByte();
      assert b == (byte)0xCB: b + " " + (byte)0xCB;
    }
    
    
    byte booleanB =is.readByte();
    optimized = (booleanB&0x01)==0x01; //this.optimized = is.readBoolean();
    hasBodies = (booleanB&0x02)!=0; //this.optimized = is.readBoolean();
    pktType = (byte)(booleanB>>2);// is.readByte();
    
    if(pktType == SyncPacket.NoCachedState){
      return;
    }
    if(!optimized){
      epoch = is.readByte();
      sender = new BranchID(-1); 
      sender.readFromStream(is, optim);
    }

    switch(pktType)
    {
    case SyncPacket.SenderHigherEpoch:
    {
      // write certificates
      short s = is.readShort();
      certificates = new LinkedList<Certificate>();
      for(int i = 0; i < s; i++){
        certificates.add(Certificate.readFromStream(is, optimized));
      }

      // write bodies
      s = is.readShort();
      if(s != 0){
        bodies = new LinkedList<IrisDataObject>();
        for(int i = 0; i < s; i++){
          short size = is.readShort();
          byte[] buf = new byte[size];
          is.readFully(buf);
          bodies.add(new IrisDataObject(buf));
        }
      }
      break;
    }
    case SyncPacket.SenderLowerEpoch:
      break;
    case SyncPacket.SenderSameEpoch:
    {
      short s;
      if(!optimized){
        //poms
        s = is.readShort();
        poms = new HashSet<ProofOfMisbehavior>();
        for(int i = 0; i < s; i++){
          poms.add(ProofOfMisbehavior.readFromStream(is, optimized));
        }

        // lastWrites
        s = is.readShort();
        lastWrites = new TreeSet<SimPreciseInv>();
        for(int i = 0; i < s; i++){
          SimPreciseInv sspi;
          if(IrisNode.enableSecurity){
            sspi = new SecureSimPreciseInv();
          }else{
            sspi = new SimPreciseInv();
          }
          sspi.readFromStream(is, optimized);
          lastWrites.add(sspi);

        }     
      }
      //writes
      s = is.readShort();
      writes = new TreeSet<SimPreciseInv>();
      for(int i = 0; i < s; i++){
        SimPreciseInv sspi;
        if(IrisNode.enableSecurity){
          sspi = new SecureSimPreciseInv();
        }else{
          sspi = new SimPreciseInv();
        }
        sspi.readFromStream(is, optimized);
        writes.add(sspi);

      }     

      break;
    }
    case SyncPacket.SyncDenied:
      break;
    default:
      assert false;
    }
    if(SangminConfig.sanityCheck){
      byte b = is.readByte();
      assert b == (byte)0xCC: b + " " + (byte)0xCC;
    }
//    this.readBodyFromStream(is, optim);
  }
  
  public void readBodyFromStream(IrisInputStream is, boolean optim) throws IOException{
    if(SangminConfig.sanityCheck){
      byte b = is.readByte();
      assert b == (byte)0xDB: b + " " + (byte)0xDB;
    }
    if(pktType == SyncPacket.SenderSameEpoch || pktType == SyncPacket.SenderHigherEpoch){
      // read bodies
      short s = is.readShort();
      if(s != 0){
        assert bodies == null;
        bodies = new LinkedList<IrisDataObject>();
        int[] sizeArr = new int[s];
        for(int i = 0; i < s; i++){
          int size = is.readInt();
          sizeArr[i] = size;
        }
        for(int i = 0; i < s; i++){
          byte[] buf = new byte[sizeArr[i]];
          is.readFully(buf);
          bodies.add(new IrisDataObject(buf));
        }
      }
    }
    if(SangminConfig.sanityCheck){
      byte b = is.readByte();
      assert b == (byte)0xDC: b + " " + (byte)0xDC;
    }
  }

  public void writeToStream(IrisOutputStream out, boolean optim) throws IOException{
    if(SangminConfig.sanityCheck){
      out.writeByte(0xCB);
     }

    byte booleanB = 0x00;
    booleanB |= (optimized?0x01:0x00); //out.writeBoolean(optimized);
    booleanB |= (hasBodies?0x02:0x00); //out.writeBoolean(optimized);
    booleanB |= pktType<<2;// out.writeByte(pktType);
    out.writeByte(booleanB);
    
    if(pktType == SyncPacket.NoCachedState){
      return;
    }
    if(!optimized){
      out.writeByte(epoch);
      sender.writeToStream(out, optimized);
    }else{
      assert pktType == SyncPacket.SenderSameEpoch ||  pktType == SyncPacket.NoCachedState;;
    }
    switch(pktType)
    {
    case SyncPacket.SenderHigherEpoch:
      // write certificates
      out.writeInt(certificates.size());
      for(Certificate c: certificates){
        Certificate.writeToStream(c, out, optimized);
      }


      break;
    case SyncPacket.SenderLowerEpoch:
      break;
    case SyncPacket.SenderSameEpoch:
      if(!optimized){
        //poms
        out.writeShort(poms.size());
        for(ProofOfMisbehavior pom: poms){
          ProofOfMisbehavior.writeToStream(pom, out, optimized);
        }

        // lastWrites
        out.writeShort(lastWrites.size());
        for(SimPreciseInv spi: lastWrites){
          spi.writeToStream(out, optimized);
        }
        
//        Env.logWrite("UNOPTIMIZED SYNC:" + out.getBytes());
      }
//      Env.logWrite("SYNC METADATA:" + out.getBytes());


      //writes
      out.writeShort(writes.size());
      for(SimPreciseInv spi: writes){
        spi.writeToStream(out, optimized);
      }

      break;
    case SyncPacket.SyncDenied:
      break;
    default:
      assert false;
    }
//    Env.logWrite("SYNC PACKET SIZE" + out.getBytes());
//    this.writeBodyToStream(out, optim);
    if(SangminConfig.sanityCheck){
      out.writeByte(0xCC);
     }
  }
  
  public void writeBodyToStream(IrisOutputStream out, boolean optimize) throws IOException{
    if(SangminConfig.sanityCheck){
      out.writeByte(0xDB);
     }
    if(pktType == SyncPacket.SenderSameEpoch || pktType == SyncPacket.SenderHigherEpoch){
      if(bodies != null){

        out.writeShort(bodies.size());
        for(IrisDataObject ido: bodies){
          byte[] buf = ido.getUnsafeBytes();
          out.writeInt(buf.length);
        }
        for(IrisDataObject ido: bodies){
          byte[] buf = ido.getUnsafeBytes();
          out.write(buf);
        }
      }else{
        out.writeShort(0);
      }
    }
    if(SangminConfig.sanityCheck){
      out.writeByte(0xDC);
     }
  }
  
  public int getBodySize(){
    if(hasBodies){
      int size = 2;
      if(bodies != null){
        for(IrisDataObject ido: bodies){
          size += 4 + ido.getUnsafeBytes().length;
        }
      }
      return size;
    }else{
      return 0;
    }
  }

  /**
   * @return the hasBodies
   */
  public boolean isHasBodies(){
    return hasBodies;
  }

}
