package code.simulator;

import java.io.ByteArrayInputStream;
import java.io.ByteArrayOutputStream;
import java.io.Externalizable;
import java.io.IOException;
import java.io.ObjectInput;
import java.io.ObjectInputStream;
import java.io.ObjectOutput;
import java.io.ObjectOutputStream;
import java.io.Serializable;
import java.lang.reflect.AccessibleObject;
import java.lang.reflect.Field;
import java.util.*;

import code.Env;
import code.AcceptStamp;
import code.AcceptVV;
import code.CounterVV;
import code.NodeId;
import code.ObjId;
import code.ObjInvalTarget;
import code.PreciseInv;
import code.VV;
import code.VVIterator;
import code.branchDetecting.BranchID;
import code.security.SangminConfig;
import code.security.ahs.DependencyVV;
import code.serialization.IrisInputStream;
import code.serialization.IrisObjectInputStream;
import code.serialization.IrisObjectOutputStream;
import code.serialization.IrisOutputStream;
import code.serialization.IrisSerializable;
import code.serialization.SerializationHelper;
import code.simulator.log.Log;
import code.simulator.protocolFilters.AgreementGCProtocol;
import code.simulator.store.Store;

public class SyncRequest implements Externalizable, IrisSerializable{

  public final static byte NoBodies = 0;
  public final static byte UncommittedLastBodies = 1;
  
  private byte bodyReq;
  private ObjId objId;
  private HashSet<ProofOfMisbehavior> poms;
  private int epoch;
  private AcceptVV startVV;
  private boolean incrementalVV;
//  final private long sender;
  private boolean optimized;
  
  public SyncRequest(AcceptVV startVV, int epoch, HashSet<ProofOfMisbehavior> poms, long sender, byte allBodies, boolean opti){
    this.startVV = startVV;
    this.epoch = epoch;
    this.poms = poms;
//    this.sender =sender;
    this.bodyReq = allBodies;
    objId = null;
    optimized = opti;
  }
  
  public SyncRequest(AcceptVV startVV, int epoch, HashSet<ProofOfMisbehavior> poms, long sender, byte allBodies, ObjId readMiss, boolean opti){
    this.startVV = startVV;
    this.epoch = epoch;
    this.poms = poms;
//    this.sender =sender;
    this.bodyReq = allBodies;
    objId = readMiss;
    optimized = opti;
  }

  public SyncRequest(){
    startVV = null;
    bodyReq = -1;
    objId = null;
    poms = null;
    epoch = -1;
//    sender = -1;
    optimized = false;
  }

  public int getEpoch(){
    return epoch;
  }

  public HashSet<ProofOfMisbehavior> getPoms(){
    return poms;
  }

  public AcceptVV getStartVV(){
    return startVV;
  }

  /* (non-Javadoc)
   * @see java.lang.Object#toString()
   */
  @Override
  public String toString(){
    return "SyncRequest [bodyReq=" + bodyReq + ", epoch=" + epoch
        + ", poms=" + poms + ", optimized=" + optimized + ", startVV=" + startVV
        + ((this.objId != null)?"objId=" + objId:"")+"]";
  }

//  public long getSender(){
//    // TODO Auto-generated method stub
//    return sender;
//  }

  /**
   * @return the allBodies
   */
  public byte getBodyReq(){
    return bodyReq;
  }

  /**
   * @return the objId
   */
  public ObjId getObjId(){
    return objId;
  }

  public void readExternal(ObjectInput in) 
  throws IOException, ClassNotFoundException{

    boolean _optimized = in.readBoolean();
    byte _bodyReq = in.readByte();
    int _epoch = -1;
    if(!_optimized){
     _epoch = in.readInt(); 
    }
    AcceptVV _startVV = SerializationHelper.readVV(in);
    
    long _sender = -1;
    HashSet<ProofOfMisbehavior> _poms = new HashSet<ProofOfMisbehavior>();
    if(!_optimized){
      short size = in.readShort();
      for(int i = 0; i < size; i++){
        ProofOfMisbehavior pom = (ProofOfMisbehavior)in.readObject();
        _poms.add(pom);
      }
//      _sender = in.readLong();
    }
    
    String str = SerializationHelper.readShortString(in);
    
    ObjId _oid = null;
    if(str != null){
      _oid = new ObjId(str);
    }
    
    
    Field[] f = new Field[6];

    try{

      f[0] = SyncRequest.class.getDeclaredField("bodyReq");
      f[1] = SyncRequest.class.getDeclaredField("objId");
      f[2] = SyncRequest.class.getDeclaredField("poms");
      f[3] = SyncRequest.class.getDeclaredField("epoch");
      f[4] = SyncRequest.class.getDeclaredField("startVV");
//      f[6] = SyncRequest.class.getDeclaredField("sender");
      f[5] = SyncRequest.class.getDeclaredField("optimized");

    }catch(NoSuchFieldException ne){
      System.err.println(ne.toString());
      ne.printStackTrace();
      System.exit(-1);
    }
    try{
      AccessibleObject.setAccessible(f, true);
    } catch (SecurityException se){
      System.err.println(se.toString());
      se.printStackTrace();
      System.exit(-1);
    }
    try{
      f[0].set(this, _bodyReq);
      f[1].set(this, _oid);
      f[2].set(this, _poms);
      f[3].set(this, _epoch);
      f[4].set(this, _startVV);
//      f[6].set(this, _sender);
      f[5].set(this, _optimized);
      
    }catch(IllegalArgumentException ie){
      System.err.println(ie.toString());
      ie.printStackTrace();
      System.exit(-1);
    }catch(IllegalAccessException iae){
      System.err.println(iae.toString());
      iae.printStackTrace();
      System.exit(-1);
    }

    try{
      AccessibleObject.setAccessible(f, false);
    } catch (SecurityException se){
      System.err.println(se.toString());
      se.printStackTrace();
      System.exit(-1);
    }

  }

  public void writeExternal(ObjectOutput out) throws IOException{
    out.writeBoolean(optimized);
    out.writeByte(bodyReq);
    if(!optimized){
      out.writeInt(epoch);
    }
    SerializationHelper.writeVV(out, startVV);
    
    if(!optimized){
      out.writeShort(poms.size());
      for(ProofOfMisbehavior pom: poms){
        out.writeObject(pom);
      }
//      out.writeLong(sender);
    }
    if(this.objId != null){
      SerializationHelper.writeShortString(objId.getPath(), out);
    }else{
      SerializationHelper.writeShortString(null, out);
    }
  }

  public static void testExt() throws IOException {
    DependencyVV dvv = new DependencyVV();
    dvv.put(new BranchID(5), 3);
    String data = "data";
    
    SyncRequest sr = new SyncRequest(new AcceptVV(dvv), 2, new HashSet<ProofOfMisbehavior>(), 0, SangminConfig.BodyConfig, new ObjId(data), false);
    ByteArrayOutputStream bs;
    ObjectOutputStream oos;

    bs = new ByteArrayOutputStream();
    oos = new IrisObjectOutputStream(bs);
    System.out.println("Serialise it");
//    oos.writeObject(sspi);
    sr.writeExternal(oos);

    oos.flush();
    //bs.toByteArray();
    System.out.println("read it " + bs.toByteArray().length);
    SyncRequest sr1 = null;
    ObjectInputStream ois = new IrisObjectInputStream(new ByteArrayInputStream(bs.toByteArray()));
    try{
//      sspi2 = (SecureSimPreciseInv)ois.readObject();
      sr1 = new SyncRequest();
      sr1.readExternal(ois);
    }catch(ClassNotFoundException e){
      e.printStackTrace();
    }

    if(sr.equals(sr1)){
      System.out.println("Success");
    } else {
      System.out.println("Fail : \n" + sr + "\n" + sr1);

    }
  }
  
  public static void testStr() throws IOException {
    DependencyVV dvv = new DependencyVV();
    dvv.put(new BranchID(5), 3);
    String data = "data";
    
    SyncRequest sr = new SyncRequest(new AcceptVV(dvv), 2, new HashSet<ProofOfMisbehavior>(), 0, SangminConfig.BodyConfig, new ObjId(data), false);
    ByteArrayOutputStream bs;
    IrisOutputStream oos;

    bs = new ByteArrayOutputStream();
    oos = new IrisOutputStream(bs);
    System.out.println("Serialise it " + sr);
//    oos.writeObject(sspi);
    sr.writeToStream(oos, false);

    oos.flush();
    //bs.toByteArray();
    System.out.println("read it " + bs.toByteArray().length);
    SyncRequest sr1 = null;
    IrisInputStream ois = new IrisInputStream(new ByteArrayInputStream(bs.toByteArray()));
    System.out.println(Arrays.toString(bs.toByteArray()));
    sr1 = new SyncRequest();
    sr1.readFromStream(ois, false);
    
    if(sr.equals(sr1)){
      System.out.println("Success");
    } else {
      System.out.println("Fail : \n" + sr + "\n" + sr1);

    }
    
  }
  public static void main(String[] args) throws IOException{

    testExt();
    testStr();
    
//    sspi.replaceIrisObject(new IrisHashObject(sspi.getData().getHash()));
//    assert sspi.generateMyHash().equals(sspi2.generateMyHash());

  }

  /* (non-Javadoc)
   * @see java.lang.Object#equals(java.lang.Object)
   */
  @Override
  public boolean equals(Object obj){
    if(this == obj){
      return true;
    }
    if(obj == null){
      return false;
    }
    if(!(obj instanceof SyncRequest)){
      return false;
    }
    SyncRequest other = (SyncRequest) obj;
    if(bodyReq != other.bodyReq){
      return false;
    }
    if(epoch != other.epoch){
      return false;
    }
    if(objId == null){
      if(other.objId != null){
        return false;
      }
    }else if(!objId.equals(other.objId)){
      return false;
    }
    if(poms == null){
      if(other.poms != null){
        return false;
      }
    }else if(!poms.equals(other.poms)){
      return false;
    }
    if(optimized != other.optimized){
      return false;
    }
    if(startVV == null){
      if(other.startVV != null){
        return false;
      }
    }else if(!startVV.equals(other.startVV)){
      return false;
    }
    return true;
  }

  public boolean isOptimized(){
    return optimized;
  }

  public void readFromStream(IrisInputStream in, boolean optim)
      throws IOException{
    if(SangminConfig.enableSanityCheck){
    byte header = in.readByte();
    assert header == (byte)0xAA: header + " "  +0xAA;
    }
    byte booleanB = in.readByte();
    
    optimized = (booleanB&1)!=0 ; //in.readBoolean();
    incrementalVV = (booleanB&2)!=0; //in.readBoolean();
    boolean containsObj = (booleanB&4)!=0; //in.readBoolean();
    bodyReq = (byte)(booleanB>>3); //in.readByte();
    epoch = -1;
    if(!optimized){
     epoch = in.readByte(); 
    }
    startVV = SerializationHelper.readVV(in);
    
//    sender = -1;
    poms = new HashSet<ProofOfMisbehavior>();
    if(!optimized){
      short size = in.readShort();
      for(int i = 0; i < size; i++){
        ProofOfMisbehavior pom = ProofOfMisbehavior.readFromStream(in, optimized);
        poms.add(pom);
      }
//      _sender = in.readLong();
    }
    
    this.objId = null;
    if(containsObj){
      String str = in.readShortString();

      if(str != null){
        objId = new ObjId(str);
      }
    }
    if(SangminConfig.enableSanityCheck){
      byte b= in.readByte();
      assert b== (byte)0xBB: b + "  " + (byte)0xBB;
    }

  }

  public void writeToStream(IrisOutputStream out, boolean optim)
      throws IOException{
    if(SangminConfig.enableSanityCheck){
    out.writeByte(0xAA);
    }
    byte booleanB = 0x00;
    booleanB |= (optimized?0x01:0x00); //out.writeBoolean(optimized);
    booleanB |= (incrementalVV?0x02:0x00); // out.writeBoolean(incrementalVV);
    booleanB |= (objId != null?0x04:0x00);  //out.writeBoolean(objId != null);
    booleanB |= bodyReq<<3;// out.writeByte(bodyReq);
    out.writeByte(booleanB);
    if(!optimized){
      out.writeByte(epoch);
    }
    SerializationHelper.writeVV(out, startVV);
    
    if(!optimized){
      out.writeShort(poms.size());
      for(ProofOfMisbehavior pom: poms){
        ProofOfMisbehavior.writeToStream(pom, out, optim);
      }
//      out.writeLong(sender);
    }
    if(objId != null){
      out.writeShortString(objId.getPath());
    }
    if(SangminConfig.enableSanityCheck){
    out.writeByte(0xBB);
    }

  }

  /**
   * @return the incrementalVV
   */
  public boolean isIncrementalVV(){
    return incrementalVV;
  }

  /**
   * @param incrementalVV the incrementalVV to set
   */
  public void setIncrementalVV(VV vv){
    this.incrementalVV = true;
    // remove the vv from incrementatlVV
    AcceptVV tmp =  startVV.project(AcceptVV.decrementAll(startVV.getRealDiff(AcceptVV.makeVVAllNegatives().cloneMaxVV(vv))).dropNegatives())
;    
    //    Env.printDebug("SetIncrementalVVDebgu: original startVV: " + startVV + " previouslySentVV: " + vv + " new startVV:" + tmp);
    startVV = tmp;
  }

  public AcceptVV resetIncrementalVV(VV cachedVV){
    assert this.incrementalVV;
    this.incrementalVV = false;
    this.startVV = startVV.cloneMaxVV(cachedVV);
    return startVV;
  }

  
}
