package code.simulator.agreement;

import java.util.HashMap;
import java.util.LinkedList;
import java.util.Random;
import java.util.Timer;
import java.util.TimerTask;
import java.util.TreeSet;
import java.util.Map.Entry;

import code.NodeId;
import code.simulator.Hash;

/**
 * this class handles all the processing for one consensus instance; 
 * i.e. receiving and processing messages, broadcasting votes, and deciding values.
 * 
 * PS: unlike the protocol described in the paper, this protocol relies on the fact that the communication is reliable and 
 * FIFO. Furthermore, after a node decides, it broadcasts the certificate and ceases to store the votes for all previous rounds and 
 * stops participating in consecutive votes.
 * 
 * @author princem
 *
 */
public class AgreementInstance implements AgreementInterface{
  public static Timer timer = new Timer();
  public static final boolean debug = true;
  public static Random r = new Random();
  
  final AgreementMonitor agreementMonitor;
  final InstanceID instanceID;
  
  private int round;
  private HashMap<Vote, TreeSet<Long>> voteMap;
  private HashMap<Long, TreeSet<Vote>> votes;
  private boolean timedOut;
  private LinkedList<Entry<Long, Vote>> futureVotes;
  private boolean decided = false;
  
  public AgreementInstance(InstanceID instanceID, AgreementMonitor am){
    this.agreementMonitor = am;
    this.instanceID = instanceID;
    votes = new HashMap<Long, TreeSet<Vote>>();
    voteMap = new HashMap<Vote, TreeSet<Long>>();
    futureVotes = new LinkedList<Entry<Long, Vote>>();
    advanceRound(0);
  }

  private void advanceRound(int r){
    this.round = r;
    votes.clear();
    voteMap.clear();
    timedOut = false;    
    
//    TimerTask task = new TimerTask(){
//      public void run(){
//        if(print)if(debug)System.out.println("Instance timer fired at " + agreementMonitor.getParams().getMyID());
//        timedOut = true; 
//        checkRoundComplete();
//      }
//    };
//    AgreementInstance.timer.schedule(task, agreementMonitor.getTimeOut());
        if(!futureVotes.isEmpty()){
      LinkedList<Entry<Long, Vote>> previousVotes = futureVotes;
      futureVotes = new LinkedList<Entry<Long, Vote>>();
      for(Entry<Long, Vote> e: previousVotes){
        this.receiveVote(e.getKey(), e.getValue());
      }
    }
  }
  
  synchronized public void notifyTimeout(){
    timedOut = true;
    checkRoundComplete();
  }
  /**
   * process a new vote
   */
  synchronized public void receiveVote(long sender, Vote v){
    assert v.getInstanceID().equals(instanceID);
    if(decided){
      return;
    }
    if(debug)System.out.println(this.agreementMonitor.getParams().getMyID() + " received vote " + v + "from " + sender);
//    if(this.agreementMonitor.getParams().getMyID() == 0){
//      if(debug)System.out.println("-----------------------");
//      if(debug)System.out.println("sender " + sender + " sends" + v);
//      if(debug)System.out.println(this);
//      if(debug)System.out.println("-----------------------");
//    }
    if(v.getRound() == round){
      if(!votes.containsKey(sender) || !votes.get(sender).contains(v)){
        if(!votes.containsKey(sender)){
          votes.put(sender, new TreeSet<Vote>());
        }
        votes.get(sender).add(v);
        if(!voteMap.containsKey(v)){
          voteMap.put(v, new TreeSet<Long>());
        }
        if(!voteMap.get(v).contains(sender)){
          voteMap.get(v).add(sender);
        }
        if(timedOut || votes.size() == agreementMonitor.getParams().getNumNodes()){
          checkRoundComplete();
        }
        if(!v.isBottom() && sender != agreementMonitor.getParams().getMyID() && !votes.containsKey(agreementMonitor.getParams().getMyID())){
          Vote myVote = agreementMonitor.getVoteFactory().createLocalVote(v);
          this.receiveVote(agreementMonitor.getParams().getMyID(), myVote);
          agreementMonitor.getChannel().broadcast(myVote);
        }
      }
      
    }else if(v.getRound() > round){
      futureVotes.add(new Tuple(sender, v));
    }else{
      assert v.getRound() < round;
    }
  }
  
  synchronized private DecisionProof createDecisionProof(Vote v){
    assert decided;
    // collect matching votes for v and notify consensus controller
    LinkedList<Vote> decidedVotes = new LinkedList<Vote>();
    for(Long l: voteMap.get(v)){
      for(Vote vote: votes.get(l)){
        if(vote.equals(v)){
          decidedVotes.add(vote);
        }
      }
    }
    assert decidedVotes.size() >= agreementMonitor.getParams().strongQuorum();
   DecisionProof dp = agreementMonitor.getVoteFactory().getDecisionProof(instanceID, decidedVotes);
   return dp;
  }
  
  private void decided(Vote v) {
    if(debug)System.out.println(this.agreementMonitor.getParams().getMyID() + " DECIDED " + v );
    if(this.agreementMonitor.getParams().getMyID() == 0 && v.isBottom()){
      if(debug)System.out.println(this.agreementMonitor.getParams().getMyID() + " DECIDED " + v );
    }
   DecisionProof dp = this.createDecisionProof(v);
   this.agreementMonitor.getChannel().broadcast(dp);
   this.agreementMonitor.notifyDecision(this.instanceID, dp);
  }
  
  synchronized public void receiveDecisionProof(DecisionProof dp){
    decided = true;
  }

  private void checkRoundComplete(){
    Vote decidedVote = null;
    synchronized(this){
      if(decided){
        return;
      }
      if(votes.size() >= agreementMonitor.getParams().strongQuorum()){
        LinkedList<Vote> majorityVotes = new LinkedList<Vote>();
        for(Vote v: this.voteMap.keySet()){
          if(voteMap.get(v).size() >= agreementMonitor.getParams().strongQuorum()){
            decidedVote = v;
            decided = true;
            break;
          }else if(voteMap.get(v).size() >= agreementMonitor.getParams().weakQuorum()){
            majorityVotes.add(v);
          }
        }
        
        if(!decided){
          Vote myNextRoundVote;
          if(majorityVotes.size() == 0){
            myNextRoundVote = agreementMonitor.getVoteFactory().getBottomVote(round, instanceID);
          }else {
            int index = AgreementInstance.r.nextInt(majorityVotes.size());
            myNextRoundVote = agreementMonitor.getVoteFactory().createNextRoundVote(majorityVotes.get(index));
          }
          advanceRound(round+1);
          this.receiveVote(agreementMonitor.getParams().getMyID(), myNextRoundVote);
          this.agreementMonitor.getChannel().broadcast(myNextRoundVote);
        }
      }else{
        if(debug)System.out.println(this.agreementMonitor.getParams().getMyID() + " " + this.getInstanceID() + " NOT DECIDING because " + votes.size() + "" + votes.keySet() + "votes received..." + agreementMonitor.getParams().strongQuorum() + " required " );
      }
    }
    if(decidedVote != null){
      this.decided(decidedVote);
    }
      
  }

  public InstanceID getInstanceID(){
    return instanceID;
  }

  public boolean isTimedOut(){
    return timedOut;
  }

  public int getRound(){
    return round;
  }

  @Override
  public String toString(){
    return "AgreementInstance [futureVotes=" + futureVotes + ", \nround=" + round
        + ", timedOut=" + timedOut + ", \nvoteMap=" + voteMap + ", \nvotes" + votes + "]";
  }
 
}
