package code.simulator.agreement.unit;

import java.util.Collection;
import java.util.LinkedList;

import code.simulator.Hash;
import code.simulator.agreement.AgreementInstance;
import code.simulator.agreement.AgreementParams;
import code.simulator.agreement.InstanceID;
import code.simulator.agreement.ValueDecisionProof;
import code.simulator.checkpoint.unit.SimulatorCheckpointUnit;
import code.simulator.unit.SimulatorUnit;
import junit.framework.Test;
import junit.framework.TestCase;
import junit.framework.TestSuite;
import junit.textui.TestRunner;

public class AgreementInstanceUnit extends TestCase{
  public static final String TEST_ALL_TEST_TYPE = "UNIT";

  /**
   * Basic constructor - called by the test runners.
   * TBD: Update constructor name to match class
   */
  public AgreementInstanceUnit (final String s) {
    super (s);
  }

  /*
   * Fixtures are run before and after each test case
   * to set up environment in which tests must run.
   */
  protected void setUp() throws Exception{
    super.setUp();
    SimulatorCheckpointUnit.makePractiConfig(0, 100);

  }

  /*
   * "new TestSuite(Class c)" constructs a test suite
   * containg every method whose name begins with "test"
   * 
   * TBD: update class name
   */
  public static Test suite(){
    TestSuite suite = new TestSuite(AgreementInstanceUnit.class);
    return suite;
  }

  LinkedList<TestCommunicationChannel> createCommunicationChanels(int num){
    LinkedList<TestCommunicationChannel> list = new LinkedList<TestCommunicationChannel>();
    LinkedList<Long> allNodes = new LinkedList<Long>();
    for(int i = 0; i < num; i++){
      allNodes.add(i, (long)i);
    }
    long timeout = 5*1000; // 5 sec
    for(int i = 0; i < num; i++){
      AgreementParams params = new AgreementParams(num, (num-1)/5, i, allNodes);
      TestCommunicationChannel tcc = new TestCommunicationChannel(i, 0, params, timeout, InstanceID.CheckpointAgreement);
      list.add(tcc);
      tcc.addInstance(new AgreementInstance(tcc.getInstanceID(), tcc));
    }

    return list;
  }

  /**
   * node receives n bottom votes
   */
  public void testBottomDecision(){
    LinkedList<TestCommunicationChannel> nodes = this.createCommunicationChanels(6);
    for(TestCommunicationChannel tc: nodes){
        tc.vote(null);
        assert tc.messages.size() == 1;
    }
    for(int i = 0; i < nodes.size(); i++){
      TestCommunicationChannel tc = nodes.get(i);
      syncAll(tc, nodes);
    }
    for(TestCommunicationChannel tc: nodes){
      assert tc.getDecisionProof() != null: tc.getMyID() + "\n tc" + tc.messages + " \n" + tc.messages.size() + " \n " + tc.getDecisionProof();
      assert (tc.getDecisionProof() instanceof ValueDecisionProof);
      ValueDecisionProof vdp = (ValueDecisionProof)tc.getDecisionProof();
      assert vdp.getDecidedValue().isBottom(): vdp.getDecidedValue();
    }
    
    
  }

  /**
   * timeout occurs after (n-1) bottom votes
   */
  public void testBottomDecision1(){
    LinkedList<TestCommunicationChannel> nodes = this.createCommunicationChanels(6);
    TestCommunicationChannel first = nodes.getFirst();
    for(TestCommunicationChannel tc: nodes){
      if(tc.getMyID() != first.getMyID()){
        tc.vote(null);
        assert tc.messages.size() == 1;
      }
    }
    for(int i = 0; i < nodes.size(); i++){
      TestCommunicationChannel tc = nodes.get(i);
      syncAll(tc, nodes);
      assert tc.getDecisionProof() == null: tc.getMyID() + "\n tc" + tc.messages + " \n" + tc.messages.size() + " \n " + tc.getDecisionProof();
      
    }
    
    try{
      Thread.sleep(10000);
    }catch(InterruptedException e){
      // TODO Auto-generated catch block
      e.printStackTrace();
    }
    for(TestCommunicationChannel tc: nodes){
      assert tc.getDecisionProof() != null: tc.getMyID() + "\n tc" + tc.messages + " \n" + tc.messages.size() + " \n " + tc.getDecisionProof();
      assert (tc.getDecisionProof() instanceof ValueDecisionProof);
      ValueDecisionProof vdp = (ValueDecisionProof)tc.getDecisionProof();
      assert vdp.getDecidedValue().isBottom(): vdp.getDecidedValue();
    }
  }

  public void testNoFailure(){
    // run the expt twice, first with quick responses from everyone..second time with slow response
    for(int i = 0; i < 2; i++){
      if(i==0){
        System.out.println("Running without timer");
      }else{
        System.out.println("Running with timer");
      }
      LinkedList<TestCommunicationChannel> nodes = this.createCommunicationChanels(6);
      TestCommunicationChannel first = nodes.getFirst();
      byte[]h = new byte[]{0, 1, 2, 3, 4, 5, 6, 7, 8, 9, 0, 1, 2, 3, 4, 5, 6, 7, 8, 9};
      Hash hash = new Hash(h, false);
      first.vote(hash);

      assert first.messages.size() == 1:"\n first" + first.messages + " \n" + first.messages.size();
      for(TestCommunicationChannel tc: nodes){
        if(tc.getMyID() != first.getMyID()){
          assert tc.messages.size() == 0;
        }
      }
      
      if(i == 1){
        try{
          Thread.sleep(10000);
        }catch(InterruptedException e){
          // TODO Auto-generated catch block
          e.printStackTrace();
        }
        assert first.instance.isTimedOut();
      }
      
      // now control message flow
      syncAll(first, nodes);

      // now everyone has the message
      for(TestCommunicationChannel tc: nodes){
        if(tc.getMyID() != first.getMyID()){
          assert tc.messages.size() == 2: tc.getMyID()  + "\n tc" + tc.messages + " \n" + tc.messages.size();
        }else{ 
          assert tc.messages.size() == 1: tc.getMyID()  + "\n tc" + tc.messages + " \n" + tc.messages.size();
        }
      }

      assert first.getDecisionProof() == null;

      // now first should decide
      int voteCount = 1;
      for(TestCommunicationChannel tc: nodes){
        if(tc.getMyID() != first.getMyID()){
          first.sync(tc);
          voteCount++;
          if(voteCount == first.getParams().strongQuorum()){
            if(i == 0){ // waiting for timeout
              assert first.getDecisionProof() == null: first.getDecisionProof();
            }else{ // timedout
              assert first.getDecisionProof() != null;
            }
          }
        }
      }

      for(TestCommunicationChannel tc: nodes){
        if(tc.getMyID() != first.getMyID()){
          assert tc.getDecisionProof() == null;
        }else{
          assert tc.getDecisionProof() != null: tc.getMyID() + "\n tc" + tc.messages + " \n" + tc.messages.size() + " \n " + tc.getDecisionProof();
          assert (tc.getDecisionProof() instanceof ValueDecisionProof);
          ValueDecisionProof vdp = (ValueDecisionProof)tc.getDecisionProof();
          assert vdp.getDecidedValue().getData().equals(hash): vdp.getDecidedValue().getData() + " hash " + hash;
        }
      }

      TestCommunicationChannel second = nodes.get(1);

      for(TestCommunicationChannel tc: nodes){
        if(tc.getMyID() != first.getMyID()){
          second.sync(tc);
        }
      }

      assert second.getDecisionProof() != null: second.getMyID() + "\n tc" + second.messages + " \n" + second.messages.size() + " \n " + second.getDecisionProof();
      assert (second.getDecisionProof() instanceof ValueDecisionProof);
      ValueDecisionProof vdp = (ValueDecisionProof)second.getDecisionProof();
      assert vdp.getDecidedValue().getData().equals(hash): vdp.getDecidedValue().getData() + " hash " + hash;

      for(TestCommunicationChannel tc: nodes){
        syncAll(tc, nodes);
        assert tc.getDecisionProof() != null: tc.getMyID() + "\n tc" + tc.messages + " \n" + tc.messages.size() + " \n " + tc.getDecisionProof();
        assert (tc.getDecisionProof() instanceof ValueDecisionProof);
        vdp = (ValueDecisionProof)tc.getDecisionProof();
        assert vdp.getDecidedValue().getData().equals(hash): vdp.getDecidedValue().getData() + " hash " + hash;
      }

    }

  }

  private void syncAll(TestCommunicationChannel src, Collection<TestCommunicationChannel> channels){
    for(TestCommunicationChannel tc: channels){
      tc.sync(src);
    }
  }

  public void testBenignFailure(){
    LinkedList<TestCommunicationChannel> nodes = this.createCommunicationChanels(6);
    TestCommunicationChannel first = nodes.getFirst();
    TestCommunicationChannel second = nodes.get(1);
    byte[]h = new byte[]{0, 1, 2, 3, 4, 5, 6, 7, 8, 9, 0, 1, 2, 3, 4, 5, 6, 7, 8, 9};

    Hash hash = new Hash(h, false);
    for(TestCommunicationChannel tc: nodes){
      if(tc.getMyID() == first.getMyID() || tc.getMyID() == second.getMyID()){
        tc.vote(null);
      }else{
        tc.vote(hash);
      }
      syncAll(tc, nodes);
      assert tc.getDecisionProof() == null: tc.getMyID() + "\n tc" + tc.messages + " \n" + tc.messages.size() + " \n " + tc.getDecisionProof();
    }
    
    for(TestCommunicationChannel tc: nodes){
      assert ((AgreementInstance)tc.instance).getRound() == 1: "\n" + tc.getMyID() + " \n" + tc.instance;
    }
    
    for(TestCommunicationChannel tc: nodes){
      syncAll(tc, nodes);
    }
    
    for(TestCommunicationChannel tc: nodes){
      assert tc.getDecisionProof() != null: tc.getMyID() + "\n tc" + tc.messages + " \n" + tc.messages.size() + " \n " + tc.getDecisionProof() + "\n" + tc.instance;
      assert (tc.getDecisionProof() instanceof ValueDecisionProof);
      ValueDecisionProof vdp = (ValueDecisionProof)tc.getDecisionProof();
      assert vdp.getDecidedValue().getData().equals(hash): vdp.getDecidedValue().getData() + " hash " + hash;
    }
  }

  public void testLieFailure(){
    LinkedList<TestCommunicationChannel> nodes = this.createCommunicationChanels(6);
    TestCommunicationChannel first = nodes.getFirst();
    TestCommunicationChannel firstClone = new TestCommunicationChannel(first.getMyID(), first.getEpoch(), first.getParams(), first.getTimeOut(), InstanceID.CheckpointAgreement);
    firstClone.addInstance(new AgreementInstance(firstClone.getInstanceID(), firstClone));
    TestCommunicationChannel second = nodes.get(1);
    TestCommunicationChannel third = nodes.get(2);
    TestCommunicationChannel fourth = nodes.get(3);
    TestCommunicationChannel fifth = nodes.get(4);
    TestCommunicationChannel sixth = nodes.get(5);
    nodes.add(firstClone);
    byte[]h = new byte[]{0, 1, 2, 3, 4, 5, 6, 7, 8, 9, 0, 1, 2, 3, 4, 5, 6, 7, 8, 9};

    Hash hash = new Hash(h, false);
    for(TestCommunicationChannel tc: nodes){
      if(tc.getMyID() != first.getMyID()){
        if(tc.getMyID() == second.getMyID() || tc.getMyID() == third.getMyID() || tc.getMyID() == fourth.getMyID()){
          tc.vote(null);
        }else{
          tc.vote(hash);
        }
        syncAll(tc, nodes);
        assert tc.getDecisionProof() == null: tc.getMyID() + "\n tc" + tc.messages + " \n" + tc.messages.size() + " \n " + tc.getDecisionProof();
      }
    }
    
    first.vote(null);
    firstClone.vote(hash);
    second.sync(first);third.sync(first);fourth.sync(first);
    fifth.sync(firstClone);sixth.sync(firstClone); // make node fifth and sixth choose randomly
    
    
    int numDecided = 0;
    while(numDecided < first.getParams().getNumNodes()){
      numDecided = 0;
      for(TestCommunicationChannel tc: nodes){
        if(tc.getDecisionProof() != null){
          numDecided++;
        }
      }
      System.out.println(numDecided + " nodes decided");
      boolean firstTime = true;
      for(TestCommunicationChannel tc: nodes){
        if(tc.getMyID() != first.getMyID()){
          syncAll(tc, nodes);
        }else if(firstTime){
          firstTime = false;
          second.sync(first);third.sync(first);fourth.sync(first);
          fifth.sync(firstClone);sixth.sync(firstClone); // make node fifth and sixth choose randomly
        }
      }
    }
    
    ValueDecisionProof vdf = (ValueDecisionProof)first.getDecisionProof();
    System.out.println("decided value " + vdf.getDecidedValue());
  }

  public void testRandomAgreement(){
    LinkedList<TestCommunicationChannel> nodes = this.createCommunicationChanels(6);
    TestCommunicationChannel first = nodes.getFirst();
    TestCommunicationChannel second = nodes.get(1);
    TestCommunicationChannel third = nodes.get(2);
    byte[]h = new byte[]{0, 1, 2, 3, 4, 5, 6, 7, 8, 9, 0, 1, 2, 3, 4, 5, 6, 7, 8, 9};

    Hash hash = new Hash(h, false);
    for(TestCommunicationChannel tc: nodes){
      if(tc.getMyID() == first.getMyID() || tc.getMyID() == second.getMyID() || tc.getMyID() == third.getMyID()){
        tc.vote(null);
      }else{
        tc.vote(hash);
      }
      syncAll(tc, nodes);
      assert tc.getDecisionProof() == null: tc.getMyID() + "\n tc" + tc.messages + " \n" + tc.messages.size() + " \n " + tc.getDecisionProof();
    }
    
    for(TestCommunicationChannel tc: nodes){
      assert ((AgreementInstance)tc.instance).getRound() == 1: "\n" + tc.getMyID() + " \n" + tc.instance;
    }
    
    for(TestCommunicationChannel tc: nodes){
      syncAll(tc, nodes);
    }
    
    int numDecided = 0;
    while(numDecided != first.getParams().getNumNodes()){
      numDecided = 0;
      for(TestCommunicationChannel tc: nodes){
        if(tc.getDecisionProof() != null){
          numDecided++;
        }
      }
      System.out.println(numDecided + " nodes decided");
      for(TestCommunicationChannel tc: nodes){
        syncAll(tc, nodes);
      }
    }
    
    ValueDecisionProof vdf = (ValueDecisionProof)first.getDecisionProof();
    System.out.println("decided value " + vdf.getDecidedValue());
  }

  /*
   * main() lets us run just this set of unit tests
   * from the comand line (you can also invoke 
   * the testrunner on this class and it will find
   * the suite())
   *
   * usage: java <classname> [-verbose] [-vverbose] [testName]*
   * 
   *   If verbose or vverbose are included, print info to screen
   *
   *   If [testName]* are included, then run test called "test[testName]"
   *   for each such [testName]. E.g., "java TestEmtpy foo" runs
   *   TwoNodesSubscriptionUnit.testfoo() as a TestCase.
   *
   * TBD: update class name
   */
  public static void main(String s[]) {
    String name = "TestAgreementInstanceUnit";
    System.err.print(name + " self test begins...");
    Test test;
    test = suite();
    TestRunner tr = new TestRunner();
    tr.doRun(test);
    System.exit(0);
  }

}
