package code.simulator.branchManager;

import java.util.*;

import code.branchDetecting.BranchID;

/**
 * Maintains a tree over elements of type E. 
 * Assumes that E implements comparable correctly, i.e. e1.equals(e2) == true iff e1 = e2. 
 * @author princem
 *
 * @param <E>
 */
public class SimpleTree<E>{

  LinkedList<SimpleTreeNode<E>> roots;
  public SimpleTree(){
    roots = new LinkedList<SimpleTreeNode<E>>();
  }
  
  public SimpleTree(Collection<SimpleTreeNode<E>> roots){
    this.roots = new LinkedList<SimpleTreeNode<E>>(roots);
  }
  
  public void markNonLeaf(E e) throws Exception{
    SimpleTreeNode<E> stn = this.findMatchingTreeNodes(e);
    stn.setLeaf(false);
  }
  
  public void markLeaf(E e) throws Exception{
    SimpleTreeNode<E> stn = this.findMatchingTreeNodes(e);
    assert stn != null: "STN is null " + e + " this: " + this.toString(); 
    stn.setLeaf(true);
    stn.getChildren().clear();
  }
  
  /**
   * Adds a new child to the tree at a given parent node
   * @param parent *Existing* parent node
   * @param child *NEW* child node
   * @throws Exception
   */
  synchronized public void addVal(E parentVal, E value) throws Exception{
    assert this.findMatchingTreeNodes(value) == null: value + " " + this;
    SimpleTreeNode<E> child = new SimpleTreeNode<E>(value);
    if(parentVal != null){
      SimpleTreeNode<E> parent = this.findMatchingTreeNodes(parentVal);
      assert parent != null;
      if(parent.isLeaf()){
        throw new Exception("Parent node is marked as leaf" + parent);
      }
      child.setParent(parent);
      parent.getChildren().add(child);
    }else{
      roots.add(child);
    }
  }
  
  synchronized public boolean isLeaf(E t) throws Exception{
    return this.findMatchingTreeNodes(t).isLeaf();
  }
  
  synchronized public Collection<E> getChildren(E e) throws Exception{
    LinkedList<E> children = new LinkedList<E>();
    Collection<SimpleTreeNode<E>> c = this.findMatchingTreeNodes(e).getChildren();
    for(SimpleTreeNode<E> stn: c){
      children.add(stn.element);
    }
    return children;
  } 
  
  synchronized public Collection<E> getAllChildren(E e) throws Exception{
    LinkedList<E> children = new LinkedList<E>();
    SimpleTreeNode<E> stn1 = this.findMatchingTreeNodes(e);
    if(stn1 == null){
      throw new Exception("node not found"  +e);
    }
    Collection<SimpleTreeNode<E>> c = stn1.getAllSubtreeNodes();
    c.remove(stn1);
    for(SimpleTreeNode<E> stn: c){
      children.add(stn.element);
    }
    return children;
  } 
  
  synchronized public TreeSet<E> getAllAncestors(E e) throws Exception{
    TreeSet<E> parents = new TreeSet<E>();
    SimpleTreeNode<E> stn = this.findMatchingTreeNodes(e);
    assert stn != null: "STN is null " + e + " this: " + this.toString(); 
    while(stn.getParent() != null){
      parents.add(stn.getParent().element);
      stn = stn.getParent();
    }
    return parents;
  } 
  
  public boolean areBranchesConcurrent(TreeSet<E> branches) throws Exception{
    for(E e: branches){
      SimpleTreeNode<E> stn = this.findMatchingTreeNodes(e);
      TreeSet<E> childrenE = new TreeSet<E>();
      Collection<SimpleTreeNode<E>> childrenSTN = stn.getAllSubtreeNodes();
      for(SimpleTreeNode<E> stnC: childrenSTN){
        if(!stnC.equals(stn))
        childrenE.add(stnC.element);
      }
      for(E e1: branches){
        if(childrenE.contains(e1)){
          return false;
        }
      }
      
    }
    return true;
  }
  
  /**
   * create a branch map that transforms the current tree leaved at validBranches into a tree that have no single-child nodes
   * 
   * @param validBranches
   * @return
   * @throws Exception
   */
  public synchronized HashMap<E, E> getBranchMap(TreeSet<E> validBranches) throws Exception{
    assert this.areBranchesConcurrent(validBranches):"\n"+validBranches;
    
    // first construct the subset of the tree that is projected by the validBranches
    HashMap<E, E> parentMap = new HashMap<E, E>();
    HashMap<E, Integer> childCount = new HashMap<E, Integer>();

    LinkedList<SimpleTreeNode<E>> cache = new LinkedList<SimpleTreeNode<E>>();
    for(E e: validBranches){
      SimpleTreeNode<E> stn = this.findMatchingTreeNodes(e);
      if(stn != null){
        cache.add(stn);
      }
      while(stn != null){
        if(stn.getParent() == null){
          parentMap.put(stn.element, null);
        }else {
          parentMap.put(stn.element, stn.getParent().element);
          if(childCount.containsKey(stn.getParent().element)){
            childCount.put(stn.getParent().element, childCount.get(stn.getParent().element)+1);
          }else{
            childCount.put(stn.getParent().element, 1);
          }
        }
        stn = stn.getParent();
      }
    }
    
    HashMap<E, E> branchMap = new HashMap<E, E>();
    
    for(SimpleTreeNode<E> stn: cache){
      SimpleTreeNode<E> iter = stn;
      while(iter != null){
        E key = iter.element;
        SimpleTreeNode<E> e = iter;
        
        while(e != null){
          // check whether e is acceptable or not?
          // if parent is null (i.e. root) or if the number of children is greater than 2
          if(parentMap.get(e.element) == null || (childCount.containsKey(e.getParent().element) && childCount.get(e.getParent().element) > 1)){
            branchMap.put(key, e.element);
            break;
          }else{
            e = e.getParent();
          }
        }
        assert e != null;
        iter = iter.getParent();
      }
    }
    
    Collection<E> coll = new LinkedList<E>(branchMap.keySet());
    for(E e: coll){
      if(branchMap.get(e).equals(e)){
        branchMap.remove(e);
      }
    }
    
    return branchMap;
  }
  
//  synchronized public void removeSubTree(E e) throws Exception{
//    SimpleTreeNode<E> stn = this.findMatchingTreeNodes(e);
//    if(stn.getParent() == null){
//      roots.remove(stn);
//    }else{
//      stn.getParent().getChildren().remove(stn);
//    }
//  }
  
  /**
   * Add a new intermediateTreeNode parent to the childTreeNode.
   * Basically Parent->childTreeNode is replaced by Parent->intermediateTreeNode->childTreeNode
   * @param childTreeNode
   * @param e New element to be inserted
   */
  synchronized public void pushNewIntermediateChild(E parentVal, E e) throws Exception{
    SimpleTreeNode<E> parent = this.findMatchingTreeNodes(parentVal);
    
    // set the parent and the newIntermediate node connection appropriately
    SimpleTreeNode<E> newIntermediateTreeNode = new SimpleTreeNode<E>(e, parent.getChildren());
    if(!parent.isLeaf()){
      newIntermediateTreeNode.setLeaf(false);
    }
    newIntermediateTreeNode.setParent(parent);
    parent.getChildren().clear();
    parent.getChildren().add(newIntermediateTreeNode);
    
    // now update the parent references in the children of the parent
    for(SimpleTreeNode<E> stn: newIntermediateTreeNode.getChildren()){
      stn.setParent(newIntermediateTreeNode);
    }
  }
  
  /**
   * remove an intermediateTreeNode from the tree. 
   * All the children of the intermediateTreeNode are migrated to the parent.
   * @param intermediateTreeNode
   */
  synchronized public void popIntermediateNode(E e) throws Exception{
    SimpleTreeNode<E> stn = this.findMatchingTreeNodes(e);
    if(stn.getParent() == null){
      // make all the children root
      roots.addAll(stn.getChildren());
    }else{
      stn.getParent().getChildren().addAll(stn.getChildren());
      stn.getParent().getChildren().remove(stn);
    }
    // update references in stn.children
    for(SimpleTreeNode<E> stnC: stn.getChildren()){
      stnC.setParent(stn.getParent());
    }
  }

  /**
   * returns a set of nodes that are neither the parent or grandparents, nor the children or grand-children of the SimpleTreeNode passed
   * @param t
   * @return
   */
  synchronized public TreeSet<E> getConcurrentNodes(E e) throws Exception{
    SimpleTreeNode<E> t = this.findMatchingTreeNodes(e);
    assert t != null;
    LinkedList<SimpleTreeNode<E>> nodes = new LinkedList<SimpleTreeNode<E>>();
    for(SimpleTreeNode<E> stn: roots){
      nodes.addAll(stn.getAllSubtreeNodes());
    }
    
    if(t != null){
      //remove children of t
      nodes.removeAll(t.getAllSubtreeNodes());
      
      // remove t's parents
      t = t.getParent();
      while(t != null){
        nodes.remove(t);
        t = t.getParent();
      }
    }
    
    TreeSet<E> values = new TreeSet<E>();
    for(SimpleTreeNode<E> stns: nodes){
      values.add(stns.element);
    }
    return values;
  }
  
  synchronized private SimpleTreeNode<E> findMatchingTreeNodes(E e) throws Exception{
    SimpleTreeNode<E> match = null;
    for(SimpleTreeNode<E> stn: roots){
      SimpleTreeNode<E> localMatch = stn.findMatchingTreeNodes(e);
      if(localMatch != null && match == null){
        match = localMatch;
      }else if(match != null && localMatch != null){
        throw new Exception("Multiple matches for " + e);
      }
    }
    
    return match;
  }
  
  synchronized public boolean containsNode(E e) {
    try{
      for(SimpleTreeNode<E> stn: roots){
        SimpleTreeNode<E> localMatch = stn.findMatchingTreeNodes(e);
        if(localMatch != null){
          return true;
        }
      }
    }catch(Exception e1){
      return true;
    }
    return false;
  }
  

  
  public SimpleTree<E> deepClone(){
    SimpleTree<E> newTree = new SimpleTree<E>();
    for(SimpleTreeNode<E> stn: roots){
      newTree.roots.add(stn.deepClone());
    }

    return newTree;
  }
  
  @Override
  public String toString(){
    return roots.toString(); 
  }

  public TreeSet<E> getAllNodes(){
    TreeSet<E> set = new TreeSet<E>();
    for(SimpleTreeNode<E> stn: roots){
      for(SimpleTreeNode<E> stn1: stn.getAllSubtreeNodes()){
        set.add(stn1.element);
      }
    }
    return set;
  }

  /* (non-Javadoc)
   * @see java.lang.Object#hashCode()
   */
  @Override
  public int hashCode(){
    final int prime = 31;
    int result = 1;
    result = prime * result + ((roots == null) ? 0 : roots.hashCode());
    return result;
  }

  /* (non-Javadoc)
   * @see java.lang.Object#equals(java.lang.Object)
   */
  @Override
  public boolean equals(Object obj){
    if(this == obj){
      return true;
    }
    if(obj == null){
      return false;
    }
    if(!(obj instanceof SimpleTree)){
      return false;
    }
    SimpleTree other = (SimpleTree) obj;
    if(roots == null){
      if(other.roots != null){
        return false;
      }
    }else if(!roots.equals(other.roots)){
      return false;
    }
    return true;
  }

  public TreeSet<E> getLeafNodes(){
    TreeSet<E> set = new TreeSet<E>();
    for(SimpleTreeNode<E> stn: roots){
      for(SimpleTreeNode<E> stn1: stn.getAllLeafNodes()){
        set.add(stn1.element);
      }
    }
    return set;
  }
}
