package code.simulator.branchManager;

import java.util.*;

/**
 * Maintains a tree over elements of type E. 
 * Assumes that E implements comparable correctly, i.e. e1.equals(e2) == true iff e1 = e2. 
 * @author princem
 *
 * @param <E>
 */
public class Tree<E>{

  HashMap<TreeNode<E>, LinkedList<TreeNode<E>>> childMap;
  
  public Tree(){
    childMap = new HashMap<TreeNode<E>, LinkedList<TreeNode<E>>>();
  }
  
  /**
   * Adds a new child to the tree at a given parent node
   * @param parent *Existing* parent node
   * @param child *NEW* child node
   * @throws Exception
   */
  synchronized public TreeNode<E> addVal(TreeNode<E> parent, E value) throws Exception{
    if(parent != null && !childMap.containsKey(parent)){
      throw new Exception("Invalid parent" + parent);
    }
    TreeNode<E> child = new TreeNode<E>(value, parent); 
    if(parent!= null){
      childMap.get(parent).add(child);
    }
    childMap.put(child, new LinkedList<TreeNode<E>>());
    return child;
  }
  
//  /**
//   * Adds a new child to the tree at a given parent node
//   * @param parent *Existing* parent node
//   * @param child *NEW* child node
//   * @throws Exception
//   */
//  public void addNode(TreeNode<E> parent, TreeNode<E> child) throws Exception{
//    if(parent != null && !childMap.containsKey(parent)){
//      throw new Exception("Invalid parent" + parent);
//    }
//    if(childMap.containsKey(child)){
//      throw new Exception("Invalid child" + child);
//    }
//    if(parent!= null){
//      childMap.get(parent).add(child);
//    }
//    childMap.put(child, new TreeSet<TreeNode<E>>());
//  }
  
  synchronized public boolean isLeaf(TreeNode<E> t) throws Exception{
    if(!childMap.containsKey(t)){
      throw new Exception("Invalid node" + t);
    }
    return childMap.get(t) == null || childMap.get(t).size() == 0;
  }
  
  synchronized public Collection<TreeNode<E>> getChildren(TreeNode<E> t) throws Exception{
    if(t == null || !childMap.containsKey(t)){
      throw new Exception("Invalid parent" + t);
    }
    return new LinkedList<TreeNode<E>>(childMap.get(t)); 
  } 
  
  synchronized public void removeSubTree(TreeNode<E> t) throws Exception{
    if(t == null || !childMap.containsKey(t)){
      throw new Exception("Invalid node" + t);
    }
    LinkedList<TreeNode<E>> nodes = new LinkedList<TreeNode<E>>();
    nodes.add(t);
    if(t.parent != null){
      childMap.get(t.parent).remove(t);
    }
    while(!nodes.isEmpty()){
      TreeNode<E> n = nodes.remove();
      nodes.addAll(childMap.get(n));
      childMap.remove(n);
    }
  }
  
  /**
   * Add a new intermediateTreeNode parent to the childTreeNode.
   * Basically Parent->childTreeNode is replaced by Parent->intermediateTreeNode->childTreeNode
   * @param childTreeNode
   * @param e New element to be inserted
   */
  synchronized public TreeNode<E> pushNewIntermediateNode(TreeNode<E> childTreeNode, E e) throws Exception{
    if(childTreeNode == null || !childMap.containsKey(childTreeNode)){
      throw new Exception("Invalid node" + childTreeNode);
    }
    
    TreeNode<E> newIntermediateTreeNode = new TreeNode<E>(e, childTreeNode.parent);
    LinkedList<TreeNode<E>> oldChildren = childMap.get(childTreeNode);

    TreeNode<E> newChildTreeNode = new TreeNode<E>(childTreeNode.element, newIntermediateTreeNode);
    childMap.remove(childTreeNode);
    childMap.put(newChildTreeNode, oldChildren);
    
    childMap.put(newIntermediateTreeNode, new LinkedList<TreeNode<E>>());
    childMap.get(newIntermediateTreeNode).add(newChildTreeNode);
    if(childTreeNode.parent != null){
      LinkedList<TreeNode<E>> children = childMap.get(childTreeNode.getParent());
      children.remove(childTreeNode);
      children.add(newIntermediateTreeNode);
    }
    
    // now update the subtree rooted at the newChildTreeNode to refresh the references
    refresh(newChildTreeNode);
    
    return newIntermediateTreeNode;
  }
  
  /**
   * update all the references in t's descendants to point to the appropriate parents
   * @param t
   */
  synchronized private void refresh(TreeNode<E> t){
    assert childMap.containsKey(t);
    LinkedList<TreeNode<E>> children = childMap.get(t);
    LinkedList<TreeNode<E>> childrenClone = new LinkedList<TreeNode<E>>(children);
    children.clear();
    for(TreeNode<E> tn: childrenClone){
      TreeNode<E> newTN = new TreeNode<E>(tn.element, t);
      LinkedList<TreeNode<E>> newTNChildren = childMap.get(tn);
      childMap.remove(tn);
      children.add(newTN);
      childMap.put(newTN, newTNChildren);
      assert childMap.containsKey(newTN);
      refresh(newTN);
    }
  }
  
  /**
   * remove an intermediateTreeNode from the tree. 
   * All the children of the intermediateTreeNode are migrated to the parent.
   * @param intermediateTreeNode
   */
  synchronized public void popIntermediateNode(TreeNode<E> intermediateTreeNode) throws Exception{
    if(intermediateTreeNode == null || !childMap.containsKey(intermediateTreeNode)){
      throw new Exception("Invalid node" + intermediateTreeNode);
    }
    LinkedList<TreeNode<E>> children = childMap.get(intermediateTreeNode);

    childMap.remove(intermediateTreeNode);
    
    if(intermediateTreeNode.parent != null){
      LinkedList<TreeNode<E>> parentChildren = childMap.get(intermediateTreeNode.getParent());
      parentChildren.remove(intermediateTreeNode);
      parentChildren.addAll(children);
      assert childMap.containsKey(intermediateTreeNode.parent);
      refresh(intermediateTreeNode.parent);
    }else{
      for(TreeNode<E> tn: children){
        TreeNode<E> newTN = new TreeNode<E>(tn.element, null);
        childMap.put(newTN, childMap.get(tn));
        childMap.remove(tn);
        refresh(newTN);
      }
    }
  }

  /**
   * returns a set of nodes that are neither the parent or grandparents, nor the children or grand-children of the TreeNode passed
   * @param t
   * @return
   */
  synchronized public Collection<TreeNode<E>> getConcurrentNodes(TreeNode<E> t) throws Exception{
    if(t == null || !childMap.containsKey(t)){
      throw new Exception("Invalid node" + t);
    }
    
    LinkedList<TreeNode<E>> nodes = new LinkedList<TreeNode<E>>();
    nodes.addAll(childMap.keySet());
    
    // now parse the tree starting from t to remove all of t's ancestors and descendants
    // first parents
    TreeNode<E> parent = t;
    while(parent != null){
      nodes.remove(parent);
      parent = parent.getParent();
    }
    
    // now children
    LinkedList<TreeNode<E>> allChildren = new LinkedList<TreeNode<E>>();
    allChildren.addAll(childMap.get(t));
    
    while(allChildren.size() > 0){
      TreeNode<E> tmp = allChildren.removeFirst();
      nodes.remove(tmp);
      assert childMap.containsKey(tmp): "tmp: "  + tmp + " \n " + childMap.keySet();
      allChildren.addAll(childMap.get(tmp));
    }
    return nodes;
  }
  
  synchronized public LinkedList<TreeNode<E>> findMatchingTreeNodes(E e){
    LinkedList<TreeNode<E>> matches = new LinkedList<TreeNode<E>>();
    for(TreeNode<E> t: childMap.keySet()){
      if(t.element.equals(e)){
        matches.add(t);
      }
    }
    return matches;
  }
  
  /**
   * 
   * @param treeNodes
   * @return
   */
  synchronized public HashMap<TreeNode<E>, TreeNode<E>> getBranchMap(Collection<TreeNode<E>> treeNodes){
    assert false;
    return null;
  }
}
