package code.simulator.checkpoint;

import java.util.HashMap;
import java.util.Iterator;
import java.util.TreeMap;
import java.util.Vector;
import java.util.TreeSet;

import java.io.ByteArrayOutputStream;
import java.io.ObjectOutput;
import java.io.ObjectOutputStream;
import java.io.Serializable;
import code.AcceptVV;
import code.AcceptStamp;
import code.CounterVV;
import code.ObjId;
import code.security.ahs.DependencyVV;
import code.serialization.SerializationHelper;
import code.branchDetecting.*;
import code.NodeId;
import code.simulator.Hash;
import code.simulator.HashedVV;
import code.simulator.IrisNode;
import code.simulator.Obj2Bytes;
import code.simulator.SecureSimPreciseInv;
import code.simulator.SimPreciseInv;
import code.simulator.protocolFilters.AgreementGCProtocol;
import code.simulator.store.*;


/**
 * A checkpoint consists of an HVV, an object store, and a partial log of unverifiable writes.
 * @author princem
 *
 */
public class Checkpoint implements Serializable, Obj2Bytes{
  public final AcceptVV omitVV;
  public final StoreInterface objectStore;
  public final TreeSet<SimPreciseInv> unverifiableWrites;
  public final HashMap<NodeId, SimPreciseInv> lastWrite;
//  public final HashMap<BranchID, DependencyVV> dependencyMap;
  public final AcceptVV endVV;
  public final int epochCount;
  public final Hash hash;
  public final TreeMap<AcceptStamp, TreeSet<Byte>> flagMap;
  public final TreeSet<SimPreciseInv> acceptableBranches;
  private transient Hash h = null; 
  
  public Checkpoint(AcceptVV omitVV, StoreInterface objectStore, 
      TreeSet<SimPreciseInv> unverifiableWrites, HashMap<NodeId, SimPreciseInv> _lastWrite, TreeMap<AcceptStamp, TreeSet<Byte>> flagMap, 
      int epochCount, AcceptVV endVV, TreeSet<SimPreciseInv> acceptableBranches //, HashMap<BranchID, DependencyVV> depMap
      ){
    this.omitVV = omitVV;
    this.objectStore = objectStore;
    this.unverifiableWrites = unverifiableWrites;
    this.lastWrite = _lastWrite;
    this.endVV = endVV;
    this.epochCount = epochCount;
    this.flagMap = flagMap;
    this.acceptableBranches = acceptableBranches;
    hash = this.getHash();
//    this.dependencyMap = depMap;
  }
  
  public int hashCode(){
    int ret = omitVV.hashCode() + endVV.hashCode() + new Integer(epochCount).hashCode() + lastWrite.hashCode() + unverifiableWrites.hashCode() + flagMap.hashCode() + this.acceptableBranches.hashCode();// + this.dependencyMap.hashCode();
    return ret;
  }
  public boolean equals(Checkpoint c){
    assert hash.equals(this.getHash());
    boolean ret = omitVV.equals(c.omitVV) && endVV.equals(c.endVV) && (epochCount == c.epochCount) && lastWrite.equals(c.lastWrite) && unverifiableWrites.equals(c.unverifiableWrites) && flagMap.equals(c.flagMap) && this.acceptableBranches.equals(c.acceptableBranches)
//    && dependencyMap.equals(c.dependencyMap)
    ;
//    if(ret){
//      for(BranchID bid: c.lastWrite.keySet()){
//        if(!lastWrite.containsKey(bid) || !lastWrite.get(bid).equals(c.lastWrite.get(bid))){
//          return false;
//        }
//      }
//      
//      for(BranchID bid: c.unverifiableWrites.keySet()){
//        if(!unverifiableWrites.containsKey(bid) || !unverifiableWrites.get(bid).equals(c.unverifiableWrites.get(bid))){
//          return false;
//        }
//      }
//    }
    return ret;
  }
  
  public boolean isUnverifiableAcceptable(SimPreciseInv spi){
    if(lastWrite.containsKey(spi.getNodeId()) && lastWrite.get(spi.getNodeId()).equals(spi)){
      return true;
    }else if(unverifiableWrites.contains(spi)){
      return true;
    }else if(acceptableBranches.contains(spi)){
      return true;
    }else{
      return false;
    }
  }

  public Checkpoint applyBranchMap(HashMap<NodeId, NodeId> bm){
    AcceptVV newOmitVV = omitVV.applyBranchMap(bm);
    AcceptVV newEndVV = endVV.applyBranchMap(bm);

    StoreInterface newStore = this.objectStore.applyBranchMap(bm);
    TreeSet<SimPreciseInv> newUnverifiableWrites = new TreeSet<SimPreciseInv>();
    HashMap<NodeId, SimPreciseInv> newLastWrite = new HashMap<NodeId, SimPreciseInv>();
    
    for(SimPreciseInv spi: unverifiableWrites){
      newUnverifiableWrites.add(spi.applyBranchMap(bm));
    }
    
    for(NodeId b: lastWrite.keySet()){
      NodeId newB;
      if(bm.containsKey(b)){
        newB = bm.get(b);
      }else{
        newB = b;
      }
      newLastWrite.put(newB, lastWrite.get(b).applyBranchMap(bm));
    }

    TreeMap<AcceptStamp, TreeSet<Byte>> newFlagMap = new TreeMap<AcceptStamp, TreeSet<Byte>>();
    for(AcceptStamp as: flagMap.keySet()){
      if(bm.containsKey(as.getNodeId())){
        newFlagMap.put(new AcceptStamp(as.getLocalClock(), bm.get(as.getNodeId())), flagMap.get(as));
      }else{
        newFlagMap.put(as, flagMap.get(as));
      }
    }
    
    TreeSet<SimPreciseInv> newAcceptableBranches = new TreeSet<SimPreciseInv>();
    for(SimPreciseInv b: acceptableBranches){
      SimPreciseInv newSimPreciseInv = ((SecureSimPreciseInv)b).applyBranchMap(bm);
      newAcceptableBranches.add(newSimPreciseInv);
    }
    
    return new Checkpoint(newOmitVV, newStore, newUnverifiableWrites, newLastWrite, newFlagMap, this.epochCount, newEndVV, newAcceptableBranches);
  }
  
  public Hash getHash(){
    if(h == null){
      h = new Hash(this);
    }
    return h;
  }

  public byte[] obj2Bytes(){
    try{
      SerializationHelper serHelper = new SerializationHelper();
      ObjectOutput oos = serHelper.getObjectOutputStream();

      if(IrisNode.useAgreementCheckpoint){
        serHelper.writeHDVV((HashedVV)omitVV);
        serHelper.writeHDVV((HashedVV)endVV);
      }else{
        serHelper.writeVV(omitVV);
        serHelper.writeVV(endVV);
      }

      oos.writeObject(flagMap);
      for(AcceptStamp as: flagMap.keySet()){
        oos.writeLong(as.getLocalClock());
        if(as.getNodeId() instanceof BranchID){
          oos.write(((BranchID)as.getNodeId()).obj2Bytes());
        }else{
          oos.writeLong(as.getNodeId().getIDint());
        }
        for(Byte b: flagMap.get(as)){
          oos.write(b);
        }
      }

      Iterator<ObjId> iter = objectStore.iterator();
      TreeSet<ObjId> objIds = new TreeSet<ObjId>();
      while(iter.hasNext()){
        objIds.add(iter.next());
      }
      for(ObjId o: objIds){
        oos.writeBytes(o.getPath());
        for(StoreEntry se: objectStore.get(o)){
          oos.write(se.obj2Bytes());
        }
      }

      for(NodeId n: new TreeSet<NodeId>(lastWrite.keySet())){
        if(n instanceof BranchID){
          oos.write(((BranchID)n).obj2Bytes());
        } else {
          oos.writeLong(n.getIDint());
        }
        oos.write((lastWrite.get(n)).obj2Bytes());
      }
      oos.writeInt(this.epochCount);

      for(SimPreciseInv sp: this.unverifiableWrites){
        SecureSimPreciseInv spi = (SecureSimPreciseInv)sp;
        oos.write(spi.generateMyHash().getHashVal());
      }

      for(SimPreciseInv n: this.acceptableBranches){
        oos.write(((SecureSimPreciseInv)n).obj2Bytes());
        if(n.getNodeId() instanceof BranchID){
          oos.write(((BranchID)n.getNodeId()).obj2Bytes());
        }else{
          oos.writeLong(n.getNodeId().getIDint());
        }
      }

      serHelper.close();
      return serHelper.toBytes();
    }catch(Exception e){
      e.printStackTrace();
    }
    assert false;
    return null;
    
  }

  @Override
  public String toString(){
    return "Checkpoint [acceptableBranches=" + acceptableBranches + ", \nendVV="
        + ((HashedVV)endVV).toLongString() + ", \nepochCount=" + epochCount + ", \nflagMap=" + flagMap
        + ", \nhash=" + hash + ", \nlastWrite=" + lastWrite + ", \nobjectStore="
        + objectStore + ", \nomitVV=" + ((HashedVV)omitVV).toLongString() + ", \nunverifiableWrites="
        + unverifiableWrites + "]";
  }

  public int getEpoch(){
    return epochCount;
  }
  
}
