package code.simulator.checkpoint.unit;

import code.branchDetecting.BranchID;
import code.security.SangminConfig;
import code.simulator.*;
import code.simulator.checkpoint.Checkpoint;
import code.simulator.log.Log;
import code.simulator.protocolFilters.GCCertificate;
import code.simulator.protocolFilters.SimpleGCProtocol;
import code.simulator.protocolFilters.InstantGCProtocol;
import code.simulator.store.StoreEntry;
import code.simulator.unit.SimulatorUnit;
import code.untrustedstorage.writeanyreadany.StorageConfig;
import code.*;
/** 
 *  Test bandwidth to subscribe to 1000 objects, 100 of which have
 *  been modified. Test created for sosp 2007 submission microbenchmark.
 **/ 

//package utils.junit;

import junit.textui.TestRunner;
import junit.framework.*;
import java.util.*;
import java.io.*;

/**
 * TBD: Update class name
 */
public class LadderedCheckpointUnit extends TestCase {
  public static final String TEST_ALL_TEST_TYPE = "UNIT";

  /**
   * Basic constructor - called by the test runners.
   * TBD: Update constructor name to match class
   */
  public LadderedCheckpointUnit (final String s) {
    super (s);
  }

  /*
   * Fixtures are run before and after each test case
   * to set up environment in which tests must run.
   */
  protected void setUp() throws Exception{
    super.setUp();
    makePractiConfig(0, 100);

  }

  public static SyncPacket getSyncPacket(Checkpoint chkPt){
    GCCertificate gcCert = new GCCertificate(chkPt, (HashedVV)chkPt.omitVV, (HashedVV)chkPt.endVV);
    SyncPacket sp = new SyncPacket(SyncPacket.SenderHigherEpoch, chkPt.epochCount, new BranchID(21), false);
    LinkedList<Certificate> certs = new LinkedList<Certificate>();
    certs.add(gcCert);
    sp.setCertificates(certs);
    return sp;
  }

  public static void makePractiConfig(long first, int total){
    Config.createEmptyConfig();

    long NODE_1_ID = first;
    int port = 9921;
    String NODE_1_IP = "localhost";

    for(int i = 0; i < total; i++){

      Config.addOneNodeConfig(new NodeId(NODE_1_ID++),
          NODE_1_IP,
          port++,
          port++,
          port++,
          port++,
          port++,
          "/test" + File.separatorChar + "local-" + 
          NODE_1_ID + ".db",
          "/*",
          -1L,
          NODE_1_IP,
          port++,
          port++,
          -1,
          Config.CACHE_SIZE_BYTES_DEFAULT,
          Config.MAX_LOG_DISK_SIZE_BYTES,
          Config.MAX_LOG_MEM_SIZE_BYTES);
    }

    Config.readKeys();//needed for secure practi

  }

  public static int UNIT_NODE_ID = 10;
  public static int MIDDLE_NODE_ID = 15;

  public static int HELPER_NODE_ID = 20;
  public static boolean vverbose = false;
  public static boolean verbose = true;


  public static String COARSE_PATH = "/coarse";


  //--------------------------------------------------------------------------- 
  // Test coarse-grained subscriptions. 
  //
  // To initialize state, Unit create 1000 objects
  // (/data-coarse/[0..9]/[0..9]/[0..9]. Then helper subscribe
  // to /* from unit and wait for /data-coarse/9/9/9 to arrive.
  //
  //
  // start helper and wait for writing to complete
  //---------------------------------------------------------------------------
  public void testSimple(){
    try{
      IrisNode.useAgreementCheckpoint = false;

      Node.dropInconsistentWrites = false;

      System.out.println("dropping inconsistency SimulatorUnit.writes " + Node.dropInconsistentWrites);
      BranchID myNodeId = NodeFactory.createNodeId(UNIT_NODE_ID+1);


      BranchID helperNodeId = NodeFactory.createNodeId(HELPER_NODE_ID);

      StorageConfig.Client2WriteServerMap.clear();
      StorageConfig.Client2WriteServerMap.put(0, 0);
      StorageConfig.Client2WriteServerMap.put(1, 1);

      if(vverbose){
        System.out.println("UNIT ID: " + myNodeId.toString());
      }
      assert(myNodeId.equals(NodeFactory.createNodeId(UNIT_NODE_ID+1)));

      Node myNode = NodeFactory.createNodeWithoutFilter(myNodeId);
      Node helperNode = NodeFactory.createNodeWithoutFilter(helperNodeId);

      //second set of SimulatorUnit.writes

      if(verbose){
        System.out.println("Hello from SimulatorCheckpointUnit");
      }

      assert myNode.getCurrentVV().equalsIgnoreNegatives(new AcceptVV());

      int ni = 5;
      //M:4
      SimulatorUnit.write(myNode, COARSE_PATH, 0, ni-1, 0);
      SimulatorUnit.checkRead(myNode, COARSE_PATH, 0, ni-1, 0);


      assert myNode.getCurrentVV().includes(new AcceptStamp(ni-1, myNodeId));
      assert !myNode.getCurrentVV().includes(new AcceptStamp(ni, myNodeId));

      //M:4
      helperNode.sync(myNode);

      SimulatorUnit.checkRead(helperNode, COARSE_PATH, 0, ni-1, 0);

      // now gabage collect myNode
      //M:4, GC:M4
      myNode.garbageCollect(myNode.getCurrentVV());

      // now do enough communication to cause garbage collection
      helperNode.sync(myNode);
      myNode.sync(helperNode);

      assert myNode.getEpoch()==1: myNode.getEpoch();
      SimulatorUnit.checkRead(myNode, COARSE_PATH, 0, ni-1, 0);
      assert myNode.getCurrentVV().includes(myNode.getOmitVV());

      //M:4, GC:M4
      helperNode.sync(myNode);
      assert helperNode.getEpoch()==1;
      SimulatorUnit.checkRead(helperNode, COARSE_PATH, 0, ni-1, 0);

      // now do SimulatorUnit.writes at helper node and sync with the main node
      //M:4, H:9. GC:M4
      SimulatorUnit.write(helperNode, COARSE_PATH, 0, ni-1, 10);
      SimulatorUnit.checkRead(helperNode, COARSE_PATH, 0, ni-1, 10);

      assert helperNode.getCurrentVV().includes(new AcceptStamp(ni-1+5, helperNodeId));
      // assert !helperNode.getCurrentVV().includes(new AcceptStamp(ni+5, helperNodeId));

      //M:4, H:9, GC:M4
      myNode.sync(helperNode);
      SimulatorUnit.checkRead(myNode, COARSE_PATH, 0, ni-1, 10);

      //M:14, H:9, GC:M4
      SimulatorUnit.write(myNode, COARSE_PATH, 0, ni-1, 20);
      SimulatorUnit.checkRead(myNode, COARSE_PATH, 0, ni-1, 20);

      //M:14, H:9, GC:M14, H9
      myNode.garbageCollect(myNode.getCurrentVV());
      helperNode.sync(myNode);
      myNode.sync(helperNode);
      SimulatorUnit.checkRead(myNode, COARSE_PATH, 0, ni-1, 20);

      //M:14, H:9, GC:M14,H9
      helperNode.sync(myNode);
      assert helperNode.getCurrentVV().includes(helperNode.getOmitVV());
      SimulatorUnit.checkRead(helperNode, COARSE_PATH, 0, ni-1, 20);

      // concurrent SimulatorUnit.writes at both nodes and show that when no unverifiable SimulatorUnit.writes are discarded, same result as if log-sync
      // both nodes do SimulatorUnit.writes and one node garbage collects
      //M:14, H:19, GC:M14, H9
      SimulatorUnit.write(helperNode, COARSE_PATH, 0, ni-1, 40);
      SimulatorUnit.checkRead(helperNode, COARSE_PATH, 0, ni-1, 40);
      helperNode.garbageCollect(helperNode.getCurrentVV());
      myNode.sync(helperNode);
      helperNode.sync(myNode);

      //M:19,H:9, GC:M14,H9
      SimulatorUnit.write(myNode, COARSE_PATH, 0, ni-1, 30);
      SimulatorUnit.checkRead(myNode, COARSE_PATH, 0, ni-1, 30);

      //M:19,H:19, GC:M14,H9
      myNode.sync(helperNode);
      if(Node.dropInconsistentWrites){
        SimulatorUnit.checkRead(myNode, COARSE_PATH, 0, ni-1, 40);
      }else{
        SimulatorUnit.checkRead(myNode, COARSE_PATH, 0, ni-1, 30);

      }

    }catch(Exception e){
      // TODO Auto-generated catch block
      e.printStackTrace();assert false;
    }
  }

  //--------------------------------------------------------------------------- 
  // Test if some concurrent SimulatorUnit.writes are getting discarded while the others being retained
  //
  // 3 nodes: A, B, C
  // A creates 5 SimulatorUnit.writes and then B syncs with A.
  // A creates another 5 SimulatorUnit.writes now and B creates 5 new SimulatorUnit.writes
  // A garbage collects 
  // B syncs with A and is forced to discard its SimulatorUnit.writes
  // start helper and wait for writing to complete
  //---------------------------------------------------------------------------
  public void dtestConcurrent1(){
    IrisNode.useAgreementCheckpoint = false;

    try{
      for(int i = 1; i <2; i++){
        Node.dropInconsistentWrites = (i%2!=0);

        if(!Node.enableSecurity && Node.dropInconsistentWrites){
          continue;
        }

        BranchID myNodeId = NodeFactory.createNodeId(UNIT_NODE_ID);

        BranchID helperNodeId = NodeFactory.createNodeId(HELPER_NODE_ID);
        BranchID thirdNodeId = NodeFactory.createNodeId(MIDDLE_NODE_ID);

        Node middleNode = NodeFactory.createNodeWithoutFilter(thirdNodeId);
        if(vverbose){
          System.out.println("UNIT ID: " + myNodeId.toString());
        }
        assert(myNodeId.equals(NodeFactory.createNodeId(UNIT_NODE_ID)));

        Node myNode = NodeFactory.createNodeWithoutFilter(myNodeId);
        ((IrisNode)myNode).addSyncFilter(new InstantGCProtocol());

        Node helperNode = NodeFactory.createNodeWithoutFilter(helperNodeId);
        ((IrisNode)helperNode).addSyncFilter(new InstantGCProtocol());



        //second set of SimulatorUnit.writes

        if(verbose){
          System.out.println("Hello from SimulatorCheckpointUnit");
        }

        assert myNode.getCurrentVV().equalsIgnoreNegatives(new AcceptVV());

        int ni = 5;

        SimulatorUnit.write(middleNode, COARSE_PATH, 0, ni-1, 10);
        SimulatorUnit.checkRead(middleNode, COARSE_PATH, 0, ni-1, 10);

        //M:4
        SimulatorUnit.write(myNode, COARSE_PATH, 0, ni-1, 0);
        SimulatorUnit.checkRead(myNode, COARSE_PATH, 0, ni-1, 0);


        assert myNode.getCurrentVV().includes(new AcceptStamp(ni-1, myNodeId));
        assert !myNode.getCurrentVV().includes(new AcceptStamp(ni, myNodeId));

        helperNode.sync(middleNode);
        SimulatorUnit.checkRead(helperNode, COARSE_PATH, 0, ni-1, 10);

        //M:4
        helperNode.sync(myNode);
        SimulatorUnit.checkRead(helperNode, COARSE_PATH, 0, ni-1, 10);

        //M:9, H:0
        SimulatorUnit.write(myNode, COARSE_PATH, 0, ni-1, 20);
        SimulatorUnit.checkRead(myNode, COARSE_PATH, 0, ni-1, 20);

        // now gabage collect myNode
        //M:4, H:0: Epoch=1
        myNode.garbageCollect(myNode.getCurrentVV());
        SimulatorUnit.checkRead(myNode, COARSE_PATH, 0, ni-1, 20);
        assert myNode.getCurrentVV().includes(myNode.getOmitVV());
        assert myNode.getEpoch() == 1;

        //M:4. H:0
        middleNode.sync(myNode);
        SimulatorUnit.checkRead(middleNode, COARSE_PATH, 0, ni-1, 20);
        assert middleNode.getLogTest().get(middleNode.getBranchID()).size() == 5; // ADD 1 when the SimulatorUnit.write on receiving a certificate is enabled-- aditional for the certificate
        assert middleNode.getEpoch() == 1;

        //M:4, H:0, T:9
        SimulatorUnit.write(middleNode, COARSE_PATH, 0, ni-1, 40);
        SimulatorUnit.checkRead(middleNode, COARSE_PATH, 0, ni-1, 40);

        //M:4, H:9
        SimulatorUnit.write(helperNode, COARSE_PATH, 0, ni-1, 30);
        SimulatorUnit.checkRead(helperNode, COARSE_PATH, 0, ni-1, 30);
        assert helperNode.getEpoch() == 0;

        // forced to discard SimulatorUnit.writes
        // M:9, H:0, 
        helperNode.sync(myNode);
        if(Node.dropInconsistentWrites){
          SimulatorUnit.checkRead(helperNode, COARSE_PATH, 0, ni-1, 20);
        }else{
          SimulatorUnit.checkRead(helperNode, COARSE_PATH, 0, ni-1, 30);
        }
        assert helperNode.getLogTest().get(thirdNodeId).size() == 5;
        assert helperNode.getEpoch() == 1;


        // M:9, H:0, T:9 
        helperNode.sync(middleNode);
        SimulatorUnit.checkRead(helperNode, COARSE_PATH, 0, ni-1, 40);

        myNode.sync(helperNode);
        SimulatorUnit.checkRead(myNode, COARSE_PATH, 0, ni-1, 40);
      }
    }catch(Exception e){
      // TODO Auto-generated catch block
      e.printStackTrace();
      assert false;
    }
  }


  public void dtestGCProtocol(){
    try{
      IrisNode.useAgreementCheckpoint = false;
      BranchID myNodeId = NodeFactory.createNodeId(UNIT_NODE_ID);

      BranchID helperNodeId = NodeFactory.createNodeId(HELPER_NODE_ID);
      BranchID thirdNodeId = NodeFactory.createNodeId(MIDDLE_NODE_ID);

      SimpleGCProtocol thirdGCProtocol = new SimpleGCProtocol(3);
      Node middleNode = NodeFactory.createNodeWithoutFilter(thirdNodeId);
      ((IrisNode)middleNode).addSyncFilter(thirdGCProtocol);
      if(vverbose){
        System.out.println("UNIT ID: " + myNodeId.toString());
      }
      assert(myNodeId.equals(NodeFactory.createNodeId(UNIT_NODE_ID)));

      Node myNode = NodeFactory.createNodeWithoutFilter(myNodeId);
      SimpleGCProtocol myGCProtocol = new SimpleGCProtocol(3);
      ((IrisNode)myNode).addSyncFilter(myGCProtocol);

      Node helperNode = NodeFactory.createNodeWithoutFilter(helperNodeId);
      SimpleGCProtocol helperGCProtocol = new SimpleGCProtocol(3);
      ((IrisNode)helperNode).addSyncFilter(helperGCProtocol);



      //second set of SimulatorUnit.writes

      if(verbose){
        System.out.println("Hello from SimulatorCheckpointUnit");
      }

      assert myNode.getCurrentVV().equalsIgnoreNegatives(new AcceptVV());

      int ni = 5;

      SimulatorUnit.write(middleNode, COARSE_PATH, 0, ni-1, 10);
      SimulatorUnit.checkRead(middleNode, COARSE_PATH, 0, ni-1, 10);

      //M:4
      SimulatorUnit.write(myNode, COARSE_PATH, 0, ni-1, 0);
      SimulatorUnit.checkRead(myNode, COARSE_PATH, 0, ni-1, 0);


      assert myNode.getCurrentVV().includes(new AcceptStamp(ni-1, myNodeId));
      //  assert !myNode.getCurrentVV().includes(new AcceptStamp(ni, myNodeId));

      helperNode.sync(middleNode);
      SimulatorUnit.checkRead(helperNode, COARSE_PATH, 0, ni-1, 10);

      //M:4
      helperNode.sync(myNode);
      SimulatorUnit.checkRead(helperNode, COARSE_PATH, 0, ni-1, 10);

      //M:9, H:0
      SimulatorUnit.write(myNode, COARSE_PATH, 0, ni-1, 20);
      SimulatorUnit.checkRead(myNode, COARSE_PATH, 0, ni-1, 20);

      // now gabage collect myNode
      //M:4, H:0
      AcceptVV gcVV = myNode.getCurrentVV();
      myNode.garbageCollect(myNode.getCurrentVV());
      assert myGCProtocol.getGCProposals().size() == 1;
      assert myGCProtocol.getGCProposals().get(gcVV).size() == 1;
      SimulatorUnit.checkRead(myNode, COARSE_PATH, 0, ni-1, 20);
      assert myNode.getEpoch() == 0;
      assert myNode.getOmitVV().equalsIgnoreNegatives(new AcceptVV()):myNode.getOmitVV();

      middleNode.sync(myNode);
      assert thirdGCProtocol.getGCProposals().size() == 1;
      assert thirdGCProtocol.getGCProposals().get(gcVV).size() == 2: thirdGCProtocol.getGCProposals().get(gcVV).size();

      SimulatorUnit.write(middleNode, COARSE_PATH, 0, ni-1, 40);
      SimulatorUnit.checkRead(middleNode, COARSE_PATH, 0, ni-1, 40);

      helperNode.sync(myNode);
      assert helperGCProtocol.getGCProposals().size() == 1;
      assert helperGCProtocol.getGCProposals().get(gcVV).size() == 2;

      myNode.sync(helperNode);
      assert myGCProtocol.getGCProposals().size() == 1;
      assert myGCProtocol.getGCProposals().get(gcVV).size() == 2;

      myNode.sync(middleNode);
      assert myGCProtocol.getGCCertificates().size() == 1;
      assert myNode.getEpoch() == 1;
    }catch(Exception e){
      // TODO Auto-generated catch block
      e.printStackTrace();
      assert false;
    }
  }

  public void dtestGCProtocol1(){
    try{
      IrisNode.useAgreementCheckpoint = false;

      BranchID myNodeId = NodeFactory.createNodeId(UNIT_NODE_ID);

      BranchID helperNodeId = NodeFactory.createNodeId(HELPER_NODE_ID);
      BranchID thirdNodeId = NodeFactory.createNodeId(MIDDLE_NODE_ID);

      SimpleGCProtocol thirdGCProtocol = new SimpleGCProtocol(4);
      Node middleNode = NodeFactory.createNodeWithoutFilter(thirdNodeId);
      ((IrisNode)middleNode).addSyncFilter(thirdGCProtocol);
      if(vverbose){
        System.out.println("UNIT ID: " + myNodeId.toString());
      }
      assert(myNodeId.equals(NodeFactory.createNodeId(UNIT_NODE_ID)));

      Node myNode = NodeFactory.createNodeWithoutFilter(myNodeId);
      SimpleGCProtocol myGCProtocol = new SimpleGCProtocol(2);
      ((IrisNode)myNode).addSyncFilter(myGCProtocol);

      Node helperNode = NodeFactory.createNodeWithoutFilter(helperNodeId);
      SimpleGCProtocol helperGCProtocol = new SimpleGCProtocol(4);
      ((IrisNode)helperNode).addSyncFilter(helperGCProtocol);



      //second set of SimulatorUnit.writes

      if(verbose){
        System.out.println("Hello from SimulatorCheckpointUnit");
      }

      assert myNode.getCurrentVV().equalsIgnoreNegatives(new AcceptVV());

      int ni = 5;

      SimulatorUnit.write(middleNode, COARSE_PATH, 0, ni-1, 10);
      SimulatorUnit.checkRead(middleNode, COARSE_PATH, 0, ni-1, 10);

      //M:4
      SimulatorUnit.write(myNode, COARSE_PATH, 0, ni-1, 0);
      SimulatorUnit.checkRead(myNode, COARSE_PATH, 0, ni-1, 0);


      assert myNode.getCurrentVV().includes(new AcceptStamp(ni-1, myNodeId));
      assert !myNode.getCurrentVV().includes(new AcceptStamp(ni, myNodeId));

      helperNode.sync(middleNode);
      SimulatorUnit.checkRead(helperNode, COARSE_PATH, 0, ni-1, 10);

      //M:4
      helperNode.sync(myNode);
      SimulatorUnit.checkRead(helperNode, COARSE_PATH, 0, ni-1, 10);

      //M:9, H:0
      SimulatorUnit.write(myNode, COARSE_PATH, 0, ni-1, 20);
      SimulatorUnit.checkRead(myNode, COARSE_PATH, 0, ni-1, 20);

      // now gabage collect myNode
      //M:4, H:0
      AcceptVV gcVV = myNode.getCurrentVV();
      myNode.garbageCollect(myNode.getCurrentVV());
      assert myGCProtocol.getGCProposals().size() == 1;
      assert myGCProtocol.getGCProposals().get(gcVV).size() == 1;
      SimulatorUnit.checkRead(myNode, COARSE_PATH, 0, ni-1, 20);
      assert myNode.getEpoch() == 0;
      assert myNode.getOmitVV().equalsIgnoreNegatives(new AcceptVV()):myNode.getOmitVV();

      middleNode.sync(myNode);
      assert thirdGCProtocol.getGCProposals().size() == 1;
      assert thirdGCProtocol.getGCProposals().get(gcVV).size() == 2: thirdGCProtocol.getGCProposals().get(gcVV).size();

      SimulatorUnit.write(middleNode, COARSE_PATH, 0, ni-1, 40);
      SimulatorUnit.checkRead(middleNode, COARSE_PATH, 0, ni-1, 40);

      helperNode.sync(myNode);
      assert helperGCProtocol.getGCProposals().size() == 1;
      assert helperGCProtocol.getGCProposals().get(gcVV).size() == 2;

      myNode.sync(helperNode);
      assert myGCProtocol.getGCCertificates().size() == 1;
      assert myNode.getEpoch() == 1;

      myNode.sync(middleNode);
      assert myGCProtocol.getGCCertificates().size() == 1;
      assert myNode.getEpoch() == 1;

      middleNode.sync(myNode);
      assert thirdGCProtocol.getGCCertificates().size() == 1;
      assert middleNode.getEpoch() == 1;

      helperNode.sync(myNode);
      assert helperGCProtocol.getGCCertificates().size() == 1;
      assert helperNode.getEpoch() == 1;
    }catch(Exception e){
      e.printStackTrace();
      assert false;
    }
  }

  public void testConcurrent(){
    IrisNode.useAgreementCheckpoint = false;

    try{
      StorageConfig.Client2WriteServerMap.clear();
      StorageConfig.Client2WriteServerMap.put(0, 0);
      StorageConfig.Client2WriteServerMap.put(1, 1);
      StorageConfig.Client2WriteServerMap.put(2, 2);

      Node.dropInconsistentWrites = false;

      BranchID myNodeId = NodeFactory.createNodeId(UNIT_NODE_ID+1);

      BranchID helperNodeId = NodeFactory.createNodeId(HELPER_NODE_ID);
      if(vverbose){
        System.out.println("UNIT ID: " + myNodeId.toString());
      }
      assert(myNodeId.equals(NodeFactory.createNodeId(UNIT_NODE_ID+1)));

      Node myNode = NodeFactory.createNodeWithoutFilter(myNodeId);
      Node helperNode = NodeFactory.createNodeWithoutFilter(helperNodeId);
      
      //second set of SimulatorUnit.writes

      if(verbose){
        System.out.println("Hello from SimulatorCheckpointUnit");
      }

      assert myNode.getCurrentVV().equalsIgnoreNegatives(new AcceptVV());

      int ni = 5;
      //M:4
      SimulatorUnit.write(myNode, COARSE_PATH, 0, ni-1, 0);
      SimulatorUnit.checkRead(myNode, COARSE_PATH, 0, ni-1, 0);


      assert myNode.getCurrentVV().includes(new AcceptStamp(ni-1, myNodeId));
      assert !myNode.getCurrentVV().includes(new AcceptStamp(ni, myNodeId));

      //M:4
      helperNode.sync(myNode);

      SimulatorUnit.checkRead(helperNode, COARSE_PATH, 0, ni-1, 0);

      //M:9, H:-1
      SimulatorUnit.write(myNode, COARSE_PATH, 0, ni-1, 10);
      SimulatorUnit.checkRead(myNode, COARSE_PATH, 0, ni-1, 10);

      // now gabage collect myNode
      //M:4, H:-1
      myNode.garbageCollect(myNode.getCurrentVV());
      helperNode.sync(myNode);
      myNode.sync(helperNode);

      SimulatorUnit.checkRead(myNode, COARSE_PATH, 0, ni-1, 10);
      assert myNode.getCurrentVV().includes(myNode.getOmitVV());

      //M:4, H:9
      SimulatorUnit.write(helperNode, COARSE_PATH, 0, ni-1, 20);
      SimulatorUnit.checkRead(helperNode, COARSE_PATH, 0, ni-1, 20);

      // forced to discard SimulatorUnit.writes
      // M:9, H:0
      helperNode.sync(myNode);

      assert !helperNode.getOmitVV().includes(helperNode.getCurrentVV());
      if(Node.dropInconsistentWrites){
        SimulatorUnit.checkRead(helperNode, COARSE_PATH, 0, ni-1, 10);
      }else{
        SimulatorUnit.checkRead(helperNode, COARSE_PATH, 0, ni-1, 20);
      }

      assert helperNode.getCurrentVV().includes(helperNode.getOmitVV());

    }catch(Exception e){
      // TODO Auto-generated catch block
      e.printStackTrace();
      assert false;
    }

  }

  /**
   * test if checkpoitn can be succesfully verfied despite differences in branches?
   */
  public void dtestCheckpointVerification(){
    try{
      BranchID myNodeId = NodeFactory.createNodeId(UNIT_NODE_ID);
      BranchID nodeId1 = NodeFactory.createNodeId(15);
      BranchID nodeId4 = NodeFactory.createNodeId(30);

      Node myNode = NodeFactory.createNodeWithoutFilter(myNodeId);
      Node node1 = NodeFactory.createNodeWithoutFilter(nodeId1);
      Node node4 = NodeFactory.createNodeWithoutFilter(nodeId4);

      ObjId oid0 = new ObjId("/test/0");
      myNode.write(oid0, new IrisDataObject(new Integer(0)));

      ObjId oid1 = new ObjId("/test/1");
      myNode.write(oid1, new IrisDataObject(new Integer(1)));

      LinkedList<Node> forks = myNode.fork(2);

      forks.get(0).write(new ObjId("/test/00"), new IrisDataObject(new Integer(2)));
      forks.get(0).write(new ObjId("/test/01"), new IrisDataObject(new Integer(3)));
      forks.get(1).write(new ObjId("/test/10"), new IrisDataObject(new Integer(4)));
      forks.get(1).write(new ObjId("/test/11"), new IrisDataObject(new Integer(5)));

      LinkedList<Node> newForks = forks.get(0).fork(2);
      newForks.get(0).write(new ObjId("/test/0000"), new IrisDataObject(new Integer(6)));
      newForks.get(0).write(new ObjId("/test/0001"), new IrisDataObject(new Integer(7)));
      newForks.get(1).write(new ObjId("/test/0010"), new IrisDataObject(new Integer(8)));
      newForks.get(1).write(new ObjId("/test/0011"), new IrisDataObject(new Integer(9)));

      node1.sync(forks.get(0));
      node1.write(new ObjId("/test/0011"), new IrisDataObject(new Integer(9)));
      node4.sync(forks.get(0));
      node4.write(new ObjId("/test/0011"), new IrisDataObject(new Integer(9)));

      node1.sync(newForks.get(0));
      node4.sync(newForks.get(0));   

      Log log = node1.getLogTest();
      CounterVV vv = new CounterVV(log.getMinimalVV(node1.getCurrentVV()));
      vv.dropNode(node1.getBranchID());
      AcceptVV cvv = new AcceptVV(vv);
      HashMap<NodeId, Hash> hashMap = node1.getLogTest().getHashMap(cvv);



      Checkpoint chkPt = node1.generateNewCheckpoint(new HashedVV(cvv, hashMap), 1);

      node1.sync(node4);

      // find out what the cvv maps to

      assert node1.verifyCheckpoint(chkPt);

    }catch(Exception e){
      // TODO Auto-generated catch block
      e.printStackTrace();
    }
  }

  /**
   * test if a checkpoint can be succesffully applied despite differences in history (branches)?
   */
  public void dtestCheckpointApplicability(){
    try{
      BranchID myNodeId = NodeFactory.createNodeId(UNIT_NODE_ID);
      BranchID nodeId1 = NodeFactory.createNodeId(15);
      BranchID nodeId2 = NodeFactory.createNodeId(20);
      BranchID nodeId3 = NodeFactory.createNodeId(25);

      IrisNode myNode = (IrisNode)NodeFactory.createNodeWithoutFilter(myNodeId);
      IrisNode node1 = (IrisNode)NodeFactory.createNodeWithoutFilter(nodeId1);
      IrisNode node2 = (IrisNode)NodeFactory.createNodeWithoutFilter(nodeId2);
      IrisNode node3 = (IrisNode)NodeFactory.createNodeWithoutFilter(nodeId3);

      ObjId oid0 = new ObjId("/test/0");
      myNode.write(oid0, new IrisDataObject(new Integer(0)));

      ObjId oid1 = new ObjId("/test/1");
      myNode.write(oid1, new IrisDataObject(new Integer(1)));

      LinkedList<Node> forks = myNode.fork(2);

      forks.get(0).write(new ObjId("/test/00"), new IrisDataObject(new Integer(2)));
      forks.get(0).write(new ObjId("/test/01"), new IrisDataObject(new Integer(3)));
      forks.get(1).write(new ObjId("/test/10"), new IrisDataObject(new Integer(4)));
      forks.get(1).write(new ObjId("/test/11"), new IrisDataObject(new Integer(5)));

      LinkedList<Node> newForks = forks.get(0).fork(2);
      newForks.get(0).write(new ObjId("/test/0000"), new IrisDataObject(new Integer(6)));
      newForks.get(0).write(new ObjId("/test/0001"), new IrisDataObject(new Integer(7)));
      newForks.get(1).write(new ObjId("/test/0010"), new IrisDataObject(new Integer(8)));
      newForks.get(1).write(new ObjId("/test/0011"), new IrisDataObject(new Integer(9)));

      node1.sync(newForks.get(0));
      node2.sync(forks.get(1));
      node3.sync(newForks.get(1));

      if(!SangminConfig.useSimulatorId){
        ((IrisNode)node1).exponentialBackoffSync(node2);
      }
      node1.sync(node2);

      Log log = node1.getLogTest();
      AcceptVV cvv = log.getMinimalVV(node1.getCurrentVV());
      HashMap<NodeId, Hash> hashMap = node1.getLogTest().getHashMap(cvv);
      Checkpoint chkPt = node1.generateNewCheckpoint(new HashedVV(cvv, hashMap), 1);
      if(!SangminConfig.useSimulatorId){
        ((IrisNode)node1).exponentialBackoffSync(node3);
      }
      node1.sync(node3);


      AcceptVV remappedVV = log.remap(new HashedVV(cvv, hashMap), true);
      AcceptVV currentVV = log.getMinimalVV(node1.getCurrentVV());
      if(Node.enableSecurity){
        assert !remappedVV.equalsIgnoreNegatives(currentVV):remappedVV +  " "   + currentVV;
      }

      node1.applySyncPacket(this.getSyncPacket(chkPt));

      log = node1.getLogTest();
      remappedVV = log.remap(new HashedVV(cvv, hashMap), true);
      currentVV = log.getMinimalVV(node1.getCurrentVV());
      CounterVV CVV = new CounterVV(currentVV);
      CVV.dropNode(node1.getBranchID());
      currentVV = new AcceptVV(CVV);
      assert remappedVV.equalsIgnoreNegatives(currentVV):remappedVV +  "\n "   + currentVV;


    }catch(Exception e){
      // TODO Auto-generated catch block
      e.printStackTrace();
    }
  }

  /**
   * Tests whether checkpoints generated are identical or not despite differences in history
   */
  public void dtestCheckpointEquivalence(){
    // create two nodes with differences in the history prior to the omitVV and request checkpoint from these two nodes
    // check if they are identical or not?

    try{
      BranchID myNodeId = NodeFactory.createNodeId(UNIT_NODE_ID);
      BranchID nodeId1 = NodeFactory.createNodeId(15);
      BranchID nodeId2 = NodeFactory.createNodeId(20);
      BranchID nodeId3 = NodeFactory.createNodeId(25);

      Node myNode = NodeFactory.createNodeWithoutFilter(myNodeId);
      Node node1 = NodeFactory.createNodeWithoutFilter(nodeId1);
      Node node2 = NodeFactory.createNodeWithoutFilter(nodeId2);
      Node node3 = NodeFactory.createNodeWithoutFilter(nodeId3);

      ObjId oid0 = new ObjId("/test/0");
      myNode.write(oid0, new IrisDataObject(new Integer(0)));

      ObjId oid1 = new ObjId("/test/1");
      myNode.write(oid1, new IrisDataObject(new Integer(1)));

      LinkedList<Node> forks = myNode.fork(2);

      forks.get(0).write(new ObjId("/test/00"), new IrisDataObject(new Integer(2)));
      forks.get(0).write(new ObjId("/test/01"), new IrisDataObject(new Integer(3)));
      forks.get(1).write(new ObjId("/test/10"), new IrisDataObject(new Integer(4)));
      forks.get(1).write(new ObjId("/test/11"), new IrisDataObject(new Integer(5)));

      LinkedList<Node> newForks = forks.get(0).fork(2);
      newForks.get(0).write(new ObjId("/test/0000"), new IrisDataObject(new Integer(6)));
      newForks.get(0).write(new ObjId("/test/0001"), new IrisDataObject(new Integer(7)));
      newForks.get(1).write(new ObjId("/test/0010"), new IrisDataObject(new Integer(8)));
      newForks.get(1).write(new ObjId("/test/0011"), new IrisDataObject(new Integer(9)));

      node1.sync(newForks.get(0));
      node2.sync(forks.get(1));
      node3.sync(newForks.get(1));

      if(!SangminConfig.useSimulatorId){
        ((IrisNode)node1).exponentialBackoffSync(node2);
      }
      node1.sync(node2);

      Log log = node1.getLogTest();
      AcceptVV cvv = log.getMinimalVV(node1.getCurrentVV());
      HashMap<NodeId, Hash> hashMap = node1.getLogTest().getHashMap(cvv);
      Checkpoint chkPt = node1.generateNewCheckpoint(new HashedVV(cvv, hashMap), 1);
      if(!SangminConfig.useSimulatorId){
        ((IrisNode)node1).exponentialBackoffSync(node3);
      }
      node1.sync(node3);

      // find out what the cvv maps to
      Checkpoint chkPt1 = node1.generateNewCheckpoint(new HashedVV(cvv, hashMap), 1, (HashedVV)chkPt.endVV);
      assert chkPt.omitVV.equals(chkPt1.omitVV): chkPt.omitVV + " " + chkPt1.omitVV;
      assert chkPt.epochCount == chkPt1.epochCount: chkPt.epochCount + " " + chkPt1.epochCount;
      assert chkPt.lastWrite.equals(chkPt1.lastWrite): chkPt.lastWrite + " " + chkPt1.lastWrite;
      assert chkPt.objectStore.equals(chkPt1.objectStore): chkPt.objectStore + " " + chkPt1.objectStore;
      assert chkPt.unverifiableWrites.equals(chkPt1.unverifiableWrites): "\n"+ chkPt.unverifiableWrites + " " + chkPt1.unverifiableWrites;
      assert chkPt.endVV.equals(chkPt1.endVV): "\n" + chkPt.endVV + "\n" + chkPt1.endVV;
      assert chkPt.flagMap.equals(chkPt1.flagMap): chkPt.flagMap + " " + chkPt1.flagMap;
      assert chkPt.equals(chkPt1);
      assert Arrays.equals(chkPt.obj2Bytes(), chkPt1.obj2Bytes()):"\n" +Arrays.toString(chkPt.obj2Bytes()) + "\n chkPt1.obj2Bytes() \n" + Arrays.toString(chkPt1.obj2Bytes());
      assert chkPt.getHash().equals(chkPt1.getHash()):"\n" +Arrays.toString(chkPt.obj2Bytes()) + "\n chkPt1.obj2Bytes() \n" + Arrays.toString(chkPt1.obj2Bytes());

    }catch(Exception e){
      e.printStackTrace();
      System.exit(0);
    }


  }

  /**
   * test whether the handle failure method is working correctly by applying new POM to the log if any is present
   */
  public void dtestHandleFailure(){
    try{
      if(!Node.enableSecurity){
        return;
      }
      IrisNode.useAgreementCheckpoint = false;

      BranchID myNodeId = NodeFactory.createNodeId(UNIT_NODE_ID);
      BranchID nodeId1 = NodeFactory.createNodeId(15);
      BranchID nodeId2 = NodeFactory.createNodeId(20);
      BranchID nodeId3 = NodeFactory.createNodeId(25);

      Node myNode = NodeFactory.createNodeWithoutFilter(myNodeId);
      Node node1 = NodeFactory.createNodeWithoutFilter(nodeId1);
      Node node2 = NodeFactory.createNodeWithoutFilter(nodeId2);
      Node node3 = NodeFactory.createNodeWithoutFilter(nodeId3);

      ObjId oid0 = new ObjId("/test/0");
      myNode.write(oid0, new IrisDataObject(new Integer(0)));

      ObjId oid1 = new ObjId("/test/1");
      myNode.write(oid1, new IrisDataObject(new Integer(1)));

      LinkedList<Node> forks = myNode.fork(2);

      forks.get(0).write(new ObjId("/test/00"), new IrisDataObject(new Integer(2)));
      forks.get(0).write(new ObjId("/test/01"), new IrisDataObject(new Integer(3)));
      forks.get(1).write(new ObjId("/test/10"), new IrisDataObject(new Integer(4)));
      forks.get(1).write(new ObjId("/test/11"), new IrisDataObject(new Integer(5)));

      LinkedList<Node> newForks = forks.get(0).fork(2);
      newForks.get(0).write(new ObjId("/test/0000"), new IrisDataObject(new Integer(6)));
      newForks.get(0).write(new ObjId("/test/0001"), new IrisDataObject(new Integer(7)));
      newForks.get(1).write(new ObjId("/test/0010"), new IrisDataObject(new Integer(8)));
      newForks.get(1).write(new ObjId("/test/0011"), new IrisDataObject(new Integer(9)));

      node1.sync(newForks.get(0));
      node2.sync(forks.get(1));
      node3.sync(newForks.get(1));

      //      if(!SangminConfig.useSimulatorId){
      //        ((IrisNode)node1).exponentialBackoffSync(node2);
      //      }
      //      node1.sync(node2); // node1 now has two branches: b00 and b1

      Log log = node1.getLogTest();
      AcceptVV cvv = log.getMinimalVV(node1.getCurrentVV());

      VVIterator vvi = cvv.getIterator();
      BranchID b = null; // corresponds to b00
      while(vvi.hasMoreElements()){
        b = (BranchID)vvi.getNext();
        if(cvv.getStampByIteratorToken(b) > 4){
          break;
        }
      }
      assert cvv.getStampByIteratorToken(b) == 5:" cvv  " + cvv + " " + b ;
      CounterVV currentVV = new CounterVV(cvv);
      currentVV.dropNode(b);
      currentVV.addMaxAS(new AcceptStamp(2, b)); //b00:2, b1:3

      HashMap<NodeId, Hash> hashMap = node1.getLogTest().getHashMap(new AcceptVV(currentVV));
      Checkpoint chkPt = node1.generateNewCheckpoint(new HashedVV(currentVV, hashMap), 1);
      if(!SangminConfig.useSimulatorId){
        ((IrisNode)node1).exponentialBackoffSync(node3);
      }
      node1.sync(node3);
      cvv = log.getMinimalVV(node1.getCurrentVV());
      ((IrisNode)node1).applySyncPacket(this.getSyncPacket(chkPt));

      CounterVV cvv1 = new CounterVV(node1.getLogTest().getMinimalVV(node1.getCurrentVV()));
      cvv1.dropNode(node1.getBranchID());

      assert new AcceptVV(cvv1).equalsIgnoreNegatives(cvv): "\n"+cvv1 + "\n cvv "  +cvv;
    }catch(Exception e){
      // TODO Auto-generated catch block
      e.printStackTrace();
    }
  }


  /*
   * "new TestSuite(Class c)" constructs a test suite
   * containg every method whose name begins with "/test"
   * 
   * TBD: update class name
   */
  public static Test suite(){
    TestSuite suite = new TestSuite(LadderedCheckpointUnit.class);
    return suite;
  }


  /*
   * main() lets us run just this set of unit tests
   * from the comand line (you can also invoke 
   * the testrunner on this class and it will find
   * the suite())
   *
   * usage: java <classname> [-verbose] [-vverbose] [testName]*
   * 
   *   If verbose or vverbose are included, print info to screen
   *
   *   If [testName]* are included, then run test called "/test[testName]"
   *   for each such [testName]. E.g., "java TestEmtpy foo" runs
   *   TwoNodesSubscriptionUnit.testfoo() as a TestCase.
   *
   * TBD: update class name
   */
  public static void main(String s[]) {
    String name = "SimulatorCheckpointUnit";
    System.err.print(name + " self test begins...");
    Test test;
    test = suite();
    TestRunner tr = new TestRunner();
    tr.doRun(test);
    System.exit(0);
  }

}



