package code.simulator.checkpoint.unit;

import code.branchDetecting.BranchID;
import code.security.SangminConfig;
import code.simulator.*;
import code.simulator.agreement.AgreementInstance;
import code.simulator.agreement.AgreementParams;
import code.simulator.agreement.InstanceID;
import code.simulator.agreement.unit.TestCommunicationChannel;
import code.simulator.checkpoint.Checkpoint;
import code.simulator.log.Log;
import code.simulator.protocolFilters.SimpleGCProtocol;
import code.simulator.protocolFilters.InstantGCProtocol;
import code.simulator.store.StoreEntry;
import code.*;
/** 
 *  Test bandwidth to subscribe to 1000 objects, 100 of which have
 *  been modified. Test created for sosp 2007 submission microbenchmark.
 **/ 

//package utils.junit;

import junit.textui.TestRunner;
import junit.framework.*;
import java.util.*;
import java.io.*;

/**
 * TBD: Update class name
 */
public class SimulatorCheckpointGenerationUnit extends TestCase {
  public static final String TEST_ALL_TEST_TYPE = "UNIT";

  /**
   * Basic constructor - called by the test runners.
   * TBD: Update constructor name to match class
   */
  public SimulatorCheckpointGenerationUnit (final String s) {
    super (s);
  }

  /*
   * Fixtures are run before and after each test case
   * to set up environment in which tests must run.
   */
  protected void setUp() throws Exception{
    super.setUp();
    makePractiConfig(0, 100);

  }

  LinkedList<IrisNode> createCommunicationChanels(int num){
    LinkedList<IrisNode> list = new LinkedList<IrisNode>();
    this.makePractiConfig(0, num);
    for(int i = 0; i < num; i++){
      IrisNode tcc = (IrisNode)NodeFactory.createIrisNode(new BranchID(i));
      list.add(tcc);
    }

    return list;
  }
  
  private void syncAll(IrisNode src, Collection<IrisNode> channels){
    for(IrisNode tc: channels){
      try{
        tc.exponentialBackoffSync(src);
        tc.sync(src);
      }catch(Exception e){
        // TODO Auto-generated catch block
        e.printStackTrace();
        assert false;
      }
    }
  }
  
  public static void makePractiConfig(long first, int total){
    Config.createEmptyConfig();

    long NODE_1_ID = first;
    int port = 9921;
    String NODE_1_IP = "localhost";
    
    for(int i = 0; i < total; i++){
      
      Config.addOneNodeConfig(new NodeId(NODE_1_ID++),
                              NODE_1_IP,
                            port++,
                            port++,
                            port++,
                            port++,
                            port++,
                            "/test" + File.separatorChar + "local-" + 
                            NODE_1_ID + ".db",
                            "/*",
                            -1L,
                            NODE_1_IP,
                            port++,
                            port++,
                            -1,
                            Config.CACHE_SIZE_BYTES_DEFAULT,
                            Config.MAX_LOG_DISK_SIZE_BYTES,
                            Config.MAX_LOG_MEM_SIZE_BYTES);
    }

    Config.readKeys();//needed for secure practi
        
  }

  public static int UNIT_NODE_ID = 10;
  public static int MIDDLE_NODE_ID = 15;

  public static int HELPER_NODE_ID = 20;
  public static boolean vverbose = false;
  public static boolean verbose = true;


  public static String COARSE_PATH = "/coarse";


  //--------------------------------------------------------------------------- 
  // Test coarse-grained subscriptions. 
  //
  // To initialize state, Unit create 1000 objects
  // (/data-coarse/[0..9]/[0..9]/[0..9]. Then helper subscribe
  // to /* from unit and wait for /data-coarse/9/9/9 to arrive.
  //
  //
  // start helper and wait for writing to complete
  //---------------------------------------------------------------------------
  public void testSimple(){
    try{
      IrisNode.useAgreementCheckpoint = true;

      for(int i = 1; i <2; i++){
        Node.dropInconsistentWrites = (i%2!=0);
        if(!Node.enableSecurity && Node.dropInconsistentWrites){
          continue;
        }

        System.out.println("dropping inconsistency writes " + Node.dropInconsistentWrites);
        BranchID myNodeId = NodeFactory.createNodeId(UNIT_NODE_ID);


        BranchID helperNodeId = NodeFactory.createNodeId(HELPER_NODE_ID);
        if(vverbose){
          System.out.println("UNIT ID: " + myNodeId.toString());
        }
        assert(myNodeId.equals(NodeFactory.createNodeId(UNIT_NODE_ID)));

        Node myNode = NodeFactory.createNodeWithoutFilter(myNodeId);
        ((IrisNode)myNode).addSyncFilter(new InstantGCProtocol());
        Node helperNode = NodeFactory.createNodeWithoutFilter(helperNodeId);
        ((IrisNode)helperNode).addSyncFilter(new InstantGCProtocol());

        //second set of writes

        if(verbose){
          System.out.println("Hello from SimulatorCheckpointUnit");
        }

        assert myNode.getCurrentVV().equalsIgnoreNegatives(new AcceptVV());

        int ni = 5;
        //M:4
        write(myNode, COARSE_PATH, 0, ni-1, 0);
        checkRead(myNode, COARSE_PATH, 0, ni-1, 0);


        assert myNode.getCurrentVV().includes(new AcceptStamp(ni-1, myNodeId));
        assert !myNode.getCurrentVV().includes(new AcceptStamp(ni, myNodeId));

        //M:4
        helperNode.sync(myNode);

        checkRead(helperNode, COARSE_PATH, 0, ni-1, 0);

        // now gabage collect myNode
        //M:4, GC:M4
        Log log = myNode.getLogTest();
        HashedVV omitVV = new HashedVV(myNode.getCurrentVV(), log.getHashMap(myNode.getCurrentVV()));
        Checkpoint chkPt = myNode.generateNewCheckpoint(omitVV, myNode.getEpoch()+1);
        ((IrisNode)myNode).applySyncPacket(SimulatorCheckpointUnit.getSyncPacket(chkPt));

        assert myNode.getEpoch()==1: myNode.getEpoch();
        checkRead(myNode, COARSE_PATH, 0, ni-1, 0);
        assert myNode.getCurrentVV().includes(myNode.getOmitVV());

        //M:4, GC:M4
        helperNode.sync(myNode);
        assert helperNode.getEpoch()==1;
        checkRead(helperNode, COARSE_PATH, 0, ni-1, 0);

        // now do writes at helper node and sync with the main node
        //M:4, H:9. GC:M4
        write(helperNode, COARSE_PATH, 0, ni-1, 10);
        checkRead(helperNode, COARSE_PATH, 0, ni-1, 10);

        assert helperNode.getCurrentVV().includes(new AcceptStamp(ni-1+5, helperNodeId));
        // assert !helperNode.getCurrentVV().includes(new AcceptStamp(ni+5, helperNodeId));

        //M:4, H:9, GC:M4
        myNode.sync(helperNode);
        checkRead(myNode, COARSE_PATH, 0, ni-1, 10);

        //M:14, H:9, GC:M4
        write(myNode, COARSE_PATH, 0, ni-1, 20);
        checkRead(myNode, COARSE_PATH, 0, ni-1, 20);

        //M:14, H:9, GC:M14, H9
        log = myNode.getLogTest();
        omitVV = new HashedVV(myNode.getCurrentVV(), log.getHashMap(myNode.getCurrentVV()));
        chkPt = myNode.generateNewCheckpoint(omitVV, myNode.getEpoch()+1);
        ((IrisNode)myNode).applySyncPacket(SimulatorCheckpointUnit.getSyncPacket(chkPt));
        checkRead(myNode, COARSE_PATH, 0, ni-1, 20);

        //M:14, H:9, GC:M14,H9
        helperNode.sync(myNode);
        assert helperNode.getCurrentVV().includes(helperNode.getOmitVV());
        checkRead(helperNode, COARSE_PATH, 0, ni-1, 20);

        // concurrent writes at both nodes and show that when no unverifiable writes are discarded, same result as if log-sync
        // both nodes do writes and one node garbage collects
        //M:14, H:19, GC:M14, H9
        write(helperNode, COARSE_PATH, 0, ni-1, 40);
        checkRead(helperNode, COARSE_PATH, 0, ni-1, 40);
        log = helperNode.getLogTest();
        omitVV = new HashedVV(helperNode.getCurrentVV(), log.getHashMap(helperNode.getCurrentVV()));
        chkPt = helperNode.generateNewCheckpoint(omitVV, helperNode.getEpoch()+1);
        ((IrisNode)helperNode).applySyncPacket(SimulatorCheckpointUnit.getSyncPacket(chkPt));

        //M:19,H:9, GC:M14,H9
        write(myNode, COARSE_PATH, 0, ni-1, 30);
        checkRead(myNode, COARSE_PATH, 0, ni-1, 30);

        //M:19,H:19, GC:M14,H9
        myNode.sync(helperNode);
        if(Node.dropInconsistentWrites){
          checkRead(myNode, COARSE_PATH, 0, ni-1, 40);
        }else{
          checkRead(myNode, COARSE_PATH, 0, ni-1, 30);

        }
      }
    }catch(Exception e){
      // TODO Auto-generated catch block
      e.printStackTrace();assert false;
    }
  }

  //--------------------------------------------------------------------------- 
  // Test if some concurrent writes are getting discarded while the others being retained
  //
  // 3 nodes: A, B, C
  // A creates 5 writes and then B syncs with A.
  // A creates another 5 writes now and B creates 5 new writes
  // A garbage collects 
  // B syncs with A and is forced to discard its writes
  // start helper and wait for writing to complete
  //---------------------------------------------------------------------------
  public void testConcurrent1(){
    IrisNode.useAgreementCheckpoint = true;

    try{
      for(int i = 1; i <2; i++){
        Node.dropInconsistentWrites = (i%2!=0);

        if(!Node.enableSecurity && Node.dropInconsistentWrites){
          continue;
        }

        BranchID myNodeId = NodeFactory.createNodeId(UNIT_NODE_ID);

        BranchID helperNodeId = NodeFactory.createNodeId(HELPER_NODE_ID);
        BranchID thirdNodeId = NodeFactory.createNodeId(MIDDLE_NODE_ID);

        Node middleNode = NodeFactory.createNodeWithoutFilter(thirdNodeId);
        if(vverbose){
          System.out.println("UNIT ID: " + myNodeId.toString());
        }
        assert(myNodeId.equals(NodeFactory.createNodeId(UNIT_NODE_ID)));

        Node myNode = NodeFactory.createNodeWithoutFilter(myNodeId);
        Node helperNode = NodeFactory.createNodeWithoutFilter(helperNodeId);
        
        //second set of writes

        if(verbose){
          System.out.println("Hello from SimulatorCheckpointUnit");
        }

        assert myNode.getCurrentVV().equalsIgnoreNegatives(new AcceptVV());

        int ni = 5;

        write(middleNode, COARSE_PATH, 0, ni-1, 10);
        checkRead(middleNode, COARSE_PATH, 0, ni-1, 10);

        //M:4
        write(myNode, COARSE_PATH, 0, ni-1, 0);
        checkRead(myNode, COARSE_PATH, 0, ni-1, 0);


        assert myNode.getCurrentVV().includes(new AcceptStamp(ni-1, myNodeId));
        assert !myNode.getCurrentVV().includes(new AcceptStamp(ni, myNodeId));

        helperNode.sync(middleNode);
        checkRead(helperNode, COARSE_PATH, 0, ni-1, 10);

        //M:4
        helperNode.sync(myNode);
        checkRead(helperNode, COARSE_PATH, 0, ni-1, 10);

        //M:9, H:0
        write(myNode, COARSE_PATH, 0, ni-1, 20);
        checkRead(myNode, COARSE_PATH, 0, ni-1, 20);

        // now gabage collect myNode
        //M:4, H:0: Epoch=1
        Log log = myNode.getLogTest();
        HashedVV omitVV = new HashedVV(myNode.getCurrentVV(), log.getHashMap(myNode.getCurrentVV()));
        Checkpoint chkPt = myNode.generateNewCheckpoint(omitVV, myNode.getEpoch()+1);
        ((IrisNode)myNode).applySyncPacket(SimulatorCheckpointUnit.getSyncPacket(chkPt));
        
        checkRead(myNode, COARSE_PATH, 0, ni-1, 20);
        assert myNode.getCurrentVV().includes(myNode.getOmitVV());
        assert myNode.getEpoch() == 1;

        //M:4. H:0
        middleNode.sync(myNode);
        checkRead(middleNode, COARSE_PATH, 0, ni-1, 20);
        assert middleNode.getLogTest().get(middleNode.getBranchID()).size() == 5; // add 1 aditional for the certificate
        assert middleNode.getEpoch() == 1;

        //M:4, H:0, T:9
        write(middleNode, COARSE_PATH, 0, ni-1, 40);
        checkRead(middleNode, COARSE_PATH, 0, ni-1, 40);

        //M:4, H:9
        write(helperNode, COARSE_PATH, 0, ni-1, 30);
        checkRead(helperNode, COARSE_PATH, 0, ni-1, 30);
        assert helperNode.getEpoch() == 0;

        // forced to discard writes
        // M:9, H:0, 
        helperNode.sync(myNode);
        if(Node.dropInconsistentWrites){
          checkRead(helperNode, COARSE_PATH, 0, ni-1, 20);
        }else{
          checkRead(helperNode, COARSE_PATH, 0, ni-1, 30);
        }
        assert helperNode.getLogTest().get(thirdNodeId).size() == 5;
        assert helperNode.getEpoch() == 1;


        // M:9, H:0, T:9 
        helperNode.sync(middleNode);
        checkRead(helperNode, COARSE_PATH, 0, ni-1, 40);

        myNode.sync(helperNode);
        checkRead(myNode, COARSE_PATH, 0, ni-1, 40);
      }
    }catch(Exception e){
      // TODO Auto-generated catch block
      e.printStackTrace();
      assert false;
    }
  }


  public void testGCProtocol(){
    try{
      IrisNode.useAgreementCheckpoint = true;

      LinkedList<IrisNode> nodes = this.createCommunicationChanels(6);
      Node middleNode = nodes.get(2);
      Node myNode = nodes.get(0);
      Node helperNode = nodes.get(1);
      BranchID myNodeId = myNode.getBranchID();

      BranchID helperNodeId = helperNode.getBranchID();
      BranchID thirdNodeId = middleNode.getBranchID();

      //second set of writes
      if(verbose){
        System.out.println("Hello from SimulatorCheckpointUnit");
      }

      assert myNode.getCurrentVV().equalsIgnoreNegatives(new AcceptVV());

      int ni = 5;

      write(middleNode, COARSE_PATH, 0, ni-1, 10);
      checkRead(middleNode, COARSE_PATH, 0, ni-1, 10);

      //M:4
      write(myNode, COARSE_PATH, 0, ni-1, 0);
      checkRead(myNode, COARSE_PATH, 0, ni-1, 0);


      assert myNode.getCurrentVV().includes(new AcceptStamp(ni-1, myNodeId));
      //  assert !myNode.getCurrentVV().includes(new AcceptStamp(ni, myNodeId));

      helperNode.sync(middleNode);
      checkRead(helperNode, COARSE_PATH, 0, ni-1, 10);

      //M:4
      helperNode.sync(myNode);
      checkRead(helperNode, COARSE_PATH, 0, ni-1, 10);

      //M:9, H:0
      write(myNode, COARSE_PATH, 0, ni-1, 20);
      checkRead(myNode, COARSE_PATH, 0, ni-1, 20);

      // now gabage collect myNode
      //M:4, H:0
      myNode.garbageCollect(myNode.getCurrentVV());
      
      for(IrisNode in: nodes){
        assert in.getOmitVV().equalsIgnoreNegatives(new AcceptVV());
      }
      
      for(int i = 0; i < 5; i++){
        for(IrisNode in: nodes){
          syncAll(in, nodes);
        }
        Thread.yield();
      }
      
      for(IrisNode in: nodes){
        assert !in.getOmitVV().equalsIgnoreNegatives(new AcceptVV()):in.getOmitVV();
        assert in.getEpoch() == 1;
        checkRead(in, COARSE_PATH, 0, ni-1, 20);

      }
      
    }catch(Exception e){
      // TODO Auto-generated catch block
      e.printStackTrace();
      assert false;
    }
  }

  public void testGCProtocol1(){
    try{
      IrisNode.useAgreementCheckpoint = true;

      LinkedList<IrisNode> nodes = this.createCommunicationChanels(6);
      Node middleNode = nodes.get(2);
      Node myNode = nodes.get(0);
      Node helperNode = nodes.get(1);
      BranchID myNodeId = myNode.getBranchID();

      BranchID helperNodeId = helperNode.getBranchID();
      BranchID thirdNodeId = middleNode.getBranchID();

      //second set of writes
      if(verbose){
        System.out.println("Hello from SimulatorCheckpointUnit");
      }

      assert myNode.getCurrentVV().equalsIgnoreNegatives(new AcceptVV());

      int ni = 5;

      write(middleNode, COARSE_PATH, 0, ni-1, 10);
      checkRead(middleNode, COARSE_PATH, 0, ni-1, 10);

      //M:4
      write(myNode, COARSE_PATH, 0, ni-1, 0);
      checkRead(myNode, COARSE_PATH, 0, ni-1, 0);


      assert myNode.getCurrentVV().includes(new AcceptStamp(ni-1, myNodeId));
      //  assert !myNode.getCurrentVV().includes(new AcceptStamp(ni, myNodeId));

      helperNode.sync(middleNode);
      checkRead(helperNode, COARSE_PATH, 0, ni-1, 10);

      //M:4
      helperNode.sync(myNode);
      checkRead(helperNode, COARSE_PATH, 0, ni-1, 10);

      //M:9, H:0
      write(myNode, COARSE_PATH, 0, ni-1, 20);
      checkRead(myNode, COARSE_PATH, 0, ni-1, 20);

      // now gabage collect myNode
      //M:4, H:0
      myNode.garbageCollect(myNode.getCurrentVV());
      
      for(IrisNode in: nodes){
        assert in.getOmitVV().equalsIgnoreNegatives(new AcceptVV());
      }
      for(int i = 0; i < 6; i++){
        System.out.println("----------------------------------------- " +"Round " + (i+1) + "----------------------------------------- ");
        for(IrisNode in: nodes){
          if(!in.getBranchID().equals(middleNode.getBranchID())){
            syncAll(in, nodes);
          }
        }
        try{
          Thread.sleep(10000);
        }catch(Exception e){
          // TODO Auto-generated catch block
          e.printStackTrace();
        }
        for(IrisNode in: nodes){
          System.out.println("Round " + (i+1) + " " + in.getBranchID() + " " + in.getGCProtocol().getPhaseTest());
        }
      }
     
      for(IrisNode in: nodes){
        if(!in.getBranchID().equals(middleNode.getBranchID())){
        assert !in.getOmitVV().equalsIgnoreNegatives(new AcceptVV()):in.getOmitVV();
        assert in.getEpoch() == 1;
        checkRead(in, COARSE_PATH, 0, ni-1, 20);
        }
      }
      
    }catch(Exception e){
      // TODO Auto-generated catch block
      e.printStackTrace();
      assert false;
    }
  }
  
 
  public void testConcurrent(){
    IrisNode.useAgreementCheckpoint = true;

    try{
      for(int i = 1; i <2; i++){
        Node.dropInconsistentWrites = (i%2!=0);

        if(!Node.enableSecurity && Node.dropInconsistentWrites){
          continue;
        }

        BranchID myNodeId = NodeFactory.createNodeId(UNIT_NODE_ID);

        BranchID helperNodeId = NodeFactory.createNodeId(HELPER_NODE_ID);
        if(vverbose){
          System.out.println("UNIT ID: " + myNodeId.toString());
        }
        assert(myNodeId.equals(NodeFactory.createNodeId(UNIT_NODE_ID)));

        Node myNode = NodeFactory.createNodeWithoutFilter(myNodeId);
        Node helperNode = NodeFactory.createNodeWithoutFilter(helperNodeId);
        ((IrisNode)myNode).addSyncFilter(new InstantGCProtocol());
        ((IrisNode)helperNode).addSyncFilter(new InstantGCProtocol());

        //second set of writes

        if(verbose){
          System.out.println("Hello from SimulatorCheckpointUnit");
        }

        assert myNode.getCurrentVV().equalsIgnoreNegatives(new AcceptVV());

        int ni = 5;
        //M:4
        write(myNode, COARSE_PATH, 0, ni-1, 0);
        checkRead(myNode, COARSE_PATH, 0, ni-1, 0);


        assert myNode.getCurrentVV().includes(new AcceptStamp(ni-1, myNodeId));
        assert !myNode.getCurrentVV().includes(new AcceptStamp(ni, myNodeId));

        //M:4
        helperNode.sync(myNode);

        checkRead(helperNode, COARSE_PATH, 0, ni-1, 0);

        //M:9, H:-1
        write(myNode, COARSE_PATH, 0, ni-1, 10);
        checkRead(myNode, COARSE_PATH, 0, ni-1, 10);

        // now gabage collect myNode
        //M:4, H:-1
        Log log = myNode.getLogTest();
        HashedVV omitVV = new HashedVV(myNode.getCurrentVV(), log.getHashMap(myNode.getCurrentVV()));
        Checkpoint chkPt = myNode.generateNewCheckpoint(omitVV, myNode.getEpoch()+1);
        ((IrisNode)myNode).applySyncPacket(SimulatorCheckpointUnit.getSyncPacket(chkPt));

        checkRead(myNode, COARSE_PATH, 0, ni-1, 10);
        assert myNode.getCurrentVV().includes(myNode.getOmitVV());

        //M:4, H:9
        write(helperNode, COARSE_PATH, 0, ni-1, 20);
        checkRead(helperNode, COARSE_PATH, 0, ni-1, 20);

        // forced to discard writes
        // M:9, H:0
        helperNode.sync(myNode);

        assert !helperNode.getOmitVV().includes(helperNode.getCurrentVV());
        if(Node.dropInconsistentWrites){
          checkRead(helperNode, COARSE_PATH, 0, ni-1, 10);
        }else{
          checkRead(helperNode, COARSE_PATH, 0, ni-1, 20);
        }

        assert helperNode.getCurrentVV().includes(helperNode.getOmitVV());
      }
    }catch(Exception e){
      // TODO Auto-generated catch block
      e.printStackTrace();
      assert false;
    }

  }


  /**
   * 
   * @param n
   * @param prefix
   * @param is
   * @param ie
   * @param startVal
   */
  public static void checkRead(Node n, 
      String prefix, 
      int is, int ie, int startVal){
    int ii;

    if(vverbose){
      System.out.println("Reading initial data: " + is + " to " + ie + " etc.");
    }
    for(ii = is; ii <= ie; ii++){

      ObjId oid = new ObjId(prefix + "/" + ii );
      if(vverbose){
        System.out.println("Reading " + oid);
      }
      LinkedList<StoreEntry> rawVals = n.rawRead(oid);
      IrisDataObject val = (IrisDataObject)code.simulator.unit.SimulatorUnit.resolveConflict(rawVals);
      assert val.getObject().equals(startVal + ii): "Expected value for obj " + oid + " was " + (startVal + ii) + " instead found " + val;
    }
  }

  /** 
   *  Write -- write 1-byte items to /prefix/[is..ie]/[js..je]/[ks..ke].
   *  Return final accept stamp.
   **/ 
  public static void write(Node n, 
      String prefix, 
      int is, int ie, 
      int startVal){
    int ii;

    if(vverbose){
      System.out.println("Writing initial data: " + is + " to " + ie + " etc.");
    }
    for(ii = is; ii <= ie; ii++){

      ObjId oid = new ObjId(prefix + "/" + ii );
      if(vverbose){
        System.out.println("Writing " + oid);
      }

      n.write(oid, new IrisDataObject(new Integer(startVal + ii)));
    }
  }


  /*
   * "new TestSuite(Class c)" constructs a test suite
   * containg every method whose name begins with "/test"
   * 
   * TBD: update class name
   */
  public static Test suite(){
    TestSuite suite = new TestSuite(SimulatorCheckpointGenerationUnit.class);
    return suite;
  }


  /*
   * main() lets us run just this set of unit tests
   * from the comand line (you can also invoke 
   * the testrunner on this class and it will find
   * the suite())
   *
   * usage: java <classname> [-verbose] [-vverbose] [testName]*
   * 
   *   If verbose or vverbose are included, print info to screen
   *
   *   If [testName]* are included, then run test called "/test[testName]"
   *   for each such [testName]. E.g., "java TestEmtpy foo" runs
   *   TwoNodesSubscriptionUnit.testfoo() as a TestCase.
   *
   * TBD: update class name
   */
  public static void main(String s[]) {
    String name = "SimulatorCheckpointUnit";
    System.err.print(name + " self test begins...");
    Test test;
    test = suite();
    TestRunner tr = new TestRunner();
    tr.doRun(test);
    System.exit(0);
  }

}



