package code.simulator.checkpoint.unit;

import code.branchDetecting.BranchID;
import code.security.SangminConfig;
import code.simulator.*;
import code.simulator.agreement.AgreementInstance;
import code.simulator.agreement.AgreementParams;
import code.simulator.agreement.InstanceID;
import code.simulator.agreement.unit.TestCommunicationChannel;
import code.simulator.checkpoint.Checkpoint;
import code.simulator.log.Log;
import code.simulator.log.LogStatus;
import code.simulator.log.NodeLog;
import code.simulator.protocolFilters.SimpleGCProtocol;
import code.simulator.protocolFilters.InstantGCProtocol;
import code.simulator.store.Store;
import code.simulator.store.StoreEntry;
import code.simulator.store.StoreInterface;
import code.*;
/** 
 *  Test bandwidth to subscribe to 1000 objects, 100 of which have
 *  been modified. Test created for sosp 2007 submission microbenchmark.
 **/ 

//package utils.junit;

import junit.textui.TestRunner;
import junit.framework.*;
import java.util.*;
import java.io.*;

/**
 * TBD: Update class name
 */
public class SimulatorMechanismCheckpointUnit extends TestCase {
  public static final String TEST_ALL_TEST_TYPE = "UNIT";

  /**
   * Basic constructor - called by the test runners.
   * TBD: Update constructor name to match class
   */
  public SimulatorMechanismCheckpointUnit (final String s) {
    super (s);
  }

  /*
   * Fixtures are run before and after each test case
   * to set up environment in which tests must run.
   */
  protected void setUp() throws Exception{
    super.setUp();
    makePractiConfig(0, 100);

  }

  public static void makePractiConfig(long first, int total){
    Config.createEmptyConfig();

    long NODE_1_ID = first;
    int port = 9921;
    String NODE_1_IP = "localhost";
    
    for(int i = 0; i < total; i++){
      
      Config.addOneNodeConfig(new NodeId(NODE_1_ID++),
                              NODE_1_IP,
                            port++,
                            port++,
                            port++,
                            port++,
                            port++,
                            "/test" + File.separatorChar + "local-" + 
                            NODE_1_ID + ".db",
                            "/*",
                            -1L,
                            NODE_1_IP,
                            port++,
                            port++,
                            -1,
                            Config.CACHE_SIZE_BYTES_DEFAULT,
                            Config.MAX_LOG_DISK_SIZE_BYTES,
                            Config.MAX_LOG_MEM_SIZE_BYTES);
    }

    Config.readKeys();//needed for secure practi
        
  }

  public static int UNIT_NODE_ID = 10;
  public static int MIDDLE_NODE_ID = 15;

  public static int HELPER_NODE_ID = 20;
  public static boolean vverbose = false;
  public static boolean verbose = true;


  public static String COARSE_PATH = "/coarse";

  public static int compareStore(StoreInterface store1, StoreInterface store2, SubscriptionSet ignoreSet){
    return compareStore(store1, store2, ignoreSet, true);
  }
  
  public static int compareStore(StoreInterface store1, StoreInterface store2, SubscriptionSet ignoreSet, boolean print){
    int mismatches = 0;
    Iterator<ObjId> iter1 = store1.iterator();
    Iterator<ObjId> iter2 = store2.iterator();
    while(iter1.hasNext() || iter2.hasNext()){
      ObjId o = null;
      if(iter1.hasNext()){
        o = iter1.next();
      }else{
        o = iter2.next();
      }
      if(ignoreSet.getIntersection(SubscriptionSet.makeSubscriptionSet(o.getPath())).isEmpty()){
        if(!store1.containsKey(o)){
          if(print)System.out.println("store1 doesn't contain key " + o + " storeEntry " + store2.get(o));
          mismatches++;
        }else if(!store2.containsKey(o)){
          if(print)System.out.println("store1 doesn't contain key " + o + " storeEntry"  + store1.get(o));
          mismatches++;
        }else if(!store1.get(o).equals(store2.get(o))){
          if(print)System.out.println("store1 and store2 differ in content for key " + o + "\n\t store1" + store1.get(o) + "\n\t store2:" + store2.get(o));
          mismatches++;
        }
      }
    }
    return mismatches;
  }
  
  public static int compareStore(StoreInterface store1, Log log1, StoreInterface store2, Log log2, SubscriptionSet ignoreSet, boolean print){
    TreeSet<ObjId> objIdSet = new TreeSet<ObjId>();
    Iterator<ObjId> iter = store1.iterator();
    while(iter.hasNext()){
      objIdSet.add(iter.next());
    }
    iter = store2.iterator();
    while(iter.hasNext()){
      objIdSet.add(iter.next());
    }
    int mismatches = 0;
    for(ObjId o: objIdSet){
      if(ignoreSet.getIntersection(SubscriptionSet.makeSubscriptionSet(o.getPath())).isEmpty()){
        if(!store1.containsKey(o)){
          if(print)System.out.println("store1 doesn't contain key " + o + " storeEntry " + store2.get(o));
          mismatches++;
        }else if(!store2.containsKey(o)){
          if(print)System.out.println("store1 doesn't contain key " + o + " storeEntry"  + store1.get(o));
          mismatches++;
        }else if(!store1.get(o).equals(store2.get(o))){
          if(print)System.out.println("store1 and store2 differ in content for key " + o);
          TreeSet<StoreEntry> ts1 = store1.get(o);
          TreeSet<StoreEntry> ts2 = store2.get(o);
          System.out.println("\n\t store1"); 
          for(StoreEntry se: ts1){
            if(print)System.out.println(se + " \t" + log1.getDeepClosure(se.getAcceptStamp()));
          }
          
          System.out.println("\n\t store2"); 
          for(StoreEntry se: ts2){
            if(print)System.out.println(se + " \t" + log2.getDeepClosure(se.getAcceptStamp()));
          }
          mismatches++;
        }
      }
    }
    return mismatches;
  }
  
  public static void applyAll(Store store, Collection<PreciseInv> invals){
     for(PreciseInv pi: invals){
       store.apply((SimPreciseInv)pi);
     }
  }

  /**
   * test whether the writes from evicted node are handled appropriately
   */
  public void testEvictedNode(){
    if(!Node.enableSecurity){
      return;
    }
    try{
      BranchID myNodeId = NodeFactory.createNodeId(UNIT_NODE_ID);
      NodeId b0 = myNodeId;
      NodeId b2 = null, b2I = null, b4 = null, b4I = null;
      BranchID nodeId1 = NodeFactory.createNodeId(15);
      BranchID nodeId2 = NodeFactory.createNodeId(20);
      BranchID nodeId3 = NodeFactory.createNodeId(25);
      BranchID nodeId4 = NodeFactory.createNodeId(30);
      BranchID nodeId5 = NodeFactory.createNodeId(25);

      Node myNode = NodeFactory.createNodeWithoutFilter(myNodeId);
      Node node1 = NodeFactory.createNodeWithoutFilter(nodeId1);
      Node node2 = NodeFactory.createNodeWithoutFilter(nodeId2);
      Node node3 = NodeFactory.createNodeWithoutFilter(nodeId3);
      Node node4 = NodeFactory.createNodeWithoutFilter(nodeId4);
      Node node5 = NodeFactory.createNodeWithoutFilter(nodeId5);

      
      ObjId oid0 = new ObjId("/test/0");
      myNode.write(oid0, new IrisDataObject(new Integer(0)));
      node1.sync(myNode);
      SimPreciseInv n1w1 = node1.write(new ObjId("/test/991"), new IrisDataObject(new Integer(-1)));
      
      ObjId oid1 = new ObjId("/test/1");
      myNode.write(oid1, new IrisDataObject(new Integer(1)));

      LinkedList<Node> forks = myNode.fork(4);
      LinkedList<Node> forks1 = new LinkedList<Node>();
      forks1.add(forks.get(2));
      forks1.add(forks.get(3));
      
      forks1.get(0).write(new ObjId("/test/fork1/0000"), new IrisDataObject(new Integer(6)));
      forks1.get(0).write(new ObjId("/test/fork1/0001"), new IrisDataObject(new Integer(7)));
      forks1.get(1).write(new ObjId("/test/fork1/0010"), new IrisDataObject(new Integer(8)));
      forks1.get(1).write(new ObjId("/test/fork1/0010"), new IrisDataObject(new Integer(9)));
      
      node4.sync(forks1.get(0));
      node5.sync(forks1.get(1));
      
      forks.get(0).write(new ObjId("/test/00"), new IrisDataObject(new Integer(2)));
      node1.sync(forks.get(0));
      SimPreciseInv n1w2 = node1.write(new ObjId("/test/991"), new IrisDataObject(new Integer(-1)));
      forks.get(0).write(new ObjId("/test/01"), new IrisDataObject(new Integer(3)));
      forks.get(1).write(new ObjId("/test/10"), new IrisDataObject(new Integer(4)));
      node2.sync(forks.get(1));
      SimPreciseInv n2w1 = node2.write(new ObjId("/test/992"), new IrisDataObject(new Integer(-2)));//depends on b2I:3
      forks.get(1).write(new ObjId("/test/11"), new IrisDataObject(new Integer(5)));

      LinkedList<Node> newForks = forks.get(0).fork(2);
      newForks.get(0).write(new ObjId("/test/0000"), new IrisDataObject(new Integer(6)));
      newForks.get(0).write(new ObjId("/test/0001"), new IrisDataObject(new Integer(7)));
      newForks.get(1).write(new ObjId("/test/0010"), new IrisDataObject(new Integer(8)));
      node3.sync(newForks.get(1));
      SimPreciseInv n3w1 = node3.write(new ObjId("/test/992"), new IrisDataObject(new Integer(-3)));
      newForks.get(1).write(new ObjId("/test/0011"), new IrisDataObject(new Integer(9)));

      node1.sync(newForks.get(0));
      node2.sync(forks.get(1));
      node3.sync(newForks.get(1));

      if(!SangminConfig.useSimulatorId){
        ((IrisNode)node1).exponentialBackoffSync(node2);
      }
      node1.sync(node2);

      AcceptVV cvv1 = node1.getCurrentVV();
      VVIterator vvi1 = cvv1.getIterator();
      while(vvi1.hasMoreElements()){
        NodeId n = vvi1.getNext();
        if(cvv1.getStampByIteratorToken(n) == 3 && n.getIDint() == b0.getIDint()){
          b2I = n;
        }else if(cvv1.getStampByIteratorToken(n) == 5 && n.getIDint() == b0.getIDint()){
          b2 = n;
        }else if(n.getIDint() == b0.getIDint()){
          assert b0.equals(n);
        }
      }

      if(!SangminConfig.useSimulatorId){
        ((IrisNode)node1).exponentialBackoffSync(node3);
      }
      node1.sync(node3);

      cvv1 = node1.getCurrentVV();
      vvi1 = cvv1.getIterator();
      while(vvi1.hasMoreElements()){
        NodeId n = vvi1.getNext();
        if(cvv1.getStampByIteratorToken(n) == 5 && n.getIDint() == b0.getIDint()){
          if(b4I == null){
            b4I = n;
          }else{
            b4 = n;
          }
        }
      }


      //initialize node logs
      NodeLog nl1 = node1.getLogTest().get(node1.getBranchID());
      assert nl1.size() == 2;
      
      NodeLog nl2 = node1.getLogTest().get(node2.getBranchID());
      assert nl2.size() == 1;
      
      NodeLog nl3 = node1.getLogTest().get(node3.getBranchID());
      assert nl3.size() == 1;
      
      NodeLog nl0 = node1.getLogTest().get(b0);
      assert nl0.size() == 2;
      
      NodeLog nl0_ = node1.getLogTest().get(b2);
      assert nl0_.size() == 2;
      
      NodeLog nl0_I = node1.getLogTest().get(b2I);
      assert nl0_I.size() ==2:nl0_I;
      
      NodeLog nl0__ = node1.getLogTest().get(b4);
      assert nl0__.size() == 2;
      
      NodeLog nl0__I = node1.getLogTest().get(b4I);
      assert nl0__I.size() == 2;
      
      Node f0 = myNode;
      Node f2 = forks.get(0);
      Node f2I = forks.get(1);
      Node f4 = newForks.get(0);
      Node f4I = newForks.get(1);
      
      NodeLog nlf0 = f0.getLogTest().get(f0.getBranchID());
      NodeLog nlf2 = f2.getLogTest().get(f2.getBranchID());
      NodeLog nlf2I = f2I.getLogTest().get(f2I.getBranchID());
      NodeLog nlf4 = f4.getLogTest().get(f4.getBranchID());
      NodeLog nlf4I = f4I.getLogTest().get(f4I.getBranchID());
      
      HashSet<ProofOfMisbehavior> poms = node1.getPOMMap();
      ProofOfMisbehavior pom2 = null, pom4 = null;
      assert poms.size() ==2 ;
      for(ProofOfMisbehavior pom: poms){
        if(pom.getParent().getAcceptStamp().getLocalClock() == 1){
          pom2 = pom;
        }else{
          pom4 = pom;
        }
      }
      
      node4.getLogTest().applyPOM(pom2, new POMRemap());
      assert node4.getCurrentVV().size() == 2:node4.getCurrentVV();
      VVIterator vvi = node4.getCurrentVV().getIterator();
      NodeId fork2II = null;
      while(vvi.hasMoreElements()){
        NodeId n = vvi.getNext();
        long ts = node4.getCurrentVV().getStampByIteratorToken(n);
        if(ts == 3){
          fork2II = n;
        }
      }
      
      Node node5Copy = NodeFactory.createNodeWithoutFilter(nodeId5);
      SyncStatus ss = node5Copy.sync(node5);
      assert ss.status == SyncStatus.SyncSuccessful:ss;
      node5Copy.getLogTest().applyPOM(pom2, new POMRemap());
      
      Node node5Copy1 = NodeFactory.createNodeWithoutFilter(nodeId5);
      ss = node5Copy1.sync(node5);
      assert ss.status == SyncStatus.SyncSuccessful:ss;
      
      node5.getLogTest().applyPOM(pom2, new POMRemap());
      assert node5.getCurrentVV().size() == 2;
      vvi = node5.getCurrentVV().getIterator();
      NodeId fork2III = null;
      while(vvi.hasMoreElements()){
        NodeId n = vvi.getNext();
        long ts = node5.getCurrentVV().getStampByIteratorToken(n);
        if(ts == 3){
          fork2III = n;
        }
      }
      
      
      // ---------------- test if the writes from the evicted nodes are included at the node creating checkpoint  --------------------
      

      Log log1 = node1.getLogTest();
      AcceptVV tVV = node1.getCurrentVV();
      AcceptVV cvv = log1.getMinimalVV(tVV);
      assert cvv.size() == 6:cvv.size();
      HashMap<NodeId, Hash> hashes = log1.getHashMap(cvv);
      
      HashedVV omitVV = new HashedVV(cvv, hashes);
      
      if(!SangminConfig.useSimulatorId){
        ((IrisNode)node1).exponentialBackoffSync(node4);
      }
      node1.sync(node4);
      
      Checkpoint chkPt = node1.generateNewCheckpoint(omitVV, 1);
      
      AcceptVV avv = log1.getMinimalVV(node1.getCurrentVV());
      assert new AcceptVV(chkPt.omitVV).equals(omitVV);
      assert new AcceptVV(chkPt.omitVV).equals(omitVV);
      assert avv.equals(new AcceptVV(chkPt.endVV)):avv + "\n" + chkPt.endVV;
      
      Log log4 = node4.getLogTest();
      assert log4.size() == 2;
      assert log4.containsKey(fork2II);
      SecureSimPreciseInv sspi = (SecureSimPreciseInv)log4.get(fork2II).get(0); 
      
      assert chkPt.acceptableBranches.size()==1:chkPt.acceptableBranches;
      assert chkPt.acceptableBranches.contains(sspi):chkPt.acceptableBranches + "\n" + sspi;
      
      StoreInterface store = node1.getStore();
      
      ((IrisNode)node1).applySyncPacket(SimulatorCheckpointUnit.getSyncPacket(chkPt));
      
      StoreInterface newStore = node1.getStore();
      SubscriptionSet ignoreSet = SubscriptionSet.makeSubscriptionSet(IrisNode.globalCertificatesString+"*");
      
      assert this.compareStore(store, newStore, ignoreSet)  == 0;
      
      CounterVV testVV = new CounterVV(node1.getCurrentVV());
      assert node1.getCurrentVV().includes(avv):"\n"+node1.getCurrentVV() +"\n" + avv;
      testVV.dropNode(node1.getBranchID());
      CounterVV acvv = new CounterVV(avv);
      acvv.dropNode(node1.getBranchID());
      assert acvv.equals(testVV);
      
//      Node node1Copy = NodeFactory.createNodeWithoutFilter(nodeId1);
//      Log log1Copy = node1Copy.getLogTest();
//      LogStatus ls = log1Copy.apply(nl1.get(0)));
//      assert ls.getApplyStatus() == LogStatus.ApplyFailed:ls;
//      if(IrisNode.enableInconsistencyNotification){
//        assert ls.getVerificationStatus() == LogStatus.InclusionFail:ls;
//      }else{
//        assert ls.getVerificationStatus() == LogStatus.DVVInclusionFail:ls;
//      }
//      ls = log1Copy.apply(nl0.get(0)));
//      assert ls.equals(LogStatus.makeDefaultLogStatus());
//      ls = log1Copy.apply(nl1.get(0)));
//      assert ls.equals(LogStatus.makeDefaultLogStatus()):ls;
//      
      //  ---------------- test if the writes from evicted node at other nodes are being dropped -------------------- 
      
      
      Log log5 = node5.getLogTest();
      log5.applyPOM(pom2, new POMRemap());
      assert log5.containsKey(fork2III);
      
      AcceptVV n5CVV = log5.getCurrentVV();
      ((IrisNode)node5).applySyncPacket(SimulatorCheckpointUnit.getSyncPacket(chkPt));

      Log log5New = node5.getLogTest();
      
      assert node5.getOmitVV().equals(chkPt.omitVV);
      assert !node5.getCurrentVV().includes(n5CVV);
      assert !node5.getCurrentVV().containsNodeId(fork2III);
      assert log5.containsKey(fork2III);
      assert !log5New.containsKey(fork2III);
      assert log5New.getBranchID((SimPreciseInv)log5.get(fork2III).get(0)) == null; 
      assert compareStore(node1.getStore(), node5.getStore(), ignoreSet) == 0;
      
      // now test the same thing with real sync
      log5 = node5Copy.getLogTest();
      log5.applyPOM(pom2, new POMRemap());
      assert log5.containsKey(fork2III);
      
      n5CVV = log5.getCurrentVV();
      ss = node5Copy.sync(node1);
      assert ss.status == SyncStatus.ReceivedNewCertificate:ss;

      log5New = node5Copy.getLogTest();
      
      assert node5Copy.getOmitVV().equals(chkPt.omitVV);
      assert !node5Copy.getCurrentVV().includes(n5CVV);
      assert !node5Copy.getCurrentVV().containsNodeId(fork2III);
      assert log5.containsKey(fork2III);
      assert !log5New.containsKey(fork2III);
      assert log5New.getBranchID((SimPreciseInv)log5.get(fork2III).get(0)) == null; 
      assert compareStore(node1.getStore(), node5Copy.getStore(), ignoreSet) == 0;

   // now test the same thing with real sync except this time POM has not been applied at the receiver
      log5 = node5Copy1.getLogTest();
      
      n5CVV = log5.getCurrentVV();
      ss = node5Copy1.sync(node1);
      assert ss.status == SyncStatus.ReceivedNewCertificate:ss;

      log5New = node5Copy1.getLogTest();
      
      assert node5Copy1.getOmitVV().equals(chkPt.omitVV);
      assert !node5Copy1.getCurrentVV().includes(n5CVV);
      assert !node5Copy1.getCurrentVV().containsNodeId(fork2III);
      assert !log5New.containsKey(fork2III);
      assert log5New.getBranchID((SimPreciseInv)log5.get(b0).get(2)) == null; 
      assert compareStore(node1.getStore(), node5Copy1.getStore(), ignoreSet) == 0;
      
    }catch(Exception e){
      e.printStackTrace();
      assert false;
    }
  }
  /**
   * test whether the writes from faulty node are handled appropriately
   */
  public void testFaultyNode(){
    if(!Node.enableSecurity){
      return;
    }
    try{
      BranchID myNodeId = NodeFactory.createNodeId(UNIT_NODE_ID);
      NodeId b0 = myNodeId;
      NodeId b2 = null, b2I = null, b4 = null, b4I = null;
      BranchID nodeId1 = NodeFactory.createNodeId(15);
      BranchID nodeId2 = NodeFactory.createNodeId(20);
      BranchID nodeId3 = NodeFactory.createNodeId(25);
      BranchID nodeId4 = NodeFactory.createNodeId(30);
      BranchID nodeId5 = NodeFactory.createNodeId(35);

      Node myNode = NodeFactory.createNodeWithoutFilter(myNodeId);
      Node node1 = NodeFactory.createNodeWithoutFilter(nodeId1);
      Node node2 = NodeFactory.createNodeWithoutFilter(nodeId2);
      Node node3 = NodeFactory.createNodeWithoutFilter(nodeId3);
      Node node4 = NodeFactory.createNodeWithoutFilter(nodeId4);
      Node node5 = NodeFactory.createNodeWithoutFilter(nodeId5);

      
      ObjId oid0 = new ObjId("/test/0");
      myNode.write(oid0, new IrisDataObject(new Integer(0)));
      node1.sync(myNode);
      SimPreciseInv n1w1 = node1.write(new ObjId("/test/991"), new IrisDataObject(new Integer(-1)));
      
      ObjId oid1 = new ObjId("/test/1");
      myNode.write(oid1, new IrisDataObject(new Integer(1)));

      LinkedList<Node> forks = myNode.fork(4);
      LinkedList<Node> forks1 = new LinkedList<Node>();
      forks1.add(forks.get(2));
      forks1.add(forks.get(3));
      
      forks1.get(0).write(new ObjId("/test/fork1/0000"), new IrisDataObject(new Integer(6)));
      forks1.get(0).write(new ObjId("/test/fork1/0001"), new IrisDataObject(new Integer(7)));
      forks1.get(1).write(new ObjId("/test/fork1/0010"), new IrisDataObject(new Integer(8)));
      forks1.get(1).write(new ObjId("/test/fork1/0010"), new IrisDataObject(new Integer(9)));
      
      node4.sync(forks1.get(0));
      node5.sync(forks1.get(1));
      
      forks.get(0).write(new ObjId("/test/00"), new IrisDataObject(new Integer(2)));
      node1.sync(forks.get(0));
      SimPreciseInv n1w2 = node1.write(new ObjId("/test/991"), new IrisDataObject(new Integer(-1)));
      Log log1 = node1.getLogTest();
      AcceptVV tVV = node1.getCurrentVV();
      AcceptVV cvv = log1.getMinimalVV(tVV);
      HashMap<NodeId, Hash> hashes = log1.getHashMap(cvv);
      
      forks.get(0).write(new ObjId("/test/01"), new IrisDataObject(new Integer(3)));

      forks.get(1).write(new ObjId("/test/10"), new IrisDataObject(new Integer(4)));
      node2.sync(forks.get(1));
      SimPreciseInv n2w1 = node2.write(new ObjId("/test/992"), new IrisDataObject(new Integer(-2)));//depends on b2I:3
      forks.get(1).write(new ObjId("/test/11"), new IrisDataObject(new Integer(5)));

      LinkedList<Node> newForks = forks.get(0).fork(2);
      newForks.get(0).write(new ObjId("/test/0000"), new IrisDataObject(new Integer(6)));
      newForks.get(0).write(new ObjId("/test/0001"), new IrisDataObject(new Integer(7)));
      newForks.get(1).write(new ObjId("/test/0010"), new IrisDataObject(new Integer(8)));
      node3.sync(newForks.get(1));
      SimPreciseInv n3w1 = node3.write(new ObjId("/test/992"), new IrisDataObject(new Integer(-3)));
      newForks.get(1).write(new ObjId("/test/0011"), new IrisDataObject(new Integer(9)));


      
      node1.sync(newForks.get(0));
      
      node2.sync(forks.get(1));
      node3.sync(newForks.get(1));

      if(!SangminConfig.useSimulatorId){
        ((IrisNode)node1).exponentialBackoffSync(node2);
      }
      node1.sync(node2);

      AcceptVV cvv1 = node1.getCurrentVV();
      VVIterator vvi1 = cvv1.getIterator();
      while(vvi1.hasMoreElements()){
        NodeId n = vvi1.getNext();
        if(cvv1.getStampByIteratorToken(n) == 3 && n.getIDint() == b0.getIDint()){
          b2I = n;
        }else if(cvv1.getStampByIteratorToken(n) == 5 && n.getIDint() == b0.getIDint()){
          b2 = n;
        }else if(n.getIDint() == b0.getIDint()){
          assert b0.equals(n);
        }
      }

      if(!SangminConfig.useSimulatorId){
        ((IrisNode)node1).exponentialBackoffSync(node3);
      }
      node1.sync(node3);

      cvv1 = node1.getCurrentVV();
      vvi1 = cvv1.getIterator();
      while(vvi1.hasMoreElements()){
        NodeId n = vvi1.getNext();
        if(cvv1.getStampByIteratorToken(n) == 5 && n.getIDint() == b0.getIDint()){
          if(b4I == null){
            b4I = n;
          }else{
            b4 = n;
          }
        }
      }


      //initialize node logs
      NodeLog nl1 = node1.getLogTest().get(node1.getBranchID());
      assert nl1.size() == 2;
      
      NodeLog nl2 = node1.getLogTest().get(node2.getBranchID());
      assert nl2.size() == 1;
      
      NodeLog nl3 = node1.getLogTest().get(node3.getBranchID());
      assert nl3.size() == 1;
      
      NodeLog nl0 = node1.getLogTest().get(b0);
      assert nl0.size() == 2;
      
      NodeLog nl0_ = node1.getLogTest().get(b2);
      assert nl0_.size() == 2;
      
      NodeLog nl0_I = node1.getLogTest().get(b2I);
      assert nl0_I.size() ==2:nl0_I;
      
      NodeLog nl0__ = node1.getLogTest().get(b4);
      assert nl0__.size() == 2;
      
      NodeLog nl0__I = node1.getLogTest().get(b4I);
      assert nl0__I.size() == 2;
      
      Node f0 = myNode;
      Node f2 = forks.get(0);
      Node f2I = forks.get(1);
      Node f4 = newForks.get(0);
      Node f4I = newForks.get(1);
      
      NodeLog nlf0 = f0.getLogTest().get(f0.getBranchID());
      NodeLog nlf2 = f2.getLogTest().get(f2.getBranchID());
      NodeLog nlf2I = f2I.getLogTest().get(f2I.getBranchID());
      NodeLog nlf4 = f4.getLogTest().get(f4.getBranchID());
      NodeLog nlf4I = f4I.getLogTest().get(f4I.getBranchID());
      
      HashSet<ProofOfMisbehavior> poms = node1.getPOMMap();
      ProofOfMisbehavior pom2 = null, pom4 = null;
      assert poms.size() ==2 ;
      for(ProofOfMisbehavior pom: poms){
        if(pom.getParent().getAcceptStamp().getLocalClock() == 1){
          pom2 = pom;
        }else{
          pom4 = pom;
        }
      }
      
      node4.getLogTest().applyPOM(pom2, new POMRemap());
      assert node4.getCurrentVV().size() == 2:node4.getCurrentVV();
      VVIterator vvi = node4.getCurrentVV().getIterator();
      NodeId fork2II = null;
      while(vvi.hasMoreElements()){
        NodeId n = vvi.getNext();
        long ts = node4.getCurrentVV().getStampByIteratorToken(n);
        if(ts == 3){
          fork2II = n;
        }
      }
      
      Node node5Copy = NodeFactory.createNodeWithoutFilter(nodeId5);
      SyncStatus ss = node5Copy.sync(node5);
      assert ss.status == SyncStatus.SyncSuccessful:ss;
      node5Copy.getLogTest().applyPOM(pom2, new POMRemap());
      
      Node node5Copy1 = NodeFactory.createNodeWithoutFilter(nodeId5);
      ss = node5Copy1.sync(node5);
      assert ss.status == SyncStatus.SyncSuccessful:ss;
      
      node5.getLogTest().applyPOM(pom2, new POMRemap());
      assert node5.getCurrentVV().size() == 2;
      vvi = node5.getCurrentVV().getIterator();
      NodeId fork2III = null;
      while(vvi.hasMoreElements()){
        NodeId n = vvi.getNext();
        long ts = node5.getCurrentVV().getStampByIteratorToken(n);
        if(ts == 3){
          fork2III = n;
        }
      }
      
      
      // ---------------- test if the writes from the faulty node are being retained when creating checkpoint  --------------------

      HashedVV omitVV = new HashedVV(cvv, hashes);
      
      if(!SangminConfig.useSimulatorId){
        ((IrisNode)node1).exponentialBackoffSync(node4);
      }
      node1.sync(node4);
      
      Checkpoint chkPt = node1.generateNewCheckpoint(omitVV, 1);


      AcceptVV avv = node1.getCurrentVV();
      CounterVV testVV = new CounterVV(avv);
      testVV.addMaxAS(new AcceptStamp(testVV.getStampByServer(b2), b0));
      testVV.dropNode(b2);
      testVV.dropNode(nodeId1);
      
      assert new AcceptVV(chkPt.omitVV).equals(omitVV);
      
      Log log4 = node4.getLogTest();
      assert log4.size() == 2;
      assert log4.containsKey(fork2II);
      SecureSimPreciseInv sspi = (SecureSimPreciseInv)log4.get(fork2II).get(0); 
      
      assert chkPt.acceptableBranches.size()==2:chkPt.acceptableBranches;
      assert chkPt.acceptableBranches.contains(sspi):chkPt.acceptableBranches + "\n" + sspi;
      assert chkPt.acceptableBranches.contains(nl0_I.get(0)):chkPt.acceptableBranches + "\n" + nl0_I.get(0);
      
      
      StoreInterface store = node1.getStore();
      
      ((IrisNode)node1).applySyncPacket(SimulatorCheckpointUnit.getSyncPacket(chkPt));
      
      StoreInterface newStore = node1.getStore();
      SubscriptionSet ignoreSet = SubscriptionSet.makeSubscriptionSet(IrisNode.globalCertificatesString+"*");
      
      assert compareStore(store, newStore, ignoreSet) == 0;
      
      CounterVV acvv  = new CounterVV(node1.getCurrentVV());
      acvv.dropNode(nodeId1);
      assert testVV.includes(acvv):"\n"+node1.getCurrentVV() +"\n" + avv;
      assert acvv.equals(testVV);
            
      // ---------------- test if the writes from the faulty node are retained when applying checkpoint  --------------------
      
      
      Log log5 = node5.getLogTest();
      log5.applyPOM(pom2, new POMRemap());
      AcceptVV origN5CVV = node5.getCurrentVV();
      assert log5.containsKey(fork2III);
      AcceptVV origVV = node5.getCurrentVV();
      StoreInterface origStore = node5.getStore();
      
      node5.notifyNewForks(node1.getPOMMap());
      
      Checkpoint chkPt1 = new Checkpoint(chkPt.omitVV, chkPt.objectStore, 
          chkPt.unverifiableWrites, chkPt.lastWrite, chkPt.flagMap, 
          chkPt.epochCount+1, chkPt.endVV, chkPt.acceptableBranches);
      ss = ((IrisNode)node5).applySyncPacket(SimulatorCheckpointUnit.getSyncPacket(chkPt));
      assert ss.status == SyncStatus.ReceivedNewCertificate:ss;
      
      testVV = new CounterVV(node1.getCurrentVV());
      ss = ((IrisNode)node5).sync(node1);
      assert ss.status == SyncStatus.SyncSuccessful:ss;
      
      newStore.putAll(origStore);
      assert this.compareStore(node5.getStore(), newStore, ignoreSet) == 0;
      
      acvv = new CounterVV(node5.getCurrentVV());
      acvv.dropNode(nodeId5);
      testVV.addMaxVV(origVV);
      assert acvv.includes(origVV):"\n"+acvv +"\n" + origVV;
      assert acvv.equals(testVV):"\n"+acvv +"\n" + testVV;
      
      
   // now test the same thing with real sync except this time POM has not been applied at the receiver
      log5 = node5Copy1.getLogTest();
      origStore = node1.getStore();
      origStore.putAll(node5Copy1.getStore());
      
      AcceptVV n5CVV = log5.getCurrentVV();
      ss = node5Copy1.sync(node1);
      assert ss.status == SyncStatus.ReceivedNewCertificate:ss;
      ss = node5Copy1.sync(node1);
      assert ss.status == SyncStatus.SyncSuccessful:ss;

      Log log5New = node5Copy1.getLogTest();
      
      assert node5Copy1.getOmitVV().equals(chkPt.omitVV);
      assert node5Copy1.getCurrentVV().includes(n5CVV); // due to the remap
      assert node5Copy1.getCurrentVV().includes(origN5CVV); // the originally n5CVV
      assert node5Copy1.getCurrentVV().containsNodeId(fork2III);
      assert log5New.containsKey(fork2III);
      assert log5New.getBranchID((SimPreciseInv)log5.get(b0).get(2)) != null; 
      assert compareStore(origStore, node5Copy1.getStore(), ignoreSet) == 0;
      
    }catch(Exception e){
      e.printStackTrace();
      assert false;
    }
  }
  
  /**
   * test whether the writes from faulty node are handled appropriately
   */
  public void testFaultyNode1(){
    if(!Node.enableSecurity){
      return;
    }
    try{
      BranchID myNodeId = NodeFactory.createNodeId(UNIT_NODE_ID);
      NodeId b0 = myNodeId;
      NodeId b2 = null, b2I = null, b4 = null, b4I = null;
      BranchID nodeId1 = NodeFactory.createNodeId(15);
      BranchID nodeId2 = NodeFactory.createNodeId(20);
      BranchID nodeId3 = NodeFactory.createNodeId(25);
      BranchID nodeId4 = NodeFactory.createNodeId(30);
      BranchID nodeId5 = NodeFactory.createNodeId(35);

      Node myNode = NodeFactory.createNodeWithoutFilter(myNodeId);
      Node node1 = NodeFactory.createNodeWithoutFilter(nodeId1);
      Node node2 = NodeFactory.createNodeWithoutFilter(nodeId2);
      Node node3 = NodeFactory.createNodeWithoutFilter(nodeId3);
      Node node4 = NodeFactory.createNodeWithoutFilter(nodeId4);
      Node node5 = NodeFactory.createNodeWithoutFilter(nodeId5);

      
      ObjId oid0 = new ObjId("/test/0");
      myNode.write(oid0, new IrisDataObject(new Integer(0)));
      node1.sync(myNode);
      SimPreciseInv n1w1 = node1.write(new ObjId("/test/991"), new IrisDataObject(new Integer(-1)));
      
      ObjId oid1 = new ObjId("/test/1");
      myNode.write(oid1, new IrisDataObject(new Integer(1)));

      LinkedList<Node> forks = myNode.fork(4);
      LinkedList<Node> forks1 = new LinkedList<Node>();
      forks1.add(forks.get(2));
      forks1.add(forks.get(3));
      
      forks1.get(0).write(new ObjId("/test/fork1/0000"), new IrisDataObject(new Integer(6)));
      forks1.get(0).write(new ObjId("/test/fork1/0001"), new IrisDataObject(new Integer(7)));
      forks1.get(1).write(new ObjId("/test/fork1/0010"), new IrisDataObject(new Integer(8)));
      forks1.get(1).write(new ObjId("/test/fork1/0010"), new IrisDataObject(new Integer(9)));
      
      node4.sync(forks1.get(0));
      node5.sync(forks1.get(1));
      
      forks.get(0).write(new ObjId("/test/00"), new IrisDataObject(new Integer(2)));
      node1.sync(forks.get(0));
      SimPreciseInv n1w2 = node1.write(new ObjId("/test/991"), new IrisDataObject(new Integer(-1)));
      forks.get(0).write(new ObjId("/test/01"), new IrisDataObject(new Integer(3)));

      forks.get(1).write(new ObjId("/test/10"), new IrisDataObject(new Integer(4)));
      node2.sync(forks.get(1));
      SimPreciseInv n2w1 = node2.write(new ObjId("/test/992"), new IrisDataObject(new Integer(-2)));//depends on b2I:3
      forks.get(1).write(new ObjId("/test/11"), new IrisDataObject(new Integer(5)));

      LinkedList<Node> newForks = forks.get(0).fork(2);
      newForks.get(0).write(new ObjId("/test/0000"), new IrisDataObject(new Integer(6)));
      
      node1.sync(newForks.get(0));
      Log log1 = node1.getLogTest();
      AcceptVV tVV = node1.getCurrentVV();
      AcceptVV cvv = log1.getMinimalVV(tVV);
      HashMap<NodeId, Hash> hashes = log1.getHashMap(cvv);
      
      SimPreciseInv spiB4 = newForks.get(0).write(new ObjId("/test/0001"), new IrisDataObject(new Integer(7)));
      newForks.get(1).write(new ObjId("/test/0010"), new IrisDataObject(new Integer(8)));
      node3.sync(newForks.get(1));
      SimPreciseInv n3w1 = node3.write(new ObjId("/test/992"), new IrisDataObject(new Integer(-3)));
      SimPreciseInv spiB4I = newForks.get(1).write(new ObjId("/test/0011"), new IrisDataObject(new Integer(9)));

      node1.sync(newForks.get(0));
      
      node2.sync(forks.get(1));
      node3.sync(newForks.get(1));

      if(!SangminConfig.useSimulatorId){
        ((IrisNode)node1).exponentialBackoffSync(node2);
      }
      node1.sync(node2);

      AcceptVV cvv1 = node1.getCurrentVV();
      VVIterator vvi1 = cvv1.getIterator();
      while(vvi1.hasMoreElements()){
        NodeId n = vvi1.getNext();
        if(cvv1.getStampByIteratorToken(n) == 3 && n.getIDint() == b0.getIDint()){
          b2I = n;
        }else if(cvv1.getStampByIteratorToken(n) == 5 && n.getIDint() == b0.getIDint()){
          b2 = n;
        }else if(n.getIDint() == b0.getIDint()){
          assert b0.equals(n);
        }
      }

      if(!SangminConfig.useSimulatorId){
        ((IrisNode)node1).exponentialBackoffSync(node3);
      }
      node1.sync(node3);

      b4 = log1.getBranchID(spiB4);
      b4I = log1.getBranchID(spiB4I);
      cvv1 = node1.getCurrentVV();
      vvi1 = cvv1.getIterator();
      while(vvi1.hasMoreElements()){
        NodeId n = vvi1.getNext();
        if(cvv1.getStampByIteratorToken(n) == 5 && n.getIDint() == b0.getIDint()){
          assert b4.equals(n) || b4I.equals(n);
        }
      }
      //initialize node logs
      NodeLog nl1 = node1.getLogTest().get(node1.getBranchID());
      assert nl1.size() == 2;
      
      NodeLog nl2 = node1.getLogTest().get(node2.getBranchID());
      assert nl2.size() == 1;
      
      NodeLog nl3 = node1.getLogTest().get(node3.getBranchID());
      assert nl3.size() == 1;
      
      NodeLog nl0 = node1.getLogTest().get(b0);
      assert nl0.size() == 2;
      
      NodeLog nl0_ = node1.getLogTest().get(b2);
      assert nl0_.size() == 2;
      
      NodeLog nl0_I = node1.getLogTest().get(b2I);
      assert nl0_I.size() ==2:nl0_I;
      
      NodeLog nl0__ = node1.getLogTest().get(b4);
      assert nl0__.size() == 2;
      
      NodeLog nl0__I = node1.getLogTest().get(b4I);
      assert nl0__I.size() == 2;
      
      Node f0 = myNode;
      Node f2 = forks.get(0);
      Node f2I = forks.get(1);
      Node f4 = newForks.get(0);
      Node f4I = newForks.get(1);
      
      NodeLog nlf0 = f0.getLogTest().get(f0.getBranchID());
      NodeLog nlf2 = f2.getLogTest().get(f2.getBranchID());
      NodeLog nlf2I = f2I.getLogTest().get(f2I.getBranchID());
      NodeLog nlf4 = f4.getLogTest().get(f4.getBranchID());
      NodeLog nlf4I = f4I.getLogTest().get(f4I.getBranchID());
      
      HashSet<ProofOfMisbehavior> poms = node1.getPOMMap();
      ProofOfMisbehavior pom2 = null, pom4 = null;
      assert poms.size() ==2 ;
      for(ProofOfMisbehavior pom: poms){
        if(pom.getParent().getAcceptStamp().getLocalClock() == 1){
          pom2 = pom;
        }else{
          pom4 = pom;
        }
      }
      
      node4.getLogTest().applyPOM(pom2, new POMRemap());
      assert node4.getCurrentVV().size() == 2:node4.getCurrentVV();
      VVIterator vvi = node4.getCurrentVV().getIterator();
      NodeId fork2II = null;
      while(vvi.hasMoreElements()){
        NodeId n = vvi.getNext();
        long ts = node4.getCurrentVV().getStampByIteratorToken(n);
        if(ts == 3){
          fork2II = n;
        }
      }
      
      Node node5Copy = NodeFactory.createNodeWithoutFilter(nodeId5);
      SyncStatus ss = node5Copy.sync(node5);
      assert ss.status == SyncStatus.SyncSuccessful:ss;
      node5Copy.getLogTest().applyPOM(pom2, new POMRemap());
      
      Node node5Copy1 = NodeFactory.createNodeWithoutFilter(nodeId5);
      ss = node5Copy1.sync(node5);
      assert ss.status == SyncStatus.SyncSuccessful:ss;
      
      node5.getLogTest().applyPOM(pom2, new POMRemap());
      assert node5.getCurrentVV().size() == 2;
      vvi = node5.getCurrentVV().getIterator();
      NodeId fork2III = null;
      while(vvi.hasMoreElements()){
        NodeId n = vvi.getNext();
        long ts = node5.getCurrentVV().getStampByIteratorToken(n);
        if(ts == 3){
          fork2III = n;
        }
      }
      
      
      // ---------------- test if the writes from the faulty node are being retained when creating checkpoint  --------------------

      HashedVV omitVV = new HashedVV(cvv, hashes);
      
      if(!SangminConfig.useSimulatorId){
        ((IrisNode)node1).exponentialBackoffSync(node4);
      }
      node1.sync(node4);
      
      Checkpoint chkPt = node1.generateNewCheckpoint(omitVV, 1);

      CounterVV avv = new CounterVV(chkPt.endVV);
      avv.dropNode(nodeId5);avv.dropNode(nodeId4);avv.dropNode(nodeId3);avv.dropNode(nodeId2);avv.dropNode(nodeId1);
      CounterVV testVV = new CounterVV();
      testVV.addMaxAS(new AcceptStamp(4, b0));
      testVV.addMaxAS(new AcceptStamp(4, b4I));
      
      assert testVV.equals(avv): "\n" + testVV +"\n" + avv;
      
      assert new AcceptVV(chkPt.omitVV).equals(omitVV);
      
      log1 = node1.getLogTest();
      assert log1.containsKey(b4I):log1;
      SecureSimPreciseInv sspi = (SecureSimPreciseInv)log1.get(b4I).get(0); 
      
      assert chkPt.acceptableBranches.size()==1:chkPt.acceptableBranches;
      assert chkPt.acceptableBranches.contains(sspi):chkPt.acceptableBranches + "\n" + sspi;
      
      StoreInterface store = node1.getStore();
      
      ((IrisNode)node1).applySyncPacket(SimulatorCheckpointUnit.getSyncPacket(chkPt));
      
      StoreInterface newStore = node1.getStore();
      SubscriptionSet ignoreSet = SubscriptionSet.makeSubscriptionSet(IrisNode.globalCertificatesString+"*");
      
      Store expectedStore = new Store();
      applyAll(expectedStore, nlf0);applyAll(expectedStore, nlf2);applyAll(expectedStore, nlf4);applyAll(expectedStore, nlf4I);applyAll(expectedStore,nl1);applyAll(expectedStore,nl2); applyAll(expectedStore,nl3); 
      
      assert this.compareStore(expectedStore, newStore, ignoreSet) == 0;
      
      Store newExpectedStore = expectedStore.clone();
      applyAll(newExpectedStore, nlf2I);
      assert !(this.compareStore(newExpectedStore, newStore, ignoreSet) == 0);
      
      CounterVV acvv  = new CounterVV(node1.getCurrentVV());
      acvv.dropNode(nodeId1);
      testVV = new CounterVV();
      testVV.addMaxAS(new AcceptStamp(5, b0));
      testVV.addMaxAS(new AcceptStamp(5, b4I));
      testVV.addMaxAS(new AcceptStamp(node2.getCurrentVV().getStampByServer(node2.getBranchID()), node2.getBranchID()));
      testVV.addMaxAS(new AcceptStamp(node3.getCurrentVV().getStampByServer(node3.getBranchID()), node3.getBranchID()));
      assert testVV.includes(acvv):"\n"+node1.getCurrentVV() +"\n" + avv;
      assert acvv.equals(testVV):"\n"+acvv +"\n" + testVV;
      

//      
      // ---------------- test if the writes from the faulty node are retained when applying checkpoint  --------------------
      
      
      Log log5 = node5.getLogTest();
      log5.applyPOM(pom2, new POMRemap());
      AcceptVV origN5CVV = node5.getCurrentVV();
      assert log5.containsKey(fork2III);
      StoreInterface origStore = node5.getStore();
      
      ss = ((IrisNode)node5).applySyncPacket(SimulatorCheckpointUnit.getSyncPacket(chkPt));
      assert ss.status == SyncStatus.ReceivedNewCertificate:ss;
      
      testVV = new CounterVV(node1.getCurrentVV());
      ss = ((IrisNode)node5).sync(node1);
      assert ss.status == SyncStatus.SyncSuccessful:ss;
      
      newStore.putAll(origStore);
      assert this.compareStore(expectedStore, node5.getStore(), ignoreSet) == 0;
      
      acvv = new CounterVV(node5.getCurrentVV());
      acvv.dropNode(nodeId5);
      assert acvv.equals(testVV):"\n"+acvv +"\n" + testVV;
      
      
   // now test the same thing with real sync except this time POM has not been applied at the receiver
      log5 = node5Copy1.getLogTest();
      
      ss = node5Copy1.sync(node1);
      assert ss.status == SyncStatus.ReceivedNewCertificate:ss;
      ss = node5Copy1.sync(node1);
      assert ss.status == SyncStatus.SyncSuccessful:ss;

      acvv = new CounterVV(node5Copy1.getCurrentVV());
      acvv.dropNode(nodeId5);
      assert acvv.equals(testVV):"\n"+acvv +"\n" + testVV;
      assert compareStore(expectedStore, node5Copy1.getStore(), ignoreSet) == 0;
      
    }catch(Exception e){
      e.printStackTrace();
      assert false;
    }
  }
  
  
  
  /**
   * test if a checkpoint can be succesffully applied despite differences in history (branches)?
   */
  public void testCheckpointApplicability(){
    try{
      BranchID myNodeId = NodeFactory.createNodeId(UNIT_NODE_ID);
      BranchID nodeId1 = NodeFactory.createNodeId(15);
      BranchID nodeId2 = NodeFactory.createNodeId(20);
      BranchID nodeId3 = NodeFactory.createNodeId(25);

      Node myNode = NodeFactory.createNodeWithoutFilter(myNodeId);
      Node node1 = NodeFactory.createNodeWithoutFilter(nodeId1);
      Node node2 = NodeFactory.createNodeWithoutFilter(nodeId2);
      Node node3 = NodeFactory.createNodeWithoutFilter(nodeId3);

      ObjId oid0 = new ObjId("/test/0");
      myNode.write(oid0, new IrisDataObject(new Integer(0)));

      ObjId oid1 = new ObjId("/test/1");
      myNode.write(oid1, new IrisDataObject(new Integer(1)));

      LinkedList<Node> forks = myNode.fork(2);

      forks.get(0).write(new ObjId("/test/00"), new IrisDataObject(new Integer(2)));
      forks.get(0).write(new ObjId("/test/01"), new IrisDataObject(new Integer(3)));
      forks.get(1).write(new ObjId("/test/10"), new IrisDataObject(new Integer(4)));
      forks.get(1).write(new ObjId("/test/11"), new IrisDataObject(new Integer(5)));

      LinkedList<Node> newForks = forks.get(0).fork(2);
      newForks.get(0).write(new ObjId("/test/0000"), new IrisDataObject(new Integer(6)));
      newForks.get(0).write(new ObjId("/test/0001"), new IrisDataObject(new Integer(7)));
      newForks.get(1).write(new ObjId("/test/0010"), new IrisDataObject(new Integer(8)));
      newForks.get(1).write(new ObjId("/test/0011"), new IrisDataObject(new Integer(9)));

      node1.sync(newForks.get(0));
      node2.sync(forks.get(1));
      node3.sync(newForks.get(1));

      if(!SangminConfig.useSimulatorId){
        ((IrisNode)node1).exponentialBackoffSync(node2);
      }
      node1.sync(node2);

      Log log = node1.getLogTest();
      AcceptVV cvv = log.getMinimalVV(node1.getCurrentVV());
      HashMap<NodeId, Hash> hashMap = node1.getLogTest().getHashMap(cvv);
      Checkpoint chkPt = node1.generateNewCheckpoint(new HashedVV(cvv, hashMap), 1);
      if(!SangminConfig.useSimulatorId){
        ((IrisNode)node1).exponentialBackoffSync(node3);
      }
      node1.sync(node3);


      AcceptVV remappedVV = log.remap(new HashedVV(cvv, hashMap), true);
      AcceptVV currentVV = log.getMinimalVV(node1.getCurrentVV());
      if(Node.enableSecurity)
      assert !remappedVV.equalsIgnoreNegatives(currentVV):remappedVV +  " "   + currentVV;

      ((IrisNode)node1).applySyncPacket(SimulatorCheckpointUnit.getSyncPacket(chkPt));
      
      log = node1.getLogTest();
      remappedVV = log.remap(new HashedVV(cvv, hashMap), true);
      currentVV = log.getMinimalVV(node1.getCurrentVV());
      CounterVV curVV = new CounterVV(currentVV);
      curVV.dropNode(node1.getBranchID());
//      assert !remappedVV.equalsIgnoreNegatives(currentVV):"\n"+remappedVV +  "\n "   + currentVV;
      assert currentVV.includes(remappedVV):"\n"+remappedVV +  "\n "   + currentVV;
      assert remappedVV.equalsIgnoreNegatives(new AcceptVV(curVV)):"\n"+remappedVV +  "\n "   + curVV;


    }catch(Exception e){
      // TODO Auto-generated catch block
      e.printStackTrace();
    }
  }

  /**
   * Tests whether checkpoints generated are identical or not despite differences in history
   */
  public void testCheckpointEquivalence1(){
    // create two nodes with differences in the history prior to the omitVV and request checkpoint from these two nodes
    // check if they are identical or not?

    try{
      BranchID myNodeId = NodeFactory.createNodeId(UNIT_NODE_ID);
      BranchID nodeId1 = NodeFactory.createNodeId(15);
      BranchID nodeId2 = NodeFactory.createNodeId(20);
      BranchID nodeId3 = NodeFactory.createNodeId(25);

      Node myNode = NodeFactory.createNodeWithoutFilter(myNodeId);
      Node node1 = NodeFactory.createNodeWithoutFilter(nodeId1);
      Node node2 = NodeFactory.createNodeWithoutFilter(nodeId2);
      Node node3 = NodeFactory.createNodeWithoutFilter(nodeId3);

      ObjId oid0 = new ObjId("/test/0");
      myNode.write(oid0, new IrisDataObject(new Integer(0)));

      ObjId oid1 = new ObjId("/test/1");
      myNode.write(oid1, new IrisDataObject(new Integer(1)));

      LinkedList<Node> forks = myNode.fork(2);

      forks.get(0).write(new ObjId("/test/00"), new IrisDataObject(new Integer(2)));
      forks.get(0).write(new ObjId("/test/01"), new IrisDataObject(new Integer(3)));
      forks.get(1).write(new ObjId("/test/10"), new IrisDataObject(new Integer(4)));
      forks.get(1).write(new ObjId("/test/11"), new IrisDataObject(new Integer(5)));
      forks.get(1).write(new ObjId("/test/11"), new IrisDataObject(new Integer(6)));
      forks.get(1).write(new ObjId("/test/11"), new IrisDataObject(new Integer(7)));
      forks.get(1).write(new ObjId("/test/11"), new IrisDataObject(new Integer(7)));
      
     

      LinkedList<Node> newForks = forks.get(0).fork(2);
      newForks.get(0).write(new ObjId("/test/0000"), new IrisDataObject(new Integer(6)));
      newForks.get(0).write(new ObjId("/test/0001"), new IrisDataObject(new Integer(7)));
      newForks.get(1).write(new ObjId("/test/0010"), new IrisDataObject(new Integer(8)));
      newForks.get(1).write(new ObjId("/test/0011"), new IrisDataObject(new Integer(9)));

      node1.sync(newForks.get(0));
      node2.sync(forks.get(1));
      node3.sync(newForks.get(1));

      Log log = node1.getLogTest();
      AcceptVV cvv = log.getMinimalVV(node1.getCurrentVV());
      HashMap<NodeId, Hash> hashMap = node1.getLogTest().getHashMap(cvv);
      
      if(!SangminConfig.useSimulatorId){
        ((IrisNode)node1).exponentialBackoffSync(node2);
      }
      SyncStatus ss = node1.sync(node2);
      assert ss.status == SyncStatus.SyncSuccessful:ss;

      AcceptVV endVVVV = log.getMinimalVV(node1.getCurrentVV());
      HashMap<NodeId, Hash> endVVHashes = log.getHashMap(endVVVV);
      HashedVV endVV = new HashedVV(endVVVV, endVVHashes);

      Checkpoint chkPt = node1.generateNewCheckpoint(new HashedVV(cvv, hashMap), 1, endVV);
      if(!SangminConfig.useSimulatorId){
        ((IrisNode)node1).exponentialBackoffSync(node3);
      }
      node1.sync(node3);

      // find out what the cvv maps to
      Checkpoint chkPt1 = node1.generateNewCheckpoint(new HashedVV(cvv, hashMap), 1, endVV);
      assert chkPt.omitVV.equals(chkPt1.omitVV): chkPt.omitVV + " " + chkPt1.omitVV;
      assert chkPt.epochCount == chkPt1.epochCount: chkPt.epochCount + " " + chkPt1.epochCount;
      assert chkPt.lastWrite.equals(chkPt1.lastWrite): chkPt.lastWrite + " " + chkPt1.lastWrite;
      assert chkPt.objectStore.equals(chkPt1.objectStore): chkPt.objectStore + " " + chkPt1.objectStore;
      assert chkPt.unverifiableWrites.equals(chkPt1.unverifiableWrites): "\n" + chkPt.unverifiableWrites + "\n" + chkPt1.unverifiableWrites;
      assert chkPt.endVV.equals(chkPt1.endVV): "\n" + chkPt.endVV + "\n" + chkPt1.endVV;
      assert chkPt.flagMap.equals(chkPt1.flagMap): "\n" + chkPt.flagMap + "\n" + chkPt1.flagMap;
      assert chkPt.equals(chkPt1);
      assert Arrays.equals(chkPt.obj2Bytes(), chkPt1.obj2Bytes()):"\n" +Arrays.toString(chkPt.obj2Bytes()) + "\n chkPt1.obj2Bytes() \n" + Arrays.toString(chkPt1.obj2Bytes());
      assert chkPt.getHash().equals(chkPt1.getHash()):"\n" +Arrays.toString(chkPt.obj2Bytes()) + "\n chkPt1.obj2Bytes() \n" + Arrays.toString(chkPt1.obj2Bytes());

    }catch(Exception e){
      // TODO Auto-generated catch block
      e.printStackTrace();
      System.exit(0);
    }


  }
  /**
   * Tests whether checkpoints generated are identical or not despite differences in history
   */
  public void testCheckpointEquivalence(){
    // create two nodes with differences in the history prior to the omitVV and request checkpoint from these two nodes
    // check if they are identical or not?

    try{
      BranchID myNodeId = NodeFactory.createNodeId(UNIT_NODE_ID);
      BranchID nodeId1 = NodeFactory.createNodeId(15);
      BranchID nodeId2 = NodeFactory.createNodeId(20);
      BranchID nodeId3 = NodeFactory.createNodeId(25);

      Node myNode = NodeFactory.createNodeWithoutFilter(myNodeId);
      Node node1 = NodeFactory.createNodeWithoutFilter(nodeId1);
      Node node2 = NodeFactory.createNodeWithoutFilter(nodeId2);
      Node node3 = NodeFactory.createNodeWithoutFilter(nodeId3);

      ObjId oid0 = new ObjId("/test/0");
      myNode.write(oid0, new IrisDataObject(new Integer(0)));

      ObjId oid1 = new ObjId("/test/1");
      myNode.write(oid1, new IrisDataObject(new Integer(1)));

      LinkedList<Node> forks = myNode.fork(2);

      forks.get(0).write(new ObjId("/test/00"), new IrisDataObject(new Integer(2)));
      forks.get(0).write(new ObjId("/test/01"), new IrisDataObject(new Integer(3)));
      forks.get(1).write(new ObjId("/test/10"), new IrisDataObject(new Integer(4)));
      forks.get(1).write(new ObjId("/test/11"), new IrisDataObject(new Integer(5)));

      LinkedList<Node> newForks = forks.get(0).fork(2);
      newForks.get(0).write(new ObjId("/test/0000"), new IrisDataObject(new Integer(6)));
      newForks.get(0).write(new ObjId("/test/0001"), new IrisDataObject(new Integer(7)));
      newForks.get(1).write(new ObjId("/test/0010"), new IrisDataObject(new Integer(8)));
      newForks.get(1).write(new ObjId("/test/0011"), new IrisDataObject(new Integer(9)));

      node1.sync(newForks.get(0));
      node2.sync(forks.get(1));
      node3.sync(newForks.get(1));

      if(!SangminConfig.useSimulatorId){
        ((IrisNode)node1).exponentialBackoffSync(node2);
      }
      node1.sync(node2);

      Log log = node1.getLogTest();
      AcceptVV endVVVV = node1.getCurrentVV();
      HashMap<NodeId, Hash> endVVHashes = log.getHashMap(endVVVV);
      HashedVV endVV = new HashedVV(endVVVV, endVVHashes);
      AcceptVV cvv = log.getMinimalVV(node1.getCurrentVV());
      HashMap<NodeId, Hash> hashMap = node1.getLogTest().getHashMap(cvv);
      Checkpoint chkPt = node1.generateNewCheckpoint(new HashedVV(cvv, hashMap), 1, endVV);
      if(!SangminConfig.useSimulatorId){
        ((IrisNode)node1).exponentialBackoffSync(node3);
      }
      node1.sync(node3);

      // find out what the cvv maps to
      Checkpoint chkPt1 = node1.generateNewCheckpoint(new HashedVV(cvv, hashMap), 1, endVV);
      assert chkPt.omitVV.equals(chkPt1.omitVV): chkPt.omitVV + " " + chkPt1.omitVV;
      assert chkPt.epochCount == chkPt1.epochCount: chkPt.epochCount + " " + chkPt1.epochCount;
      assert chkPt.lastWrite.equals(chkPt1.lastWrite): chkPt.lastWrite + " " + chkPt1.lastWrite;
      assert chkPt.objectStore.equals(chkPt1.objectStore): chkPt.objectStore + " " + chkPt1.objectStore;
      assert chkPt.unverifiableWrites.equals(chkPt1.unverifiableWrites): chkPt.unverifiableWrites + " " + chkPt1.unverifiableWrites;
      assert chkPt.endVV.equals(chkPt1.endVV): "\n" + chkPt.endVV + "\n" + chkPt1.endVV;
      assert chkPt.flagMap.equals(chkPt1.flagMap): chkPt.flagMap + " " + chkPt1.flagMap;
      assert chkPt.acceptableBranches.equals(chkPt1.acceptableBranches): chkPt.acceptableBranches + " " + chkPt1.acceptableBranches;
      assert chkPt.equals(chkPt1);
      assert Arrays.equals(chkPt.obj2Bytes(), chkPt1.obj2Bytes()):"\n" +Arrays.toString(chkPt.obj2Bytes()) + "\n chkPt1.obj2Bytes() \n" + Arrays.toString(chkPt1.obj2Bytes());
      assert chkPt.getHash().equals(chkPt1.getHash()):"\n" +Arrays.toString(chkPt.obj2Bytes()) + "\n chkPt1.obj2Bytes() \n" + Arrays.toString(chkPt1.obj2Bytes());

    }catch(Exception e){
      // TODO Auto-generated catch block
      e.printStackTrace();
      System.exit(0);
    }


  }

  /**
   * test whether the handle failure method is working correctly by applying new POM to the log if any is present
   */
  public void testHandleFailure(){
    try{
      IrisNode.useAgreementCheckpoint = true;

      BranchID myNodeId = NodeFactory.createNodeId(UNIT_NODE_ID);
      BranchID nodeId1 = NodeFactory.createNodeId(15);
      BranchID nodeId2 = NodeFactory.createNodeId(20);
      BranchID nodeId3 = NodeFactory.createNodeId(25);

      Node myNode = NodeFactory.createNodeWithoutFilter(myNodeId);
      Node node1 = NodeFactory.createNodeWithoutFilter(nodeId1);
      Node node2 = NodeFactory.createNodeWithoutFilter(nodeId2);
      Node node3 = NodeFactory.createNodeWithoutFilter(nodeId3);

      ObjId oid0 = new ObjId("/test/0");
      myNode.write(oid0, new IrisDataObject(new Integer(0)));

      ObjId oid1 = new ObjId("/test/1");
      myNode.write(oid1, new IrisDataObject(new Integer(1)));

      LinkedList<Node> forks = myNode.fork(2);

      forks.get(0).write(new ObjId("/test/00"), new IrisDataObject(new Integer(2)));
      forks.get(0).write(new ObjId("/test/01"), new IrisDataObject(new Integer(3)));
      forks.get(1).write(new ObjId("/test/10"), new IrisDataObject(new Integer(4)));
      forks.get(1).write(new ObjId("/test/11"), new IrisDataObject(new Integer(5)));

      LinkedList<Node> newForks = forks.get(0).fork(2);
      newForks.get(0).write(new ObjId("/test/0000"), new IrisDataObject(new Integer(6)));
      newForks.get(0).write(new ObjId("/test/0001"), new IrisDataObject(new Integer(7)));
      newForks.get(1).write(new ObjId("/test/0010"), new IrisDataObject(new Integer(8)));
      newForks.get(1).write(new ObjId("/test/0011"), new IrisDataObject(new Integer(9)));

      node1.sync(newForks.get(0));
      node2.sync(forks.get(1));
      node3.sync(newForks.get(1));

      if(!SangminConfig.useSimulatorId){
        ((IrisNode)node1).exponentialBackoffSync(node2);
      }
      node1.sync(node2);

      Log log = node1.getLogTest();
      AcceptVV cvv = log.getMinimalVV(node1.getCurrentVV());

      VVIterator vvi = cvv.getIterator();
      BranchID b = null;
      while(vvi.hasMoreElements()){
        b = (BranchID)vvi.getNext();
        if(cvv.getStampByIteratorToken(b) > 4){
          break;
        }
      }
      assert cvv.getStampByIteratorToken(b) == 5:" cvv  " + cvv + " " + b ;
      CounterVV currentVV = new CounterVV(cvv);
      currentVV.dropNode(b);
      currentVV.addMaxAS(new AcceptStamp(2, b));

      HashMap<NodeId, Hash> hashMap = node1.getLogTest().getHashMap(new AcceptVV(currentVV));
      Checkpoint chkPt = node1.generateNewCheckpoint(new HashedVV(currentVV, hashMap), 1);
      cvv = log.getMinimalVV(node1.getCurrentVV());

      if(!SangminConfig.useSimulatorId){
        ((IrisNode)node1).exponentialBackoffSync(node3);
      }
      node1.sync(node3);
      ((IrisNode)node1).applySyncPacket(SimulatorCheckpointUnit.getSyncPacket(chkPt));
      
      log = node1.getLogTest();
      AcceptVV cvv1 = log.getMinimalVV(node1.getCurrentVV());
      CounterVV currVV = new CounterVV(cvv1);
      currVV.dropNode(node1.getBranchID());
      cvv1 = new AcceptVV(currVV);
      assert cvv1.equalsIgnoreNegatives(cvv): "\n"+cvv1 + "\ncvv "  +cvv;
    }catch(Exception e){
      // TODO Auto-generated catch block
      e.printStackTrace();
    }
  }

  /**
   * 
   * @param n
   * @param prefix
   * @param is
   * @param ie
   * @param startVal
   */
  public static void checkRead(Node n, 
      String prefix, 
      int is, int ie, int startVal){
    int ii;

    if(vverbose){
      System.out.println("Reading initial data: " + is + " to " + ie + " etc.");
    }
    for(ii = is; ii <= ie; ii++){

      ObjId oid = new ObjId(prefix + "/" + ii );
      if(vverbose){
        System.out.println("Reading " + oid);
      }
      LinkedList<StoreEntry> rawVals = n.rawRead(oid);
      Object val = code.simulator.unit.SimulatorUnit.resolveConflict(rawVals);
      assert val.equals(startVal + ii): "Expected value for obj " + oid + " was " + (startVal + ii) + " instead found " + val;
      
    }
  }

  /** 
   *  Write -- write 1-byte items to /prefix/[is..ie]/[js..je]/[ks..ke].
   *  Return final accept stamp.
   **/ 
  public static void write(Node n, 
      String prefix, 
      int is, int ie, 
      int startVal){
    int ii;

    if(vverbose){
      System.out.println("Writing initial data: " + is + " to " + ie + " etc.");
    }
    for(ii = is; ii <= ie; ii++){

      ObjId oid = new ObjId(prefix + "/" + ii );
      if(vverbose){
        System.out.println("Writing " + oid);
      }

      n.write(oid, new IrisDataObject(new Integer(startVal + ii)));
    }
  }


  /*
   * "new TestSuite(Class c)" constructs a test suite
   * containg every method whose name begins with "/test"
   * 
   * TBD: update class name
   */
  public static Test suite(){
    TestSuite suite = new TestSuite(SimulatorMechanismCheckpointUnit.class);
    return suite;
  }


  /*
   * main() lets us run just this set of unit tests
   * from the comand line (you can also invoke 
   * the testrunner on this class and it will find
   * the suite())
   *
   * usage: java <classname> [-verbose] [-vverbose] [testName]*
   * 
   *   If verbose or vverbose are included, print info to screen
   *
   *   If [testName]* are included, then run test called "/test[testName]"
   *   for each such [testName]. E.g., "java TestEmtpy foo" runs
   *   TwoNodesSubscriptionUnit.testfoo() as a TestCase.
   *
   * TBD: update class name
   */
  public static void main(String s[]) {
    String name = "SimulatorCheckpointUnit";
    System.err.print(name + " self test begins...");
    Test test;
    test = suite();
    TestRunner tr = new TestRunner();
    tr.doRun(test);
    System.exit(0);
  }

}



