package code.simulator.log;

import java.util.*;

import code.AcceptVV;
import code.NoSuchEntryException;
import code.NodeId;
import code.VVIterator;

import code.branchDetecting.*;
import code.*;
import code.security.SangminConfig;
import code.security.ahs.DependencyVV;
import code.simulator.*;
import code.simulator.branchManager.*;
import code.simulator.checkpoint.Checkpoint;

import java.util.Map.Entry;
import code.CounterVV;


/**
 * Log provides functions for 
 * (1) applying a write conditional on certain checks passing
 * (2) removing a write
 * (3) getCurrentVV
 * (4) generating ProofOfMisbehavior
 * (5) applying POM
 * 
 * @author princem
 *
 */
public class Log extends HashMap<NodeId, NodeLog>{// implements ForkableLog, SegmentCheckInterface{

  private CounterVV cvv;
  private HashedVV omitVV;
  //BranchKnowledge bk;

  private HashMap<Long, SimpleTree<BranchID>> branchKnowledge;

  public static final byte ApplyIfAllPass = 0;
  public static final byte ApplyIfFIFOPass = 1;
  public static final byte Apply = 2;


  private HashSet<ProofOfMisbehavior> pomList;
  private HashSet<Long> evictedNodes;
  private HashMap<Long, Long> baseOmitVV; 
  //  public CounterVV forkedMap;
  public Log(){
    super();
    //    forkedMap = new CounterVV();
    cvv = new CounterVV();
    omitVV = new HashedVV();
    pomList  = new HashSet<ProofOfMisbehavior>();
    branchKnowledge = new HashMap<Long, SimpleTree<BranchID>>();
    evictedNodes = new HashSet<Long>();
    baseOmitVV = new HashMap<Long, Long>();
    //  bk = new BranchKnowledge();
  }

  public AcceptVV getCurrentVV(){
    return new AcceptVV(cvv);
  }

  public void applyCheckpoint(Checkpoint chkPt){
    assert chkPt.omitVV.includes(omitVV);
    cvv.advanceTimestamps(chkPt.omitVV);
    omitVV = (HashedVV)chkPt.omitVV;
    baseOmitVV = this.getBaseVV(omitVV);
    //    // create branches corresponding to the chkPt
  }

  public void addEvictedNodes(NodeId n){
    evictedNodes.add(n.getIDint());

  }

  public LogStatus apply(PreciseInv pi){
    return this.apply(pi, Log.ApplyIfAllPass);
  }

  //  /**
  //   * Check if the longer segment includes the short one.
  //   * If not, ConflictingSegmentsException is thrown.
  //   * If so, and also if the short one is a non-leaf segment,
  //   * it returns Hash-TimeStamp tuple for the start of the uncommon 
  //   * part of the long segment.
  //   * If the short one is a leaf segment, returns null;
  //   */
  //  public HashTSTuple checkCompatibility(Segment segShort, Segment segLong) 
  //  throws ConflictingSegmentsException{
  //    if(segShort.getEndTS() == segLong.getEndTS()){
  //      if(segShort.getEndHash().equals(segLong.getEndHash())){
  //        return null;
  //      } else {
  //        throw new ConflictingSegmentsException();
  //      }
  //    }
  //
  //    HashTSTuple ht = getSummaryHashAtOrBefore(segShort.getBid(), segShort.getEndTS());
  //
  //
  //    if(ht.getTimeStamp() != segShort.getEndTS() ||
  //        !ht.getHash().equals(segShort.getEndHash())){
  //      throw new ConflictingSegmentsException();
  //    }
  //
  //    if(segShort.isLeaf()){
  //      return null;
  //    }    
  //
  //    HashTSTuple tuple = getSummaryHashAfter(
  //        segShort.getBid(), segShort.getEndTS());
  //    if(Node.shouldFail) assert tuple!=null;
  //
  //    return tuple;
  //  }

  /**
   * 
   * @return cloned version of knowledge
   */
  public BranchKnowledge getKnowledge(){
    return null;
    //  return bk.clone();
  }

  /**
   * debug function used to check if the logs until the accept stamp (in the accept stamp order) are identical or not
   * @param l
   * @param as
   * @return
   */
  public boolean compare(Log l, AcceptStamp as){
    System.out.println("compareLog called for " + as);
    NodeLog myLog = this.get(as.getNodeId());
    int index = myLog.binarySearch(as.getLocalClock()).getValue();
    SimPreciseInv spi = (SimPreciseInv)myLog.get(index);
    assert spi.getAcceptStamp().equals(as);
    VVIterator vvi = spi.getDVV().getIterator();
    while(vvi.hasMoreElements()){
      NodeId n = vvi.getNext();
      if(!l.containsKey(n)){
        System.out.println("remote node doesn't have a component for node " + n);
      }else{
        int index1 = l.get(n).binarySearch(spi.getDVV().getStampByIteratorToken(n)).getValue();
        if(index1 < 0 || index1 >= l.get(n).size()){
          System.out.println("remote node doesn't have as " + spi.getDVV().getStampByIteratorToken(n));
          System.out.println(l.get(n));
        }else{
          SimPreciseInv pi = (SimPreciseInv)l.get(n).get(index1);
          NodeLog nl = this.get(n);
          SimPreciseInv actualpi = (SimPreciseInv)nl.get(nl.binarySearch(spi.getDVV().getStampByIteratorToken(n)).getValue());

          if(!actualpi.equalsIgnoreBranches(pi)){
            System.out.println("PROBLEM: " + actualpi + " AND " + pi + " are mismatching....there must exist an earlier conflict");
          }
        }
      }
    }
    return true;
  }

  /**
   * This function applies the given invalidate under the specified applyCondition. If the applyCondition is Apply then the invalidate is applied 
   * without caring about security checks. On the other hand, if the applyCondition is ApplyIfFIFO then the invalidate is applied if FIFO 
   * checks pass. Finally, if the applyCondition is ApplyIfAllPass then the invalidate is only applied if all checks pass. 
   * The returned LogStatus contains three components. First one 
   * indicates whether the invalidate was successfully applied (ApplySuccessful/PseudoApplySuccessful) or not (ApplyFailed)
   * PseudoApplySuccessful is returned when the same invalidate was already applied.
   * 
   * Second one indicates the verification status
   * and the third one provides the flags of the writes corresponding to the DVV components
   * @param pi
   * @param applyCondition
   * @return
   */
  public LogStatus apply(PreciseInv pi, byte applyCondition){
    HashMap<NodeId, SimPreciseInv> invals = null;
    LogStatus ls = null;
    byte success = LogStatus.VerificationPassed;
    if(Node.shouldFail) assert pi instanceof SimPreciseInv;
    SimPreciseInv spi = (SimPreciseInv)pi;
    
    // the DVV check will guard against the reordering of writes
    
    if(!Node.enableSecurity){
      if(this.getCurrentVV().includes(pi.getAcceptStamp())){
        LogStatus.makeLogStatusPassedVerificationStatus(LogStatus.PseudoApplySuccessful);
      }else if(!getCurrentVV().includes(spi.getDVV())){
        return LogStatus.makeLogStatusFailedVerificationStatus(LogStatus.DVVInclusionFail);
      }
      applyCondition = Log.Apply;
    }

    if(spi.getAcceptStamp().getLocalClock() > System.currentTimeMillis()*1000){
      return LogStatus.makeLogStatusFailedVerificationStatus(LogStatus.TimeStampExhaustionCheckFailed);
    }
    if(!getCurrentVV().includes(spi.getDVV()) && (applyCondition == Log.ApplyIfAllPass) && !SangminConfig.enableInconsistencyNotification){
      System.err.println("DVV inclusion check failed");
      return LogStatus.makeLogStatusFailedVerificationStatus(LogStatus.DVVInclusionFail);
    }

    // check if the accept stamp is larger than all prior accept stamps
    VV dvv = spi.getDVV();
    VVIterator vvi = dvv.getIterator();
    long localClock = spi.getAcceptStamp().getLocalClock();
    boolean foundLocal = false;
    long id = spi.getAcceptStamp().getNodeId().getIDint();
    while(vvi.hasMoreElements()){
      NodeId n = vvi.getNext();
      if(dvv.getStampByIteratorToken(n) >= localClock){
        return LogStatus.makeLogStatusFailedVerificationStatus(LogStatus.ImproperLamportClock);
      }
      if(n.getIDint() == id){
        if(foundLocal){
          return LogStatus.makeLogStatusFailedVerificationStatus(LogStatus.ImproperDVV);
        }else{
          foundLocal = true;
        }
      }
    }

    if(Node.shouldFail) assert spi instanceof SimPreciseInv;

    SimPreciseInv sspi = (SimPreciseInv)spi;

    if(containsKey(sspi.getNodeId())){
      NodeLog nl = this.get(sspi.getNodeId());
      int index = nl.binarySearch(sspi.getAcceptStamp().getLocalClock()).getValue();
      if((index >= 0 && index <= (nl.size() - 1)) && sspi.equalsIgnoringFlags(nl.get(index))){
        if(Node.shouldFail) assert nl.size() > 0;
        return LogStatus.makeLogStatusPassedVerificationStatus(LogStatus.PseudoApplySuccessful);
      }
    }

    invals = this.getInvals(spi.getDVV());
//    System.out.println(sspi);
//    System.out.println(this.get(sspi.getNodeId()).get(4));

    if((ls = inclusionCheck(sspi, invals, applyCondition)).getApplyStatus() != LogStatus.ApplySuccessful){
      return ls;
    }
//    System.out.println(sspi);
//    System.out.println(this.get(sspi.getNodeId()).get(4));
//    System.out.println(sspi.equals(this.get(sspi.getNodeId()).get(4)));
    
    success = ls.getVerificationStatus();
    if(applyCondition != Log.Apply){
      //    compatibility check
      if((ls = compatibilityCheck(spi, invals)).getApplyStatus() != LogStatus.ApplySuccessful){
        return ls;
      }
    }else if(cvv.includes(spi.getAcceptStamp())){
      return LogStatus.makeLogStatusPassedVerificationStatus(LogStatus.PseudoApplySuccessful);
    }


    if((ls = this.applyBranchKnowledge(spi, applyCondition)).getApplyStatus() != LogStatus.ApplySuccessful){
      return ls;
    }

    cvv.advanceTimestamps(spi.getAcceptStamp());
    if(!this.containsKey(spi.getNodeId())){
      this.put(spi.getNodeId(), new NodeLog());
    }
    this.get(spi.getNodeId()).add(spi);
    if(Node.enableSecurity && (applyCondition == Log.ApplyIfAllPass)){
      sspi = (SecureSimPreciseInv)spi;
      assert invals != null: spi;
      sspi.getFlags().addAll(this.getFlags(invals.values(), sspi.getEpoch()));  
    }


    if(success == LogStatus.VerificationPassed){
      ls = LogStatus.makeDefaultLogStatus();
      return ls;
    }else{
      ls = LogStatus.makeLogStatus(LogStatus.ApplySuccessful, success);
      return ls;
    }
  }


  /**
   * returns whether the NodeId form of VV would include the nodeId form of as or not?
   * @param vv
   * @param as
   * @return
   */
  private boolean baseStrongIncludes(AcceptStamp as){
    long id = as.getNodeId().getIDint();
    long ts = as.getLocalClock();
    if(baseOmitVV.containsKey(id) && baseOmitVV.get(id) > ts){
      return true;
    }
    return false;
  }

  /**
   * returns whether the NodeId form of VV would include the nodeId form of as or not?
   * @param vv
   * @param as
   * @return
   */
  public HashMap<Long, Long> getBaseVV(VV vv){
    HashMap<Long, Long> baseVV = new HashMap<Long, Long>();
    VVIterator vvi = vv.getIterator();
    while(vvi.hasMoreElements()){
      NodeId n = vvi.getNext();
      long id = n.getIDint();
      long ts = vv.getStampByIteratorToken(n);
      if(!baseVV.containsKey(id) || (baseVV.get(id) < ts)){
        baseVV.put(id, ts);
      }
    }
    return baseVV;
  }

  private LogStatus inclusionCheck(SimPreciseInv sspi, HashMap<NodeId, SimPreciseInv> invals, byte applyCondition){
    byte success = LogStatus.VerificationPassed;
    if(!Node.enableSecurity){
      return LogStatus.makeDefaultLogStatus();
    }
    Hash h = null;
    if(invals.size() == sspi.getDVV().getSize()){
      HashMap<NodeId, Hash> hashes = getHashMap(invals);
      HashedVV dvv = (HashedVV)sspi.getDVV();
      if(!dvv.isInitialized()){
        if(!dvv.initialize(hashes)){
          return LogStatus.makeLogStatusFailedVerificationStatus(LogStatus.OptimizedSyncFailed);
        }
      }
      h = getHash(dvv.createMatchingHashedVV(hashes));
    }else{
      System.out.println("invals" + invals + "\n" + sspi);
      System.err.println("invals" + invals + "\n" + sspi);
    }

    LogStatus ls = null;
    if(h == null || !h.equals(sspi.getHash())){
      if(applyCondition == Log.ApplyIfAllPass){
        boolean accept = false;
        if(SangminConfig.enableInconsistencyNotification){
          accept = true;
          // conditions under which an unverifiable write will be accepted 
          // if for each DVV components, one or more of the following conditinos are true
          // (1) its hash matches local hash 
          // (2) its as (remapped) lies inside omitVV
          // (3) the corresponding node is faulty and the hash is not present in the local history
          HashedVV hdvv = (HashedVV)sspi.getDVV();
          for(AcceptStamp as: hdvv.getAllStamps()){
            if(invals.containsKey(as.getNodeId())){
              SecureSimPreciseInv sspi1 = (SecureSimPreciseInv)invals.get(as.getNodeId());
              if(!sspi1.generateMyHash().equals(hdvv.getHashByIteratorToken(as.getNodeId()))){
                accept = false;
                break;
              }
            }else {
              if(!(baseStrongIncludes(as) && applyCondition==Log.ApplyIfAllPass) && // as is included in the omitVV and the node is not faulty    
                  !(evictedNodes.contains(as.getNodeId().getIDint()) && this.getBranchID(as.getNodeId(), as.getLocalClock(), hdvv.getHashByIteratorToken(as.getNodeId())) == null)){
                accept = false;
                break;
              }
            }
          }
        }

        if(!accept){
          //TODO: address the case when DVV is not precise
          System.out.println("spi.getDVV(): " + sspi.getDVV());
          System.out.println("SH check (inclusion) failed" + (h!=null?h:"NULL") + " \n " + sspi.getHash());
          ls = LogStatus.makeLogStatusFailedVerificationStatus(LogStatus.InclusionFail);
          return ls;
        }else{
          sspi.getFlags().add(Flags.InconsistentWrite);
          success = LogStatus.FIFOPassedInclusionUnverifiable;
        }
      }else if(applyCondition == Log.ApplyIfFIFOPass){
        VVIterator vvi  = sspi.getDVV().getIterator();
        NodeId n = null;
        AcceptStamp as = null;
        long id = sspi.getAcceptStamp().getNodeId().getIDint();
        while(vvi.hasMoreElements()){
          n = vvi.getNext();
          if(n.getIDint() == id){
            assert as == null;
            as = new AcceptStamp(sspi.getDVV().getStampByIteratorToken(n), n);
          }
        }

        try{
          if(as != null && !this.getHash(as).equals(sspi.getHash(as.getNodeId()))){
            ls = LogStatus.makeLogStatusFailedVerificationStatus(LogStatus.FIFOFail);
            return ls;
          }else{
            success = LogStatus.FIFOPassedInclusionUnverifiable;
          }
        }catch(NoSuchEntryException e){
          e.printStackTrace();
          assert false;
        }
      }
    }
    ls = LogStatus.makeLogStatus(LogStatus.ApplySuccessful, success);

    return ls;
  }


  private LogStatus compatibilityCheck(SimPreciseInv sspi, HashMap<NodeId, SimPreciseInv> invals){
    if(!Node.enableSecurity){
      return LogStatus.makeDefaultLogStatus();
    }
    if(containsKey(sspi.getNodeId())){


      NodeLog nl = null;
      nl = this.get(sspi.getNodeId());
      Entry<Boolean, Integer> search = nl.binarySearch(sspi.getAcceptStamp().getLocalClock());
      int index = search.getValue();
      if(index < (nl.size()-1) ){
        if(index < 0 || !sspi.equals(nl.get(index))){
          System.err.println("compatibility check failed because trying to insert in the middle of the log");
          LogStatus ls = LogStatus.makeLogStatusFailedVerificationStatus(LogStatus.CompatibilityFail);
          return ls;
        }else{
          LogStatus ls = LogStatus.makeLogStatusPassedVerificationStatus(LogStatus.PseudoApplySuccessful);
          return ls;
        }
      }else{

        //check that the DVV includes component for the last write
        if(sspi.getDVV().containsNodeId(sspi.getNodeId())){
          assert index != -1:"\n"+sspi +"\n" +nl;
          try{

            if(sspi.getDVV().getStampByServer(sspi.getNodeId()) != nl.get(index).getAcceptStamp().getLocalClock()){
              System.err.println("compatibility check failed because dvv doesn't include the last local write");
              LogStatus ls = LogStatus.makeLogStatusFailedVerificationStatus(LogStatus.CompatibilityFail);
              return ls;
            }
          }catch(NoSuchEntryException e){
            e.printStackTrace();
            if(Node.shouldFail) assert false;
          }
        }else{
          if(index >= 0){
            System.err.println("compatibility check failed because dvv doesn't include the last local write");
            LogStatus ls = LogStatus.makeLogStatusFailedVerificationStatus(LogStatus.CompatibilityFail);
            return ls;
          }
        }

      }
    }
    return LogStatus.makeDefaultLogStatus();

  }

  private SimpleTree<BranchID> getTree(long id){
    if(!branchKnowledge.containsKey(id)){
      SimpleTree<BranchID> tree = new SimpleTree<BranchID>();
      branchKnowledge.put(id, tree);
      return tree;
    }else{
      return branchKnowledge.get(id);
    }
  }

  private LogStatus applyBranchKnowledge(SimPreciseInv spi, byte applyCondition){
    // do the branchKnowledge based verification here
    // 1) check if the added inval is not being added to a forked branch
    // and 2) if the added inval is going to a new branch then appropriate POM has been applied for its parent
    // if not then return the appropriate verification failure status

    SimpleTree<BranchID> tree = getTree(spi.getAcceptStamp().getNodeId().getIDint());

    try{
      BranchID bid = (BranchID)spi.getNodeId();
      if(tree.containsNode(bid)){
        if(!tree.isLeaf(bid)){
          LogStatus ls = LogStatus.makeLogStatusFailedVerificationStatus(LogStatus.WriteToForkedParentBranch);
          return ls;
        }
      }else{
        // new branch needs to be added
        LinkedList<BranchID> parents = this.getParents(spi);

        if((parents.size() == 0 && !baseOmitVV.containsKey(bid.getIDint()))|| //(parents.size() == 1 && parents.contains(spi.getAcceptStamp().getNodeId()) && 
            applyCondition == Log.Apply//)
        ){
          tree.addVal(null, bid);
        }else{

          if(parents.size() != 1){
            LogStatus ls = LogStatus.makeLogStatusFailedVerificationStatus(LogStatus.ImproperDVV);
            assert parents.size() == 1;
            return ls;
          }
          LinkedList<BranchID> mappedParents = new LinkedList<BranchID>(); // the set of parents NOT in the omitVV range
          NodeId localOmitVVNode = null;
          VVIterator vvi = omitVV.getIterator();
          while(vvi.hasMoreElements()){
            NodeId n = vvi.getNext();
            if(n.getIDint() == bid.getIDint()){
              assert localOmitVVNode == null;
              localOmitVVNode = n;
            }
          }

          BranchID b = parents.getFirst();
          try{
            long ts = spi.getDVV().getStampByServer(b);
            if(localOmitVVNode == null || omitVV.getStampByServer(localOmitVVNode) <= ts){
              mappedParents.add(b);
            }
          }catch(Exception e){
            e.printStackTrace();
            assert false: parents +"\n" + spi.getDVV();
            return null;
          }


          if(mappedParents.size() > 0 && tree.containsNode(mappedParents.getFirst())){ // parent different from the child
            assert mappedParents.size() == 1;
            if(tree.isLeaf(mappedParents.getFirst())){
              boolean found = false;
              for(ProofOfMisbehavior pom:pomList){
                if(!(pom instanceof ConcurrentPOM) && pom.getParent().getAcceptStamp().getNodeId().getIDint() == spi.getAcceptStamp().getNodeId().getIDint()){
                  // see if any of the POMs must be reapplied to address the rollback issue---state rollback 
                  // may leave some POMs that are unapplied
                  BranchID nbid = this.getBranchID(pom.getParent());
                  if(nbid.equals(mappedParents.getFirst())){
                    found = this.applyPOM(pom, new POMRemap());
                    break;
                  }
                }
              }
              if(!found){
                LogStatus ls = LogStatus.makeLogStatusFailedVerificationStatus(LogStatus.MissingPOMForForkedChildBranch);
                return ls;
              }
            }else{
              tree.addVal(mappedParents.getFirst(), bid);
            }
          }else if(applyCondition != Log.Apply){
            // here check if the reason parent is different from the child is that a checkpoint has been applied
            // and if the branch name of child is appropriate

            // Conditions for acceptable inval:
            // (1) only one branch from spi creator in omitVV
            // (2) spi.ts >= omitVV[spi.nodeId.idInt] and spi.hash != lastWrites[spi.nodeId]
            // (3) spi.dvv[spi.nodeId.idInd] < omitVV[spi.nodeId.idInt]
            // (4) spi.nodeId = Hash(spi)
            assert !this.evictedNodes.contains(bid.getIDint());

            assert localOmitVVNode != null;
            if(spi.getAcceptStamp().getLocalClock() < omitVV.getStampByServer(localOmitVVNode) || 
                (spi.getAcceptStamp().getLocalClock() == omitVV.getStampByServer(localOmitVVNode) && 
                    spi.generateMyHash().equals(omitVV.getHashByServer(localOmitVVNode)))){
              LogStatus ls = LogStatus.makeLogStatusFailedVerificationStatus(LogStatus.ImproperBranchID);
              return ls;
            }

            if(spi.getDVV().getStampByServer(parents.getFirst()) >= omitVV.getStampByServer(localOmitVVNode)){
              assert false; // the inclusion check should have invalidated this write
              LogStatus ls = LogStatus.makeLogStatusFailedVerificationStatus(LogStatus.WriteFromForkedNodeWithInvalidBranchHistory);
              return ls;
            }

            BranchID expectedBID = new BranchID(bid.getIDint(), spi.getAcceptStamp().getLocalClock(), ((SimPreciseInv)spi).generateMyHash());
            if(!expectedBID.equals(bid)){
              LogStatus ls = LogStatus.makeLogStatusFailedVerificationStatus(LogStatus.ImproperBranchID);
              return ls; 
            }

            boolean foundMatchingPOM = false;
            for(ProofOfMisbehavior pom: pomList){
              if(pom instanceof ConcurrentPOM){
                ConcurrentPOM cpom = (ConcurrentPOM)pom;
                SimPreciseInv sspi = (SimPreciseInv)cpom.getFaultyBranch();
                if(sspi.generateMyHash().equals(((SimPreciseInv)spi).generateMyHash())){
                  foundMatchingPOM = true;
                }
              }
            }

            if(!foundMatchingPOM){
              LogStatus ls = LogStatus.makeLogStatusFailedVerificationStatus(LogStatus.MissingPOMForForkedChildBranch);
              return ls;
            }

            tree.addVal(null, bid);
          }else{
            assert false: "\nthis.bk" + this.branchKnowledge + " \nthis" + this + "\n spi" + spi;
          }
        }
      }
    }catch(Exception e){
      e.printStackTrace();
      assert false: "\nthis.bk" + this.branchKnowledge + " \nthis" + this + "\n spi" + spi;
    }
    return LogStatus.makeDefaultLogStatus();
  }

  public TreeSet getFlags(AcceptStamp as){
    NodeLog nl = this.get(as.getNodeId());
    int index = nl.binarySearch(as.getLocalClock()).getValue();
    return ((SimPreciseInv)nl.get(index)).getFlags();
  }

  public void addFlag(AcceptStamp as, Object o){
    NodeLog nl = this.get(as.getNodeId());
    int index = nl.binarySearch(as.getLocalClock()).getValue();
    ((SimPreciseInv)nl.get(index)).getFlags().add(o);
  }

  public void removeFlag(AcceptStamp as, Object o){
    NodeLog nl = this.get(as.getNodeId());
    int index = nl.binarySearch(as.getLocalClock()).getValue();
    ((SimPreciseInv)nl.get(index)).getFlags().remove(o);
  }

  public void removeAllFlags(Object o){
    for(NodeId b: this.keySet()){
      NodeLog nl = this.get(b);
      for(PreciseInv pi: nl){
        SimPreciseInv spi = (SimPreciseInv)pi;
        spi.getFlags().remove(o);
      }
    }
  }

  public LinkedList<BranchID> getParents(SimPreciseInv spi){
    VV dvv = spi.getDVV();
    BranchID bid = (BranchID)spi.getAcceptStamp().getNodeId();
    VVIterator vvi = dvv.getIterator();
    LinkedList<BranchID> parents = new LinkedList<BranchID>();
    while(vvi.hasMoreElements()){
      NodeId n = vvi.getNext();
      if(n.getIDint() == bid.getIDint() && !parents.contains(n)){
        parents.add((BranchID)n);
      }
    }
    return parents;
  }

  // cases on forking point
  // DVV not precise : currently not handled
  // AS_u !> all prior AS: write corresponding to DVV_u
  // DVV_u != last smaller write: write corresponding to DVV_u
  public ProofOfMisbehavior getProofOfMisbehavior(SimPreciseInv sspi){
    // two incompatible updates and all the hashes necessary to verify their connection to immediately common ancestor
    // parent is the update with ts = DVV or smaller
    // if smaller, the node violated the contract and therefore we have a POM
    // if DVV has no local component, then dummy update is the parent

    assert Node.enableSecurity;
    assert sspi instanceof SecureSimPreciseInv;

    NodeLog nl = this.get(sspi.getNodeId());
    SimPreciseInv forkingPoint = null;
    SimPreciseInv child2 = null;
    if(sspi.getDVV().containsNodeId(sspi.getNodeId())){
      try{
        int index = nl.binarySearch(sspi.getDVV().getStampByServer(sspi.getNodeId())).getValue();
        if(Node.shouldFail) assert (nl.size()-1) > index;
        forkingPoint = (SimPreciseInv)nl.get(index);
        child2 = (SimPreciseInv)nl.get(index+1);
        if(Node.shouldFail) assert forkingPoint.getAcceptStamp().getLocalClock() == sspi.getDVV().getStampByServer(sspi.getNodeId());
      }catch(NoSuchEntryException e){
        e.printStackTrace();
        if(Node.shouldFail) assert false;
      }
    }else{
      forkingPoint = DummySimPreciseInv.getSecureSimPreciseInv(new AcceptStamp(-1, sspi.getNodeId()));
    }

    return new ProofOfMisbehavior(forkingPoint, sspi, child2);
  }

  /**
   * used by the applyCheckpoint and applyEvictionCertificate
   * @param as
   * @return
   */
  public boolean dropWrite(AcceptStamp as){
    NodeLog nl = this.get(as.getNodeId());
    int index = nl.binarySearch(as.getLocalClock()).getValue();
    if(nl.get(index).getAcceptStamp().equals(as)){
      nl.remove(index);
      return true;
    }else{
      return false;
    }
  }

  /** 
   *  getSummaryHash() 
   **/ 
  public Hash getHash(AcceptStamp as){

    if(!Node.enableSecurity){
      return Hash.NullHash;
    }
    if(as == null || as.getLocalClock() < 0){      
      return new Hash("EmptyDVV".getBytes(), true);
    }

    NodeId nodeId = as.getNodeId();
    //ByteArrayOutputStream tempBuffer = new ByteArrayOutputStream();
    long ts = as.getLocalClock(); 
    NodeLog nl = this.get(nodeId);
    int index = nl.binarySearch(ts).getValue();

    if(Node.shouldFail) assert index >= 0: ts + " nl " + nl;
    PreciseInv p = nl.get(index); 

    if(!(p instanceof SecureSimPreciseInv)){
      if(Node.shouldFail) assert false;
      return ((SimPreciseInv)p).generateMyHash();
    }else{
      SimPreciseInv pi = (SimPreciseInv)p;
      return pi.generateMyHash();
    }

  }

  /** 
   *  getSummaryHash() 
   *  returns null when hash can't be computed based on the local state
   **/ 
  public static Hash getHash(HashedVV hvv){

    if(!Node.enableSecurity){
      return Hash.NullHash;
    }
    return hvv.getHash();
  }

  /**
   * returns whether the immediately preceeding dependencies are consistent or not. Consistent here means that there fields have not been 
   * modified since their hash was computed. Meant to track any werid SH mismatch or signature verification failure due to modification of 
   * an inval. 
   *  
   * @param dvv
   * @return
   */
  public boolean isConsistent(DependencyVV dvv){

    if(!Node.enableSecurity){
      return true;
    }
    if(dvv.getSize() == 0 || dvv.getMaxTimeStamp() < 0){      
      return true;
    }

    VVIterator vvi = dvv.getIterator();
    while(vvi.hasMoreElements()){
      NodeId nodeId = vvi.getNext();
      //ByteArrayOutputStream tempBuffer = new ByteArrayOutputStream();
      long ts = dvv.getStampByIteratorToken(nodeId); 
      if(ts<0){
        continue;
      }
      NodeLog nl = this.get(nodeId);
      int index = nl.binarySearch(ts).getValue();
      if(index < 0){
        System.out.println();
      }
      if(Node.shouldFail) assert index >= 0: ts + " nl " + nl;
      PreciseInv p = nl.get(index); 

      if(!(p instanceof SecureSimPreciseInv)){
        if(Node.shouldFail) assert false;
      }else{
        SecureSimPreciseInv pi = (SecureSimPreciseInv)p;
        if(!pi.isConsistent()){
          return false;
        }
      }
    }
    return true;
  }

  /**
   * Return a tuple of a smallest time stamp that is greater than the given
   * time stamp and the corresponding summary hash
   */
  public HashTSTuple getSummaryHashAfter(BranchID bid, long timestamp){
    if(Node.shouldFail) assert this.containsKey(bid);
    NodeLog nl = this.get(bid);
    int index = nl.binarySearch(timestamp).getValue(); // index of a matching timestamp or the smaller one
    if(Node.shouldFail) assert index +1< nl.size();
    PreciseInv pi = nl.get(index+1);
    long ts = pi.getAcceptStamp().getLocalClock();
    Hash hash = ((SimPreciseInv)pi).generateMyHash();    
    return new HashTSTuple(ts, hash);
  }


  /**
   * Return a tuple of a largest time stamp that is equal to or less than the given
   * time stamp and the corresponding summary hash
   */
  public HashTSTuple getSummaryHashAtOrBefore(BranchID bid, long timestamp){
    if(Node.shouldFail) assert timestamp >= 0;
    if(Node.shouldFail) assert this.containsKey(bid);
    NodeLog nl = this.get(bid);
    int index = nl.binarySearch(timestamp).getValue(); // index of a matching timestamp or the smaller one
    if(Node.shouldFail) assert index >= 0;
    PreciseInv pi = nl.get(index);

    return new HashTSTuple(pi.getAcceptStamp().getLocalClock(), ((SimPreciseInv)pi).generateMyHash());      
  }

  public BranchID getBranchID(SimPreciseInv spi){
    NodeId baseId = ((BranchID)spi.getNodeId()).getBaseId();
    BranchID localBranchID = null;
    long as = spi.getAcceptStamp().getLocalClock();
    for(NodeId nid: this.keySet()){
      if(Node.shouldFail) assert nid instanceof BranchID;
      BranchID bid = (BranchID)nid;
      if(bid.contains(baseId)){
        NodeLog nl = this.get(bid);
        int index = nl.binarySearch(as).getValue();
        if(index >= 0 && index < nl.size() && 
            ((SimPreciseInv)nl.get(index)).equalsIgnoreBranches(spi)){
          if(Node.shouldFail) assert localBranchID == null;
          localBranchID = bid;
        }
      }
    }
    return localBranchID;
  }

  /**
   * return the BranchID for the given acceptStamp and the hash value: return null if no match found
   * @param baseId
   * @param as
   * @param hash
   * @return
   */
  public BranchID getBranchID(NodeId baseId, long as, Hash hash){
    BranchID localBranchID = null;
    for(NodeId nid: this.keySet()){
      if(Node.shouldFail) assert nid instanceof BranchID;
      BranchID bid = (BranchID)nid;
      if(bid.contains(baseId)){
        NodeLog nl = this.get(bid);
        int index = nl.binarySearch(as).getValue();
        if(index >= 0 && index < nl.size() && 
            ((SimPreciseInv)nl.get(index)).generateMyHash().equals(hash)){
          if(Node.shouldFail) assert localBranchID == null;
          localBranchID = bid;
        }
      }
    }
    return localBranchID;
  }

  public NodeLog get(NodeId bid){
    if(this.containsKey(bid)){
      return super.get(bid);
    }else{
      if(((BranchID)bid).isForked()){
        assert false;
        return null;
      }else{
        super.put(bid, new NodeLog());
        return super.get(bid);
      }
    }
  }

  public void put(BranchID bid, NodeLog nl){
    assert false;
  }

  /**
   * computes a branch map that converts local logs to map to the version vector present in the checkpoint 
   * 
   * @param vv
   * @param lastWrites
   * @return
   */
  public HashMap<BranchID, BranchID> getBranchMap(AcceptVV vv, HashMap<BranchID, Hash> lastHashes){
    // search among all the logs to see if the vv is present in local state and if yes, whether the hash matches the hash of local state

    HashMap<BranchID, BranchID> branchMap = new HashMap<BranchID, BranchID>();
    VVIterator vvi = vv.getIterator();

    while(vvi.hasMoreElements()){
      BranchID b = (BranchID)vvi.getNext();
      assert lastHashes.containsKey(b);
      branchMap.put(b, this.getBranchID(b.getBaseId(), vv.getStampByIteratorToken(b), lastHashes.get(b)));
    }

    return branchMap;
  }

  public HashMap<NodeId, NodeId> getAcceptableBranchMap(VV omitVV) throws Exception{
    HashMap<NodeId, NodeId> branchMap = new HashMap<NodeId, NodeId>();
    HashMap<Long, TreeSet<BranchID>> validBranches = new HashMap<Long, TreeSet<BranchID>>();
    VVIterator vvi = omitVV.getIterator();
    while(vvi.hasMoreElements()){
      NodeId n = vvi.getNext();
      long l = n.getIDint();
      if(!validBranches.containsKey(l)){
        validBranches.put(l, new TreeSet<BranchID>());
      }
      if(branchKnowledge.containsKey(l)){
        validBranches.get(l).add((BranchID)n);
      }
    }

    for(Long l: validBranches.keySet()){
      branchMap.putAll(branchKnowledge.get(l).getBranchMap(validBranches.get(l)));
    }

    return branchMap;
  }

  public TreeSet<NodeId> getMinimalRoots(Collection<NodeId> nodes) {
    TreeSet<NodeId> minId = new TreeSet<NodeId>(nodes);
    try{
      LinkedList<NodeId> removedIds = new LinkedList<NodeId>();
      for(NodeId n: nodes){
        SimpleTree<BranchID> tree = this.getTree(n.getIDint());
        Collection<BranchID> children = tree.getAllChildren((BranchID)n);
        assert !children.contains(n);
        removedIds.addAll(children);
      }
      minId.removeAll(removedIds);
    }catch(Exception e){
      // TODO Auto-generated catch block
      e.printStackTrace();
      assert false;
    }

    return minId;
  }
  /**
   * eliminates the multiple ordered branches for a given node
   * @param vv
   * @return
   * @throws Exception
   */
  public AcceptVV getMinimalVV(VV vv) throws Exception{
    HashMap<Long, HashMap<NodeId, TreeSet<BranchID>>> nodeIdMap = new HashMap<Long, HashMap<NodeId, TreeSet<BranchID>>>();
    CounterVV cvv = new CounterVV(vv);
    VVIterator vvi = vv.getIterator();
    while(vvi.hasMoreElements()){
      NodeId n = vvi.getNext();
      if(!nodeIdMap.containsKey(n.getIDint())){
        nodeIdMap.put(n.getIDint(), new HashMap<NodeId, TreeSet<BranchID>>());
      }
      TreeSet<BranchID> ancestors = this.branchKnowledge.get(n.getIDint()).getAllAncestors((BranchID)n);
      ancestors.add((BranchID)n);

      // remove all the entries that are contained by this ancestors
      // if any of the ancestor contains this branchID then remove this
      HashMap<NodeId, TreeSet<BranchID>> nMap = nodeIdMap.get(n.getIDint());
      boolean ignore = false;
      LinkedList<NodeId> removed = new LinkedList<NodeId>();
      for(NodeId n1: nMap.keySet()){
        if(nMap.get(n1).contains(n)){
          ignore = true;
        }else if(ancestors.contains(n1)){
          removed.add(n1);
        }

      }

      if(!ignore){
        for(NodeId n1:removed){
          nMap.remove(n1);
          cvv.dropNode(n1);
        }
        nMap.put(n, ancestors);
      }else{
        cvv.dropNode(n);
      }
    }

    return new AcceptVV(cvv);
  }

  public TreeSet<BranchID> findIllegalBranches(Collection<NodeId> mappedBranches, Collection<NodeId> unmappedBranches) throws Exception{
    TreeSet<BranchID> illegalBranches = new TreeSet<BranchID>();
    HashMap<Long, LinkedList<TreeSet<BranchID>>> illegalBranchMap = new HashMap<Long, LinkedList<TreeSet<BranchID>>>();
    for(NodeId n: mappedBranches){
      long l = n.getIDint();
      if(!illegalBranchMap.containsKey(l)){
        illegalBranchMap.put(l, new LinkedList<TreeSet<BranchID>>());
      }
      if(branchKnowledge.containsKey(l)){
        illegalBranchMap.get(l).add(branchKnowledge.get(l).getConcurrentNodes((BranchID)n));
      }
    }

    for(NodeId n: unmappedBranches){
      long l = n.getIDint();
      if(!illegalBranchMap.containsKey(l)){
        illegalBranchMap.put(l, new LinkedList<TreeSet<BranchID>>());
      }
      if(branchKnowledge.containsKey(l)){
        illegalBranchMap.get(l).add(branchKnowledge.get(l).getAllNodes());
      }
    }

    for(Long l: illegalBranchMap.keySet()){
      LinkedList<TreeSet<BranchID>> branches = illegalBranchMap.get(l);
      if(branches.size() > 0){
        TreeSet<BranchID> intersection = branches.removeFirst();
        while(branches.size() > 0){
          intersection.retainAll(branches.removeFirst());
        }

        illegalBranches.addAll(intersection);
      }
    }

    return illegalBranches;
  }

  /**
   * computes a branch map that converts local logs to map to the version vector present in the checkpoint 
   * 
   * @param vv
   * @param lastWrites
   * @param strict this parameter determines if the conversion should be best effort or strict?
   * @return
   */
  public AcceptVV remap(HashedVV vv, boolean strict) throws Exception{
    // search among all the logs to see if the vv is present in local state and if yes, whether the hash matches the hash of local state

    VVIterator vvi = vv.getIterator();
    CounterVV cvv = new CounterVV();

    while(vvi.hasMoreElements()){
      BranchID b = (BranchID)vvi.getNext();
      BranchID myB = this.getBranchID(b.getBaseId(), vv.getStampByIteratorToken(b), vv.getHashByIteratorToken(b));
      if(myB != null){
        cvv.addMaxAS(new AcceptStamp(vv.getStampByIteratorToken(b), myB));
      }else if(strict){
        throw new Exception("Couldnt find a match for the last write " + b + " " + vv.getHashByIteratorToken(b));
      }
    }

    return new AcceptVV(cvv);
  }

  /**
   * Given an acceptVV "vv" that corresponds to the local state, return an acceptVV that is the closure of this VV
   * The closure is computed mainly to gather the non-leaf branches and doesn't result in cross-node closure computation
   * @param vv
   * @return
   */
  public AcceptVV getClosure(VV vv){
    CounterVV cvv = new CounterVV(vv);
    VVIterator vvi = vv.getIterator();
    try{
      while(vvi.hasMoreElements()){
        BranchID n = (BranchID)vvi.getNext();
        assert this.branchKnowledge.containsKey(n.getIDint());
        Collection<BranchID> parents = this.branchKnowledge.get(n.getIDint()).getAllAncestors(n);
        for(BranchID b: parents){
          cvv.addMaxAS(new AcceptStamp(this.cvv.getStampByServer(b), b));
        }
      }
    }catch(NoSuchEntryException e){
      // TODO Auto-generated catch block
      e.printStackTrace();
      assert false;
    }catch(Exception e){
      // TODO Auto-generated catch block
      e.printStackTrace();
      assert false;
    }
    return new AcceptVV(cvv);
  }

  /**
   * apply a POM and return the remap object in pr
   * @param pom
   * @param pr
   * @return
   */
  public boolean applyPOM(ProofOfMisbehavior pom, POMRemap pr){

    if(pomList.contains(pom)){
	System.out.println("applyPOM: returning because pomList already contains POM");
      return false;
    }
    if(pom instanceof ConcurrentPOM){
      pomList.add(pom);
      return true;
    }
  
    //can only be applied if we are not using simulator id..otherwise, writes are already applied
    if(!SangminConfig.useSimulatorId){
      BranchID bid = getBranchID(pom.getParent());
      if(bid == null){
        System.out.println("Forking point corresponding to "  + pom + " not found in local history");
        return false;
      }else{
        pomList.add(pom);
      }


      //returns the branch in which we believe an inval exists
      //this.forkedMap.addMaxAS(new AcceptStamp(pom.getParent().getAcceptStamp().getLocalClock(), bid));
      if(this.containsKey(bid)){
        long endTS = pom.getParent().getAcceptStamp().getLocalClock();

        int index = this.get(bid).binarySearch(endTS).getValue();
        if(index == (this.get(bid).size() -1)){
          // in this case, we have not yet seen any forked write
          try{
            this.branchKnowledge.get(bid.getIDint()).markNonLeaf(bid);
          }catch(Exception e){
            e.printStackTrace();
            assert false;
          }

          return true;

        }else{

          //          long startTSOfNewBranch = this.get(index)

          if(Node.shouldFail) assert this.containsKey(bid);
          NodeLog nl = this.get(bid);

          cvv.dropNode(bid);

          //System.out.println("Before forking :\n " + nl);
          if(Node.shouldFail) assert nl.size() > 1;

          Iterator<PreciseInv> iter = nl.iterator();

          if(Node.shouldFail) assert iter.hasNext();
          PreciseInv pi = iter.next();
          LogStatus status;
          long preciseEndTS = -1;
          while(pi.getAcceptStamp().getLocalClock() <= endTS){
            cvv.addMaxAS(pi.getAcceptStamp());
            preciseEndTS = pi.getAcceptStamp().getLocalClock();
            if(iter.hasNext()){
              pi = iter.next();
            }else{
              pi = null;
              break;
            }
          }
          nl.reset(preciseEndTS);

          try{
            this.getTree(bid.getIDint()).markNonLeaf(bid);
          }catch(Exception e){
            e.printStackTrace();
            assert false;
          }

          //      Segment seg = bk.getSegment(bid);
          //      seg.makeNonLeaf();

          if(Node.shouldFail) assert pi == null || pi.getAcceptStamp().getLocalClock() > endTS;

          NodeLog newNL = new NodeLog();

          if(pi!=null){
            /*
             * Start of new branch
             */
            if(Node.shouldFail) assert pi instanceof SimPreciseInv;
            Hash hash =  ((SimPreciseInv)pi).generateMyHash(); 
            if(Node.shouldFail) assert hash !=null;

            long startTSOfNewBranch = pi.getAcceptStamp().getLocalClock();
            BranchID newBranchID = new BranchID(bid.getIDint(), startTSOfNewBranch, hash);        


            if(Node.shouldFail) assert !this.containsKey(newBranchID);

            super.put(newBranchID, newNL);

            pr.init(bid, newBranchID, startTSOfNewBranch);

            SimPreciseInv orgSPI = (SimPreciseInv)pi;
            //System.out.println("Org SPI : " + orgSPI);
            SimPreciseInv spi = orgSPI.cloneWithNewNodeId(newBranchID);   
            spi = spi.updateDVV(pr);
            //System.out.println("New SPI : " + spi);
            //securityFilter.verifyAndApply(spi, newBranchID);
            status = this.apply(spi);

            if(status.getApplyStatus() == LogStatus.ApplyFailed){// note that in presence of checkpoints, this check may not always pass
              if(Node.shouldFail) assert false;
            }

            while(iter.hasNext()){
              orgSPI = (SecureSimPreciseInv)iter.next();
              //System.out.println("Org SPI : " + orgSPI);
              spi = orgSPI.cloneWithNewNodeId(newBranchID);   
              spi = spi.updateDVV(pr);
              //System.out.println("New SPI : " + spi);
              //securityFilter.verifyAndApply(spi, newBranchID);
              status = this.apply(spi);
              if(status.getApplyStatus() == LogStatus.ApplyFailed){// note that in presence of checkpoints, this check may not always pass
                if(Node.shouldFail) assert false: spi + " status " + status;
              }

            }

            HashMap<NodeId, NodeLog> copy = (HashMap<NodeId, NodeLog>)this.clone();
            this.clear();
            for(NodeId n: copy.keySet()){
              NodeLog nll = copy.get(n);
              NodeLog newNLL = new NodeLog();
              this.put(n, newNLL);

              for(PreciseInv pii: nll){
                if(Node.shouldFail) assert pii instanceof SimPreciseInv;
                SimPreciseInv sspii = (SimPreciseInv)pii;
                sspii = sspii.updateDVV(pr);
                newNLL.add(sspii);
              }
            }
            //          }
          }
        }
        try{
          VV vv = this.remap(omitVV, true);
          omitVV = new HashedVV(vv, this.getHashMap(vv));
        }catch(Exception e){
          // TODO Auto-generated catch block
          e.printStackTrace();
          assert false;
        }
        //System.out.println("After forking :\n " + newNL + "\n" + get(bid));
      }else{
        System.out.println("Received a POM for future");
      }
    }else{
      pomList.add(pom);
    }
    return true;

  }

  /**
   * find last writes larger than or equal to minVV
   * @param minVV
   * @return
   */
  public TreeSet<SimPreciseInv> findLastWritesIncludingLastWrite(AcceptVV minVV){
    TreeSet<SimPreciseInv> lastWrites = new TreeSet<SimPreciseInv>();

    VVIterator vvi = minVV.getIterator();
    while(vvi.hasMoreElements()){
      NodeId n = vvi.getNext();
      if(this.containsKey(n)){
        long ts = minVV.getStampByIteratorToken(n);
        try{
          if(this.cvv.getStampByServer(n) >= ts){
            NodeLog nl = this.get(n);
            int index = nl.binarySearch(ts).getValue();

            if(index >= 0 && nl.get(index).getAcceptStamp().getLocalClock() == ts){
              lastWrites.add((SimPreciseInv)nl.get(index));
            }else{
//              if(index == (nl.size() - 1)){
//                System.out.println(nl.get(index));
//              }
//              assert index < nl.size(): nl.size() + "  " + index  + nl + " " + ts;
//              lastWrites.add((SimPreciseInv)nl.get(index + 1));
              if(index >= 0){
                lastWrites.add((SimPreciseInv)nl.get(index));
              }
            
            }
          }
        }catch(NoSuchEntryException e){
          // TODO Auto-generated catch block
          assert false;
        }
      }
    }
    return lastWrites;
  }


  public boolean containsPOM(ProofOfMisbehavior pom){
    // TODO Auto-generated method stub
    return pomList.contains(pom);
  }

  public HashMap<NodeId, Hash> getHashMap(VV vv) throws Exception{
    HashMap<NodeId, Hash> hashMap = new HashMap<NodeId, Hash>();
    HashMap<NodeId, SimPreciseInv> hashes = this.getInvals(vv);
    if((hashes.size() != vv.getSize()) || !hashes.keySet().containsAll(vv.getNodes())){
      throw new Exception("\nInvalid omitVV" + vv + "\n" + hashes.keySet());
    }
    for(NodeId n: hashes.keySet()){
      hashMap.put(n, ((SimPreciseInv)hashes.get(n)).generateMyHash());
    }
    return hashMap;
  }

  public static HashMap<NodeId, Hash> getHashMap(HashMap<NodeId, SimPreciseInv> hashes) {
    HashMap<NodeId, Hash> hashMap = new HashMap<NodeId, Hash>();
    for(NodeId n: hashes.keySet()){
      hashMap.put(n, ((SimPreciseInv)hashes.get(n)).generateMyHash());
    }
    return hashMap;
  }

  /**
   * returns invals correponding to the VV...best effort
   * @param vv
   * @return
   */
  public HashMap<NodeId, SimPreciseInv> getInvals(VV vv){
    HashMap<NodeId, SimPreciseInv> hashMap = new HashMap<NodeId, SimPreciseInv>();
    VVIterator vvi = vv.getIterator();
    while(vvi.hasMoreElements()){
      NodeId n = vvi.getNext();
      long as = vv.getStampByIteratorToken(n);
      if(this.containsKey(n)){
        int index = get(n).binarySearch(as).getValue();
        if(index < get(n).size() && index >= 0){
          SimPreciseInv spi = (SimPreciseInv)get(n).get(index);
          if(spi.getAcceptStamp().getLocalClock() == as){
            hashMap.put(n, spi);
          }
        }
      }
    }
    return hashMap;
  }

  //  /**
  //   * returns a best effort closure based on the logs and the depedencyVV
  //   * @param endVV
  //   * @return
  //   */
  //  public AcceptVV getDeepClosure(VV endVV){
  //    CounterVV closure = new CounterVV(endVV);
  //    CounterVV newVV = new CounterVV(endVV);
  //    while(!newVV.equalsIgnoreNegatives(new CounterVV())){
  //      CounterVV newClosure = new CounterVV(closure);
  //      VVIterator vvi = newVV.getIterator();
  //      while(vvi.hasMoreElements()){
  //        NodeId n = vvi.getNext();
  //        NodeLog nl = this.get(n);
  //        for(PreciseInv pi: nl){
  //          if(newVV.includes(pi.getAcceptStamp())){
  //            SimPreciseInv spi = (SimPreciseInv)pi;
  //            newClosure.addMaxVV(spi.getDVV());
  //          }
  //        }
  //      }
  //      if(newClosure.equals(closure)){
  //        newVV = new CounterVV();
  //      }else{
  //        AcceptVV avv = new AcceptVV(newClosure);
  //        for(AcceptStamp as: avv.getAllStamps()){
  //          if(!closure.includes(as)){
  //            newVV.addMaxAS(as);
  //          }
  //        }
  //      }
  //      closure.addMaxVV(newClosure);
  //    }
  //
  //    return new AcceptVV(closure);
  //  }

  /**
   * returns a best effort closure based on the logs and the depedencyVV
   * @param endVV
   * @return
   */
  public AcceptVV getDeepClosure(VV endVV){
    VVIterator vvi = endVV.getIterator();
    long maxTS = -1;
    while(vvi.hasMoreElements()){
      NodeId n = vvi.getNext();
      long ts = endVV.getStampByIteratorToken(n);
      if(ts > maxTS){
        maxTS = ts;
      }
    }
    CounterVV maxEndVV = new CounterVV(endVV);
    //    for(NodeId n: this.keySet()){
    //      NodeLog nl = this.get(n);
    //      Entry<Boolean, Integer> search = nl.binarySearch(maxTS);
    //      if(search.getValue() < nl.size() && search.getValue() >=0){
    //        maxEndVV.addMaxAS(nl.get(search.getValue()).getAcceptStamp());
    //      }
    //    }
    return new AcceptVV(maxEndVV);
  }
  
  /**
   * returns a best effort closure based on the logs and the depedencyVV
   * @param endVV
   * @return
   */
  public AcceptVV getDeepClosure(AcceptStamp as){

    CounterVV maxEndVV = new CounterVV();
    maxEndVV.addMaxAS(as);
    if(this.containsKey(as.getNodeId())){
      for(PreciseInv pi: this.get(as.getNodeId())){
        if(pi.getAcceptStamp().getLocalClock() <= as.getLocalClock())
          maxEndVV.addMaxVV(((SimPreciseInv)pi).getDVV());
      }
    }
    return new AcceptVV(maxEndVV);
  }

  /**
   * Get the flags for an inval belonging to "epoch" epoch and having immediate dependencies on "invals" invals  
   * @param invals
   * @param epoch
   * @return
   */
  synchronized protected TreeSet getFlags(Collection<SimPreciseInv> invals, int epoch){
    TreeSet ts = new TreeSet();
    for(SimPreciseInv spi: invals){
      SimPreciseInv sspi = (SimPreciseInv)spi;
      if(sspi.getEpoch() == epoch){
        ts.addAll(sspi.getFlags());
      }
    }
    return ts;
  }

  /* (non-Javadoc)
   * @see java.lang.Object#hashCode()
   */
  @Override
  public int hashCode(){
    final int prime = 31;
    int result = super.hashCode();
    result = prime * result
    + ((branchKnowledge == null) ? 0 : branchKnowledge.hashCode());
    result = prime * result + ((cvv == null) ? 0 : cvv.hashCode());
    result = prime * result
    + ((evictedNodes == null) ? 0 : evictedNodes.hashCode());
    result = prime * result + ((omitVV == null) ? 0 : omitVV.hashCode());
    result = prime * result + ((pomList == null) ? 0 : pomList.hashCode());
    return result;
  }

  /* (non-Javadoc)
   * @see java.lang.Object#equals(java.lang.Object)
   */
  @Override
  public boolean equals(Object obj){
    if(this == obj){
      return true;
    }
    if(!super.equals(obj)){
      return false;
    }
    if(!(obj instanceof Log)){
      return false;
    }
    Log other = (Log) obj;
    if(branchKnowledge == null){
      if(other.branchKnowledge != null){
        return false;
      }
    }else if(!branchKnowledge.equals(other.branchKnowledge)){
      return false;
    }
    if(cvv == null){
      if(other.cvv != null){
        return false;
      }
    }else if(!cvv.equals(other.cvv)){
      return false;
    }
    if(evictedNodes == null){
      if(other.evictedNodes != null){
        return false;
      }
    }else if(!evictedNodes.equals(other.evictedNodes)){
      return false;
    }
    if(omitVV == null){
      if(other.omitVV != null){
        return false;
      }
    }else if(!omitVV.equals(other.omitVV)){
      return false;
    }
    if(pomList == null){
      if(other.pomList != null){
        return false;
      }
    }else if(!pomList.equals(other.pomList)){
      return false;
    }
    return true;
  }

  /**
   * reset the log state to writes included in oldVV
   * @param oldVV
   * @param hashes
   * @throws Exception
   */
  public void resetState(HashedVV oldVV) throws Exception{
    AcceptVV mappedOldVVPartial = this.remap(oldVV, true);
    assert this.getCurrentVV().includes(mappedOldVVPartial): this.getCurrentVV() + " \n " + mappedOldVVPartial;
    AcceptVV mappedOldVV = this.getClosure(mappedOldVVPartial);

    //reset CVV
    TreeSet<NodeId> removedIds = new TreeSet<NodeId>();
    VVIterator vvi = this.cvv.getIterator();
    while(vvi.hasMoreElements()){
      NodeId n = vvi.getNext();
      if(!mappedOldVV.containsNodeId(n)){
        removedIds.add(n);
      }
    }

    for(AcceptStamp as: mappedOldVV.getAllStamps()){
      cvv.dropNode(as.getNodeId());
      cvv.setStampByNodeId(as.getNodeId(), as.getLocalClock());
    }
    
    for(NodeId n: removedIds){
      cvv.dropNode(n);
    }

    //reset node log
    for(NodeId n: this.keySet()){
      NodeLog nl = this.get(n);
      if(mappedOldVV.containsNodeId(n)){
        nl.reset(mappedOldVV.getStampByServer(n));
      }else if(removedIds.contains(n)){
        nl.clear();
      }
    }



    // reset branch knowledge
    vvi = mappedOldVVPartial.getIterator();
    while(vvi.hasMoreElements()){
      NodeId n = vvi.getNext();
      SimpleTree<BranchID> tree = this.branchKnowledge.get(n.getIDint());
      tree.markLeaf((BranchID)n);
    }
  }

  public HashedVV getOmitVV(){
    return this.omitVV;
  }

}
