package code.simulator.log;

import code.NodeId;
import java.util.*;
import code.simulator.SimPreciseInv;

public class LogStatus{

  // apply status
  public static final byte ApplySuccessful = 0;
  public static final byte PseudoApplySuccessful = 1;
  public static final byte ApplyFailed = -1;
  
  // verification status
  public static final byte VerificationPassed = 13;
  public static final byte DVVInclusionFail = 2;
  public static final byte InclusionFail = 3;
  public static final byte FIFOFail = 5;
  public static final byte CompatibilityFail = 4;
  public static final byte SignatureVerificationFail = 11;
  public static final byte FIFOPassedInclusionUnverifiable = 12; // when DVV falls inside omitVV for at least one component
  public static final byte WriteToForkedParentBranch = 14; // when DVV falls inside omitVV for at least one component
  public static final byte MissingPOMForForkedChildBranch = 15; // when DVV falls inside omitVV for at least one component
  public static final byte WriteFromForkedNodeWithInvalidBranchHistory = 16; // when DVV falls inside omitVV for at least one component
  public static final byte ImproperLamportClock = 17; // when DVV falls inside omitVV for at least one component
  public static final byte ImproperEpochNumber = 18; // when DVV falls inside omitVV for at least one component
  public static final byte ImproperDVV = 19; //DVV has more than one local components
  public static final byte ImproperBranchID = 20; // the BranchID of a write that is inconsistent due to checkpoint is incorrect
  public static final byte OptimizedSyncFailed = 21; // the BranchID of a write that is inconsistent due to checkpoint is incorrect
  public static final byte TimeStampExhaustionCheckFailed = 23; // the time stamp is higher than the expected value
  
  private final byte applyStatus;
  private final byte verificationStatus;
  
  
  protected LogStatus(byte applyStatus, byte verificationStatus){
    this.applyStatus = applyStatus;
    this.verificationStatus = verificationStatus;
  }
  
  public static LogStatus makeLogStatusFailedVerificationStatus(byte verificationStatus){
    return new LogStatus(LogStatus.ApplyFailed,verificationStatus);
  }
  
  public static LogStatus makeLogStatusPassedVerificationStatus(byte applyStatus){
    return new LogStatus(applyStatus, LogStatus.VerificationPassed);
  }
  
  public static LogStatus makeDefaultLogStatus(){
    return new LogStatus(LogStatus.ApplySuccessful, LogStatus.VerificationPassed);
  }
  
  public static LogStatus makeLogStatus(byte applyStatus, byte verificationStatus){
    return new LogStatus(applyStatus, verificationStatus);
  }
  
  public byte getApplyStatus(){
    return applyStatus;
  }
  
  public byte getVerificationStatus(){
    return verificationStatus;
  }

  public String toString(){
    String str = "LogStatus: [ApplyStatus=";
    switch(applyStatus){
      case ApplySuccessful: str += "ApplySuccessful]"; break;
      case PseudoApplySuccessful: str += "PseudoApplySuccessful]";break;
      case ApplyFailed: str += "ApplyFailed]"; break; 
    }
    str += "], [VerificationStatus=";
    switch(this.verificationStatus){
      case VerificationPassed: str += "VerificationPassed]"; break;
      case DVVInclusionFail: str += "DVVInclusionFail]"; break;
      case InclusionFail: str += "InclusionFail]"; break;
      case FIFOFail: str += "FIFOFail]"; break;
      case CompatibilityFail: str += "CompatibilityFail]"; break;
      case SignatureVerificationFail: str += "SignatureVerificationFail]"; break;
      case FIFOPassedInclusionUnverifiable: str += "FIFOPassedInclusionUnverifiable]"; break;
      case WriteToForkedParentBranch: str += "WriteToForkedParentBranch]"; break;
      case MissingPOMForForkedChildBranch: str += "MissingPOMForForkedChildBranch]"; break;
      case WriteFromForkedNodeWithInvalidBranchHistory: str += "WriteFromForkedNodeWithInvalidBranchHistory]"; break;
      case ImproperLamportClock: str += "ImproperLamportClock]"; break;
      case ImproperDVV: str += "ImproperDVV]"; break;
      case ImproperBranchID: str += "ImproperBranchID]"; break;
      case OptimizedSyncFailed: str += "OptimizedSyncFailed]"; break;
      case TimeStampExhaustionCheckFailed: str += "TimeStampExhaustionCheckFailed]"; break;
      
    }
    str += "] ";
    return str;
  }

  @Override
  public int hashCode(){
    final int prime = 31;
    int result = 1;
    result = prime * result + applyStatus;
    result = prime * result + verificationStatus;
    return result;
  }

  @Override
  public boolean equals(Object obj){
    if(this == obj){
      return true;
    }
    if(obj == null){
      return false;
    }
    if(!(obj instanceof LogStatus)){
      return false;
    }
    LogStatus other = (LogStatus) obj;
    if(applyStatus != other.applyStatus){
      return false;
    }
    if(verificationStatus != other.verificationStatus){
      return false;
    }
    return true;
  }
  
}
