package code.simulator.log.unit;

import java.util.*;

import junit.framework.Test;
import junit.framework.TestCase;
import junit.framework.TestSuite;
import junit.textui.TestRunner;
import code.*;
import code.branchDetecting.BranchID;
import code.security.SangminConfig;
import code.simulator.checkpoint.Checkpoint;
import code.simulator.checkpoint.unit.SimulatorCheckpointUnit;
import code.simulator.ConcurrentPOM;
import code.simulator.Flags;
import code.simulator.Hash;
import code.simulator.HashedVV;
import code.simulator.IrisDataObject;
import code.simulator.IrisNode;
import code.simulator.IrisObject;
import code.simulator.Node;
import code.simulator.NodeFactory;
import code.simulator.POMRemap;
import code.simulator.ProofOfMisbehavior;
import code.simulator.SecureSimPreciseInv;
import code.simulator.SimPreciseInv;
import code.simulator.SyncPacket;
import code.simulator.SyncRequest;
import code.simulator.SyncStatus;
import code.simulator.log.Log;
import code.simulator.log.LogStatus;
import code.simulator.log.NodeLog;

public class LogUnit extends TestCase{
  public static final String TEST_ALL_TEST_TYPE = "UNIT";

  /**
   * Basic constructor - called by the test runners.
   * TBD: Update constructor name to match class
   */
  public LogUnit (final String s) {
    super (s);
  }

  /*
   * Fixtures are run before and after each test case
   * to set up environment in which tests must run.
   */
  protected void setUp() throws Exception{
    super.setUp();
    code.simulator.checkpoint.unit.SimulatorCheckpointUnit.makePractiConfig(0, 100);

  }


  public static int UNIT_NODE_ID = 10;
  public static int MIDDLE_NODE_ID = 15;

  public static int HELPER_NODE_ID = 20;
  public static boolean vverbose = false;
  public static boolean verbose = true;


  public static String COARSE_PATH = "/coarse";


  //--------------------------------------------------------------------------- 
  // Test coarse-grained subscriptions. 
  //
  // To initialize state, Unit create 1000 objects
  // (/data-coarse/[0..9]/[0..9]/[0..9]. Then helper subscribe
  // to /* from unit and wait for /data-coarse/9/9/9 to arrive.
  //
  //
  // start helper and wait for writing to complete
  //---------------------------------------------------------------------------
  public void testSimple(){
    try{
      System.out.println("/testSimple");
//      if(!Node.enableSecurity){
//        return;
//      }
      assert SangminConfig.useSimulator && Node.enableSecurity;
      SangminConfig.usePersistentStore = false;
      
      BranchID myNodeId = NodeFactory.createNodeId(UNIT_NODE_ID);
      NodeId b0 = myNodeId;
      NodeId b2 = null, b2I = null, b4 = null, b4I = null;
      BranchID nodeId1 = NodeFactory.createNodeId(15);
      BranchID nodeId2 = NodeFactory.createNodeId(20);
      BranchID nodeId3 = NodeFactory.createNodeId(25);

      Node myNode = NodeFactory.createNodeWithoutFilter(myNodeId);
      Node node1 = NodeFactory.createNodeWithoutFilter(nodeId1);
      Node node2 = NodeFactory.createNodeWithoutFilter(nodeId2);
      Node node3 = NodeFactory.createNodeWithoutFilter(nodeId3);

      ObjId oid0 = new ObjId("/test/0");
      myNode.write(oid0, new IrisDataObject(new Integer(0)));

      ObjId oid1 = new ObjId("/test/1");
      myNode.write(oid1, new IrisDataObject(new Integer(1)));

      LinkedList<Node> forks = myNode.fork(2);

      forks.get(0).write(new ObjId("/test/00"), new IrisDataObject(new Integer(2)));
      forks.get(0).write(new ObjId("/test/01"), new IrisDataObject(new Integer(3)));
      forks.get(1).write(new ObjId("/test/10"), new IrisDataObject(new Integer(4)));
      forks.get(1).write(new ObjId("/test/11"), new IrisDataObject(new Integer(5)));

      LinkedList<Node> newForks = forks.get(0).fork(2);
      newForks.get(0).write(new ObjId("/test/0000"), new IrisDataObject(new Integer(6)));
      newForks.get(0).write(new ObjId("/test/0001"), new IrisDataObject(new Integer(7)));
      newForks.get(1).write(new ObjId("/test/0010"), new IrisDataObject(new Integer(8)));
      newForks.get(1).write(new ObjId("/test/0011"), new IrisDataObject(new Integer(9)));

      node1.sync(newForks.get(0));
      node2.sync(forks.get(1));
      node3.sync(newForks.get(1));

      if(!SangminConfig.useSimulatorId){
        ((IrisNode)node1).exponentialBackoffSync(node2);
      }
      node1.sync(node2);

      AcceptVV cvv1 = node1.getCurrentVV();
      VVIterator vvi1 = cvv1.getIterator();
      while(vvi1.hasMoreElements()){
        NodeId n = vvi1.getNext();
        if(cvv1.getStampByIteratorToken(n) == 3){
          b2I = n;
        }else if(cvv1.getStampByIteratorToken(n) == 5){
          b2 = n;
        }else{
          assert b0.equals(n);
        }
      }

      if(!SangminConfig.useSimulatorId){
        ((IrisNode)node1).exponentialBackoffSync(node3);
      }
      node1.sync(node3);

      cvv1 = node1.getCurrentVV();
      vvi1 = cvv1.getIterator();
      while(vvi1.hasMoreElements()){
        NodeId n = vvi1.getNext();
        if(cvv1.getStampByIteratorToken(n) == 5){
          if(b4I == null){
            b4I = n;
          }else{
            b4 = n;
          }
        }
      }

      Log log1 = node1.getLogTest();


      // test find illegal branches
      //------------------------------

      // when CVV = all then no illegal branches
      assert log1.findIllegalBranches(myNode.getCurrentVV().getNodes(), new LinkedList<NodeId>()).size() == 0: log1.findIllegalBranches(myNode.getCurrentVV().getNodes(), new LinkedList<NodeId>());

      // when CVV includes one branch then other branches should be illegal
      CounterVV cvv = new CounterVV(node2.getCurrentVV());
      cvv.dropNode(myNodeId);
      assert log1.findIllegalBranches(cvv.getNodes(), new LinkedList<NodeId>()).size() == 3: log1.findIllegalBranches(cvv.getNodes(), new LinkedList<NodeId>());

      //create a new random branchID with same node id as myNodeId
      BranchID bid = new BranchID(myNodeId.getIDint(), 20, Hash.NullHash);
      AcceptStamp as = new AcceptStamp(10, bid);
      cvv = new CounterVV();
      cvv.addMaxAS(as);
      assert log1.findIllegalBranches(new LinkedList<NodeId>(), cvv.getNodes()).size() == 5: log1.findIllegalBranches(new LinkedList<NodeId>(), cvv.getNodes());

      // test getClosure
      //-----------------------
      cvv = new CounterVV(node2.getCurrentVV());
      cvv.dropNode(myNodeId);
      assert log1.getClosure(cvv).equals(node2.getCurrentVV()): log1.getClosure(cvv);

      // test getAcceptableBranchMap
      HashMap<NodeId, NodeId> bm  = log1.getAcceptableBranchMap(cvv);
      assert bm.size() == 1: bm;
      NodeId b; // b is the branchID of the branch forked starting at 2
      VVIterator vvi = cvv.getIterator();
      assert cvv.getSize() == 1;
      assert vvi.hasMoreElements();
      b = vvi.getNext();
      assert bm.containsKey(b):bm;
      assert bm.get(b).equals(myNodeId): bm.get(b);

      // test2 for getAcceptableBranchMap
      bm  = log1.getAcceptableBranchMap(myNode.getCurrentVV());
      assert bm.size() == 0: bm;

      // test3 for getAcceptableBranchMap
      cvv = new CounterVV(node1.getCurrentVV());
      cvv.dropNode(myNodeId);
      cvv.dropNode(b);
      assert cvv.getSize()==3;
      vvi = cvv.getIterator();
      NodeId []b5 = new NodeId[2]; // the two branches starting at 4
      NodeId b3= null; // the other branch starting at 2
      int count = 0;
      while(vvi.hasMoreElements()){
        NodeId n = vvi.getNext();
        if(cvv.getStampByIteratorToken(n) == 5){
          b5[count++]=n;
        }
        if(cvv.getStampByIteratorToken(n) == 3)b3=n;
      }
      cvv.dropNode(b3);
      cvv.dropNode(b5[0]);
      assert cvv.getSize() == 1;
      cvv.addMaxAS(new AcceptStamp(3, b));
      bm  = log1.getAcceptableBranchMap(cvv);

      assert bm.size()==1:bm;
      assert bm.containsKey(b5[1]);
      assert bm.get(b5[1]).equals(b3):bm + " " + b3;

      // test getHashMap
      AcceptVV vv1 = node1.getCurrentVV();
      HashMap<NodeId, Hash> hashMap = log1.getHashMap(vv1);
      vvi = vv1.getIterator();
      while(vvi.hasMoreElements()){
        NodeId n = vvi.getNext();
        assert hashMap.containsKey(n): hashMap + " vv " + vv1 + " n " +n;
        assert hashMap.get(n).equals(log1.getHash(new AcceptStamp(vv1.getStampByIteratorToken(n), n)));
      }

      node2.getLogTest().getHash(new AcceptStamp(vv1.getStampByServer(b), b)).equals(log1.getHash(new AcceptStamp(vv1.getStampByServer(b), b)));

      //test getBranchID
      NodeId baseIDB = ((BranchID)b).getBaseId();
      assert b.equals(node2.getLogTest().getBranchID(baseIDB, vv1.getStampByServer(b), log1.getHash(new AcceptStamp(vv1.getStampByServer(b), b))));

      // test getMinimalVV
      AcceptVV vv = node1.getCurrentVV().dropNegatives();
      AcceptVV minVV = node1.getLogTest().getMinimalVV(vv);
      assert minVV.getAllStamps().size() == 3:vv + " minVV: " + minVV;

      // test getDeepClosure()
      node1.write(new ObjId("/test/0000"), new IrisDataObject(new Integer(6)));
      vv = node1.getCurrentVV();
      long ts = vv.getStampByServer(node1.getBranchID());
      cvv = new CounterVV();
      cvv.addMaxAS(new AcceptStamp(ts, node1.getBranchID()));
//      assert log1.getDeepClosure(cvv).equalsIgnoreNegatives(vv): log1.getDeepClosure(cvv) + " " + vv;

      // test getMinimalRoots
      LinkedList<NodeId> nodes = new LinkedList<NodeId>();
      nodes.add(b0);nodes.add(b2);nodes.add(b2I);nodes.add(b4);nodes.add(b4I);
      TreeSet<NodeId> minNodes = log1.getMinimalRoots(nodes);
      assert minNodes.size() ==1;
      assert minNodes.contains(b0);
      
      nodes.clear();
      nodes.add(b2);nodes.add(b2I);nodes.add(b4I);
      minNodes = log1.getMinimalRoots(nodes);
      assert minNodes.size() ==2: minNodes;
      assert minNodes.contains(b2);
      assert minNodes.contains(b2I);
      
      nodes.clear();
      nodes.add(b4);nodes.add(b2I);nodes.add(b4I);
      minNodes = log1.getMinimalRoots(nodes);
      assert minNodes.size() ==3: minNodes;
      assert minNodes.containsAll(nodes);
      
    }catch(Exception e){
      e.printStackTrace();
      assert false;
    }




  }

  public boolean apply(Log log, SimPreciseInv spi, byte applyCondition, LogStatus expectedStatus){
    LogStatus ls = log.apply(spi, applyCondition);
    if(ls.equals(expectedStatus)){
      return true;
    }else{
      return false;
    }
  }
  
  /**
   * test whether the log issues appropriate error messages
   */
  public void testInconsistencyLogError(){
    try{
//      if(!Node.enableSecurity){
//        return;
//      }
      assert SangminConfig.useSimulator && Node.enableSecurity;
      SangminConfig.usePersistentStore = false;
      SangminConfig.enableInconsistencyNotification = true;

      BranchID myNodeId = NodeFactory.createNodeId(UNIT_NODE_ID);
      NodeId b0 = myNodeId;
      NodeId b2 = null, b2I = null, b4 = null, b4I = null;
      BranchID nodeId1 = NodeFactory.createNodeId(15);
      BranchID nodeId2 = NodeFactory.createNodeId(20);
      BranchID nodeId3 = NodeFactory.createNodeId(25);

      Node myNode = NodeFactory.createNodeWithoutFilter(myNodeId);
      Node node1 = NodeFactory.createNodeWithoutFilter(nodeId1);
      Node node2 = NodeFactory.createNodeWithoutFilter(nodeId2);
      Node node3 = NodeFactory.createNodeWithoutFilter(nodeId3);

      ObjId oid0 = new ObjId("/test/0");
      myNode.write(oid0, new IrisDataObject(new Integer(0)));
      node1.sync(myNode);
      SimPreciseInv n1w1 = node1.write(new ObjId("/test/991"), new IrisDataObject(new Integer(-1)));
      
      ObjId oid1 = new ObjId("/test/1");
      myNode.write(oid1, new IrisDataObject(new Integer(1)));

      LinkedList<Node> forks = myNode.fork(2);
      
      forks.get(0).write(new ObjId("/test/00"), new IrisDataObject(new Integer(2)));
      node1.sync(forks.get(0));
      SimPreciseInv n1w2 = node1.write(new ObjId("/test/991"), new IrisDataObject(new Integer(-1)));
      forks.get(0).write(new ObjId("/test/01"), new IrisDataObject(new Integer(3)));
      forks.get(1).write(new ObjId("/test/10"), new IrisDataObject(new Integer(4)));
      node2.sync(forks.get(1));
      SimPreciseInv n2w1 = node2.write(new ObjId("/test/992"), new IrisDataObject(new Integer(-2)));//depends on b2I:3
      forks.get(1).write(new ObjId("/test/11"), new IrisDataObject(new Integer(5)));

      LinkedList<Node> newForks = forks.get(0).fork(2);
      newForks.get(0).write(new ObjId("/test/0000"), new IrisDataObject(new Integer(6)));
      newForks.get(0).write(new ObjId("/test/0001"), new IrisDataObject(new Integer(7)));
      newForks.get(1).write(new ObjId("/test/0010"), new IrisDataObject(new Integer(8)));
      node3.sync(newForks.get(1));
      SimPreciseInv n3w1 = node3.write(new ObjId("/test/992"), new IrisDataObject(new Integer(-3)));
      newForks.get(1).write(new ObjId("/test/0011"), new IrisDataObject(new Integer(9)));

      node1.sync(newForks.get(0));
      node2.sync(forks.get(1));
      node3.sync(newForks.get(1));

      if(!SangminConfig.useSimulatorId){
        ((IrisNode)node1).exponentialBackoffSync(node2);
      }
      node1.sync(node2);

      AcceptVV cvv1 = node1.getCurrentVV();
      VVIterator vvi1 = cvv1.getIterator();
      while(vvi1.hasMoreElements()){
        NodeId n = vvi1.getNext();
        if(cvv1.getStampByIteratorToken(n) == 3 && n.getIDint() == b0.getIDint()){
          b2I = n;
        }else if(cvv1.getStampByIteratorToken(n) == 5 && n.getIDint() == b0.getIDint()){
          b2 = n;
        }else if(n.getIDint() == b0.getIDint()){
          assert b0.equals(n);
        }
      }

      if(!SangminConfig.useSimulatorId){
        ((IrisNode)node1).exponentialBackoffSync(node3);
      }
      node1.sync(node3);

      cvv1 = node1.getCurrentVV();
      vvi1 = cvv1.getIterator();
      while(vvi1.hasMoreElements()){
        NodeId n = vvi1.getNext();
        if(cvv1.getStampByIteratorToken(n) == 5 && n.getIDint() == b0.getIDint()){
          if(b4I == null){
            b4I = n;
          }else{
            b4 = n;
          }
        }
      }


      //initialize node logs
      NodeLog nl1 = node1.getLogTest().get(node1.getBranchID());
      assert nl1.size() == 2;
      
      NodeLog nl2 = node1.getLogTest().get(node2.getBranchID());
      assert nl2.size() == 1;
      
      NodeLog nl3 = node1.getLogTest().get(node3.getBranchID());
      assert nl3.size() == 1;
      
      NodeLog nl0 = node1.getLogTest().get(b0);
      assert nl0.size() == 2;
      
      NodeLog nl0_ = node1.getLogTest().get(b2);
      assert nl0_.size() == 2;
      
      NodeLog nl0_I = node1.getLogTest().get(b2I);
      assert nl0_I.size() ==2:nl0_I;
      
      NodeLog nl0__ = node1.getLogTest().get(b4);
      assert nl0__.size() == 2;
      
      NodeLog nl0__I = node1.getLogTest().get(b4I);
      assert nl0__I.size() == 2;
      
      Node f0 = myNode;
      Node f2 = forks.get(0);
      Node f2I = forks.get(1);
      Node f4 = newForks.get(0);
      Node f4I = newForks.get(1);
      
      NodeLog nlf0 = f0.getLogTest().get(f0.getBranchID());
      NodeLog nlf2 = f2.getLogTest().get(f2.getBranchID());
      NodeLog nlf2I = f2I.getLogTest().get(f2I.getBranchID());
      NodeLog nlf4 = f4.getLogTest().get(f4.getBranchID());
      NodeLog nlf4I = f4I.getLogTest().get(f4I.getBranchID());
      
      HashSet<ProofOfMisbehavior> poms = node1.getPOMMap();
      ProofOfMisbehavior pom2 = null, pom4 = null;
      assert poms.size() ==2 ;
      for(ProofOfMisbehavior pom: poms){
        if(pom.getParent().getAcceptStamp().getLocalClock() == 1){
          pom2 = pom;
        }else{
          pom4 = pom;
        }
      }
      
      // ---------------- test DVVInclusionFail --------------------
      Node node1Copy = NodeFactory.createNodeWithoutFilter(nodeId1);
      Log log1Copy = node1Copy.getLogTest();
      LogStatus ls = log1Copy.apply(nl1.get(0));
      assert ls.getApplyStatus() == LogStatus.ApplyFailed:ls;
      if(SangminConfig.enableInconsistencyNotification){
        assert ls.getVerificationStatus() == LogStatus.InclusionFail:ls;
      }else{
        assert ls.getVerificationStatus() == LogStatus.DVVInclusionFail:ls;
      }
      ls = log1Copy.apply(nl0.get(0));
      assert ls.equals(LogStatus.makeDefaultLogStatus());
      ls = log1Copy.apply(nl1.get(0));
      assert ls.equals(LogStatus.makeDefaultLogStatus()):ls;
      
      //  ---------------- test InclusionFail -------------------- 
      ls = log1Copy.apply(nl0.get(1));
      assert ls.equals(LogStatus.makeDefaultLogStatus());
      
      // now apply the write from the forked branch 2I 
      ls = log1Copy.apply(nlf2I.get(2));
      assert ls.equals(LogStatus.makeDefaultLogStatus());
      
      // now the inclusion should fail for n1w2
      ls = log1Copy.apply(n1w2);
      assert ls.equals(LogStatus.makeLogStatusFailedVerificationStatus(LogStatus.InclusionFail)):ls;
      
      //  ---------------- test FIFOFail -------------------- 
      // try applying nlf2.get(3) with FIFO only applyCondition
      ls = log1Copy.apply(nlf2.get(3), Log.ApplyIfFIFOPass);
      assert ls.equals(LogStatus.makeLogStatusFailedVerificationStatus(LogStatus.FIFOFail)):ls;

      //  ---------------- test FIFOPassedInclusionUnverifiable -------------------- 
      // applying n1w2 with fifo only check should succeeed
      ls = log1Copy.apply(n1w2, Log.ApplyIfFIFOPass);
      assert ls.equals(LogStatus.makeLogStatus(LogStatus.ApplySuccessful, LogStatus.FIFOPassedInclusionUnverifiable)):ls;
      
      //  ---------------- test CompatibilityFail -------------------- 
      //  try applying nlf2.get(2)  
      ls = log1Copy.apply(nlf2.get(2), Log.ApplyIfFIFOPass);
      assert ls.equals(LogStatus.makeLogStatusFailedVerificationStatus(LogStatus.CompatibilityFail)):ls;
      
      ls = log1Copy.apply(nlf2.get(2));
      assert ls.equals(LogStatus.makeLogStatusFailedVerificationStatus(LogStatus.CompatibilityFail)):ls;
      
      //  ---------------- test MissingPOMForForkedChildBranch -------------------- 
      //try applying nl0_[2]
      ls = log1Copy.apply(nl0_.get(0));
      assert ls.equals(LogStatus.makeLogStatusFailedVerificationStatus(LogStatus.MissingPOMForForkedChildBranch)):ls;
      
      //  ---------------- test WriteToForkedParentBranch -------------------- 
      assert log1Copy.applyPOM(pom2, new POMRemap());
      
      ls = log1Copy.apply(nlf2.get(2));
      assert ls.equals(LogStatus.makeLogStatusFailedVerificationStatus(LogStatus.WriteToForkedParentBranch)):ls;
      
      assert ((SecureSimPreciseInv)nl0_.get(0)).generateMyHash().equals(((SecureSimPreciseInv)nlf2.get(2)).generateMyHash());
      
      ls = log1Copy.apply(nl0_.get(0));
      assert ls.equals(LogStatus.makeDefaultLogStatus()):ls;
      
      //  ---------------- test ImproperLamportClock -------------------- 
      // make a copy of n2w1 with an improper lamport clock
      SecureSimPreciseInv sn2w1 = (SecureSimPreciseInv)nl2.get(0); // the version with branch map applied
      SecureSimPreciseInv newn2wi = new SecureSimPreciseInv(sn2w1.getObjId(), new AcceptStamp(sn2w1.getAcceptStamp().getLocalClock()-1, sn2w1.getNodeId()), sn2w1.getDVV(), sn2w1.getData(), sn2w1.getEpoch(), sn2w1.getObjectHashes());
      
      ls = log1Copy.apply(newn2wi);
      assert ls.equals(LogStatus.makeLogStatusFailedVerificationStatus(LogStatus.ImproperLamportClock)):ls;
      
//      //  ---------------- test ImproperEpochNumber -------------------- 
      //PS:THIS CHECK IS CURRENTLY BEING DONE IN THE NODE
//      
//      newn2wi = new SecureSimPreciseInv(sn2w1.getObjId(), sn2w1.getAcceptStamp(), sn2w1.getDVV(), sn2w1.getData(), sn2w1.getHash(), sn2w1.getEpoch()+1);
//      
//      ls = log1Copy.apply(newn2wi);
//      assert ls.equals(LogStatus.makeLogStatusFailedVerificationStatus(LogStatus.ImproperEpochNumber)):ls;
      
      ls = log1Copy.apply(sn2w1);
      assert ls.equals(LogStatus.makeDefaultLogStatus()):ls;
      
      
      //  ---------------- test ImproperDVV -------------------- 
      SecureSimPreciseInv sspi = (SecureSimPreciseInv)nl0_.get(1);
      AcceptVV vv = log1Copy.getCurrentVV();
      HashMap<NodeId, Hash> hashMap = log1Copy.getHashMap(vv);
      HashedVV dvv = new HashedVV(vv, hashMap);
      Hash h = Log.getHash(dvv);
      VVIterator vvi = dvv.getIterator();
      long lc = -1;
      while(vvi.hasMoreElements()){
        NodeId n = vvi.getNext();
        long ts = dvv.getStampByIteratorToken(n);
        if(ts > lc){
          lc = ts;
        }
      }
      lc++;
      
      AcceptStamp as = new AcceptStamp(lc, sspi.getNodeId());
      SecureSimPreciseInv nsspi = new SecureSimPreciseInv(sspi.getObjId(), as, dvv, sspi.getData(), sspi.getEpoch(), sspi.getObjectHashes());
      
      ls = log1Copy.apply(nsspi);
      assert ls.equals(LogStatus.makeLogStatusFailedVerificationStatus(LogStatus.ImproperDVV)):ls;

      //  ---------------- test ImproperBranchID -------------------- 
      Node node1Copy2 = NodeFactory.createNodeWithoutFilter(nodeId1);
      Log log1Copy2 = node1Copy2.getLogTest();
      node1Copy2.sync(f0);
      
      ls = log1Copy2.apply(nlf2.get(2));
      assert ls.equals(LogStatus.makeDefaultLogStatus()):ls;
 
      ls = log1Copy2.apply(nlf2.get(3));
      assert ls.equals(LogStatus.makeDefaultLogStatus()):ls;
      
      HashedVV hvv = new HashedVV(node1Copy2.getCurrentVV(), log1Copy2.getHashMap(node1Copy2.getCurrentVV()));
      Checkpoint chkPt = node1Copy2.generateNewCheckpoint(hvv, 1);
      ((IrisNode)node1Copy2).applySyncPacket(SimulatorCheckpointUnit.getSyncPacket(chkPt));
      //checkpoint omitVV is b0:1;b2:2
      // now create an inval that violates the BranchID check: b2I:3 is the ts but don't rename the branch
      log1Copy2 = node1Copy2.getLogTest();
      
      sspi = (SecureSimPreciseInv)nl0_I.get(1);
      sspi.getFlags().clear();
      ls = log1Copy2.apply(sspi);
      assert ls.equals(LogStatus.makeLogStatusFailedVerificationStatus(LogStatus.ImproperBranchID)):ls;
      
      // now create an inval that violates the ts check
      ls = log1Copy2.apply(nl0_I.get(0));
      assert ls.equals(LogStatus.makeLogStatusFailedVerificationStatus(LogStatus.ImproperBranchID)):ls;

      //  ---------------- test Inconsistency -------------------- 
      Node node1Copy3 = NodeFactory.createNodeWithoutFilter(nodeId1);
      Log log1Copy3 = node1Copy3.getLogTest();
      node1Copy3.sync(f0);
      
      ls = log1Copy3.apply(nlf2.get(2));
      assert ls.equals(LogStatus.makeDefaultLogStatus()):ls;
 
      hvv = new HashedVV(node1Copy3.getCurrentVV(), log1Copy3.getHashMap(node1Copy3.getCurrentVV()));
      chkPt = node1Copy3.generateNewCheckpoint(hvv, 1);
      ((IrisNode)node1Copy3).applySyncPacket(SimulatorCheckpointUnit.getSyncPacket(chkPt));
      //checkpoint omitVV is b0:1;b2:2
      log1Copy3 = node1Copy3.getLogTest();
      
      //application of a later write not concurrent with the omitVV should fail the inclusion check
      sspi = (SecureSimPreciseInv)nl0_I.get(1);
      sspi.getFlags().clear();
      ls = log1Copy3.apply(sspi);
      assert ls.equals(LogStatus.makeLogStatusFailedVerificationStatus(LogStatus.InclusionFail)):ls;
      
      //  ---------------- test MissingPOMForForkedChildBranch -------------------- 
      // --- for the scenario when ConcurrentPOM is used
      sspi = (SecureSimPreciseInv)nl0_I.get(0);
      sspi.getFlags().clear();
      ls = log1Copy3.apply(sspi);
      assert ls.equals(LogStatus.makeLogStatus(LogStatus.ApplyFailed, LogStatus.MissingPOMForForkedChildBranch)):ls;
      
      //  ---------------- test Inconsistency -------------------- 
      // write starting a new branch concurrent with omitVV should be inconsistent
      // apply a ConcurrentPOM now
      SimPreciseInv sspiConcur = null;
      for(NodeId n: chkPt.lastWrite.keySet()){
        if(n.getIDint() == sspi.getNodeId().getIDint()){
          sspiConcur = chkPt.lastWrite.get(n);
        }
      }
      ConcurrentPOM cpom = new ConcurrentPOM(sspi, sspiConcur);
      log1Copy3.applyPOM(cpom, new POMRemap());
      ls = log1Copy3.apply(sspi);
      assert ls.equals(LogStatus.makeLogStatus(LogStatus.ApplySuccessful, LogStatus.FIFOPassedInclusionUnverifiable)):ls;
      assert sspi.getFlags().contains(Flags.InconsistentWrite):sspi.getFlags();
      
      //write in the old epoch should be inconsistent
      sspi = (SecureSimPreciseInv)nl0_I.get(1);
      sspi.getFlags().clear();
      
      ls = log1Copy3.apply(sspi);
      assert ls.equals(LogStatus.makeDefaultLogStatus()):ls;
      assert sspi.getFlags().contains(Flags.InconsistentWrite):sspi.getFlags();
      
      // write in the new epoch should be consistent
      node1Copy3.setLamportClockTest(5);
      SecureSimPreciseInv spi = (SecureSimPreciseInv)node1Copy3.write(oid1, new IrisDataObject(new Integer(1)));
      NodeLog nl = log1Copy3.get(node1Copy3.getBranchID());
      assert nl.size()==1:nl; //2 when the certificate writing in log is enabled--1 for the certificate
      sspi = (SecureSimPreciseInv)nl.get(0); //1 when certificate writing is enabled
      assert sspi.getFlags().isEmpty();
      assert spi.getFlags().isEmpty();
      
    }catch(Exception e){
      e.printStackTrace();
      assert false;
    }
  }
  /**
   * test whether the log issues appropriate error messages
   */
  public void testLogError(){
    try{
//      if(!Node.enableSecurity){
//        return;
//      }
      assert SangminConfig.useSimulator && Node.enableSecurity;
      SangminConfig.usePersistentStore = false;
      SangminConfig.enableInconsistencyNotification = false;
      BranchID myNodeId = NodeFactory.createNodeId(UNIT_NODE_ID);
      NodeId b0 = myNodeId;
      NodeId b2 = null, b2I = null, b4 = null, b4I = null;
      BranchID nodeId1 = NodeFactory.createNodeId(15);
      BranchID nodeId2 = NodeFactory.createNodeId(20);
      BranchID nodeId3 = NodeFactory.createNodeId(25);

      Node myNode = NodeFactory.createNodeWithoutFilter(myNodeId);
      Node node1 = NodeFactory.createNodeWithoutFilter(nodeId1);
      Node node2 = NodeFactory.createNodeWithoutFilter(nodeId2);
      Node node3 = NodeFactory.createNodeWithoutFilter(nodeId3);

      ObjId oid0 = new ObjId("/test/0");
      myNode.write(oid0, new IrisDataObject(new Integer(0)));
      node1.sync(myNode);
      SimPreciseInv n1w1 = node1.write(new ObjId("/test/991"), new IrisDataObject(new Integer(-1)));
      
      ObjId oid1 = new ObjId("/test/1");
      myNode.write(oid1, new IrisDataObject(new Integer(1)));

      LinkedList<Node> forks = myNode.fork(2);
      
      forks.get(0).write(new ObjId("/test/00"), new IrisDataObject(new Integer(2)));
      node1.sync(forks.get(0));
      SimPreciseInv n1w2 = node1.write(new ObjId("/test/991"), new IrisDataObject(new Integer(-1)));
      forks.get(0).write(new ObjId("/test/01"), new IrisDataObject(new Integer(3)));
      forks.get(1).write(new ObjId("/test/10"), new IrisDataObject(new Integer(4)));
      node2.sync(forks.get(1));
      SimPreciseInv n2w1 = node2.write(new ObjId("/test/992"), new IrisDataObject(new Integer(-2)));//depends on b2I:3
      forks.get(1).write(new ObjId("/test/11"), new IrisDataObject(new Integer(5)));

      LinkedList<Node> newForks = forks.get(0).fork(2);
      newForks.get(0).write(new ObjId("/test/0000"), new IrisDataObject(new Integer(6)));
      newForks.get(0).write(new ObjId("/test/0001"), new IrisDataObject(new Integer(7)));
      newForks.get(1).write(new ObjId("/test/0010"), new IrisDataObject(new Integer(8)));
      node3.sync(newForks.get(1));
      SimPreciseInv n3w1 = node3.write(new ObjId("/test/992"), new IrisDataObject(new Integer(-3)));
      newForks.get(1).write(new ObjId("/test/0011"), new IrisDataObject(new Integer(9)));

      node1.sync(newForks.get(0));
      node2.sync(forks.get(1));
      node3.sync(newForks.get(1));

      if(!SangminConfig.useSimulatorId){
        ((IrisNode)node1).exponentialBackoffSync(node2);
      }
      node1.sync(node2);

      AcceptVV cvv1 = node1.getCurrentVV();
      VVIterator vvi1 = cvv1.getIterator();
      while(vvi1.hasMoreElements()){
        NodeId n = vvi1.getNext();
        if(cvv1.getStampByIteratorToken(n) == 3 && n.getIDint() == b0.getIDint()){
          b2I = n;
        }else if(cvv1.getStampByIteratorToken(n) == 5 && n.getIDint() == b0.getIDint()){
          b2 = n;
        }else if(n.getIDint() == b0.getIDint()){
          assert b0.equals(n);
        }
      }

      if(!SangminConfig.useSimulatorId){
        ((IrisNode)node1).exponentialBackoffSync(node3);
      }
      node1.sync(node3);

      cvv1 = node1.getCurrentVV();
      vvi1 = cvv1.getIterator();
      while(vvi1.hasMoreElements()){
        NodeId n = vvi1.getNext();
        if(cvv1.getStampByIteratorToken(n) == 5 && n.getIDint() == b0.getIDint()){
          if(b4I == null){
            b4I = n;
          }else{
            b4 = n;
          }
        }
      }


      //initialize node logs
      NodeLog nl1 = node1.getLogTest().get(node1.getBranchID());
      assert nl1.size() == 2;
      
      NodeLog nl2 = node1.getLogTest().get(node2.getBranchID());
      assert nl2.size() == 1;
      
      NodeLog nl3 = node1.getLogTest().get(node3.getBranchID());
      assert nl3.size() == 1;
      
      NodeLog nl0 = node1.getLogTest().get(b0);
      assert nl0.size() == 2;
      
      NodeLog nl0_ = node1.getLogTest().get(b2);
      assert nl0_.size() == 2;
      
      NodeLog nl0_I = node1.getLogTest().get(b2I);
      assert nl0_I.size() ==2:nl0_I;
      
      NodeLog nl0__ = node1.getLogTest().get(b4);
      assert nl0__.size() == 2;
      
      NodeLog nl0__I = node1.getLogTest().get(b4I);
      assert nl0__I.size() == 2;
      
      Node f0 = myNode;
      Node f2 = forks.get(0);
      Node f2I = forks.get(1);
      Node f4 = newForks.get(0);
      Node f4I = newForks.get(1);
      
      NodeLog nlf0 = f0.getLogTest().get(f0.getBranchID());
      NodeLog nlf2 = f2.getLogTest().get(f2.getBranchID());
      NodeLog nlf2I = f2I.getLogTest().get(f2I.getBranchID());
      NodeLog nlf4 = f4.getLogTest().get(f4.getBranchID());
      NodeLog nlf4I = f4I.getLogTest().get(f4I.getBranchID());
      
      HashSet<ProofOfMisbehavior> poms = node1.getPOMMap();
      ProofOfMisbehavior pom2 = null, pom4 = null;
      assert poms.size() ==2 ;
      for(ProofOfMisbehavior pom: poms){
        if(pom.getParent().getAcceptStamp().getLocalClock() == 1){
          pom2 = pom;
        }else{
          pom4 = pom;
        }
      }
      
      // ---------------- test DVVInclusionFail --------------------
      Node node1Copy = NodeFactory.createNodeWithoutFilter(nodeId1);
      Log log1Copy = node1Copy.getLogTest();
      LogStatus ls = log1Copy.apply(nl1.get(0));
      assert ls.getApplyStatus() == LogStatus.ApplyFailed:ls;
      if(SangminConfig.enableInconsistencyNotification){
        assert ls.getVerificationStatus() == LogStatus.InclusionFail:ls;
      }else{
        assert ls.getVerificationStatus() == LogStatus.DVVInclusionFail:ls;
      }
      ls = log1Copy.apply(nl0.get(0));
      assert ls.equals(LogStatus.makeDefaultLogStatus());
      ls = log1Copy.apply(nl1.get(0));
      assert ls.equals(LogStatus.makeDefaultLogStatus()):ls;

      // ---------------- test TimeStampExhaustionCheckFailed --------------------
      SecureSimPreciseInv sspiTest = (SecureSimPreciseInv)nl0.get(1);
      long rejectedTS = (System.currentTimeMillis()+10000000)*1000;
      sspiTest = new SecureSimPreciseInv(sspiTest.getObjId(), new AcceptStamp(rejectedTS, sspiTest.getNodeId()), 
          sspiTest.getDVV(), sspiTest.getData(), sspiTest.getEpoch(), sspiTest.getObjectHashes());
      ls = log1Copy.apply(sspiTest);
      assert ls.equals(LogStatus.makeLogStatusFailedVerificationStatus(LogStatus.TimeStampExhaustionCheckFailed)): ls;

      //  ---------------- test InclusionFail -------------------- 
      ls = log1Copy.apply(nl0.get(1));
      assert ls.equals(LogStatus.makeDefaultLogStatus());
      
      // now apply the write from the forked branch 2I 
      ls = log1Copy.apply(nlf2I.get(2));
      assert ls.equals(LogStatus.makeDefaultLogStatus());
      
      // now the inclusion should fail for n1w2
      ls = log1Copy.apply(n1w2);
      assert ls.equals(LogStatus.makeLogStatusFailedVerificationStatus(LogStatus.InclusionFail)):ls;
      
      //  ---------------- test FIFOFail -------------------- 
      // try applying nlf2.get(3) with FIFO only applyCondition
      ls = log1Copy.apply(nlf2.get(3), Log.ApplyIfFIFOPass);
      assert ls.equals(LogStatus.makeLogStatusFailedVerificationStatus(LogStatus.FIFOFail)):ls;

      //  ---------------- test FIFOPassedInclusionUnverifiable -------------------- 
      // applying n1w2 with fifo only check should succeeed
      ls = log1Copy.apply(n1w2, Log.ApplyIfFIFOPass);
      assert ls.equals(LogStatus.makeLogStatus(LogStatus.ApplySuccessful, LogStatus.FIFOPassedInclusionUnverifiable)):ls;
      
      //  ---------------- test CompatibilityFail -------------------- 
      //  try applying nlf2.get(2)  
      ls = log1Copy.apply(nlf2.get(2), Log.ApplyIfFIFOPass);
      assert ls.equals(LogStatus.makeLogStatusFailedVerificationStatus(LogStatus.CompatibilityFail)):ls;
      
      ls = log1Copy.apply(nlf2.get(2));
      assert ls.equals(LogStatus.makeLogStatusFailedVerificationStatus(LogStatus.CompatibilityFail)):ls;
      
      //  ---------------- test MissingPOMForForkedChildBranch -------------------- 
      //try applying nl0_[2]
      ls = log1Copy.apply(nl0_.get(0));
      assert ls.equals(LogStatus.makeLogStatusFailedVerificationStatus(LogStatus.MissingPOMForForkedChildBranch)):ls;
      
      //  ---------------- test WriteToForkedParentBranch -------------------- 
      assert log1Copy.applyPOM(pom2, new POMRemap());
      
      ls = log1Copy.apply(nlf2.get(2));
      assert ls.equals(LogStatus.makeLogStatusFailedVerificationStatus(LogStatus.WriteToForkedParentBranch)):ls;
      
      assert ((SecureSimPreciseInv)nl0_.get(0)).generateMyHash().equals(((SecureSimPreciseInv)nlf2.get(2)).generateMyHash());
      
      ls = log1Copy.apply(nl0_.get(0));
      assert ls.equals(LogStatus.makeDefaultLogStatus()):ls;
      
      //  ---------------- test ImproperLamportClock -------------------- 
      // make a copy of n2w1 with an improper lamport clock
      SecureSimPreciseInv sn2w1 = (SecureSimPreciseInv)nl2.get(0); // the version with branch map applied
      SecureSimPreciseInv newn2wi = new SecureSimPreciseInv(sn2w1.getObjId(), new AcceptStamp(sn2w1.getAcceptStamp().getLocalClock()-1, sn2w1.getNodeId()), sn2w1.getDVV(), sn2w1.getData(), sn2w1.getEpoch(), sn2w1.getObjectHashes());
      
      ls = log1Copy.apply(newn2wi);
      assert ls.equals(LogStatus.makeLogStatusFailedVerificationStatus(LogStatus.ImproperLamportClock)):ls;
      
//      //  ---------------- test ImproperEpochNumber -------------------- 
      //PS:THIS CHECK IS CURRENTLY BEING DONE IN THE NODE
//      
//      newn2wi = new SecureSimPreciseInv(sn2w1.getObjId(), sn2w1.getAcceptStamp(), sn2w1.getDVV(), sn2w1.getData(), sn2w1.getHash(), sn2w1.getEpoch()+1);
//      
//      ls = log1Copy.apply(newn2wi);
//      assert ls.equals(LogStatus.makeLogStatusFailedVerificationStatus(LogStatus.ImproperEpochNumber)):ls;
      
      ls = log1Copy.apply(sn2w1);
      assert ls.equals(LogStatus.makeDefaultLogStatus()):ls;
      
      
      //  ---------------- test ImproperDVV -------------------- 
      SecureSimPreciseInv sspi = (SecureSimPreciseInv)nl0_.get(1);
      AcceptVV vv = log1Copy.getCurrentVV();
      HashMap<NodeId, Hash> hashMap = log1Copy.getHashMap(vv);
      HashedVV dvv = new HashedVV(vv, hashMap);
      Hash h = Log.getHash(dvv);
      VVIterator vvi = dvv.getIterator();
      long lc = -1;
      while(vvi.hasMoreElements()){
        NodeId n = vvi.getNext();
        long ts = dvv.getStampByIteratorToken(n);
        if(ts > lc){
          lc = ts;
        }
      }
      lc++;
      
      AcceptStamp as = new AcceptStamp(lc, sspi.getNodeId());
      SecureSimPreciseInv nsspi = new SecureSimPreciseInv(sspi.getObjId(), as, dvv, sspi.getData(), sspi.getEpoch(), sspi.getObjectHashes());
      
      ls = log1Copy.apply(nsspi);
      assert ls.equals(LogStatus.makeLogStatusFailedVerificationStatus(LogStatus.ImproperDVV)):ls;

      //  ---------------- test ImproperBranchID -------------------- 
      Node node1Copy2 = NodeFactory.createNodeWithoutFilter(nodeId1);
      Log log1Copy2 = node1Copy2.getLogTest();
      node1Copy2.sync(f0);
      
      ls = log1Copy2.apply(nlf2.get(2));
      assert ls.equals(LogStatus.makeDefaultLogStatus()):ls;
 
      ls = log1Copy2.apply(nlf2.get(3));
      assert ls.equals(LogStatus.makeDefaultLogStatus()):ls;

      //  ---------------- test MissingPOMForForkedChildBranch -------------------- 
      // --- for the scenario when ConcurrentPOM is used
      Node node1Copy3 = NodeFactory.createNodeWithoutFilter(nodeId1);
      Log log1Copy3 = node1Copy3.getLogTest();
      node1Copy3.sync(f0);
      sspi = (SecureSimPreciseInv)nl0_I.get(0);
      sspi.getFlags().clear();
      ls = log1Copy3.apply(sspi);
      assert ls.equals(LogStatus.makeLogStatus(LogStatus.ApplyFailed, LogStatus.MissingPOMForForkedChildBranch)):ls;
      
      
    }catch(Exception e){
      e.printStackTrace();
      assert false;
    }
  }

  /**
   * test whether the log issues appropriate error messages
   */
  public void testResetState(){
    try{
//      if(!Node.enableSecurity){
//        return;
//      }
      assert SangminConfig.useSimulator && Node.enableSecurity;
      SangminConfig.usePersistentStore = false;
      SangminConfig.enableInconsistencyNotification = false;
      BranchID myNodeId = NodeFactory.createNodeId(UNIT_NODE_ID);
      NodeId b0 = myNodeId;
      NodeId b2 = null, b2I = null, b4 = null, b4I = null;
      BranchID nodeId1 = NodeFactory.createNodeId(15);
      BranchID nodeId2 = NodeFactory.createNodeId(20);
      BranchID nodeId3 = NodeFactory.createNodeId(25);
      BranchID nodeId4 = NodeFactory.createNodeId(5);

      Node myNode = NodeFactory.createNodeWithoutFilter(myNodeId);
      Node node1 = NodeFactory.createNodeWithoutFilter(nodeId1);
      Node node2 = NodeFactory.createNodeWithoutFilter(nodeId2);
      Node node3 = NodeFactory.createNodeWithoutFilter(nodeId3);
      Node node4 = NodeFactory.createNodeWithoutFilter(nodeId4);

      ObjId oid0 = new ObjId("/test/0");
      myNode.write(oid0, new IrisDataObject(new Integer(0)));
      node1.sync(myNode);
      SimPreciseInv n1w1 = node1.write(new ObjId("/test/991"), new IrisDataObject(new Integer(-1)));
      
      ObjId oid1 = new ObjId("/test/1");
      myNode.write(oid1, new IrisDataObject(new Integer(1)));

      LinkedList<Node> forks = myNode.fork(2);
      
      forks.get(0).write(new ObjId("/test/00"), new IrisDataObject(new Integer(2)));
      node1.sync(forks.get(0));
      SimPreciseInv n1w2 = node1.write(new ObjId("/test/991"), new IrisDataObject(new Integer(-1)));
      forks.get(0).write(new ObjId("/test/01"), new IrisDataObject(new Integer(3)));
      forks.get(1).write(new ObjId("/test/10"), new IrisDataObject(new Integer(4)));
      node2.sync(forks.get(1));
      SimPreciseInv n2w1 = node2.write(new ObjId("/test/992"), new IrisDataObject(new Integer(-2)));//depends on b2I:3
      forks.get(1).write(new ObjId("/test/11"), new IrisDataObject(new Integer(5)));

      LinkedList<Node> newForks = forks.get(0).fork(2);
      newForks.get(0).write(new ObjId("/test/0000"), new IrisDataObject(new Integer(6)));
      newForks.get(0).write(new ObjId("/test/0001"), new IrisDataObject(new Integer(7)));
      newForks.get(1).write(new ObjId("/test/0010"), new IrisDataObject(new Integer(8)));
      node3.sync(newForks.get(1));
      SimPreciseInv n3w1 = node3.write(new ObjId("/test/992"), new IrisDataObject(new Integer(-3)));
      newForks.get(1).write(new ObjId("/test/0011"), new IrisDataObject(new Integer(9)));

      node1.sync(newForks.get(0));
      node2.sync(forks.get(1));
      node3.sync(newForks.get(1));

      node4.write(new ObjId("/test/992"), new IrisDataObject(new Integer(-2)));//depends on b2I:3
      node4.write(new ObjId("/test/992"), new IrisDataObject(new Integer(-2)));//depends on b2I:3
      // ---------------- test resetState --------------------
      // ---- first without forks
      Log log1 = node1.getLogTest();
      HashedVV oldVV = node1.getHashedDependencyVV(log1.getMinimalVV(node1.getCurrentVV()));
      SyncStatus ss = node1.sync(node4);
      assert ss.status == SyncStatus.SyncSuccessful: ss;
      assert node1.getCurrentVV().includes(oldVV): "oldVV " + oldVV + "\ncvv"+node1.getCurrentVV();
      assert !oldVV.includes(node1.getCurrentVV()): "oldVV " + oldVV + "\ncvv"+node1.getCurrentVV();
      log1.resetState(oldVV);
      assert node1.getCurrentVV().includes(oldVV): "oldVV " + oldVV + "\ncvv"+node1.getCurrentVV();
      assert oldVV.includes(node1.getCurrentVV()): "oldVV " + oldVV + "\ncvv"+node1.getCurrentVV();
      
      // ---- then with forks     
      oldVV = node1.getHashedDependencyVV(log1.getMinimalVV(node1.getCurrentVV()));
      
      if(!SangminConfig.useSimulatorId){
        ((IrisNode)node1).exponentialBackoffSync(node3);
      }
      ss = node1.sync(node3);
      assert ss.status == SyncStatus.SyncSuccessful: ss;
      AcceptVV remapedOldVV = log1.getClosure(node1.getHashedDependencyVV(log1.getMinimalVV(log1.remap(oldVV, true))));
      assert node1.getCurrentVV().includes(remapedOldVV): "oldVV " + oldVV + "\ncvv"+node1.getCurrentVV();
      assert !remapedOldVV.includes(node1.getCurrentVV()): "oldVV " + oldVV + "\ncvv"+node1.getCurrentVV();
      log1.resetState(oldVV);
      remapedOldVV = log1.getClosure(node1.getHashedDependencyVV(log1.getMinimalVV(log1.remap(oldVV, true))));
      assert node1.getCurrentVV().includes(remapedOldVV): "oldVV " + oldVV + "\ncvv"+node1.getCurrentVV();
      assert remapedOldVV.includes(node1.getCurrentVV()): "oldVV " + oldVV + "\ncvv"+node1.getCurrentVV();
      
      // ---- now with localWrites     
      oldVV = node1.getHashedDependencyVV(log1.getMinimalVV(node1.getCurrentVV()));
      
      if(!SangminConfig.useSimulatorId){
        ((IrisNode)node1).exponentialBackoffSync(node3);
      }
      ss = node1.sync(node3);
      assert ss.status == SyncStatus.SyncSuccessful: ss;
      node1.write(new ObjId("/test/0011"), new IrisDataObject(new Integer(9)));
      remapedOldVV = log1.getClosure(node1.getHashedDependencyVV(log1.getMinimalVV(log1.remap(oldVV, true))));
      assert node1.getCurrentVV().includes(remapedOldVV): "oldVV " + oldVV + "\ncvv"+node1.getCurrentVV();
      assert !remapedOldVV.includes(node1.getCurrentVV()): "oldVV " + oldVV + "\ncvv"+node1.getCurrentVV();
      log1.resetState(oldVV);
      remapedOldVV = log1.getClosure(node1.getHashedDependencyVV(log1.getMinimalVV(log1.remap(oldVV, true))));
      assert node1.getCurrentVV().includes(remapedOldVV): "oldVV " + oldVV + "\ncvv"+node1.getCurrentVV();
      assert remapedOldVV.includes(node1.getCurrentVV()): "oldVV " + oldVV + "\ncvv"+node1.getCurrentVV();
        
    }catch(Exception e){
      e.printStackTrace();
      assert false;
    }
  }
  /*
   * "new TestSuite(Class c)" constructs a test suite
   * containg every method whose name begins with "/test"
   * 
   * TBD: update class name
   */
  public static Test suite(){
    TestSuite suite = new TestSuite(LogUnit.class);
    return suite;
  }


  /*
   * main() lets us run just this set of unit tests
   * from the comand line (you can also invoke 
   * the testrunner on this class and it will find
   * the suite())
   *
   * usage: java <classname> [-verbose] [-vverbose] [testName]*
   * 
   *   If verbose or vverbose are included, print info to screen
   *
   *   If [testName]* are included, then run test called "/test[testName]"
   *   for each such [testName]. E.g., "java TestEmtpy foo" runs
   *   TwoNodesSubscriptionUnit.testfoo() as a TestCase.
   *
   * TBD: update class name
   */
  public static void main(String s[]) {
    String name = "LogUnit";
    System.err.print(name + " self test begins...");
    Test test;
    test = suite();
    TestRunner tr = new TestRunner();
    tr.doRun(test);
    System.exit(0);
  }

}
