package code.simulator.persistentLog;
//---------------------------------------------------------------
/* PersistentLog
 * 
 * Provide stable on-disk invalidate sequences
 * Use BerkeleyDB to manage data storage
 *
 */
//---------------------------------------------------------------
import code.AcceptStamp;
import code.Env;
import code.NodeId;
import code.ObjId;
import code.ResultStats;
import code.branchDetecting.BranchID;
import code.lasr.db.*;
import code.security.SangminConfig;
import code.security.ahs.DependencyVV;
import code.serialization.IrisObjectInputStream;
import code.serialization.IrisObjectOutputStream;
import code.serialization.SerializationHelper;
import code.simulator.Hash;
import code.simulator.HashedVV;
import code.simulator.IrisDataObject;

import java.io.File;
import java.io.IOException;

import java.util.Arrays;
import java.util.Random;
import java.util.ArrayList;
import java.util.Collections;

public class PersistentStore{

  public final DbHandle statusDbHandle; 
  public final DbHandle logDbHandle; 
  public final DbHandle storeDbHandle; 

  final private static String statusDbName = "statusDb";
  final private static String logDbName = "logDb";
  final private static String storeDbName = "storeDb";

  final public static String CheckpointKey = "Checkpoint";
  final public static String SeqNumKey = "SeqNum";

  final Database db;
  private int seqNum;
  //-----------------------------------------------------------
  // constructor
  // open/create databases
  // for omitVV, if it's new-created, put the initial VV
  //----------------------------------------------------------- 
  public 
  PersistentStore(String path,
      long cacheSizeBytes,
      boolean reset, 
      boolean sync)
  throws IOException, DbException{

    File dbDir = new File(path);
    if(dbDir.exists()&&reset){
      try{
        assert dbDir.isDirectory();

        File[] dbFiles = dbDir.listFiles();
        for(int i=0; i<dbFiles.length; i++){
          assert dbFiles[i].isFile();
          dbFiles[i].delete();
        }
      }catch (SecurityException se){
        throw new IOException(se.toString());
      }
    }
    //    dbDir.mkdirs();

    //
    // Initialize environment and databases
    //
    db = new Database(path, cacheSizeBytes, sync);

    seqNum = 0;

    statusDbHandle = Database.getHandle(path+statusDbName, String.class, byte[].class);
    logDbHandle = Database.getHandle(path+logDbName, Integer.class, byte[].class);
    storeDbHandle = Database.getHandle(path+storeDbName, byte[].class, byte[].class);
    db.openDatabase(statusDbHandle);
    db.openDatabase(logDbHandle);
    db.openDatabase(storeDbHandle);
  }

  final public Database getDB(){
    return db;
  }

  public 
  void append(DbTransaction txn, byte[] buf){
    try{
      int mySeqNum = -1;
      synchronized(this){
        mySeqNum = seqNum++;
      }
      txn.write(logDbHandle, seqNum++, buf);
    }catch(DbException e){
      e.printStackTrace();
      System.exit(1);
    }
  }

  //--------------------------------------------------------
  // close() -- flush everything to disk and close.
  //--------------------------------------------------------
  public void
  close(){
    db.closeDatabase(statusDbHandle);
    db.closeDatabase(logDbHandle);
    db.closeDatabase(storeDbHandle);
    db.close();
  }

  /**
   * @return the checkpointDbHandle
   */
  public DbHandle getStatusDbHandle(){
    return statusDbHandle;
  }

  /**
   * @return the logDbHandle
   */
  public DbHandle getLogDbHandle(){
    return logDbHandle;
  }

  /**
   * @return the storeDbHandle
   */
  public DbHandle getStoreDbHandle(){
    return storeDbHandle;
  }

  private static final int ALL = 0;
  private static final int SerialRead = 1;
  private static final int SerialWrite = 2;
  private static final int RandomWrite = 3;
  private static final int RandomRead = 4;
  private static long delay = 0;

  public static void main(String[] args) throws IOException, DbException{
      try{
      Thread.sleep(2000);
      }catch(Exception e){
      }
    if(args.length == 0){
      System.out.print("usage: PersistentStore <storepath> -type <TxnType> -size <size> -reset <reset?> -numOps <NUM> -useHashKeys <UseHashKeys> -useDelay <delay>");
      System.exit(1);
    }else{
      System.out.println("args" + args[0]);
    }
    if(args.length > 0){
      SangminConfig.configFile = args[0] + "/";
    }

    ResultStats.logData = true;
    int size = -1;

    int config = ALL;
    int NUM = 1000;
    boolean reset = true;
    boolean useoper1 = false;
    for(int i = 0; i < args.length; i++){
      String s = args[i];
      if(s.equals("-type")){
        assert i < args.length-1;
        if(args[i+1].equals("SerialRead")){
          config = SerialRead;
        }else if(args[i+1].equals("SerialWrite")){
          config = SerialWrite;
        }else if(args[i+1].equals("RandomWrite")){
          config = RandomWrite;
        }else if(args[i+1].equals("RandomRead")){
          config = RandomRead;
        }
        i++;
      }
      else if(s.equals("-size")){
        assert i < args.length-1;
        size = Integer.parseInt(args[i+1]);
        i++;
      }else if(s.equals("-numOps")){
        assert i < args.length-1;
        NUM = Integer.parseInt(args[i+1]);
        i++;
      }else if(s.equals("-reset")){
        assert i < args.length-1;
        reset = Boolean.parseBoolean(args[i+1]);
        i++;
      }else if(s.equals("-useHashKeys")){
        assert i < args.length-1;
        useoper1 = Boolean.parseBoolean(args[i+1]);
        i++;
      }else if(s.equals("-useDelay")){
        assert i < args.length-1;
        delay = Integer.parseInt(args[i+1]);
        i++;
      }
    }

    System.out.println("config: "  + config + " size: "  + size + " NUM " + NUM + " reset "  + reset + " delay " +delay);
    byte[] testD = null;
    if(size != -1){
      testD = new byte[size];
      for(int c = 0; c < size; c++){
        testD[c] = (byte)(c%128);
      }
    }

    // create buffers of size 100 Bytes, 1K, 10K, 100K, 1MB, 10MB
    String data = "";
    for(int base = 0; base < 10; base++){
      data += base;
    }

    Random r = new Random();
    long startTime, endTime;
    Hash h = new Hash("test".getBytes(), true);
    byte[] key = h.getHashVal();

    switch(config){
    case ALL:
    {
      for(int bufSize= 0; bufSize < 5; bufSize++){
        DependencyVV dvv = new DependencyVV();
        dvv.put(new BranchID(5), 3);

        String newData ="";
        for(int repeat = 0; repeat < 10; repeat++){
          newData += data;
        }
        data = newData; 

        byte[] buf = data.getBytes();
        for(int j = 0; j <1; j++){
          boolean durable = (j%2)==0;
          PersistentStore ps = new PersistentStore(SangminConfig.configFile+"test", (long)1024*1024*100, reset, durable);

          startTime = System.currentTimeMillis();
          rs = new ResultStats();
          long startTime1;
          for(int k = 0; k < NUM; k++){
            startTime1 = System.currentTimeMillis();
            PersistentStore.oper(key, k, buf, false, ps);
            endTime = System.currentTimeMillis();
            rs.enter(endTime-startTime1);
          }
          endTime = System.currentTimeMillis();
          System.out.println("sequential write transaction for " + (durable?"durable":"non-durable") + " " + buf.length + " took " + (endTime-startTime) +"ms");

          startTime = System.currentTimeMillis();

          for(int k = 0; k < NUM; k++){
            PersistentStore.oper(key, r.nextInt(NUM), buf, false, ps);
          }
          endTime = System.currentTimeMillis();
          System.out.println("random write transaction for " + (durable?"durable":"non-durable") + " " + buf.length + " took " + (endTime-startTime) +"ms");
          System.out.println("mean0" + rs.getMean(0));
          System.out.println("mean1" + rs.getMean(1));
          System.out.println(rs.getAverage90());
          System.out.println("std0" + rs.getStandardDeviation(0));
          System.out.println("std1" +rs.getStandardDeviation(1));

          rs = new ResultStats();
          startTime = System.currentTimeMillis();
          for(int k = 0; k < NUM; k++){
            PersistentStore.oper(key, k, buf, true, ps);
          }
          //        System.exit(1);
          endTime = System.currentTimeMillis();
          System.out.println("sequential read transaction for " + (durable?"durable":"non-durable") + " " + buf.length + " took " + (endTime-startTime) +"ms");

          startTime = System.currentTimeMillis();
          for(int k = 0; k < NUM; k++){
            PersistentStore.oper(key, r.nextInt(NUM), buf, true, ps);
          }
          //        System.exit(1);
          ps.close();
          endTime = System.currentTimeMillis();
          System.out.println("random read transaction for " + (durable?"durable":"non-durable") + " " + buf.length + " took " + (endTime-startTime) +"ms");
        }
      }
    }        
    break;

    case SerialWrite:
    {
      PersistentStore ps = new PersistentStore(SangminConfig.configFile+"test", (long)1024*1024*100, reset, true);

      ArrayList<Integer> listPK = new ArrayList<Integer>(NUM);
      for(int i = 0; i < NUM; i++){
	  listPK.add(i);
      }
      Collections.shuffle(listPK);

      startTime = System.currentTimeMillis();
      for(int k = 0; k < NUM; k++){
        if(useoper1){
	    PersistentStore.oper1(key, listPK.get(k), testD, false, ps);
        }else{
	    PersistentStore.oper(key, listPK.get(k), testD, false, ps);
        }
      }
      //        System.exit(1);
      ps.close();
      endTime = System.currentTimeMillis();
      System.out.println("serial write transaction for " + " " + testD.length + " took " + (endTime-startTime) +"ms");

    }
    break;

    case RandomWrite:
    {
      PersistentStore ps = new PersistentStore(SangminConfig.configFile+"test", (long)1024*1024*100, reset, true);

      ArrayList<Integer> listPK = new ArrayList<Integer>(NUM);
      for(int i = 0; i < NUM; i++){
	  listPK.add(i);
      }
      Collections.shuffle(listPK);
		
      startTime = System.currentTimeMillis();
      for(int k = 0; k < NUM; k++){
        if(useoper1){
	    PersistentStore.oper1(key, listPK.get(k), testD, false, ps);
        }else{
	    PersistentStore.oper(key, listPK.get(k), testD, false, ps);
        }
      }
      //        System.exit(1);
      ps.close();
      endTime = System.currentTimeMillis();
      System.out.println("random write transaction for " + " " + testD.length + " took " + (endTime-startTime) +"ms");

    }
    break;

    case SerialRead:
    {
      PersistentStore ps = new PersistentStore(SangminConfig.configFile+"test", (long)1024*1024*100, reset, true);

      startTime = System.currentTimeMillis();
      for(int k = 0; k < NUM; k++){
        if(useoper1){
          PersistentStore.oper1(key, k, testD, true, ps);
        }else{
          PersistentStore.oper(key, k, testD, true, ps);
        }
      }
      //        System.exit(1);
      ps.close();
      endTime = System.currentTimeMillis();
      System.out.println("serial read transaction for " + " " + testD.length + " took " + (endTime-startTime) +"ms");

    }
    break;

    case RandomRead:
    {
      PersistentStore ps = new PersistentStore(SangminConfig.configFile+"test", (long)1024*1024*100, reset, true);

      startTime = System.currentTimeMillis();
      for(int k = 0; k < NUM; k++){
        if(useoper1){
          PersistentStore.oper1(key, k, testD, true, ps);
        }else{
          PersistentStore.oper(key, k, testD, true, ps);
        }
      }
      //        System.exit(1);
      ps.close();
      endTime = System.currentTimeMillis();
      System.out.println("random read transaction for " + " " + testD.length + " took " + (endTime-startTime) +"ms");

    }
    break;
    }
    System.out.println("mean0" + rs.getMean(0));
    System.out.println("mean1" + rs.getMean(1));
    System.out.println(rs.getAverage90());
    System.out.println("std0" + rs.getStandardDeviation(0));
    System.out.println("std1" +rs.getStandardDeviation(1));
    /*    ArrayList ar = rs.getAll();
    for(Object o: ar){
	System.out.println(o);
	}*/
    try{
      Thread.sleep(2000);
      }catch(Exception e){
    }
  }

  public static void oper(byte[] key, int keyMask, byte[] buf, boolean read, PersistentStore ps) throws DbException{
    key[0] = (byte)(keyMask&0xFF);
    key[1] = (byte)((keyMask>>8)&0xFF);
    long startTime = System.currentTimeMillis();
    //key = new Hash(buf, true).getHashVal();
    DbTransaction txn = ps.db.newTransaction();
    if(read){
      byte[] readBuf = (byte[])txn.read(ps.storeDbHandle, key);
      //          System.out.println("read data of size " + readBuf.length);
      //          assert Arrays.equals(readBuf, buf);
      if(Arrays.equals(readBuf, buf)){
        Env.logWrite("read key "+keyMask);
      }else{
        //        Env.logWrite("read NULL for key " + keyMask);
	  assert false:"Key: " + Arrays.toString(key) + "readBuf: " + readBuf;
      }
    }else{
      txn.write(ps.storeDbHandle, key, buf);
      Env.logWrite("writing key "+keyMask);

    }
//    txn.commit();
    txn.sync();
    long endTime = System.currentTimeMillis();
    rs.enter(endTime-startTime);
    //    System.out.println("disk read " + Arrays.toString(key) + " took " + (endTime-startTime));
    if(delay != 0){
      try{
        Thread.sleep(delay);
      }catch(InterruptedException e){
      }
    }
  }

  private static ResultStats rs = new ResultStats();
  public static void oper1(byte[] key, int keyMask, byte[] buf, boolean read, PersistentStore ps) throws DbException{
    //  key[0] = (byte)(keyMask%128);
    //  key[1] = (byte)((keyMask>>8)%128);
    long startTime = System.currentTimeMillis();

    DbTransaction txn = ps.db.newTransaction();
    if(read){
      int mySeq = SerializationHelper.getInt((byte[])txn.read(ps.logDbHandle, key), 0);
      byte[] readBuf = (byte[])txn.read(ps.storeDbHandle, mySeq);
      //          System.out.println("read data of size " + readBuf.length);
      //          assert Arrays.equals(readBuf, buf);
      if(Arrays.equals(readBuf, buf)){
        Env.logWrite("read key "+keyMask);
      }else{
        //      Env.logWrite("read NULL for key " + keyMask);
        assert false;
      }
    }else{
      int mySeq = ps.seqNum++;
      txn.write(ps.logDbHandle, mySeq, buf);
      key = new Hash(buf, true).getHashVal();
      txn.write(ps.storeDbHandle, key, SerializationHelper.getBytesFromInt(mySeq));
      Env.logWrite("writing key "+keyMask);
      long endTime = System.currentTimeMillis();
      rs.enter(endTime-startTime);
    }
    txn.commit();txn.sync();
    if(delay != 0){
      try{
        Thread.sleep(delay);
      }catch(InterruptedException e){
      }
    }
  }

}

