/**
 * 
 */
package code.simulator.protocolFilters;

import java.util.*;
import java.util.Map.Entry;

import code.AcceptVV;
import code.ObjId;
import code.SubscriptionSet;
import code.branchDetecting.BranchID;
import code.simulator.Certificate;
import code.simulator.Hash;
import code.simulator.HashedVV;
import code.simulator.IrisDataObject;
import code.simulator.IrisNode;
import code.simulator.NamespaceWatch;
import code.simulator.SimPreciseInv;
import code.simulator.agreement.AgreementInstance;
import code.simulator.agreement.AgreementInterface;
import code.simulator.agreement.AgreementMonitor;
import code.simulator.agreement.AgreementParams;
import code.simulator.agreement.CommunicationChannel;
import code.simulator.agreement.DecisionProof;
import code.simulator.agreement.InstanceID;
import code.simulator.agreement.ListAgreementInstance;
import code.simulator.agreement.ListDecisionProof;
import code.simulator.agreement.Tuple;
import code.simulator.agreement.ValueDecisionProof;
import code.simulator.agreement.Vote;
import code.simulator.agreement.VoteFactory;
import code.simulator.checkpoint.Checkpoint;
import code.simulator.store.StoreEntry;

/**
 * @author princem
 */
public class AgreementGCProtocol implements NamespaceWatch, AgreementMonitor, CommunicationChannel{

  /**
   * ObjIds of the objects used to communicate in this protocol
   * gcProposals store the proposals given by each node for each other faulty node
   *    - if we haven't received a proposal from a given node for a given fault and we receive a synchronization packet, we 
   *    notify that node by calling the receivePOM function at that node and discard that application of that sync
   * on receiving a globalGCCertificate, we add this node to the evicted node list in our local node
   */
  public static HashMap<Long, ObjId> gcProposalsFiles;
  public static String gcProposalDir = "/iris/gc/proposals/";

  private AgreementInterface curAgreementInstance;
  private byte phase;
  
  private final AgreementParams params;
  private int epoch;
  final long timeout;
  final IrisNode myNode;
  private int msgCount;
  private final VoteFactory voteFactory;
  private HashedVV omitVV;
  private HashedVV endVV;
  private Hash checkpointHash;
  private LinkedList<Entry<Long, Vote>> futureVotes;
  private Checkpoint lastCheckpoint; 

  public AgreementGCProtocol(AgreementParams params, int epoch, long timeout, IrisNode node){
    this.params = params;
    this.myNode = node;
    this.epoch = epoch;
    this.timeout = timeout;
    this.msgCount = 0;
    this.phase = InstanceID.NullPhase;
    futureVotes = new LinkedList<Entry<Long, Vote>>();
    lastCheckpoint = null;

    gcProposalsFiles = new HashMap<Long, ObjId>();
    for(long i = 0; i < IrisNode.numNodes; i++){
      gcProposalsFiles.put(i, new ObjId("/iris/gc/proposals/"+i));
    }
    voteFactory = new VoteFactory(this.params);
    omitVV = null;
    endVV = null;
    checkpointHash = null;
  }

  private ObjId getObjId(){
    return new ObjId(gcProposalDir+"/"+params.getMyID()+"/"+epoch+"/"+(msgCount++));
  }

  synchronized public void startGC(HashedVV omitVV) throws Exception{
    if(phase == InstanceID.NullPhase){
      System.out.println(this.params.getMyID() + " received startGC");
      assert this.phase == InstanceID.NullPhase: this;
      assert !myNode.getOmitVV().includes(omitVV): this;
      assert !myNode.getOmitVV().isAnyPartGreaterThan(omitVV): this;
      //ensure that the omitVV matches oldOmitVV in all the evited nodes
      TimerTask task = new TimerTask(){
        public void run(){
          handleTimeout();
        }
      };
      AgreementInstance.timer.scheduleAtFixedRate(task, this.getTimeOut(), this.getTimeOut());
      advancePhaseAndVote(omitVV);
    }else{
      System.out.println("POSSIBLE PROBLEM: A STARTGC REQUEST WAS IGNORED");
    }
  }
  
  synchronized private void handleTimeout(){
    if(this.curAgreementInstance != null){
      curAgreementInstance.notifyTimeout();
    }
  }

  /**
   * 
   * @param omitVV2: makes sense in the local state
   * @return
   */
  private void advancePhaseAndVote(HashedVV omitVV2) throws Exception{
    InstanceID instanceID;
    Vote vote = null;
    switch(phase){
    case InstanceID.NullPhase: 
      this.phase = InstanceID.OmitVVAgreement;
      instanceID = new InstanceID(epoch+1, phase, params.getMyID());
      vote = new Vote<HashedVV>(instanceID, 0, omitVV2);
      this.curAgreementInstance = new ListAgreementInstance(new InstanceID(instanceID.getEpoch(), instanceID.getPhase(), InstanceID.NullNode), this);
      break;
      
    case InstanceID.OmitVVAgreement: 
      this.phase = InstanceID.EndVVAgreement;
      instanceID = new InstanceID(epoch+1, phase, params.getMyID());
      omitVV = omitVV2;
      Checkpoint chkPt = myNode.generateNewCheckpoint(omitVV2, epoch+1);
      if(!omitVV.equals(chkPt.omitVV))
      {
        System.err.println(omitVV + "\n" + chkPt.omitVV);
        System.out.println(chkPt.lastWrite);
        System.out.println(chkPt.unverifiableWrites);
      }
      vote = new Vote<HashedVV>(instanceID, 0, (HashedVV)chkPt.endVV);
      this.curAgreementInstance = new ListAgreementInstance(new InstanceID(instanceID.getEpoch(), instanceID.getPhase(), InstanceID.NullNode), this);
      break;
      
    case InstanceID.EndVVAgreement: 
      this.phase = InstanceID.CheckpointAgreement;
      endVV = omitVV2;
      instanceID = new InstanceID(epoch+1, phase, InstanceID.NullNode);
      Checkpoint chkPt1 = myNode.generateNewCheckpoint(this.omitVV, epoch+1, endVV);
      if(!omitVV.equals(chkPt1.omitVV))
      {
        System.err.println(omitVV + "\n" + chkPt1.omitVV);
        System.out.println(chkPt1.lastWrite);
        System.out.println(chkPt1.unverifiableWrites);
      }
      
      assert this.lastCheckpoint == null;
      this.lastCheckpoint = chkPt1;
      this.checkpointHash = chkPt1.getHash();
//      System.out.println("CHECKPOINT hash : " + this.checkpointHash.toLongString());
//      System.out.println("CHECKPOINT : " + chkPt1);
//      System.out.println("-------------------");
//      System.out.println("CHECKPOINT byte : " +Arrays.toString(chkPt1.obj2Bytes()));

      vote = new Vote<Hash>(instanceID, 0, this.checkpointHash);
      this.curAgreementInstance = new AgreementInstance(instanceID, this);
      break;
      
    default: assert false: this;
    }
    this.handleNetworkVote(params.getMyID(), vote);
    this.broadcast(vote);
    // handle future votes
    handleFutureVotes();
  }

  private void handleFutureVotes(){
    if(!futureVotes.isEmpty()){
      LinkedList<Entry<Long, Vote>> oldFutureVotes = futureVotes;
      futureVotes = new LinkedList<Entry<Long, Vote>>();
      for(Entry<Long, Vote> e: oldFutureVotes){
        this.handleNetworkVote(e.getKey(), e.getValue());
      }
    }
  }
  
  public final AgreementParams getParams(){
    return params;
  }

  synchronized public void notifyCheckpoint(int epoch){
    assert this.epoch <= epoch;
    System.out.println(this.myNode.getBranchID() + " received checkpoint with epoch " + epoch);
    this.epoch = epoch; 
    AgreementInstance.timer.cancel();
    AgreementInstance.timer = new Timer();
    msgCount = 0;
    this.phase = InstanceID.NullPhase;
    //perhaps should clean up the earlier writes
    futureVotes.clear();
    this.lastCheckpoint = null;
  }

  /**
   * handle a new decision proof from the network...see if its valid and call local decision proof handler too
   * @param dp
   */
  private void handleNetworkDecisionProof(DecisionProof dp){
    if(dp.getInstanceID().getEpoch() != (this.epoch+1) || dp.getInstanceID().getPhase() != this.phase){
      return; 
    }
    
    if(dp.verify()){
      // notify the corresponding agreementMonitor
      if(phase == InstanceID.CheckpointAgreement){
          System.out.println(this.myNode.getBranchID().getIDint() 
              + " received a CheckpointAgreement vote "  
              + ": DecisionProof : " + dp);
        this.notifyDecision(dp.getInstanceID(), dp);
      }else if(phase == InstanceID.OmitVVAgreement || phase == InstanceID.EndVVAgreement){
        ((AgreementMonitor)this.curAgreementInstance).notifyDecision(dp.getInstanceID(), dp);
      }else{
        assert false;
        return;
      }

    }
    
  }
  
  private void handleNetworkVote(long sender, Vote v){
    if(v.getInstanceID().getEpoch() != (this.epoch+1) || phase > v.getInstanceID().getPhase()){ // invalid epoch or (past) phase
      return; 
    }
    
    if(phase == InstanceID.CheckpointAgreement){
      System.out.println(this.myNode.getBranchID().getIDint() 
          + " received a CheckpointAgreement vote from " + sender 
          + ": " + v);
    }
    if(phase == InstanceID.NullPhase && v.getInstanceID().getPhase() == InstanceID.OmitVVAgreement){ // initiate agreement
      assert v.getData() instanceof HashedVV;
      try{
        HashedVV hvv = (HashedVV)v.getData();
        this.startGC(hvv);
      }catch(Exception e){
        e.printStackTrace();
        assert false;
      }
    }else if(phase != v.getInstanceID().getPhase()){
      futureVotes.add(new Tuple<Long, Vote>(sender, v));
      return;
    }

    assert(curAgreementInstance != null);

    assert v != null;
    // now based on the phase, ensure that the vote is legitimate and if so, forward it to the agreement cluster
    if(phase == InstanceID.EndVVAgreement || phase == InstanceID.OmitVVAgreement){
      if(v.getData() != null){
      assert v.getData() instanceof HashedVV:v.getData();
      HashedVV hdvv = (HashedVV)v.getData();
      try{
        HashedVV newHDVV = this.myNode.remapHashedDependencyVV(hdvv, false); //if the vote is valid, deliver it
        if(newHDVV.getSize() == hdvv.getSize()){ // deliver the vote if it can be successfully mapped
          this.curAgreementInstance.receiveVote(sender, v);
        }
      }catch(Exception e){
        e.printStackTrace();
        assert false;
        System.exit(0);
      }
      }else{
        this.curAgreementInstance.receiveVote(sender, v);
      }

    }else if(phase == InstanceID.CheckpointAgreement){
      assert v.getData() instanceof Hash;
      Hash h = (Hash)v.getData();
      if(h.equals(this.checkpointHash)){//if the vote is valid, deliver it
        this.curAgreementInstance.receiveVote(sender, v);
      } else {
        System.out.println("VOTE hash doesn't match, hash: " + this.checkpointHash + " , vote : " + v);
      }
    }
  }
  
  /**
   * receive new vote from the communication channel
   */
  synchronized public void notifyWrite(int epoch, ObjId o, StoreEntry se){
    assert epoch <= epoch;
    assert o.getPath().startsWith(gcProposalDir);
    assert se.getData() instanceof IrisDataObject:se + "\n" + o;
    Object obj = ((IrisDataObject)se.getData()).getObject();
    if(obj != null){
      if(obj instanceof DecisionProof){
        handleNetworkDecisionProof((DecisionProof)obj);
      }else if(obj instanceof Vote){
        handleNetworkVote(se.getAcceptStamp().getNodeId().getIDint(), (Vote)obj);
      }else{
        assert false;
      }
    }else{
      assert false;
    }
  }

  /**
   * receive decision proof from the agreement instance or from the channel
   */
  synchronized public void notifyDecision(InstanceID instanceID, DecisionProof proof){
    if(instanceID.getEpoch() == (this.epoch+1) && phase != InstanceID.NullPhase && 
        instanceID.getPhase() == this.phase && proof.verify()){
      assert curAgreementInstance != null;
      this.curAgreementInstance.receiveDecisionProof(proof);

      if(phase == InstanceID.OmitVVAgreement || phase == InstanceID.EndVVAgreement){
        assert(proof instanceof ListDecisionProof);
        ListDecisionProof ldf = (ListDecisionProof)proof;
        try{
          HashedVV vv = getVV(ldf);
          System.out.println("DECIDED " + phase + " vv : " + vv);
          this.advancePhaseAndVote(vv);
        }catch(Exception e){
          e.printStackTrace();
          assert false;
        }

      }else if(phase == InstanceID.CheckpointAgreement){
        assert (proof instanceof ValueDecisionProof);
        try{
          ValueDecisionProof vdf = (ValueDecisionProof)proof;
          // this checkpoint generation might fail if someone else sneaks in between minimalVV call and checkpoint generation
          Checkpoint chkPt =  
            myNode.generateNewCheckpoint(this.omitVV, epoch+1, endVV);
          //hack for accomodating faulty nodes
          System.out.println(this.params.getMyID() + " created certificate");

          if(myNode.getBranchID().getIDint() != -1){
            Checkpoint chkPt1 = lastCheckpoint;
            if(!chkPt.omitVV.equals(chkPt1.omitVV))
            {
              System.err.println(chkPt.omitVV + "\n" + chkPt1.omitVV +"\n" + omitVV);
              System.err.println("cur ChkPt");
              System.out.println(chkPt.lastWrite);
              System.out.println(chkPt.unverifiableWrites);
              System.err.println("prev ChkPt");
              System.out.println(chkPt1.lastWrite);
              System.out.println(chkPt1.unverifiableWrites);
            }
            assert chkPt.omitVV.equals(chkPt1.omitVV): "\n" + omitVV+ "\n" + chkPt.omitVV + "\n" + chkPt1.omitVV +"\n" + ((HashedVV)chkPt.omitVV).getHashes() + "\n" + ((HashedVV)chkPt1.omitVV).getHashes() + "\n" + ((HashedVV)omitVV).getHashes();
            assert chkPt.epochCount == chkPt1.epochCount: chkPt.epochCount + " " + chkPt1.epochCount;
            assert chkPt.lastWrite.equals(chkPt1.lastWrite): chkPt.lastWrite + " " + chkPt1.lastWrite;
            if(!chkPt.objectStore.equals(chkPt1.objectStore)){
              int num = code.simulator.checkpoint.unit.SimulatorMechanismCheckpointUnit.compareStore(chkPt.objectStore, myNode.getLogTest(), chkPt1.objectStore, myNode.getLogTest(), SubscriptionSet.makeEmptySet(), true);
              System.out.println(this.params.getMyID() + " found " + num + "conflicts\n" + myNode.rawRead(new ObjId("/0/10")));
              System.out.println(this.params.getMyID() + " found " + num + "conflicts\n" + myNode.rawRead(new ObjId("/0/89")));
            }
            assert chkPt.unverifiableWrites.equals(chkPt1.unverifiableWrites): chkPt.unverifiableWrites + " " + chkPt1.unverifiableWrites;
            assert chkPt.endVV.equals(chkPt1.endVV): "\n" + chkPt.endVV + "\n" + chkPt1.endVV;
            assert chkPt.flagMap.equals(chkPt1.flagMap): chkPt.flagMap + " " + chkPt1.flagMap;
            assert chkPt.acceptableBranches.equals(chkPt1.acceptableBranches): chkPt.acceptableBranches + " " + chkPt1.acceptableBranches;
            assert Arrays.equals(chkPt.obj2Bytes(), chkPt1.obj2Bytes()):"\n" +Arrays.toString(chkPt.obj2Bytes()) + "\n chkPt1.obj2Bytes() \n" + Arrays.toString(chkPt1.obj2Bytes());
            assert chkPt.getHash().equals(chkPt1.getHash()):"\n" +Arrays.toString(chkPt.obj2Bytes()) + "\n chkPt1.obj2Bytes() \n" + Arrays.toString(chkPt1.obj2Bytes());
            
            assert this.lastCheckpoint.equals(chkPt):"\n" + this.lastCheckpoint +"\n" + chkPt;
            assert this.checkpointHash.equals(chkPt.getHash()):this.checkpointHash  +"\n" + chkPt.getHash() +"\n" + vdf.getDecidedValue().getData()+"\n" + this.lastCheckpoint +"\n" + chkPt;
            assert vdf.getDecidedValue().getData().equals(chkPt.getHash()):vdf.getDecidedValue() +"\n" + chkPt.getHash();
            myNode.applyNewLocalCertificate(new GCCertificate(chkPt, omitVV, endVV));
          }
          System.out.println(this.params.getMyID() + " Garbage collection complete");
        }catch(Exception e){
          e.printStackTrace();
          assert false;
        }
      }
    }
  }

  /**
   * returns the locally mapped minimal VV using the ldf
   * @param ldf
   * @return
   */
  private HashedVV getVV(ListDecisionProof ldf) throws Exception{

    LinkedList<HashedVV> hashedVVs = new LinkedList<HashedVV>();
    for(DecisionProof dp: ldf.getProofs()){
      assert dp instanceof ValueDecisionProof;
      Object d = ((ValueDecisionProof)dp).getDecidedValue().getData();
      if(d != null){
        hashedVVs.add((HashedVV)d);
      }
    }
    return myNode.getMaxVV(hashedVVs);
  }

  public CommunicationChannel getChannel(){
    return this;
  }

  public long getTimeOut(){
    return this.timeout;
  }

  public VoteFactory getVoteFactory(){
    return this.voteFactory;
  }

  public void addNamespaceWatch(NamespaceWatch mr, SubscriptionSet ss){
    this.myNode.addNamespaceWatch(mr, ss);
  }

  public void broadcast(Object o){
    this.myNode.write(this.getObjId(), new IrisDataObject(o));
  }

  @Override
  public String toString(){
    return "AgreementGCProtocol [\ncheckpointHash=" + checkpointHash
        + ", \ncurAgreementInstance=" + curAgreementInstance + ", \nendVV=" + endVV
        + ", \nepoch=" + epoch + ", \nfutureVotes=" + futureVotes + ", \nomitVV="
        + omitVV + ", \nphase=" + InstanceID.phaseString(phase) + "]";
  }

  public AgreementInterface getCurAgreementInstanceTest(){
    return curAgreementInstance;
  }

  public byte getPhaseTest(){
    return phase;
  }

  public HashedVV getOmitVVTest(){
    return omitVV;
  }

  public HashedVV getEndVVTest(){
    return endVV;
  }


}
