/**
 * 
 */
package code.simulator.protocolFilters;

import java.util.*;

import code.AcceptVV;
import code.ObjId;
import code.NodeId;
import code.SubscriptionSet;
import code.CounterVV;
import code.PreciseInv;
import code.branchDetecting.*;
import code.simulator.Certificate;
import code.simulator.IrisDataObject;
import code.simulator.IrisNode;
import code.simulator.POMRemap;
import code.simulator.ProofOfMisbehavior;
import code.simulator.SimPreciseInv;
import code.simulator.agreement.Tuple;
import code.simulator.store.StoreEntry;

/**
 * @author princem
 * This class provides the implementation of the algorithm for distributing and collecting eviction certificates. 
 * The certificates are written at pre-specified objects/filenames 
 * This class enforces the following features:
 *      - we create an eviction proposal on receiving the first POM
 *      - we force nodes to create eviction proposal by refusing to accept writes from other nodes in the same epoch
 *      - we invoke consensus to convert eviction proposals into eviction certificate
 *      - we discard eviction proposals on collecting eviction certificate 
 */
public class CopyOfEvictionProtocol implements SyncFilter, POMFilter, DiscardedWriteFilter{

  /**
   * POMs for various nodes against whom the eviction is currently active
   */
  HashMap<Long, ProofOfMisbehavior> poms;
  
  HashMap<Long, TreeSet<EvictionProposal>> evictionProposals;
  HashMap<Long, TreeSet<PreciseInv>> acceptableWrites;
  
  HashMap<Long, EvictionCertificate> evictionCertificates;
  LinkedList<EvictionProposal> pendingEvictionCertificates;

  int numNodes = -1;
  /**
   * ObjIds of the objects used to communicate in this protocol
   * evictionProposals store the proposals given by each node for each other faulty node
   *    - if we haven't received a proposal from a given node for a given fault and we receive a synchronization packet, we 
   *    notify that node by calling the receivePOM function at that node and discard that application of that sync
   * on receiving a globalEvictionCertificate, we add this node to the evicted node list in our local node
   */
  public static HashMap<Long, ObjId> evictionProposalsFiles;
  public static String evictionProposalDir = "/iris/eviction/proposals/";
  
  public CopyOfEvictionProtocol(){
    poms = new HashMap<Long, ProofOfMisbehavior>();
    evictionProposalsFiles = new HashMap<Long, ObjId>();
    for(long i = 0; i < IrisNode.numNodes; i++){
      evictionProposalsFiles.put(i, new ObjId("/iris/eviction/proposals/"+i));
    }
    evictionProposals = new HashMap<Long, TreeSet<EvictionProposal>>();
    acceptableWrites = new HashMap<Long, TreeSet<PreciseInv>>();
    evictionCertificates = new HashMap<Long, EvictionCertificate>();
    pendingEvictionCertificates = new LinkedList<EvictionProposal>();

  }
  
  public CopyOfEvictionProtocol(int numNodes){
    this();
    this.numNodes = numNodes;
  }
  
  /**
   * observe any eviction certificates 
   * @throws Exception 
   */
  public void notifyEpochSync(IrisNode myNode, Certificate c) throws Exception{
    boolean updated = false;
    
    if (c instanceof EvictionCertificate){
      EvictionCertificate ec = (EvictionCertificate)c;
      if(!evictionCertificates.containsKey(ec.nodeId.getIDint())){
        long id = ec.getNodeId().getIDint();
        evictionCertificates.put(id, ec);
        //update eviction proposals and acceptableWrites 
        evictionProposals.remove(id);
        poms.remove(id);
        acceptableWrites.get(id).clear();
        for(BranchID bid: ec.getForkedWrites().keySet()){
          acceptableWrites.get(id).addAll(ec.getForkedWrites().get(bid));
        }
        updated = true;
      }
    }

    if(updated){
      updateEvictionProposalsFile(myNode);
    }
  }
  
  public HashMap<Long, TreeSet<EvictionProposal>> getEvictionProposals(){
    return evictionProposals;
    
  }
  
  private void updateEvictionProposalsFile(IrisNode myNode){
    ObjId objId = evictionProposalsFiles.get(myNode.getBranchID().getIDint());
    HashMap<Long, EvictionProposal> evictionProposalsList = new HashMap<Long, EvictionProposal>(); 
//      (HashMap<Long, EvictionProposal>)myNode.read(objId);
//    if(evictionProposalsList == null){
//      evictionProposalsList = new HashMap<Long, EvictionProposal>();
//    }     
//    
//    // remove any proposals for which we have obtained certificate
//    for(Long l: evictionProposalsList.keySet()){
//      if(this.evictionCertificates.containsKey(l)){
//        evictionProposalsList.remove(l);
//      }
//    }
    
    for(Long l: evictionProposals.keySet()){
//      if(!evictionProposalsList.containsKey(l)){
        for(EvictionProposal ep: evictionProposals.get(l)){
          if(ep.bid.getIDint() == myNode.getBranchID().getIDint()){
            evictionProposalsList.put(l, ep);
            break;
          }
        }
//      }
      assert evictionProposalsList.containsKey(l);
    }
    myNode.write(objId, new IrisDataObject(evictionProposalsList));
  }
  
  /* (non-Javadoc)
   * @see code.simulator.SyncFilter#notifySync(code.simulator.BranchID, code.AcceptVV, code.AcceptVV, code.simulator.Checkpoint, java.util.TreeSet, code.AcceptVV)
   */
  public boolean notifySync(IrisNode myNode, AcceptVV myCVV, BranchID otherNodeId,
      //, AcceptVV otherCVV,
//      Checkpoint lastCheckpoint, 
      TreeSet<SimPreciseInv> newUpdates
//      ,
//      AcceptVV omitVV
      , Collection<Tuple<ObjId, TreeSet<StoreEntry>>> addedEntries
      ){
    
    // ensure that each new branch is accompanied by a POM [ignore for now]
    // ensure that the sender has as many evictionProposals as we do
    // - if not, then call notifyPOM on the sender and abort the syn
    
      // if eviction is active for someone
      // then ensure that either this packet contains a proposal for the faulty node or an eviction certificate
    // otherwire reject and notify sender
    // further, ensure that all new writes for the faulty node are covered by at least one proposal
    // in addition, for any new proposal, add the proposal

    //1: scan the packet for any eviction certificates/eviction proposals/
    //2: ensure that (a) the sender has provided us with proposals for all evictions
    //               (b) each write from a currently known faulty node is accompanied by a proposal covering it
    //               (c) TODO: no new write from an evicted node is accepted

    for(SimPreciseInv spi: newUpdates){
      assert spi != null;
      assert spi.getObjId() != null:spi;
      assert spi.getObjId().getPath() != null:spi;
      if(spi.getObjId().getPath().startsWith(CopyOfEvictionProtocol.evictionProposalDir)){
        //HashMap<Long, EvictionProposal> evictionProposal = (HashMap<Long, EvictionProposal>)spi.data;
        assert spi.getData() instanceof IrisDataObject;
        IrisDataObject irisObj = (IrisDataObject)spi.getData();
        HashMap<Long, EvictionProposal> evictionProposal = (HashMap<Long, EvictionProposal>)irisObj.getObject();
        //process eviction proposal
        for(Long l: evictionProposal.keySet()){
          if(evictionCertificates.containsKey(l) || !evictionProposal.get(l).pom.verify()){
            continue;
          }
          if(!evictionProposals.containsKey(l)){
            // found a new eviction proposal that we didn't have before
            receivePOM(myNode, evictionProposal.get(l).pom);
          }
          this.handleEvictionProposal(myNode, evictionProposal.get(l));

        }
      }
    }
    
    if(!poms.isEmpty() || !evictionCertificates.isEmpty()){
      // now do (2)
      // (a)
      for(Long l: evictionProposals.keySet()){
        boolean found = false;
        for(EvictionProposal ep: evictionProposals.get(l)){
          if(ep.bid.getIDint() == otherNodeId.getIDint()){
            found = true;
            break;
          }
        }
        if(!found){
          //otherNode.notifyNewForks(myNode.getPOMMap());
          return false;
        }
      }
      
  
      // now 2(b)
      for(SimPreciseInv spi: newUpdates){
        Long l = spi.getNodeId().getIDint();
        if(acceptableWrites.containsKey(l)){
          if(!acceptableWrites.get(l).contains(spi)){
            //assert false : acceptableWrites + " ## "  + spi;
            return false;
          }
        }
        
      }
    }
    return true;
  }
  
  
  /**
   * called by the remote node when a new POM is created or by the local node when it creates a new POM
   */
  public void receivePOM(IrisNode myNode, ProofOfMisbehavior pom){
    // write pom to the list of my POMs'
    long id = pom.getParent().getNodeId().getIDint();
    if(evictionCertificates.containsKey(id)){
      return;
    }    
    System.out.println("new POM received");
    if(!poms.containsKey(id)){
      System.out.println("Eviction Protocol Initiated at " + myNode.getBranchID());
      // initiate eviction
      HashMap<BranchID, TreeSet<PreciseInv>> forkedWrites = myNode.getAcceptedForkedWrites(((BranchID)pom.getParent().getNodeId()).getBaseId());
      EvictionProposal ep = new EvictionProposal(forkedWrites, pom, myNode.getBranchID().getBaseId());
      poms.put(id, pom);
      assert !evictionProposals.containsKey(id);
      evictionProposals.put(id, new TreeSet<EvictionProposal>());
      acceptableWrites.put(id, new TreeSet<PreciseInv>());
      evictionProposals.get(id).add(ep);
      for(BranchID bid: forkedWrites.keySet()){
        acceptableWrites.get(id).addAll(forkedWrites.get(bid));
      }
      this.updateEvictionProposalsFile(myNode);
    }
  }

  /**
   * see if any optimized writes were thrown away because of a concurrent agreement instance
   */
  public void notifyDiscarded(IrisNode myNode, LinkedList<SimPreciseInv> spiSet){
    // if any eviction proposals for active requests were discarded, then reissue them
    
    for(SimPreciseInv spi:spiSet){
      if(spi.getObjId().equals(evictionProposals.get(myNode.getBranchID().getIDint()))){
        this.updateEvictionProposalsFile(myNode);
        break;
      }
    }
  }
  
  /**
   * trigger the agreement protocol or wait
   * @param ep
   */
  public void handleEvictionProposal(IrisNode myNode, EvictionProposal ep){
    long id = ep.pom.getParent().getNodeId().getIDint();
    if(evictionCertificates.containsKey(id)){
      return;
    }
    System.out.println("Eviction Proposal Recieved at " + myNode.getBranchID());
    if(!evictionProposals.get(id).contains(ep) && ep.bid.getIDint() != ep.pom.getParent().getNodeId().getIDint()){
      evictionProposals.get(id).add(ep);
      assert poms.containsKey(id);
      for(BranchID bid: ep.forkedWrites.keySet()){
        acceptableWrites.get(id).addAll(ep.forkedWrites.get(bid));
      }
      if(evictionProposals.get(id).size() == this.numNodes){
        this.pendingEvictionCertificates.add(ep);
      }
    }
  }
  
  public void syncOver(IrisNode myNode, boolean success) throws Exception{
    if(success){
    for(EvictionProposal ep:pendingEvictionCertificates){
      this.createEvictionCertificate(myNode, ((BranchID)ep.pom.getParent().getNodeId()));
    }
    this.pendingEvictionCertificates.clear();
    }
  }

  public void createEvictionCertificate(IrisNode myNode, NodeId bid) throws Exception{
    
    // first merge all the eviction proposals that I have received
    // then create a certificate consisting of all the writes from the faulty node that nodes must accept
    // and finally install that certificate and rollback our state
    
    long id = bid.getIDint();
    if(!this.evictionCertificates.containsKey(id)){
      assert evictionProposals.containsKey(id);
      System.out.println("Eviction Certificate is created at " + myNode.getBranchID() + " for " + bid);
      HashMap<BranchID, TreeSet<PreciseInv>> acceptableWrites = new HashMap<BranchID, TreeSet<PreciseInv>>();
      for(EvictionProposal ep: evictionProposals.get(id)){
        for(BranchID bid1: ep.forkedWrites.keySet()){
          if(!acceptableWrites.containsKey(bid1)){
            acceptableWrites.put(bid1, new TreeSet<PreciseInv>());
          }
          acceptableWrites.get(bid1).addAll(ep.forkedWrites.get(bid1));
        }
      }

      if(bid instanceof BranchID){
        bid = ((BranchID)bid).getBaseId();
      }
      EvictionCertificate ec = new EvictionCertificate(bid, acceptableWrites, myNode.getEpoch()+1);
      myNode.applyNewLocalCertificate(ec);
    }
    
  }

  public void receivePOMRemap(IrisNode myNode, POMRemap pomRemap){
    // TODO Auto-generated method stub
    
  }
}
