/**
 * 
 */
package code.simulator.protocolFilters;

import java.util.*;

import code.AcceptVV;
import code.ObjId;
import code.NodeId;
import code.SubscriptionSet;
import code.CounterVV;
import code.PreciseInv;
import code.branchDetecting.*;
import code.simulator.Certificate;
import code.simulator.Hash;
import code.simulator.IrisDataObject;
import code.simulator.IrisNode;
import code.simulator.POMRemap;
import code.simulator.ProofOfMisbehavior;
import code.simulator.SecureSimPreciseInv;
import code.simulator.SimPreciseInv;
import code.simulator.agreement.Tuple;
import code.simulator.store.StoreEntry;

/**
 * @author princem
 * This class provides the implementation of the algorithm for distributing and collecting eviction certificates. 
 * The certificates are written at pre-specified objects/filenames 
 * This class enforces the following features:
 *      - we create an eviction proposal on receiving the first POM
 *      - we force nodes to create eviction proposal by refusing to accept writes from other nodes in the same epoch
 *      - we invoke consensus to convert eviction proposals into eviction certificate
 *      - we discard eviction proposals on collecting eviction certificate 
 */
public class EvictionProtocol implements SyncFilter, POMFilter, DiscardedWriteFilter{

  HashMap<Long, TreeSet<Long>> evictionProposalsProvider;
  HashMap<Long, EvictionProposal> evictionProposals;
  HashMap<Long, TreeSet<Hash>> acceptableWrites;
  
  LinkedList<ProofOfMisbehavior> newPOMS;

  int numNodes = -1;
  /**
   * ObjIds of the objects used to communicate in this protocol
   * evictionProposals store the proposals given by each node for each other faulty node
   *    - if we haven't received a proposal from a given node for a given fault and we receive a synchronization packet, we 
   *    notify that node by calling the receivePOM function at that node and discard that application of that sync
   * on receiving a globalEvictionCertificate, we add this node to the evicted node list in our local node
   */
  public static HashMap<Long, ObjId> evictionProposalsFiles;
  public static String evictionProposalDir = "/iris/eviction/proposals/";
  
  public EvictionProtocol(){
    evictionProposalsFiles = new HashMap<Long, ObjId>();
    for(long i = 0; i < IrisNode.numNodes; i++){
      evictionProposalsFiles.put(i, new ObjId("/iris/eviction/proposals/"+i));
    }
    
    evictionProposals = new HashMap<Long, EvictionProposal>();
    evictionProposalsProvider = new HashMap<Long, TreeSet<Long>>();
    acceptableWrites = new HashMap<Long, TreeSet<Hash>>();
    newPOMS = new LinkedList<ProofOfMisbehavior>();
  }
  
  public EvictionProtocol(int numNodes){
    this();
    this.numNodes = numNodes;
  }
  
  /**
   * observe any eviction certificates 
   * @throws Exception 
   */
  public void notifyEpochSync(IrisNode myNode, Certificate certs) throws Exception{
    assert myNode.getEpoch() == certs.getEpoch();
    newPOMS.clear();
    if(!this.evictionProposals.isEmpty()){
      evictionProposals.clear();
      acceptableWrites.clear();   
      evictionProposalsProvider.clear();
      updateEvictionProposalsFile(myNode);
    }
  }
  
  public HashMap<Long, TreeSet<Long>> getEvictionProposals(){
    return evictionProposalsProvider;
    
  }
  
  private void updateEvictionProposalsFile(IrisNode myNode){
    ObjId objId = evictionProposalsFiles.get(myNode.getBranchID().getIDint());
    HashMap<Long, EvictionProposal> evictionProposalsList = new HashMap<Long, EvictionProposal>(); 

    for(Long l: evictionProposals.keySet()){
      EvictionProposal ep = evictionProposals.get(l);
      assert ep.bid.getIDint() == myNode.getBranchID().getIDint();
      evictionProposalsList.put(l, ep);
      assert evictionProposalsList.containsKey(l);
    }
    myNode.write(objId, new IrisDataObject(evictionProposalsList));
  }
  
  /* (non-Javadoc)
   * @see code.simulator.SyncFilter#notifySync(code.simulator.BranchID, code.AcceptVV, code.AcceptVV, code.simulator.Checkpoint, java.util.TreeSet, code.AcceptVV)
   */
  public boolean notifySync(IrisNode myNode, AcceptVV myCVV, BranchID otherNodeId,
      //, AcceptVV otherCVV,
//      Checkpoint lastCheckpoint, 
      TreeSet<SimPreciseInv> newUpdates
//      ,
//      AcceptVV omitVV
      , Collection<Tuple<ObjId, TreeSet<StoreEntry>>> addedEntries
      ){
    
    // ensure that each new branch is accompanied by a POM [ignore for now]
    // ensure that the sender has as many evictionProposals as we do
    // - if not, then call notifyPOM on the sender and abort the syn
    
      // if eviction is active for someone
      // then ensure that either this packet contains a proposal for the faulty node or an eviction certificate
    // otherwire reject and notify sender
    // further, ensure that all new writes for the faulty node are covered by at least one proposal
    // in addition, for any new proposal, add the proposal

    //1: scan the packet for any eviction certificates/eviction proposals/
    //2: ensure that (a) the sender has provided us with proposals for all evictions
    //               (b) each write from a currently known faulty node is accompanied by a proposal covering it
    //               (c) TODO: no new write from an evicted node is accepted

    newPOMS.clear();
    for(SimPreciseInv spi: newUpdates){
      assert spi != null;
      assert spi.getObjId() != null:spi;
      assert spi.getObjId().getPath() != null:spi;
      if(spi.getObjId().getPath().startsWith(EvictionProtocol.evictionProposalDir)){
        //HashMap<Long, EvictionProposal> evictionProposal = (HashMap<Long, EvictionProposal>)spi.data;
        assert spi.getData() instanceof IrisDataObject;
        IrisDataObject irisObj = (IrisDataObject)spi.getData();
        HashMap<Long, EvictionProposal> evictionProposal = (HashMap<Long, EvictionProposal>)irisObj.getObject();
        //process eviction proposal
        for(Long l: evictionProposal.keySet()){
          
          if(!evictionProposals.containsKey(l)){
            // found a new eviction proposal that we didn't have before
            newPOMS.add(evictionProposal.get(l).pom);
          }
          if(!this.handleEvictionProposal(myNode, evictionProposal.get(l))){
            return false;
          }

        }
      }
    }
    
    if(!this.evictionProposalsProvider.keySet().isEmpty()){
      // now do (2)
      // (a)
      for(Long l: evictionProposalsProvider.keySet()){
        boolean found = false;
        for(Long provider: evictionProposalsProvider.get(l)){
          if(provider == otherNodeId.getIDint()){
            found = true;
            break;
          }
        }
        if(!found){
          return false;
        }
      }
      
  
      // now 2(b)
      for(SimPreciseInv spi: newUpdates){
        Long l = spi.getNodeId().getIDint();
        if(this.evictionProposalsProvider.containsKey(l) || myNode.isEvicted(l) || myNode.isFaulty(l)){
          if(acceptableWrites.containsKey(l)){
            Hash h = ((SecureSimPreciseInv)spi).generateMyHash();
            if(!acceptableWrites.get(l).contains(h)){
              return false;
            }
          }else{
            return false;
          }
        }
        
      }
    }
    return true;
  }
  
  
  /**
   * called by the remote node when a new POM is created or by the local node when it creates a new POM
   */
  public void receivePOM(IrisNode myNode, ProofOfMisbehavior pom){
    // write pom to the list of my POMs'
    long id = pom.getParent().getNodeId().getIDint();
       
    assert !myNode.isEvicted(id);
    
    System.out.println("new POM received");
    if(!this.evictionProposals.containsKey(id)){
      System.out.println("Eviction Protocol Initiated at " + myNode.getBranchID());
      // initiate eviction
      HashMap<BranchID, TreeSet<PreciseInv>> forkedWrites = myNode.getAcceptedForkedWrites(((BranchID)pom.getParent().getNodeId()).getBaseId());
      EvictionProposal ep = new EvictionProposal(forkedWrites, pom, myNode.getBranchID().getBaseId());
      assert !evictionProposals.containsKey(id);
      evictionProposals.put(id, ep);
      if(!evictionProposalsProvider.containsKey(id)){
        evictionProposalsProvider.put(id, new TreeSet<Long>());
      }
      if(!acceptableWrites.containsKey(id)){
        acceptableWrites.put(id, new TreeSet<Hash>());
      }
      evictionProposalsProvider.get(id).add(myNode.getBranchID().getIDint());
      for(BranchID bid: forkedWrites.keySet()){
        for(PreciseInv pi: forkedWrites.get(bid)){
          Hash h = ((SecureSimPreciseInv)pi).generateMyHash();
          acceptableWrites.get(id).add(h);
        }
      }
      this.updateEvictionProposalsFile(myNode);
    }
  }

  /**
   * see if any optimized writes were thrown away because of a concurrent agreement instance
   */
  public void notifyDiscarded(IrisNode myNode, LinkedList<SimPreciseInv> spiSet){
    // do nothing
  }
  
  /**
   * trigger the agreement protocol or wait
   * @param ep
   */
  public boolean handleEvictionProposal(IrisNode myNode, EvictionProposal ep){
    long id = ep.pom.getParent().getNodeId().getIDint();
    System.out.println("Eviction Proposal Recieved at " + myNode.getBranchID());
    if(!myNode.getFaultyNodes().contains(id) || myNode.isEvicted(id)){
      return false;
    }
    if(!evictionProposalsProvider.containsKey(id)){
      evictionProposalsProvider.put(id, new TreeSet<Long>());
    }
    if(!acceptableWrites.containsKey(id)){
      acceptableWrites.put(id, new TreeSet<Hash>());
    }
    if(!evictionProposalsProvider.get(id).contains(ep.bid.getIDint()) && 
        ep.bid.getIDint() != ep.pom.getParent().getNodeId().getIDint()){
      evictionProposalsProvider.get(id).add(ep.bid.getIDint());
      for(BranchID bid: ep.forkedWrites.keySet()){
        for(PreciseInv pi: ep.forkedWrites.get(bid)){
          Hash h = ((SecureSimPreciseInv)pi).generateMyHash();
          acceptableWrites.get(id).add(h);
        }
      }
      
    }
    
    return true;
  }
  
  public void syncOver(IrisNode myNode, boolean success) throws Exception{
    if(success){
      for(ProofOfMisbehavior pom: newPOMS){
        if(!evictionProposals.containsKey(pom.getParent().getNodeId().getIDint())){
          receivePOM(myNode, pom);
        }
      }
    }
  }

  public void receivePOMRemap(IrisNode myNode, POMRemap pomRemap){
    
  }

 
}
