/**
 * 
 */
package code.simulator.protocolFilters;

import java.util.*;

import code.AcceptVV;
import code.CounterVV;
import code.NodeId;
import code.ObjId;
import code.branchDetecting.BranchID;
import code.simulator.Certificate;
import code.simulator.Hash;
import code.simulator.HashedVV;
import code.simulator.IrisDataObject;
import code.simulator.IrisNode;
import code.simulator.NamespaceWatch;
import code.simulator.POMRemap;
import code.simulator.ProofOfMisbehavior;
import code.simulator.SimPreciseInv;
import code.simulator.agreement.Tuple;
import code.simulator.checkpoint.Checkpoint;
import code.simulator.store.StoreEntry;

/**
 * @author princem
 * This class provides the implementation of the algorithm for distributing and collecting gc certificates. 
 * The certificates are written at pre-specified objects/filenames 
 * This class enforces the following features:
 *      - we create an gc proposal on receiving the first POM
 *      - we force nodes to create gc proposal by refusing to accept writes from other nodes in the same epoch
 *      - we invoke consensus to convert gc proposals into gc certificate
 *      - we discard gc proposals on collecting gc certificate 
 *      
 *      ASSUMPTIONS: CONCURRENT GC WILL NOT BE ISSUED
 *      ONLY ONE ATTEMPT TO GC IN A DAY
 *      ONLY ONE VV FOR A GIVEN DAY
 *      
 */
public class LadderedGCProtocol implements POMFilter, GCProtocol{

  private int dayTime;

  /**
   * to keep all the proposals we have received from a given node--nodes may be faulty and 
   * send multiple proposals
   */
  private TreeSet<SimPreciseInv> gcProposals;

  /**
   * essentially a list for maintaining the proposals/certificates taht I have gathered
   */
  private HashMap<Integer, Object> gcCertificates;

  /**
   * the max proposed VV so far
   */
  private HashedVV lastProposedVV;

  /**
   * Proposed/Agreed/None
   */
  private byte phase;

  public static final byte Proposed = 0x00;
  public static final byte Agreed = 0x01;
  public static final byte None = 0x02;

  /**
   * myNode 
   */
  IrisNode node;

  /**
   * Checkpoint corresponding to the last checkpoint
   */
  Checkpoint chkPt;

  /**
   * ObjIds of the objects used to communicate in this protocol
   * gcProposals store the proposals given by each node for each other faulty node
   *    - if we haven't received a proposal from a given node for a given fault and we receive a synchronization packet, we 
   *    notify that node by calling the receivePOM function at that node and discard that application of that sync
   * on receiving a globalLadderedGCCertificate, we add this node to the evicted node list in our local node
   */
  public static HashMap<Long, ObjId> gcProposalsFiles;
  public static String gcProposalDir = "/iris/gc/proposals/";
  private int numResponse;
  
  /**
   * temporary data structure to hold invals until syncover is called
   */
  private TreeSet<SimPreciseInv> invals = new TreeSet<SimPreciseInv>();

  public LadderedGCProtocol(int numResponse, IrisNode node){
    this.numResponse = numResponse;
    this.node = node;
    gcProposalsFiles = new HashMap<Long, ObjId>();
    for(long i = 0; i < IrisNode.numNodes; i++){
      gcProposalsFiles.put(i, new ObjId("/iris/gc/proposals/"+i));
    }
    gcProposals = new TreeSet<SimPreciseInv>();
    gcCertificates = new HashMap<Integer, Object>();
    lastProposedVV = new HashedVV();
    phase = None;
    chkPt = null;
    dayTime = 0;
  }

  public synchronized void advanceDayTime(int dayTime){
    assert dayTime >= this.dayTime;
    this.dayTime = dayTime;
    gcProposals.clear();
    phase = None;
    chkPt = null;
  }

  public synchronized final HashMap<Integer, Object> getLadderedGCCertificates(){
    return (HashMap<Integer, Object>)this.gcCertificates.clone();
  }

  synchronized public void startGC(IrisNode myNode, AcceptVV omitVV, int epoch)
  throws Exception{
    if(gcCertificates.containsKey(epoch)){
      System.out.println("ERROR: already started the gc for daytime " + dayTime + "\n" + gcCertificates.get(dayTime));
      assert false;
      return;
    }
    assert phase == this.None:phase;
    assert epoch == dayTime+1;
    chkPt = node.generateNewCheckpoint((HashedVV)omitVV, epoch);

    LadderedGCLocalCertificate ep = new LadderedGCLocalCertificate((HashedVV)omitVV, (HashedVV)chkPt.endVV, 
        node.getBranchID(), epoch, chkPt.getHash());

    ObjId objId = gcProposalsFiles.get(node.getBranchID().getIDint()); // find hte object
    SimPreciseInv spi1 = node.write(objId, new IrisDataObject(ep)); // write to the log
    gcProposals.add(spi1);
    phase = this.Proposed;
  }

  private synchronized void startGC(LadderedGCLocalCertificate lgp) throws Exception{
    if(gcCertificates.containsKey(dayTime)){
      System.out.println("ERROR: already started the gc for daytime " + dayTime + "\n" + gcCertificates.get(dayTime));
      assert false;
      return;
    }
    if(phase == this.None){
      System.out.println("GC started at node " + node.getBranchID() + " for acceptVV " + lgp.omitVV + " dayTime " + dayTime);
      chkPt = node.generateNewCheckpoint(lgp.omitVV, lgp.dayNum);
      if(this.lastProposedVV.includes(chkPt.omitVV)){
        System.out.println("ERROR: new proposed vv is smaller than the previos VV");
        chkPt = null;
        assert false;
        return;
      }
      lastProposedVV = (HashedVV)chkPt.omitVV;
      if(chkPt.endVV.equals(lgp.endVV)){
        LadderedGCLocalCertificate ep = new LadderedGCLocalCertificate(lgp.omitVV, (HashedVV)chkPt.endVV, node.getBranchID(), dayTime, chkPt.getHash());
        ObjId objId = gcProposalsFiles.get(node.getBranchID().getIDint()); // find hte object
        SimPreciseInv spi1 = node.write(objId, new IrisDataObject(ep)); // write to the log
        gcProposals.add(spi1);
        this.gcCertificates.put(lgp.dayNum, ep);
        phase = this.Proposed;
      }
    }
  }

  public synchronized  void notifyPOMRemap(POMRemap pr) throws Exception{
    lastProposedVV = node.remapHashedDependencyVV(lastProposedVV);
  }

  /**
   * trigger the agreement protocol or wait
   * @param ep
   */
  private synchronized void handleLadderedGCProposal(SimPreciseInv spi) throws Exception{
    LadderedGCLocalCertificate ep = (LadderedGCLocalCertificate)((IrisDataObject)spi.getData()).getObject();
    if(ep.dayNum != (dayTime+1)){ // sanity check
      System.out.println("Proposal with invalid dayNum received");
      return;
    }

    // if the sender is valid; and omitVV is larger than any previously observed values 
    if(ep.bid.getIDint()%this.numResponse == (this.dayTime+1)%this.numResponse){
      System.out.println("LadderedGCProposal collected at " + node.getBranchID() + " , " + gcProposals.size() + " from " + ep.bid);
      // add to our list
      if(phase == None){
        startGC(ep);
      }
    }

    if(phase == Proposed){
      if(((HashedVV)chkPt.omitVV).getHash().equals(ep.omitVV.getHash())){
        gcProposals.add(spi);
      }

      if(gcProposals.size() == this.numResponse){
        this.createLadderedGCCertificate(ep.dayNum);
      }
    }
  }

  private synchronized void createLadderedGCCertificate(int dayNum) throws Exception{
    assert phase == Proposed;
    // first merge all the gc proposals that I have received
    // then create a certificate consisting of all the writes from the faulty node that nodes must accept
    // and finally install that certificate and rollback our state
    System.out.println("Create GC Certificate at node "  + node);
    LadderedGCCertificate ec = new LadderedGCCertificate(chkPt);
    this.gcCertificates.put(dayNum, ec);
    node.applyNewLocalCertificate(ec);
    this.advanceDayTime(dayNum);
  }

  private void notifyWrite(SimPreciseInv spi) throws Exception{
    if(spi.getObjId().getPath().startsWith(LadderedGCProtocol.gcProposalDir)){
      //HashMap<AcceptVV, LadderedGCProposal> gcProposal = (HashMap<AcceptVV, LadderedGCProposal>)spi.data;
      IrisDataObject irisObj = (IrisDataObject)spi.getData();
      LadderedGCLocalCertificate gcProposal = (LadderedGCLocalCertificate)irisObj.getObject();
      //process gc proposal
      if(gcProposal.dayNum == (this.dayTime+1)){
        this.handleLadderedGCProposal(spi);
      }
    }
  }

  public void receivePOM(IrisNode myNode, ProofOfMisbehavior pom){
    
  }

  public synchronized void receivePOMRemap(IrisNode myNode, POMRemap pomRemap) throws Exception{
    lastProposedVV = node.remapHashedDependencyVV(lastProposedVV);
  }

  synchronized public void notifyEpochSync(IrisNode myNode, Certificate cert)
      throws Exception{
    if(cert instanceof LadderedGCCertificate && ((this.dayTime+1) == cert.getEpoch()) 
        && phase == Proposed){
      this.advanceDayTime(cert.getEpoch());
      this.gcCertificates.put(cert.getEpoch(), cert);
    }
  }

  synchronized public boolean notifySync(IrisNode myNode, AcceptVV myCVV,
      BranchID otherNodeId, TreeSet<SimPreciseInv> newUpdates,
      Collection<Tuple<ObjId, TreeSet<StoreEntry>>> addedEntries)
      throws Exception{
    for(SimPreciseInv spi: newUpdates){
      invals.clear();
      if(spi.getObjId().getPath().startsWith(AgreementGCProtocol.gcProposalDir)){
        invals.add(spi);
      }
    }
    return true;
  }

  synchronized public void syncOver(IrisNode myNode, boolean success) throws Exception{
    for(SimPreciseInv spi: invals){
      this.notifyWrite(spi);
    }
    invals.clear();
  }

}
