/**
 * 
 */
package code.simulator.protocolFilters;

import java.util.*;

import code.AcceptVV;
import code.ObjId;
import code.branchDetecting.BranchID;
import code.simulator.Certificate;
import code.simulator.IrisDataObject;
import code.simulator.IrisNode;
import code.simulator.SimPreciseInv;
import code.simulator.agreement.Tuple;
import code.simulator.checkpoint.Checkpoint;
import code.simulator.store.StoreEntry;

/**
 * @author princem
 * This class provides the implementation of the algorithm for distributing and collecting gc certificates. 
 * The certificates are written at pre-specified objects/filenames 
 * This class enforces the following features:
 *      - we create an gc proposal on receiving the first POM
 *      - we force nodes to create gc proposal by refusing to accept writes from other nodes in the same epoch
 *      - we invoke consensus to convert gc proposals into gc certificate
 *      - we discard gc proposals on collecting gc certificate 
 */
public class SimpleGCProtocol implements GCProtocol{

  /**
   * POMs for various nodes against whom the gc is currently active
   */
  
  HashMap<AcceptVV, TreeSet<GCProposal>> gcProposals;
  HashMap<AcceptVV, GCCertificate> gcCertificates;
  LinkedList<GCProposal> pendingGCCertificates;
  
  /**
   * ObjIds of the objects used to communicate in this protocol
   * gcProposals store the proposals given by each node for each other faulty node
   *    - if we haven't received a proposal from a given node for a given fault and we receive a synchronization packet, we 
   *    notify that node by calling the receivePOM function at that node and discard that application of that sync
   * on receiving a globalGCCertificate, we add this node to the evicted node list in our local node
   */
  public static HashMap<Long, ObjId> gcProposalsFiles;
  public static String gcProposalDir = "/iris/gc/proposals/";
  int numResponses;
  public SimpleGCProtocol(int numResponses){
    this.numResponses = numResponses;
    gcProposalsFiles = new HashMap<Long, ObjId>();
    for(long i = 0; i < IrisNode.numNodes; i++){
      gcProposalsFiles.put(i, new ObjId("/iris/gc/proposals/"+i));
    }
    gcProposals = new HashMap<AcceptVV, TreeSet<GCProposal>>();
    gcCertificates = new HashMap<AcceptVV, GCCertificate>();
    pendingGCCertificates = new LinkedList<GCProposal>();
  }
  
  public HashMap<AcceptVV, TreeSet<GCProposal>> getGCProposals(){
    return gcProposals;
  }
  
  public HashMap<AcceptVV, GCCertificate> getGCCertificates(){
    return this.gcCertificates;
  }
  


  /**
   * observe any gc certificates 
   */
  public void notifyEpochSync(IrisNode myNode, Certificate c){
    if (c instanceof GCCertificate){
      GCCertificate ec = (GCCertificate)c;
      gcProposals.remove(ec.checkpoint.omitVV);
      gcCertificates.put(ec.checkpoint.omitVV, ec);
    }
  }
  
 
  /* (non-Javadoc)
   * @see code.simulator.SyncFilter#notifySync(code.simulator.BranchID, code.AcceptVV, code.AcceptVV, code.simulator.Checkpoint, java.util.TreeSet, code.AcceptVV)
   */
  public boolean notifySync(IrisNode myNode, AcceptVV myCVV, BranchID otherNodeId, 
//      AcceptVV otherCVV,
//      Checkpoint lastCheckpoint, 
      TreeSet<SimPreciseInv> newUpdates
//      ,      AcceptVV omitVV
      , Collection<Tuple<ObjId, TreeSet<StoreEntry>>> addedEntries
      ) throws Exception{
    
    // ensure that each new branch is accompanied by a POM [ignore for now]
    // ensure that the sender has as many gcProposals as we do
    // - if not, then call notifyPOM on the sender and abort the syn
    
      // if gc is active for someone
      // then ensure that either this packet contains a proposal for the faulty node or an gc certificate
      // otherwire reject and notify sender
      // further, ensure that all new writes for the faulty node are covered by at least one proposal
      // in addition, for any new proposal, add the proposal
      
      //1: scan the packet for any gc certificates/gc proposals/
      //2: ensure that (a) the sender has provided us with proposals for all gcs
      //               (b) each write from a currently known faulty node is accompanied by a proposal covering it

    for(SimPreciseInv spi: newUpdates){
      assert spi != null;
      assert spi.getObjId() != null:spi;
      assert spi.getObjId().getPath() != null:spi;
      if(spi.getObjId().getPath().startsWith(SimpleGCProtocol.gcProposalDir)){
        //HashMap<AcceptVV, GCProposal> gcProposal = (HashMap<AcceptVV, GCProposal>)spi.data;
        IrisDataObject irisObj = (IrisDataObject)spi.getData();
        HashMap<AcceptVV, GCProposal> gcProposal = (HashMap<AcceptVV, GCProposal>)irisObj.getObject();
        //process gc proposal
        for(AcceptVV avv: gcProposal.keySet()){
          if(!gcCertificates.containsKey(avv)){
            if(!gcProposals.containsKey(avv)){
              // found a new gc proposal that we didn't have before
              startGC(myNode, avv, gcProposal.get(avv).epoch);
            }
            this.handleGCProposal(myNode, gcProposal.get(avv));
          }
        }
      }
    }
    
//    this.syncOver(myNode, true); PRINCEM commented this line out after making the store COW as this was causing store to be replaced--
    // i expect this line be called when a sync finishes
    return true;
  }
  
  public void startGC(IrisNode myNode, AcceptVV avv, int epoch) throws Exception{
    if(gcCertificates.containsKey(avv)){
      return;
    }
    if(!gcProposals.containsKey(avv)){
      System.out.println("GC started at node " + myNode + " for acceptVV " + avv);
      GCProposal ep = new GCProposal(avv, myNode.getBranchID().getBaseId(), epoch);
      
      gcProposals.put(avv, new TreeSet<GCProposal>());
      gcProposals.get(avv).add(ep);
      
      this.updateGCProposalsFile(myNode);
    }
  }
  
  private void updateGCProposalsFile(IrisNode myNode){
    ObjId objId = gcProposalsFiles.get(myNode.getBranchID().getIDint());
    HashMap<AcceptVV, GCProposal> gcProposalsList = new HashMap<AcceptVV, GCProposal>(); 

    
    for(AcceptVV avv: gcProposals.keySet()){
        for(GCProposal ep: gcProposals.get(avv)){
          if(ep.bid.getIDint() == myNode.getBranchID().getIDint()){
            gcProposalsList.put(avv, ep);
            break;
          }
        }
      assert gcProposalsList.containsKey(avv);
    }
    myNode.write(objId, new IrisDataObject(gcProposalsList));
  }
  
  /**
   * trigger the agreement protocol or wait
   * @param ep
   */
  private void handleGCProposal(IrisNode myNode, GCProposal ep){
    if(!gcProposals.get(ep.omitVV).contains(ep) ){
      gcProposals.get(ep.omitVV).add(ep);

      System.out.println("GCProposal collected at " + myNode.getBranchID() + " , " + gcProposals.get(ep.omitVV).size() + " from " + ep.bid);
      if(gcProposals.get(ep.omitVV).size() == this.numResponses){
        this.pendingGCCertificates.add(ep);
      }
    }
  }
  
  public void syncOver(IrisNode myNode, boolean success) throws Exception{
    if(success){
      for(GCProposal ep:pendingGCCertificates){
        this.createGCCertificate(myNode, ep.omitVV, ep.epoch);
      }
      this.pendingGCCertificates.clear();
    }
  }

  public void createGCCertificate(IrisNode myNode, AcceptVV avv, int epoch) throws Exception{
    if(!gcCertificates.containsKey(avv)){
      // first merge all the gc proposals that I have received
      // then create a certificate consisting of all the writes from the faulty node that nodes must accept
      // and finally install that certificate and rollback our state
      System.out.println("Create GC Certificate at node "  + myNode);
      Checkpoint chkPt = myNode.generateCheckpoint(avv, epoch);
      GCCertificate ec = new GCCertificate(chkPt);
      myNode.applyNewLocalCertificate(ec);
    }
    
  }
}
