package code.simulator.store;

import java.util.Collection;
import java.util.HashMap;
import java.util.Iterator;
import java.util.LinkedList;
import java.util.Map;
import java.util.Random;
import java.util.Set;
import java.util.TreeMap;
import java.util.TreeSet;

import code.AcceptStamp;
import code.AcceptVV;
import code.NodeId;
import code.ObjId;
import code.SubscriptionSet;
import code.branchDetecting.BranchID;
import code.security.SangminConfig;
import code.simulator.DeleteObject;
import code.simulator.Hash;
import code.simulator.IrisDataObject;
import code.simulator.IrisHashObject;
import code.simulator.IrisObject;
import code.simulator.POMRemap;
import code.simulator.SimPreciseInv;
import code.simulator.agreement.Tuple;
import code.simulator.branchManager.LogBranchSegment;
import code.simulator.checkpoint.Checkpoint;
import code.simulator.log.Log;

import java.io.Serializable;

public class CopyOfStore extends TreeMap<ObjId, TreeSet<StoreEntry>> 
implements StoreInterface, Serializable{

  public static Random rand =  new Random();
  transient private long storeId;
  transient public static final DeleteObject deleteObject = new DeleteObject();

  /**
   * Information related to snapshotting and making this store support efficient rollbacks
   */
  transient private boolean inTxn;
  transient private HashMap<ObjId, TreeSet<StoreEntry>> undoLogModifiedEntries;
  transient private TreeSet<ObjId> undoLogAddedKeys;
  
  SubscriptionSet ignoreSet = null;
  public CopyOfStore(){
    super();
    storeId = rand.nextLong();
    inTxn = false;
    undoLogModifiedEntries = new HashMap<ObjId, TreeSet<StoreEntry>>();
    this.undoLogAddedKeys = new TreeSet<ObjId>();
    
  }

  public CopyOfStore(Map<? extends ObjId, ? extends TreeSet<StoreEntry>> m){
    super(m);
    storeId = rand.nextLong();
    inTxn = false;
    undoLogModifiedEntries = new HashMap<ObjId, TreeSet<StoreEntry>>();
    this.undoLogAddedKeys = new TreeSet<ObjId>();
    
  }
  
  synchronized public StoreEntry apply(SimPreciseInv spi){
    if(ignoreSet != null && 
        !ignoreSet.getIntersection(SubscriptionSet.makeSubscriptionSet(spi.getObjId().getPath())).isEmpty()){
      return new StoreEntry(spi);
    }
    StoreEntry se = null;
    TreeSet<StoreEntry> newSt = new TreeSet<StoreEntry>();
    se = new StoreEntry(spi);
    storeId++;
    newSt.add(se);
    if(containsKey(spi.getObjId())){
      TreeSet<StoreEntry> st = get(spi.getObjId());
      for(StoreEntry s: st){
        if(SangminConfig.lastWriterWin){
          if(spi.getAcceptStamp().lt(s.getAcceptStamp())){
            newSt.clear();
            newSt.add(s);
          }
        }else if(!spi.getObjectHashes().contains(s.getHash())){
          newSt.add(s);
        }else{
          assert spi.getAcceptStamp().gt(s.getAcceptStamp()): spi + " s " + s;
        }
      }
    }
    this.put(spi.getObjId(), newSt);
    if(spi.getData() instanceof IrisDataObject && SangminConfig.separateBodies){
      spi.replaceIrisObject(new IrisHashObject(spi.getData().getHash()));
    }
    return se;
  }

  synchronized public boolean removeBody(ObjId o, AcceptStamp as){
    if(!this.containsKey(o)){
      return false;
    }else{
      TreeSet<StoreEntry> ste = this.get(o);
      for(StoreEntry se: ste){
        if(as.eq(se.getAcceptStamp())){
          se.dangerousSetBody(new IrisHashObject(se.getData().getHash()));
          return true;
        }
      }
    }
    
    return false;
  }
  
  synchronized public boolean addBody(ObjId o, IrisDataObject body){
    if(!this.containsKey(o)){
      return false;
    }else{
      TreeSet<StoreEntry> ste = this.get(o);
      for(StoreEntry se: ste){
        if(se.getData().getHash().equals(body.getHash())){
          se.dangerousSetBody(body);
          return true;
        }
      }
    }
    
    return false;
  }
  
  synchronized public IrisObject getBody(ObjId o, Hash hash){
    if(!this.containsKey(o)){
      return null;
    }else{
      TreeSet<StoreEntry> ste = this.get(o);
      for(StoreEntry se: ste){
        if(se.getData().getHash().equals(hash)){
          return se.getData();
        }
      }
    }
    return null;
  }
  
  synchronized public StoreEntry apply(SimPreciseInv spi, Log log){
    if(ignoreSet != null && 
        !ignoreSet.getIntersection(SubscriptionSet.makeSubscriptionSet(spi.getObjId().getPath())).isEmpty()){
      return new StoreEntry(spi);
    }
    storeId++;
    StoreEntry se = null;
    TreeSet<StoreEntry> newSt = new TreeSet<StoreEntry>();
    AcceptVV closureVV = log.getDeepClosure(spi.getAcceptStamp());
    se = new StoreEntry(spi);
    newSt.add(se);
    if(containsKey(spi.getObjId())){
      TreeSet<StoreEntry> st = get(spi.getObjId());
      for(StoreEntry s: st){
        if(SangminConfig.lastWriterWin){
          if(spi.getAcceptStamp().lt(s.getAcceptStamp())){
            newSt.clear();
            newSt.add(s);
          }
        }else if(!spi.getObjectHashes().contains(s.getHash())){
          newSt.add(s);
          assert !closureVV.includes(s.getAcceptStamp()): "\n"+closureVV +"\n"+ s +"\n" + spi;
        }else{
          assert spi.getAcceptStamp().gt(s.getAcceptStamp()): spi + " s " + s;
        }
      }
    }
    this.put(spi.getObjId(), newSt);
    if(spi.getData() instanceof IrisDataObject && SangminConfig.separateBodies){
      spi.replaceIrisObject(new IrisHashObject(spi.getData().getHash()));
    }
    return se;
  }
  
  @Override
  synchronized public boolean equals(Object o){
    return super.equals(o);
  }

  @Override
  synchronized public int hashCode(){
    return super.hashCode();
  }

  @Override
  synchronized public String toString(){
    return super.toString();
  }
  
  synchronized public CopyOfStore clone(){
    assert !inTxn:"Inside a txn";
    CopyOfStore newStore = new CopyOfStore(this);
    newStore.setIgnoreSet(ignoreSet);
    return newStore;
  }
  
  synchronized public void setIgnoreSet(SubscriptionSet ignoreSet2){
    assert !inTxn:"Inside a txn";
    this.ignoreSet = ignoreSet2;
  }

  synchronized public CopyOfStore applyBranchMap(HashMap<NodeId, NodeId> branchMap){
    assert !inTxn:"Inside a txn";
    CopyOfStore store = new CopyOfStore();
    for(ObjId o: this.keySet()){
      TreeSet<StoreEntry> oldSt = this.get(o);
      TreeSet<StoreEntry> newSt = new TreeSet<StoreEntry>();
      for(StoreEntry se: oldSt){
        newSt.add(se.applyBranchMap(branchMap));
      }
      store.put(o, newSt);
    }
    store.setIgnoreSet(this.ignoreSet);
    return store;
  }
  
  synchronized public void applyPOMRemap(POMRemap pr){
    TreeMap<ObjId, TreeSet<StoreEntry>> oldStore = ((TreeMap<ObjId, TreeSet<StoreEntry>>)super.clone());
    for(ObjId o: oldStore.keySet()){
      TreeSet<StoreEntry> oldSt = oldStore.get(o);
      TreeSet<StoreEntry> newSt = new TreeSet<StoreEntry>();
      for(StoreEntry se: oldSt){
        newSt.add(se.applyPOMRemap(pr));
      }
      super.put(o, newSt);
    }
    if(inTxn){
      HashMap<ObjId, TreeSet<StoreEntry>> oldUndoLog = (HashMap<ObjId, TreeSet<StoreEntry>>)undoLogModifiedEntries.clone();
      for(ObjId o: oldUndoLog.keySet()){
        TreeSet<StoreEntry> oldSt = oldUndoLog.get(o);
        TreeSet<StoreEntry> newSt = new TreeSet<StoreEntry>();
        for(StoreEntry se: oldSt){
          newSt.add(se.applyPOMRemap(pr));
        }
        undoLogModifiedEntries.put(o, newSt);
      }
    }
  }

  synchronized public TreeSet<Hash> getObjectHashes(ObjId objId){
    TreeSet<Hash> objHashes = new TreeSet<Hash>();
    if(this.containsKey(objId)){
      TreeSet<StoreEntry> tse = this.get(objId);
      for(StoreEntry se: tse){
        objHashes.add(se.getHash());
      }
    }
    return objHashes;
  }

  synchronized public LinkedList<IrisObject> getData(ObjId objId){
    LinkedList<IrisObject> data = new LinkedList<IrisObject>();
    if(this.containsKey(objId)){
      TreeSet<StoreEntry> tse = this.get(objId);
      for(StoreEntry se: tse){
        if(se.isDelete()){
          data.add(deleteObject);
        }else{
          data.add(se.getData());
        }
      }
    }
    return data;
  }

  synchronized public LinkedList<StoreEntry> getRawData(ObjId objId){
    LinkedList<StoreEntry> data = new LinkedList<StoreEntry>();
    if(this.containsKey(objId)){
      TreeSet<StoreEntry> tse = this.get(objId);
      for(StoreEntry se: tse){
        data.add(se);
      }
    }
    return data;  
  }

  synchronized public SubscriptionSet getIgnoreSet(){
     return ignoreSet;
  }

  synchronized public long getStoreId(){
    return storeId;
  }
  
  synchronized public void startTxn(){
    assert !inTxn:"Already inside a txn";
    assert undoLogModifiedEntries.isEmpty();
    assert undoLogAddedKeys.isEmpty();
    inTxn = true;
  }
  
  synchronized public void endTxn(boolean commit){    
    assert inTxn:"Not inside a txn";
    if(!commit){
      // remove added keys
      for(ObjId o: undoLogAddedKeys){
        super.remove(o);
      }
      //replace overwritten keys
      for(ObjId o: undoLogModifiedEntries.keySet()){
        super.put(o, undoLogModifiedEntries.get(o));
      }
    }
    undoLogAddedKeys.retainAll(undoLogModifiedEntries.keySet());
    assert undoLogAddedKeys.size() == 0; // entries should not be repeated between addedKeys and modifiedEntries

    undoLogModifiedEntries.clear();
    undoLogAddedKeys.clear();
    inTxn = false;
  }

  /**
   * obtains the entries added by a given txns
   * Expected that the returned value is treated as immutable object
   * @return
   */
  synchronized public LinkedList<Tuple<ObjId, TreeSet<StoreEntry>>> getAddedEntries(){
    assert inTxn:"Not inside a txn";
    LinkedList<Tuple<ObjId, TreeSet<StoreEntry>>> addedEntries = new LinkedList<Tuple<ObjId, TreeSet<StoreEntry>>>();
    // all the added entries qualify
    for(ObjId o: undoLogAddedKeys){
      addedEntries.add(new Tuple<ObjId, TreeSet<StoreEntry>>(o, this.get(o)));
    }
    // only the updated entries qualify--deletes don't qualify
    for(ObjId o: undoLogModifiedEntries.keySet()){
      if(this.containsKey(o)){
        addedEntries.add(new Tuple<ObjId, TreeSet<StoreEntry>>(o, this.get(o)));
      }
    }
    return addedEntries;
  }
  
  synchronized public HashMap<ObjId, TreeSet<StoreEntry>> getRemovedEntries(){
    assert inTxn:"Not inside a txn";
    return this.undoLogModifiedEntries;
  }
  
  @Override
  synchronized public void clear(){
    assert !inTxn:"Clear not supported inside a txn";
    super.clear();
  }

  @Override
  synchronized public TreeSet<StoreEntry> put(ObjId key, TreeSet<StoreEntry> value){
    // if in a txn and this is not the first modification to this key
    if(inTxn && (!undoLogAddedKeys.contains(key) && !undoLogModifiedEntries.containsKey(key))){
      //add undo records
      if(this.containsKey(key)){
//        undoLogModifiedEntries.put(key, this.get(key));
        undoLogModifiedEntries.put(key, (TreeSet<StoreEntry>)this.get(key).clone());
      }else{
        undoLogAddedKeys.add(key);
      }
    }
    return super.put(key, value);
  }

  @Override
  synchronized public void putAll(Map<? extends ObjId, ? extends TreeSet<StoreEntry>> map){
    assert !inTxn:"putAll not supported inside a txn";
    super.putAll(map);
  }


  @Override
  synchronized public TreeSet<StoreEntry> remove(Object key){
    assert !inTxn:"remove not supported inside a txn";
    return super.remove(key);
  }

  /* (non-Javadoc)
   * @see java.util.TreeMap#get(java.lang.Object)
   */
  @Override
  public TreeSet<StoreEntry> get(Object key){
    return super.get(key);
  }

  /* (non-Javadoc)
   * @see java.util.AbstractMap#isEmpty()
   */
  @Override
  public boolean isEmpty(){
    return super.isEmpty();
  }

  /* (non-Javadoc)
   * @see code.simulator.store.StoreInterface#containsKey(code.ObjId)
   */
  public boolean containsKey(ObjId key){
    return false;
  }

  /* (non-Javadoc)
   * @see code.simulator.store.StoreInterface#get(code.ObjId)
   */
  public TreeSet<StoreEntry> get(ObjId key){
    return null;
  }

  public Iterator<ObjId> iterator(){
    return this.keySet().iterator();
  }

  public void putAll(StoreInterface si){
    // TODO Auto-generated method stub
    
  }

}
