package code.simulator.store;


import code.ObjId;
import code.simulator.Hash;
import code.simulator.checkpoint.Checkpoint;

import com.sleepycat.je.DatabaseException;
import com.sleepycat.je.Environment;
import com.sleepycat.je.EnvironmentConfig;
import com.sleepycat.bind.serial.StoredClassCatalog;
import com.sleepycat.je.Database;
import com.sleepycat.je.DatabaseConfig;
import com.sleepycat.bind.EntryBinding;
import com.sleepycat.bind.serial.ClassCatalog;
import com.sleepycat.bind.serial.SerialBinding;
import com.sleepycat.collections.StoredEntrySet;
import com.sleepycat.collections.StoredMap;


import java.io.File;
import java.io.FileNotFoundException;


public class PersistentEnv{

  private Environment env;
  
  private static final String CLASS_CATALOG = "java_class_catalog";
  private StoredClassCatalog javaCatalog;
  
  private static final String BODY_STORE = "body_store";
  private static final String CHECKPOINT_STORE = "checkpoint_store";
  private static final String APPEND_LOG = "append_log";
  
  private final Database bodyStoreDb;
  private final Database checkpointDb;
  private final Database appendLogDb;
  
  public PersistentEnv(String homeDirectory) throws DatabaseException, FileNotFoundException{
    System.out.println("Opening environment in: " + homeDirectory);

    EnvironmentConfig envConfig = new EnvironmentConfig();
    envConfig.setTransactional(true);
    envConfig.setAllowCreate(true);

    env = new Environment(new File(homeDirectory), envConfig);

    DatabaseConfig dbConfig = new DatabaseConfig();
    dbConfig.setTransactional(true);
    dbConfig.setAllowCreate(true);

    bodyStoreDb = env.openDatabase(null, BODY_STORE, dbConfig);
    checkpointDb = env.openDatabase(null, CHECKPOINT_STORE, dbConfig);
    appendLogDb = env.openDatabase(null, APPEND_LOG, dbConfig);

    
    Database catalogDb = env.openDatabase(null, CLASS_CATALOG, 
                                          dbConfig);

    javaCatalog = new StoredClassCatalog(catalogDb);

    EntryBinding<byte[]> byteBinding =
        new SerialBinding<byte[]>(javaCatalog, byte[].class);
    EntryBinding<Hash> hashBinding =
        new SerialBinding<Hash>(javaCatalog, Hash.class);
    EntryBinding<Checkpoint> checkpointBinding =
        new SerialBinding<Checkpoint>(javaCatalog, Checkpoint.class);
    EntryBinding<Integer> seqBinding =
      new SerialBinding<Integer>(javaCatalog, Integer.class);
    EntryBinding<ObjId> oidBinding =
      new SerialBinding<ObjId>(javaCatalog, ObjId.class);
    
  }
  
  public final Environment getEnvironment()
  {
      return env;
  }
  
  public final StoredClassCatalog getClassCatalog() {
    return javaCatalog;
  } 

  public final Database getBodyStoreDb()
  {
      return bodyStoreDb;
  }

  public final Database getCheckpointDb()
  {
      return checkpointDb;
  }

  public final Database getAppendLogDb()
  {
      return appendLogDb;
  }
  
  public void close()
      throws DatabaseException
  {
    bodyStoreDb.close();
    checkpointDb.close();
    appendLogDb.close();
    javaCatalog.close();
    env.close();
  }

 
}
