
package code.simulator.store;


import java.io.ByteArrayOutputStream;
import java.io.ObjectOutputStream;
import java.io.Serializable;
import java.util.HashMap;

import code.Immutable;
import code.AcceptStamp;
import code.NodeId;
import code.serialization.SerializationHelper;
import code.simulator.Hash;
import code.simulator.IrisDataObject;
import code.simulator.IrisHashObject;
import code.simulator.IrisObject;
import code.simulator.Obj2Bytes;
import code.simulator.POMRemap;
import code.simulator.SecureSimPreciseInv;
import code.simulator.SimPreciseInv;

public class StoreEntry implements Serializable, Immutable, Comparable<StoreEntry>, Obj2Bytes{

  private IrisObject data;
  /**
   * the hash of the last invalidate that affected this objId 
   */
  private final Hash hash;
  private final AcceptStamp as;
  private final boolean delete;
  
  public StoreEntry(SimPreciseInv spi){
    as = spi.getAcceptStamp();
    data = spi.getData();
    hash = spi.generateMyHash();
    this.delete = spi.isDelete();
  }
  
  public StoreEntry(IrisObject data, AcceptStamp as, Hash hash, boolean delete){
    this.as = as;
    this.data = data;
    this.hash = hash;
    this.delete = delete;
  }

  public IrisObject getData(){
    return data;
  }

  public AcceptStamp getAcceptStamp(){
    return as;
  }

  @Override
  public int hashCode(){
    final int prime = 31;
    int result = 1;
    result = prime * result + ((as == null) ? 0 : as.hashCode());
    result = prime * result + ((data == null) ? 0 : data.hashCode());
    result = prime * result + (delete ? 1231 : 1237);
    result = prime * result + ((hash == null) ? 0 : hash.hashCode());
    return result;
  }

  @Override
  public boolean equals(Object obj){
    if(this == obj){
      return true;
    }
    if(obj == null){
      return false;
    }
    if(!(obj instanceof StoreEntry)){
      return false;
    }
    StoreEntry other = (StoreEntry) obj;
    if(as == null){
      if(other.as != null){
        return false;
      }
    }else if(!as.equals(other.as)){
      return false;
    }
    if(data == null){
      if(other.data != null){
        return false;
      }
    }else if(!data.equals(other.data)){
      return false;
    }
    if(delete != other.delete){
      return false;
    }
    if(hash == null){
      if(other.hash != null){
        return false;
      }
    }else if(!hash.equals(other.hash)){
      return false;
    }
    return true;
  }

  public StoreEntry applyBranchMap(HashMap<NodeId, NodeId> branchMap){
    if(branchMap.containsKey(as.getNodeId())){
      return new StoreEntry(data, new AcceptStamp(as.getLocalClock(), branchMap.get(as.getNodeId())), hash, delete);
    }else{
      return this;
    }
  }
  
  @Override
  public String toString(){
    return "StoreEntry [as=" + as + ", data=" + data + ", delete=" + delete
        + ", hash=" + hash + "]";
  }

  public StoreEntry applyPOMRemap(POMRemap pr){
    AcceptStamp newAS = pr.applyPOMRemap(as);
    if(newAS.eq(as)){
      return this;
    }else{
      return new StoreEntry(data, newAS, hash, delete);
    }
  }

  public int compareTo(StoreEntry se){
    int ret = new Boolean(this.delete).compareTo(se.delete);
    if(ret != 0){
      return ret;
    }else{
      ret = this.hash.compareTo(se.hash);
      if(ret != 0){
        return ret;
      }else{
        ret = new Long(as.getLocalClock()).compareTo(se.as.getLocalClock());
        if(ret != 0){
          return ret;
        }else{
          ret = new Long(as.getNodeId().getIDint()).compareTo(se.as.getNodeId().getIDint());
          if(ret != 0){
            return ret;
          }
          else if(data.getHash().equals(se.data.getHash())){
            return 0;
          }else{
            assert this.equals(se) : "SHOULD NOT REACH HERE : " + se + ", " + this;
          return 0;
          }
        }
      }
    }

  }

  public Hash getHash(){
    return hash;
  }

  public byte[] obj2Bytes(){
    byte[] ret = null;
    try{
      ByteArrayOutputStream bs = new ByteArrayOutputStream();
      ObjectOutputStream oos = new ObjectOutputStream(bs);
      oos.writeLong(as.getLocalClock());
      oos.writeLong(as.getNodeId().getIDint());
      oos.write(hash.obj2Bytes());
      
      oos.write(data.obj2Bytes());
      oos.flush();
      ret = bs.toByteArray();
      bs.close();
      oos.close();
      return ret;
    } catch (Exception e){
      e.printStackTrace();
    }
    assert false;
    return null;
  }

  public boolean isDelete(){
    return delete;
  }

  public void dangerousSetBody(IrisObject body){
    assert this.data.getHash().equals(body.getHash());
    this.data = body;
  }
  
  public StoreEntry addBody(IrisObject body){
    assert this.data.getHash().equals(body.getHash());
    return new StoreEntry(body, this.as, this.hash, this.delete);
  }
}
