package code.simulator.unit;

import code.branchDetecting.BranchID;
import code.simulator.*;
import code.simulator.SyncRequest;
import code.simulator.checkpoint.unit.SimulatorCheckpointUnit;
import code.simulator.store.StoreEntry;
import code.*;
import code.security.SangminConfig;

 /** 
 *  Test bandwidth to subscribe to 1000 objects, 100 of which have
 
 *  been modified. Test created for sosp 2007 submission microbenchmark.
 
 **/ 

//package utils.junit;

import junit.textui.TestRunner;
import junit.framework.*;
import java.util.*;
import java.io.*;

/**
 * TBD: Update class name
 */
public class SimulatorUnit extends TestCase {
  public static final String TEST_ALL_TEST_TYPE = "UNIT";
  
  /**
   * Basic constructor - called by the test runners.
   * TBD: Update constructor name to match class
   */
  public SimulatorUnit (final String s) {
    super (s);
  }

  /*
   * Fixtures are run before and after each test case
   * to set up environment in which tests must run.
   */
  protected void setUp() throws Exception{
    super.setUp();
    SimulatorCheckpointUnit.makePractiConfig(0, 100);
 
  }


  public static int UNIT_NODE_ID = 10;
  public static int MIDDLE_NODE_ID = 15;

  public static int HELPER_NODE_ID = 20;
  public static boolean vverbose = false;
  public static boolean verbose = true;


  public static String COARSE_PATH = "/coarse";


  //--------------------------------------------------------------------------- 
  // Test coarse-grained subscriptions. 
  //
  // To initialize state, Unit create 1000 objects
  // (/data-coarse/[0.9]/[0.9]/[0.9]. Then helper subscribe
  // to /* from unit and wait for /data-coarse/9/9/9 to arrive.
  //
  //
  // start helper and wait for writing to complete
  //---------------------------------------------------------------------------
  public void testSimple(){
    try{
      System.out.println("testSimple");

      BranchID myNodeId = NodeFactory.createNodeId(UNIT_NODE_ID);
      
      BranchID helperNodeId = NodeFactory.createNodeId(HELPER_NODE_ID);
      if(vverbose){
        System.out.println("UNIT ID: " + myNodeId.toString());
      }
      assert(myNodeId.equals(NodeFactory.createNodeId(UNIT_NODE_ID)));
  
      Node myNode = NodeFactory.createNodeWithoutFilter(myNodeId);
      Node helperNode = NodeFactory.createNodeWithoutFilter(helperNodeId);

      //second set of writes
      
      if(verbose){
        System.out.println("Hello from SimulatorUnit");
      }

      assert myNode.getCurrentVV().equalsIgnoreNegatives(new AcceptVV());

      int ni = 5;
      write(myNode, COARSE_PATH, 0, ni-1, 0);
      checkRead(myNode, COARSE_PATH, 0, ni-1, 0);
      SyncRequest sr = new SyncRequest(new AcceptVV(), 0, new HashSet<ProofOfMisbehavior> (), myNode.getBranchID().getIDint(), SangminConfig.BodyConfig, false);
      SyncPacket sp = ((IrisNode)myNode).getSyncPacket(sr);
      SyncStatus ss = ((IrisNode)myNode).applySyncPacket(sp);
      System.out.println(ss);
      assert ss.status == SyncStatus.SyncSuccessful;
      
      assert myNode.getCurrentVV().includes(new AcceptStamp(ni-1, myNodeId));
      assert !myNode.getCurrentVV().includes(new AcceptStamp(ni, myNodeId));

      helperNode.sync(myNode);

      checkRead(helperNode, COARSE_PATH, 0, ni-1, 0);
      
      // now do writes at helper node and sync with the main node
      
      write(helperNode, COARSE_PATH, 0, ni-1, 10);
      checkRead(helperNode, COARSE_PATH, 0, ni-1, 10);

      assert helperNode.getCurrentVV().includes(new AcceptStamp(ni-1+5, helperNodeId));
      assert !helperNode.getCurrentVV().includes(new AcceptStamp(ni+5, helperNodeId));

      myNode.sync(helperNode);
      checkRead(myNode, COARSE_PATH, 0, ni-1, 10);
    }catch(Exception e){
      e.printStackTrace();
      assert false;
    }
    
  }
  
  //--------------------------------------------------------------------------- 
  // Test conflict resolution by ensuring that writes with same time stamp and to same objects are resolved based on the nodeId
  //---------------------------------------------------------------------------
  public void testConflictResolution(){
    try{
      System.out.println("testConflictResolution");
      BranchID myNodeId = NodeFactory.createNodeId(UNIT_NODE_ID);
      BranchID thirdNodeId = NodeFactory.createNodeId(MIDDLE_NODE_ID);

      BranchID helperNodeId = NodeFactory.createNodeId(HELPER_NODE_ID);
      if(vverbose){
        System.out.println("UNIT ID: " + myNodeId.toString());
      }
      assert(myNodeId.equals(NodeFactory.createNodeId(UNIT_NODE_ID)));
  
      Node myNode = NodeFactory.createNodeWithoutFilter(myNodeId);
      Node helperNode = NodeFactory.createNodeWithoutFilter(helperNodeId);
      Node thirdNode = NodeFactory.createNodeWithoutFilter(thirdNodeId);

      //second set of writes
      
      if(verbose){
        System.out.println("Hello from SimulatorUnit");
      }

      assert myNode.getCurrentVV().equalsIgnoreNegatives(new AcceptVV());

      int ni = 5;
      write(myNode, COARSE_PATH, 0, ni-1, 0);
      checkRead(myNode, COARSE_PATH, 0, ni-1, 0);
      assert myNode.getLogTest().get(myNode.getBranchID()).size() == 5;
      assert myNode.getLogTest().get(helperNodeId).size() == 0;
      assert myNode.getLogTest().get(thirdNodeId).size() == 0;
      
      assert myNode.getCurrentVV().includes(new AcceptStamp(ni-1, myNodeId));
      assert !myNode.getCurrentVV().includes(new AcceptStamp(ni, myNodeId));

      thirdNode.sync(myNode);
      assert thirdNode.getLogTest().get(myNode.getBranchID()).size() == 5;
      assert thirdNode.getLogTest().get(helperNodeId).size() == 0;
      assert thirdNode.getLogTest().get(thirdNodeId).size() == 0;
      checkRead(myNode, COARSE_PATH, 0, ni-1, 0, 1);

      // now do writes at helper node and sync with the main node
      ni=10;
      write(helperNode, COARSE_PATH, 0, ni-1, 10);
      assert helperNode.getLogTest().get(myNode.getBranchID()).size() == 0;
      assert helperNode.getLogTest().get(helperNodeId).size() == 10;
      assert helperNode.getLogTest().get(thirdNodeId).size() == 0;
      
      checkRead(helperNode, COARSE_PATH, 0, ni-1, 10, 1);

      assert helperNode.getCurrentVV().includes(new AcceptStamp(ni-1, helperNodeId));
      assert !helperNode.getCurrentVV().includes(new AcceptStamp(ni, helperNodeId));

      myNode.sync(helperNode);
      assert myNode.getLogTest().get(myNode.getBranchID()).size() == 5;
      assert myNode.getLogTest().get(helperNodeId).size() == 10;
      assert myNode.getLogTest().get(thirdNodeId).size() == 0;
      
      //ensure that local writes got overwritten
      checkRead(myNode, COARSE_PATH, 0, 4, 10, 2);
      checkRead(myNode, COARSE_PATH, 5, 9, 10, 1);
      
      // even after local writes, it shouldn't matter
      write(thirdNode, COARSE_PATH, 5, ni-1, 20);
      assert thirdNode.getLogTest().get(myNode.getBranchID()).size() == 5;
      assert thirdNode.getLogTest().get(helperNodeId).size() == 0;
      assert thirdNode.getLogTest().get(thirdNodeId).size() == 5;
      checkRead(thirdNode, COARSE_PATH, 5, ni-1, 20, 1);

      thirdNode.sync(helperNode);
      checkRead(thirdNode, COARSE_PATH, 0, ni-1, 10, 2);

      helperNode.sync(thirdNode);
      checkRead(helperNode, COARSE_PATH, 0, ni-1, 10, 2);

      assert helperNode.getCurrentVV().equalsIgnoreNegatives(thirdNode.getCurrentVV());
    }catch(Exception e){
      // TODO Auto-generated catch block
      e.printStackTrace();
      assert false;
    }
    
  }
  
  //--------------------------------------------------------------------------- 
  // Test if attaining eventual consistency despite forking
  //---------------------------------------------------------------------------
  public void testForking(){
    try{
      if(!Node.enableSecurity){
        return;
      }
      System.out.println("testForking");
      BranchID myNodeId = NodeFactory.createNodeId(UNIT_NODE_ID);
      BranchID thirdNodeId = NodeFactory.createNodeId(MIDDLE_NODE_ID);

      BranchID helperNodeId = NodeFactory.createNodeId(HELPER_NODE_ID);
      if(vverbose){
        System.out.println("UNIT ID: " + myNodeId.toString());
      }
      assert(myNodeId.equals(NodeFactory.createNodeId(UNIT_NODE_ID)));
  
      Node myNode = NodeFactory.createNodeWithoutFilter(myNodeId);
      Node helperNode = NodeFactory.createNodeWithoutFilter(helperNodeId);
      Node thirdNode = NodeFactory.createNodeWithoutFilter(thirdNodeId);

      //second set of writes
      
      if(verbose){
        System.out.println("Hello from SimulatorUnit");
      }

      assert myNode.getCurrentVV().equalsIgnoreNegatives(new AcceptVV());

      int ni = 5;
      write(myNode, COARSE_PATH, 0, ni-1, 0);
      checkRead(myNode, COARSE_PATH, 0, ni-1, 0);

      assert myNode.getCurrentVV().includes(new AcceptStamp(ni-1, myNodeId));
      assert !myNode.getCurrentVV().includes(new AcceptStamp(ni, myNodeId));


      ni=10;
      int numForks = 2;
      LinkedList<Node> forkedPseudoNodes = myNode.fork(numForks);
      
      assert forkedPseudoNodes.size() == numForks;
      int i = 0;
      for(Node n: forkedPseudoNodes){
        checkRead(n, COARSE_PATH, 0, 4, 0);
        write(n, COARSE_PATH+i, 5, ni-1, 0);
        checkRead(n, COARSE_PATH+i, 5, ni-1, 0);
        i++;
      }
      
      helperNode.sync(forkedPseudoNodes.get(0));
      checkRead(helperNode, COARSE_PATH, 0, 4, 0);
      checkRead(helperNode, COARSE_PATH+0, 5, ni-1, 0);
      
      thirdNode.sync(forkedPseudoNodes.get(1));
      checkRead(thirdNode, COARSE_PATH, 0, 4, 0);
      checkRead(thirdNode, COARSE_PATH+1, 5, ni-1, 0);
      

      
      if(!SangminConfig.useSimulatorId){
        ((IrisNode)helperNode).exponentialBackoffSync(thirdNode);
      }
      
      helperNode.sync(thirdNode);
      checkRead(helperNode, COARSE_PATH, 0, 4, 0);
      checkRead(helperNode, COARSE_PATH+1, 5, ni-1, 0);
      
      thirdNode.sync(helperNode);
      checkRead(thirdNode, COARSE_PATH, 0, 4, 0);
      checkRead(thirdNode, COARSE_PATH+0, 5, ni-1, 0);
    }catch(Exception e){
      e.printStackTrace();
      assert false;
    }
    
  }
  
  public void testFaultSecurity(){
    if(Node.enableSecurity){
      return;
      //PRINCEM: IF WE ARE USING THIS FOR GENERATING FORKING GRAPH THEN COMMENT THE enableSecurity CHECK

    }
    try{
      System.out.println("testForking");
      BranchID myNodeId = NodeFactory.createNodeId(UNIT_NODE_ID);
      BranchID thirdNodeId = NodeFactory.createNodeId(MIDDLE_NODE_ID);
      BranchID fourthNodeId = NodeFactory.createNodeId(25);

      BranchID helperNodeId = NodeFactory.createNodeId(HELPER_NODE_ID);
      if(vverbose){
        System.out.println("UNIT ID: " + myNodeId.toString());
      }
      assert(myNodeId.equals(NodeFactory.createNodeId(UNIT_NODE_ID)));
  
      Node myNode = NodeFactory.createNodeWithoutFilter(myNodeId);
      Node helperNode = NodeFactory.createNodeWithoutFilter(helperNodeId);
      Node thirdNode = NodeFactory.createNodeWithoutFilter(thirdNodeId);

      //second set of writes
      
      if(verbose){
        System.out.println("Hello from SimulatorUnit");
      }

      assert myNode.getCurrentVV().equalsIgnoreNegatives(new AcceptVV());

      System.out.println("Round " + 0 + " --------");

      int ni = 2;
      AcceptStamp as = write(myNode, COARSE_PATH, 0, ni-1, 0);
      checkRead(myNode, COARSE_PATH, 0, ni-1, 0);
      printRead(myNode, COARSE_PATH, 0, ni-1);

      thirdNode.sync(myNode);
      printRead(thirdNode, COARSE_PATH, 0, ni-1);

      assert myNode.getCurrentVV().includes(new AcceptStamp(ni-1, myNodeId));
      assert !myNode.getCurrentVV().includes(new AcceptStamp(ni, myNodeId));

      for(int i = 0; i < 5; i++){
        System.out.println("Round " + (i+1) + " --------");
        
        int value = (i+1)*ni;
        as = write(myNode, COARSE_PATH, 0, ni-1, value);
        checkRead(myNode, COARSE_PATH, 0, ni-1, value);

        helperNode.sync(myNode);
        checkRead(helperNode, COARSE_PATH, 0, ni-1, value);

        AcceptStamp dropAS = new AcceptStamp(as.getLocalClock()-1, as.getNodeId());
        if(!helperNode.getLogTest().dropWrite(dropAS)){
          System.err.println("DROP FAILED");
        }
        thirdNode.sync(helperNode);
        
        thirdNode.sync(myNode);
        printRead(myNode, COARSE_PATH, 0, ni-1);
        printRead(thirdNode, COARSE_PATH, 0, ni-1);
      }
    }catch(Exception e){
      // TODO Auto-generated catch block
      e.printStackTrace();
      assert false;
    }
  }
  
  public void testForkSecurity(){
  if(!Node.enableSecurity){
    return;
    //PRINCEM: IF WE ARE USING THIS FOR GENERATING FORKING GRAPH THEN COMMENT THE enableSecurity CHECK
  }
  try{
    System.out.println("testForking");
    BranchID myNodeId = NodeFactory.createNodeId(UNIT_NODE_ID);
    BranchID thirdNodeId = NodeFactory.createNodeId(MIDDLE_NODE_ID);

    BranchID helperNodeId = NodeFactory.createNodeId(HELPER_NODE_ID);
    if(vverbose){
      System.out.println("UNIT ID: " + myNodeId.toString());
    }
    assert(myNodeId.equals(NodeFactory.createNodeId(UNIT_NODE_ID)));
 
    Node myNode = NodeFactory.createNodeWithoutFilter(myNodeId);
    Node helperNode = NodeFactory.createNodeWithoutFilter(helperNodeId);
    Node thirdNode = NodeFactory.createNodeWithoutFilter(thirdNodeId);

    //second set of writes
    
    if(verbose){
      System.out.println("Hello from SimulatorUnit");
    }

    assert myNode.getCurrentVV().equalsIgnoreNegatives(new AcceptVV());

    System.out.println("Round " + 0 + " --------");

    int ni = 1;
    write(myNode, COARSE_PATH, 0, ni-1, 0);

    int numForks = 2;
    LinkedList<Node> forkedPseudoNodes = myNode.fork(numForks);
    
    assert forkedPseudoNodes.size() == numForks;
    
    int MAX = 100;
    
    write(forkedPseudoNodes.get(0), COARSE_PATH, 0, ni-1, 0);
    write(forkedPseudoNodes.get(1), COARSE_PATH, 0, ni-1, 100);
    
    helperNode.sync(forkedPseudoNodes.get(0));
    
    thirdNode.sync(forkedPseudoNodes.get(1));
    
    printRead(helperNode, COARSE_PATH, 0, ni-1);
    printRead(thirdNode, COARSE_PATH, 0, ni-1);

    
    for(int i = 0; i < 5; i++){
      System.out.println("Round " + (i+1) + " --------");
      
      int value = (i+1)*ni;
      int value1 = MAX-value;
      write(forkedPseudoNodes.get(0), COARSE_PATH, 0, ni-1, value);
      write(forkedPseudoNodes.get(1), COARSE_PATH, 0, ni-1, value1);
      
      helperNode.sync(forkedPseudoNodes.get(0));
      
      thirdNode.sync(forkedPseudoNodes.get(1));
      
      if(!SangminConfig.useSimulatorId){
        ((IrisNode)thirdNode).exponentialBackoffSync(helperNode);
        ((IrisNode)thirdNode).exponentialBackoffSync(helperNode);
      }
      if(!SangminConfig.useSimulatorId){
        ((IrisNode)helperNode).exponentialBackoffSync(thirdNode);
        ((IrisNode)helperNode).exponentialBackoffSync(thirdNode);
      }
//    printRead(forkedPseudoNodes.get(0), COARSE_PATH, 0, ni-1);
//    printRead(forkedPseudoNodes.get(1), COARSE_PATH, 0, ni-1);
      printRead(helperNode, COARSE_PATH, 0, ni-1);
      printRead(thirdNode, COARSE_PATH, 0, ni-1);

    }
  }catch(Exception e){
    // TODO Auto-generated catch block
    e.printStackTrace();
    assert false;
  }
}

  public static void checkRead(Node n, 
      String prefix, 
      int is, int ie, int startVal, int numVersions){
      int ii;

    if(vverbose){
      System.out.println("Reading initial data: " + is + " to " + ie + " etc.");
    }
    for(ii = is; ii <= ie; ii++){

      ObjId oid = new ObjId(prefix + "/" + ii );
      if(vverbose){
        System.out.println("Reading " + oid);
      }
      LinkedList<StoreEntry> rawVals = n.rawRead(oid);
      if(numVersions != -1){
        assert rawVals.size() == numVersions:oid + "expected " + numVersions + " got " + rawVals.size() + " vals:" + rawVals;
      }
      Object val = resolveConflict(rawVals);
      assert val instanceof IrisDataObject: val;
      Object data = ((IrisDataObject)val).getObject();
      assert data.equals(startVal + ii): "Expected value for obj " + oid + " was " + (startVal + ii) + " instead found " +  data + " rawValues " + rawVals;
    }
  }
  
  public static void checkRead(Node n, 
      String prefix, 
      int is, int ie, int startVal){
      checkRead(n, prefix, is, ie, startVal, -1);
  }
  
  public static IrisObject resolveConflict(LinkedList<StoreEntry> ste){
    assert ste.size() > 0:ste;
    AcceptStamp as = ste.getFirst().getAcceptStamp();
    IrisObject data = ste.getFirst().getData();
    for(StoreEntry se: ste){
      if(as.lt(se.getAcceptStamp())){
        data = se.getData();
        as = se.getAcceptStamp();
      }
    }
    return data;
  }

  public static void printRead(Node n, 
      String prefix, 
      int is, int ie){
      int ii;

    if(vverbose){
      System.out.println("Reading initial data: " + is + " to " + ie + " etc.");
    }
    for(ii = is; ii <= ie; ii++){

      ObjId oid = new ObjId(prefix + "/" + ii );
      if(vverbose){
        System.out.println("Reading " + oid);
      }
      Object o = n.read(oid);
      if(o != null){
        System.out.println("at node " + n + " Object " + oid + " has value " + o);
      }
    }
  }
  
  static byte[] testD = new byte[1000000];
  static{
    for(int c = 0; c < 1000000; c++){
      testD[c] = (byte)(c%128);
    }
  }
 /** 
 *  Write -- write 1-byte items to /prefix/[is.ie]/[js.je]/[ks.ke].
 
 *  Return final accept stamp.
 
 **/ 
  public static AcceptStamp write(Node n, 
                           String prefix, 
                           int is, int ie, 
                           int startVal){
    int ii;

    AcceptStamp as = null;
    if(vverbose){
      System.out.println("Writing initial data: " + is + " to " + ie + " etc.");
    }
    for(ii = is; ii <= ie; ii++){

      ObjId oid = new ObjId(prefix + "/" + ii );
      if(vverbose){
        System.out.println("Writing " + oid);
      }
     
      SimPreciseInv spi = n.write(oid, new IrisDataObject(
//          new Integer(startVal + ii)));
          new Integer(startVal + ii)));
      as = spi.getAcceptStamp();
    }
    return as;
  }
    

  /*
   * "new TestSuite(Class c)" constructs a test suite
   * containg every method whose name begins with "test"
   * 
   * TBD: update class name
   */
  public static Test suite(){
    TestSuite suite = new TestSuite(SimulatorUnit.class);
    return suite;
  }


  /*
   * main() lets us run just this set of unit tests
   * from the comand line (you can also invoke 
   * the testrunner on this class and it will find
   * the suite())
   *
   * usage: java <classname> [-verbose] [-vverbose] [testName]*
   * 
   *   If verbose or vverbose are included, print info to screen
   *
   *   If [testName]* are included, then run test called "test[testName]"
   *   for each such [testName]. E.g., "java TestEmtpy foo" runs
   *   TwoNodesSubscriptionUnit.testfoo() as a TestCase.
   *
   * TBD: update class name
   */
  public static void main(String s[]) {
    String name = "SimulatorUnit";
    System.err.print(name + " self test begins..");
    Test test;
    test = suite();
    TestRunner tr = new TestRunner();
    tr.doRun(test);
    System.exit(0);
  }

}



