package code.untrustedstorage.writeanyreadany;

import java.util.Collection;
import java.util.HashMap;
import java.util.Iterator;
import java.util.PriorityQueue;
import java.util.TreeSet;

import code.AcceptStamp;
import code.AcceptVV;
import code.CounterVV;
import code.NoSuchEntryException;
import code.NodeId;
import code.ObjId;
import code.branchDetecting.BranchID;
import code.simulator.Certificate;
import code.simulator.IrisHashObject;
import code.simulator.IrisNode;
import code.simulator.POMRemap;
import code.simulator.ProofOfMisbehavior;
import code.simulator.SimPreciseInv;
import code.simulator.agreement.Tuple;
import code.simulator.protocolFilters.POMFilter;
import code.simulator.protocolFilters.SyncFilter;
import code.simulator.store.StoreEntry;

public class CommitCutFilter implements SyncFilter, POMFilter{
  
  final static int quorumSize;
  static {
    quorumSize = StorageConfig.CommitQuorumSize;//Math.min(2, StorageConfig.numServers());
  }
  private HashMap<BranchID, PriorityQueue<TaggedTimeStamp>> queueMap;
  private CounterVV commitCut;
  
  private AcceptVV origCommitCut;
  private HashMap<BranchID, PriorityQueue<TaggedTimeStamp>> origQueueMap;
  
  private CachingPolicy cachingPolicy;
  
  public CommitCutFilter(CachingPolicy cp){
    queueMap = new HashMap<BranchID, PriorityQueue<TaggedTimeStamp>>();
    commitCut = new CounterVV();
    this.cachingPolicy = cp;
  }
  
  public synchronized AcceptVV getCommitCut(){
    return new AcceptVV(commitCut);
  }
  
  private void updateQueueMap(AcceptVV dvv, NodeId sId){
    
    for(NodeId id : dvv.getNodes()){
      PriorityQueue<TaggedTimeStamp> q;
      if(queueMap.containsKey(id)){
        q = queueMap.get(id);
      } else {
        q = new PriorityQueue<TaggedTimeStamp>();
        queueMap.put((BranchID)id, q);
      }
      long ts = -1L;
      try{
        ts = dvv.getStampByServer(id);
      }catch(NoSuchEntryException e){
        assert false;
      }
      Iterator<TaggedTimeStamp> iter = q.iterator();
      TaggedTimeStamp tts = null;
      while(iter.hasNext()){
        tts = iter.next();
        if(tts.serverId == sId){
          iter.remove();
          break;          
        }
      }
      if(tts == null || tts.timeStamp < ts){
        q.add(new TaggedTimeStamp(ts, sId));
      } else {
        q.add(tts);
      }
    }
    
    return;
    
  }
  
  /**
   * Update the commit cut consistent with the queueMap
   */
  private void updateCommitCut(){
    
    for(NodeId id : queueMap.keySet()){
      PriorityQueue<TaggedTimeStamp> q = queueMap.get(id);
      for(int i=0; i<quorumSize-1; i++){
        q.poll();
      }
      TaggedTimeStamp tts = q.poll();
      if(tts == null){
        commitCut.setStampByServer(id, AcceptStamp.BEFORE_TIME_BEGAN);
      } else {
        commitCut.setStampByServer(id, tts.timeStamp);
      }      
    }
    
    return;
    
  }
  
  public synchronized boolean receivedNewWrites(IrisNode irisNode, TreeSet<SimPreciseInv> newUpdates, Collection<Tuple<ObjId, TreeSet<StoreEntry>>> addedEntries){
    
    // Backup my state
    origCommitCut = new AcceptVV(commitCut);
    origQueueMap = new HashMap<BranchID, PriorityQueue<TaggedTimeStamp>>();
    for(BranchID id : queueMap.keySet()){
      origQueueMap.put(id, new PriorityQueue<TaggedTimeStamp>(queueMap.get(id)));
    }
    
    // update commit cut
    
    for(SimPreciseInv spi : newUpdates){
      String path = spi.getObjId().getPath();
      if(path.startsWith(NamespaceLayout.COMMIT_OBJ_ROOT)){
        AcceptVV dvv = spi.getDVV();
        updateQueueMap(dvv, 
            new BranchID(NamespaceLayout.getServerIdFromCommitObjId(spi.getObjId())));        
      }      
    }
    
//    CounterVV originalCommitCut = new CounterVV(commitCut);
    updateCommitCut();
    
    Iterator<Tuple<ObjId, TreeSet<StoreEntry>>> iter = addedEntries.iterator();
    while(iter.hasNext()){
      Tuple<ObjId, TreeSet<StoreEntry>> tuple = iter.next();
      for(StoreEntry se : tuple.getValue()){
        if(!commitCut.includes(se.getAcceptStamp()) && se.getData() instanceof IrisHashObject){
//          commitCut = originalCommitCut;
          return false;
        } 
      }      
    }
    
//    // Check if all writes newer than the commit cut have their body attached
//    for(SimPreciseInv spi : newUpdates){
//      if(!commitCut.includes(spi.getAcceptStamp())){
//        if(spi.getData() instanceof IrisHashObject){
//          // restore the commit cut
//          commitCut = originalCommitCut;
//          return false;
//        }
//      }
//    }
    
    if(this.cachingPolicy != null){
      this.cachingPolicy.cache(irisNode, origCommitCut, new AcceptVV(commitCut));
    }
    
    return true;
  }
  
  private synchronized void remapNodeId(POMRemap remap){
    
    PriorityQueue<TaggedTimeStamp> oldQ = queueMap.get(remap.getOrigNodeId());
    PriorityQueue<TaggedTimeStamp> newQ = new PriorityQueue<TaggedTimeStamp>(oldQ);
    
    long ts = remap.getTs();
    
    Iterator<TaggedTimeStamp> iter = newQ.iterator();
    TaggedTimeStamp tts = null;
    while(iter.hasNext()){
      tts = iter.next();
      if(tts.timeStamp < ts){
        iter.remove();
        break;          
      }
    }

    queueMap.put((BranchID) remap.getNewNodeId(), newQ);
    
    updateCommitCut();
    
    
  }

  public void receivePOM(IrisNode myNode, ProofOfMisbehavior pom){
    // Nothing To do
  }

  public void receivePOMRemap(IrisNode myNode, POMRemap pomRemap){
    remapNodeId(pomRemap);
    
  }

  public void notifyEpochSync(IrisNode myNode, Certificate cert)
      throws Exception{
    // Nothing to do
  }

//  @Override
//  public boolean notifySync(IrisNode myNode, AcceptVV myCVV,
//      BranchID otherNodeId, TreeSet<SimPreciseInv> newUpdates) throws Exception{
//    return receivedNewWrites(myNode, newUpdates);
//  }

  public void syncOver(IrisNode myNode, boolean success) throws Exception{
    if(!success){
      commitCut = new CounterVV(origCommitCut);
      queueMap = origQueueMap;
    }
    origCommitCut = null;
    origQueueMap = null;
  }  
  
  private class TaggedTimeStamp implements Comparable<TaggedTimeStamp>{
    final long timeStamp;
    final NodeId serverId;
    
    TaggedTimeStamp(long ts, NodeId sId){
      timeStamp = ts;
      serverId = sId;
    }

    public int compareTo(TaggedTimeStamp o){
      return (new Long(this.timeStamp)).compareTo(o.timeStamp);
    }
    
  }

  public boolean notifySync(IrisNode myNode, AcceptVV myCVV,
      BranchID otherNodeId, TreeSet<SimPreciseInv> newUpdates,
      Collection<Tuple<ObjId, TreeSet<StoreEntry>>> addedEntries)
      throws Exception{
    // TODO Auto-generated method stub
    return receivedNewWrites(myNode, newUpdates, addedEntries);
  }

}

