package code.untrustedstorage.writeanyreadany.client;

import java.io.ByteArrayInputStream;
import java.io.ByteArrayOutputStream;
import java.io.File;
import java.io.IOException;
import java.io.InputStream;
import java.io.ObjectInputStream;
import java.io.ObjectOutputStream;
import java.io.Serializable;
import java.net.Socket;
import java.net.UnknownHostException;
import java.rmi.Naming;
import java.rmi.NotBoundException;
import java.rmi.RemoteException;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.HashMap;
import java.util.HashSet;
import java.util.LinkedList;
import java.util.List;
import java.util.Random;
import java.util.Set;
import java.util.TreeSet;
import java.util.Vector;

import code.lasr.db.DbException;
import code.lasr.db.DbTransaction;
import code.lasr.db.TxnFailedException;
import code.simulator.agreement.Tuple;
import code.simulator.irisnetty.IrisNetworkQueue;
import code.simulator.irisnetty.NetworkHandler;
import code.simulator.log.Log;
import code.simulator.netty.NettyTCPReceiver;

import code.AcceptVV;
import code.Config;
import code.Env;
import code.NodeId;
import code.ObjId;
import code.SubscriptionSet;
import code.branchDetecting.BranchID;
import code.security.SangminConfig;
import code.security.SangminConfig.DebugLevel;
import code.serialization.IrisInputStream;
import code.serialization.IrisObjectInputStream;
import code.serialization.IrisObjectOutputStream;
import code.serialization.IrisOutputStream;
import code.serialization.SerializationHelper;
import code.simulator.Hash;
import code.simulator.IrisDataObject;
import code.simulator.IrisHashObject;
import code.simulator.IrisNode;
import code.simulator.IrisObject;
import code.simulator.NamespaceWatch;
import code.simulator.NodeFactory;
import code.simulator.SimPreciseInv;
import code.simulator.SyncRequest;
import code.simulator.SyncStatus;
import code.simulator.netty.NettyTCPSender;
import code.simulator.netty.NetworkQueue;
import code.simulator.persistentLog.PersistentStore;
import code.simulator.store.BodyStore;
import code.simulator.store.BodyStoreInterface;
import code.simulator.store.PersistentBodyStore;
import code.simulator.store.StoreEntry;
import code.untrustedstorage.writeanyreadany.BodyRequestData;
import code.untrustedstorage.writeanyreadany.CommitCutFilter;
import code.untrustedstorage.writeanyreadany.NamespaceLayout;
import code.untrustedstorage.writeanyreadany.NewWriteRequestData;
import code.untrustedstorage.writeanyreadany.P2PBodyRequestData;
import code.untrustedstorage.writeanyreadany.Reply;
import code.untrustedstorage.writeanyreadany.Request;
import code.untrustedstorage.writeanyreadany.S3Inval;
import code.untrustedstorage.writeanyreadany.StorageConfig;
import code.untrustedstorage.writeanyreadany.WatcherForLatency;
import code.untrustedstorage.writeanyreadany.Request.RequestType;
import code.untrustedstorage.writeanyreadany.client.ServerChooserInterface.OpType;
import code.untrustedstorage.writeanyreadany.server.ServerListener;

public class ClientNode implements NetworkQueue{

  private BranchID myId;

    public static boolean printException = true;
  private Set<BranchID> syncServerIdSet;
  private Set<BranchID> syncClientIdSet;
  //  private HashMap<BranchID, ServerRMIInterface> serverRMIs;

  private IrisNode irisNode;
  
  // This thread periodically syncs with servers
  private SyncThread syncThread;
  private BeaconWriter beaconWriter;
  private BeaconWatcher beaconWatcher;

  public boolean ServerClientPartitionPhase = false;
  Random r = new Random();

  NetworkHandler networkHandler;

  private boolean shutdown = false;
  /**
   * Listener for a client to handle read requests
   */
  ServerListener sl;

  ServerChooserInterface serverChooser;
  ClientConnectionManager clientConnMgr;

  public ClientNode(int myId, int confIdx, int primaryWriteServerId, int primaryReadServerId, Set<Integer> syncServerIdSet, Set<Integer> clientIdSet, Set<Integer> serverIdSet)
  throws NotBoundException, UnknownHostException, IOException{
	  init(myId, confIdx, primaryWriteServerId, primaryReadServerId, syncServerIdSet, clientIdSet, serverIdSet);
  }
  
  public ClientNode(int myId, int primaryWriteServerId, int primaryReadServerId, Set<Integer> syncServerIdSet, Set<Integer> clientIdSet, Set<Integer> serverIdSet)
  throws NotBoundException, UnknownHostException, IOException{
	  init(myId, myId, primaryWriteServerId, primaryReadServerId, syncServerIdSet, clientIdSet, serverIdSet);
  }
  
  private void init(int myId, int confIdx, int primaryWriteServerId, int primaryReadServerId, Set<Integer> syncServerIdSet, Set<Integer> clientIdSet, Set<Integer> serverIdSet)
  throws NotBoundException, UnknownHostException, IOException{

    this.myId = NodeFactory.createNodeId(myId);
    
    BranchID configId = NodeFactory.createNodeId(confIdx);
    
    if(StorageConfig.S3Emulation){
      // hack to avoid berkeley db creation on S3Emulations
      boolean tmp = SangminConfig.usePersistentStore;
      SangminConfig.usePersistentStore = false;
      this.irisNode = (IrisNode) NodeFactory.createIrisNode(this.myId, configId);
      SangminConfig.usePersistentStore = tmp;
    }else{
      this.irisNode = (IrisNode) NodeFactory.createIrisNode(this.myId, configId);
    }

    if(StorageConfig.useNetty){
      networkHandler = new NetworkHandler(confIdx, new IrisNetworkQueue(this));
    }

    this.syncServerIdSet = new HashSet<BranchID>();
    for(Integer id : syncServerIdSet){
      BranchID serverId = NodeFactory.createNodeId(id);
      this.syncServerIdSet.add(serverId);
    }

    this.syncClientIdSet = new HashSet<BranchID>();
    for(Integer id : clientIdSet){
      if ( id == myId) continue;
      BranchID serverId = NodeFactory.createNodeId(id);
      this.syncClientIdSet.add(serverId);
    }

    serverChooser = new ServerChooser(serverIdSet, serverIdSet, this.syncServerIdSet, syncClientIdSet);
    clientConnMgr = new ClientConnectionManager(primaryWriteServerId, primaryReadServerId);

    syncThread = this.new SyncThread();
    syncThread.start();

    // Start thread for beacon
    if(StorageConfig.useBeacon){
      beaconWatcher = new BeaconWatcher(this, clientIdSet);
      irisNode.addNamespaceWatch(beaconWatcher, SubscriptionSet.makeSubscriptionSet(NamespaceLayout.BEACON_OBJ_ROOT + "/*"));
      beaconWriter = new BeaconWriter(this, StorageConfig.BeaconInterval);
      beaconWriter.start();      
    }

    // Install commit cut filter
    if(StorageConfig.useCommit){
      CommitCutFilter cf;
      if(StorageConfig.useClientBodyCleanup){
        cf = new CommitCutFilter(new ClientCachingPolicy());
      } else {
        cf = new CommitCutFilter(null);
      }

      irisNode.addSyncFilter(cf);
      irisNode.addPOMFilter(cf);

    }
    if(StorageConfig.NoBodyOnClientServerSync){
      Config.setBodyConfig(new NodeId(myId), SyncRequest.NoBodies);
    }    

    //irisNode.addNamespaceWatch(new WatcherForLatency(this.myId, "RECEIVED WRITE"), SubscriptionSet.makeSubscriptionSet("/*"));
    /**
     * we don't need to have "otherServers" because a client doesn't handle write requests that need to be forwarded.
     * Setting null to networkHandler ensures that the messages are not forwarded 
     */
    sl = new ServerListener(irisNode, new TreeSet<BranchID>(), null);

    sl.start();
  }

  public BranchID getID(){
    return myId;
  }

  /**
   * helper method
   * @param nodeId
   * @return
   */
  public SyncStatus syncTo(int nodeId){
    SyncStatus ss = null;
    try{
      ss = irisNode.exponentialBackoffSync(NodeFactory.createNodeId(nodeId), irisNode.getCurrentVV());
    }catch(Exception e){
      e.printStackTrace();
      return null;
    }
    return ss;
  }

  /**
   * helper method
   * @param nodeId
   * @param o
   * @return
   */
  public SyncStatus syncTo(int nodeId, ObjId o){
    SyncStatus ss = null;
    try{
      ss = irisNode.exponentialBackoffSync(NodeFactory.createNodeId(nodeId), irisNode.getCurrentVV(), o);
    }catch(Exception e){
      e.printStackTrace();
      return null;
    }
    return ss;
  }

  /**
   * notify at least one server about the write
   * @param spi
   * @return
   */
  private synchronized boolean notifyPrimaryWriteServer(Object spi) {    

    long sentBytes = 0;
    long receivedBytes = 0;
    IrisDataObject io = null;
    ObjId oid;
    long start = System.currentTimeMillis();
    long writeFinish = -1;
    long receivedReply = -1;
    int bodysize = -1;
    Request req;
    if(!StorageConfig.S3Emulation){
      req = new Request(RequestType.NEW_WRITES, new NewWriteRequestData((SimPreciseInv)spi, 
          (IrisDataObject)((SimPreciseInv)spi).getData()));
      oid = ((SimPreciseInv)spi).getObjId();
      io = ((IrisDataObject)((SimPreciseInv)spi).getData());
      bodysize = ((IrisDataObject)((SimPreciseInv)spi).getData()).getUnsafeBytes().length;
    } else {
      assert spi instanceof S3Inval;
      req = new Request(RequestType.NEW_WRITES, spi);
      oid = ((S3Inval)spi).getObjId();
      io = ((S3Inval)(spi)).getValue();
      bodysize = ((S3Inval)(spi)).getValue().getUnsafeBytes().length;
    }

    IrisClientServerConnection sc = null;
    LinkedList<Integer> servers = serverChooser.getServers(OpType.Write);
    System.out.println();
    if(servers.size() > 0){
      sc = clientConnMgr.getNextConnection(servers, OpType.Write);
    }
    boolean serverNotified = false;
    while(sc != null && sc.isConnected() && !serverNotified){
      try{
        if(SangminConfig.debugLevel == DebugLevel.Verbose)System.out.println(this.myId + " using " + sc.getServerId() + " for writes");
        IrisOutputStream ios = sc.getIos();
        IrisInputStream ois = sc.getIis();

        long sendStart = 0, recvStart = 0;
        if(SangminConfig.useByteArraysForIO){
          sc.getSocket().getOutputStream().write(SerializationHelper.getBufWithLength(req));
        }
        else{
          sendStart = ios.getBytes();
          recvStart = ois.getBytesRead();
          req.writeToStream(ios, true);
        }
        if(SangminConfig.debugLevel == DebugLevel.Verbose)System.out.println(System.currentTimeMillis() + " " + this.myId + " sent write " + req);

        sentBytes += ios.getBytes()-sendStart;

        ios.flush();
        sc.getSocket().getOutputStream().flush();
        writeFinish = System.currentTimeMillis();
        Env.logWrite(this.myId + " sent " + (sentBytes-bodysize) + " as NewWriteRequest-metadata to " + sc.getServerId());
        Env.logWrite(this.myId + " sent " + bodysize + " as NewWriteRequest-bodies to " + sc.getServerId());

        Reply rep = new Reply();
        if(SangminConfig.useByteArraysForIO){
          ois = new IrisInputStream(new ByteArrayInputStream(SerializationHelper.getBufWithoutLength(sc.getSocket().getInputStream())));
        }
        rep.readFromStream(ois, false);
        assert rep.getType() == req.type;
        if(SangminConfig.debugLevel == DebugLevel.Verbose)System.out.println(System.currentTimeMillis() + " " + this.myId + " received reply " + rep);
        receivedBytes += ois.getBytesRead()-recvStart;
        serverNotified = (Boolean)rep.getData();
        if(!serverNotified){
          System.err.println("Server " + sc + " notification failed " + spi);
        }
      }catch(IOException e){
	  if(printException)e.printStackTrace();
        try {
          sc.getSocket().close();
        } catch (IOException e1) {
	    if(printException)e1.printStackTrace();
        }
        sc.setConnected(false);
      }

      receivedReply = System.currentTimeMillis();
      if(SangminConfig.fineGrainedTracking)System.out.println("WRITE SEND TIME = " + (writeFinish-start));
      if(SangminConfig.fineGrainedTracking)System.out.println("SERVER PROCESSSING AND TRANSFER COST = " + (receivedReply - writeFinish));
      if(!serverNotified){
        sc = clientConnMgr.getNextConnection(OpType.Write);
      }
    }
    /**
     * if we weren't alredy in p2p mode and we failed to notify any server then 
     * turn on the p2p mode
     */
    if(!serverNotified && servers.size() > 0){
      this.serverChooser.notifyAllServersFailed();
    }
    Env.logEvent(System.currentTimeMillis() + " PUT myNodeId: " + this.myId + " oid: [" + oid + "] " + getVV() + 
        " " + io.getHash() + " sentBytes: " + sentBytes +  " receivedBytes: " + receivedBytes);
    return serverNotified;
  }
  
  public void write(ObjId objId, Object value){
    assert value instanceof byte[];
    this.writeWithStat(objId, new IrisDataObject((byte[])value));
  }
  
  public Tuple<Long, IrisDataObject> writeWithStat(ObjId objId, IrisDataObject ido){
    Env.logWrite(this.myId + " ISSUED WRITE " + ido.getHash());
    long localCompletionTime;
    if(StorageConfig.S3Emulation){
      localCompletionTime = System.nanoTime();
      notifyPrimaryWriteServer(new S3Inval(objId, ido));
    } else {
      SimPreciseInv spi = irisNode.write(objId, ido);
      localCompletionTime = System.nanoTime();
      Env.logWrite(this.myId + " WRITE LOCAL RETURN " + ido.getHash());
      //      if(SangminConfig.debugLevel == DebugLevel.Verbose)System.out.println("Local node " + this.getID() + " wrote: "  + spi);
      if(SangminConfig.separateBodies){
        spi = spi.cloneWithNewNodeId(spi.getNodeId());
        spi.replaceIrisObject(ido);
      }
      // Notify my primary server of this write
      notifyPrimaryWriteServer(spi);
    }
    return new Tuple<Long, IrisDataObject>(localCompletionTime, ido);
  }

  /**
   * obtain body from the specified connection and return the stats
   * @param oid
   * @param hashObjList
   * @param sc
   * @return
   */
  private synchronized Tuple<AccessStat, LinkedList<IrisObject>> getBodyFromServer(
      ObjId oid, List<IrisHashObject> hashObjList, IrisClientServerConnection sc, boolean p2pMode) 
      {
    IrisInputStream iis = sc.getIis();
    IrisOutputStream ios = sc.getIos();
    if(SangminConfig.debugLevel == DebugLevel.Verbose)System.out.println(this.myId + " using " + sc.getServerId() + " for reads");
    long sendStart =0, recvStart = 0;
    long t0,t1,t2,t3;

    long sentBytes = 0;
    long receivedBytes = 0;
    LinkedList<IrisObject> data = null;

    t0 = System.currentTimeMillis();
    Request req; 
    if(p2pMode){
      req = new Request(RequestType.REQUEST_P2P_BODY, new P2PBodyRequestData(oid, hashObjList));
    }else{
      req = new Request(RequestType.REQUEST_BODY, new BodyRequestData(oid, hashObjList));
    }
    sendStart = ios.getBytes();
    recvStart = iis.getBytesRead();

    try{
      t1 = System.currentTimeMillis();

      ByteArrayOutputStream bs;
      if(SangminConfig.useByteArraysForIO){
        bs = new ByteArrayOutputStream(1024);
        ios = new IrisOutputStream(bs);
        sendStart = ios.getBytes();
      }

      if(SangminConfig.debugLevel == DebugLevel.Verbose)System.out.println(System.currentTimeMillis() + " " + this.myId + " sent request " + req);
      req.writeToStream(ios, true);
      ios.flush();
      if(SangminConfig.useByteArraysForIO){
        sc.getSocket().getOutputStream().write(SerializationHelper.getBufWithLength(bs.toByteArray()));
      }
      Env.logWrite(this.myId + " sent " + (ios.getBytes()-sendStart) + " as BodyRequest to " + sc.getServerId());

      t2 = System.currentTimeMillis();
      Reply rep = new Reply();
      if(SangminConfig.useByteArraysForIO){
        iis = new IrisInputStream(new ByteArrayInputStream(SerializationHelper.getBufWithoutLength(sc.getSocket().getInputStream())));
        recvStart = iis.getBytesRead();
      }

      rep.readFromStream(iis, false);
      if(SangminConfig.debugLevel == DebugLevel.Verbose)System.out.println(System.currentTimeMillis() + " " + this.myId + " received reply " + rep);
      assert rep.getType() == req.type: "reply type " + rep.getType() + " expected type " + req.type;
      t3 = System.currentTimeMillis();
      data = (LinkedList<IrisObject>)rep.getData();

      if(SangminConfig.fineGrainedTracking)System.out.println("GETBODY REQUEST BEFORE SERIALIZATION = " + (t1-t0));
      if(SangminConfig.fineGrainedTracking)System.out.println("GETBODY REQUEST SERIALIZATION = " + (t2-t1));
      if(SangminConfig.fineGrainedTracking)System.out.println("GETBODY REPLY SERIALIZATION = " + (t3-t2));
    }catch(Exception e){
      if(printException)e.printStackTrace();
    }

    sentBytes = ios.getBytes()-sendStart;
    receivedBytes = iis.getBytesRead()-recvStart;

    return new Tuple<AccessStat, LinkedList<IrisObject>>(new AccessStat(sentBytes, receivedBytes), data);
      }

  /**
   * returns the objects for which the argument list contains IrisHashObject---
   * used to test if a read was successful or not
   * @param list
   * @return
   */
  private List<IrisHashObject> getHashObject(List<IrisObject> list){
    List<IrisHashObject> ret = new LinkedList<IrisHashObject>();
    for(IrisObject io : list){
      if(io instanceof IrisHashObject){
        ret.add((IrisHashObject) io);
      }
    }
    return ret;
  }

  /*
   * UNUSED AND BADLY NAMED FUNCTION
  private LinkedList<IrisObject> replaceHashWithData(List<IrisObject> l1, List<IrisDataObject> dataList){

    HashMap<Integer, IrisObject> map = new HashMap<Integer, IrisObject>();
    for(IrisObject io : l1){
      map.put(io.hashCode(), io);
    }
    for(IrisObject ido : dataList){
      assert ido instanceof IrisDataObject;
      map.put(ido.hashCode(), ido);
    }
    return new LinkedList<IrisObject>(map.values());
  }
   */

  /**
   * eval related function
   */
  public String getVV(){
    return irisNode.getCurrentVV().toShortString();
  }
  
  public AcceptVV getVVRaw(){
    return irisNode.getCurrentVV();
  }

  public Tuple<AccessStat, LinkedList<Object>> readS3(ObjId objId, IrisClientServerConnection sc) throws DataNotFound{
    //Env.logWrite(myId + " READ ISSUED with oid " +objId);
    assert(StorageConfig.S3Emulation);

    final long startTime = System.nanoTime();

    Tuple<AccessStat, LinkedList<IrisObject>> getBodyReply = getBodyFromServer(objId, null, sc, false);

    LinkedList<Object> ret = new LinkedList<Object>();
    AccessStat as = getBodyReply.getKey(); 
    as.setSuccess(true);
    if(getBodyReply.getValue() != null){
      for(IrisObject io : getBodyReply.getValue()){
        assert io instanceof IrisDataObject;
        ret.add(((IrisDataObject)io).getUnsafeBytes());
        as.addHash(io.getHash());
      }
    }
    assert ret != null;
    long endTime = System.nanoTime();
    Env.logWrite(myId + " COMPLETE READ LATENCY " + (((double)(endTime -startTime)/1000000)));
    return new Tuple<AccessStat, LinkedList<Object>>(as, ret);
  }

  public Tuple<AccessStat, LinkedList<Object>> readPeer(ObjId objId) throws DataNotFound{
    LinkedList<StoreEntry> seList = irisNode.efficientRawRead(objId);
    LinkedList<IrisDataObject> list = new LinkedList<IrisDataObject>();
    AccessStat readStat = new AccessStat(0, 0);
    LinkedList<IrisObject> tmpRes;
    LinkedList<IrisHashObject> hashObjList = new LinkedList<IrisHashObject>();
    Tuple<AccessStat, LinkedList<IrisObject>> irisRet;
    try{
      for(StoreEntry se : seList){
        if(!se.getAcceptStamp().getNodeId().equals(irisNode.getBranchID())){
          hashObjList.add((IrisHashObject)se.getData());
          irisRet = this.getBodyFromServer(objId, hashObjList, this.clientConnMgr.getConnection((int)se.getAcceptStamp().getNodeId().getIDint()), true);
          readStat.addReadStat(irisRet.getKey());
          hashObjList.clear();
          tmpRes = irisRet.getValue();
        }else{
          LinkedList<IrisObject> tmpIrisObjList = new LinkedList<IrisObject>();
          tmpIrisObjList.add(se.getData());
          tmpRes = irisNode.immutableRead(tmpIrisObjList);
        }
        if(tmpRes.size() == 1 && (tmpRes.getFirst() instanceof IrisDataObject)){
          list.add((IrisDataObject)tmpRes.getFirst());
        }else{
          // failed to obtain bodies at some point so return null
          throw new DataNotFound("slow p2p read failed");
          //        return new Tuple<AccessStat, LinkedList<Object>>(readStat, null);
        }
      }
    }catch(UnknownHostException e){
	if(printException)e.printStackTrace();
      throw new DataNotFound("slow p2p read failed");
    }catch(IOException e){
	if(printException)e.printStackTrace();
      throw new DataNotFound("slow p2p read failed");
    }
    LinkedList<Object> ret = new LinkedList<Object>();
    for(IrisObject io : list){
      assert io instanceof IrisDataObject;
      ret.add(((IrisDataObject)io).getUnsafeBytes());
      readStat.addHash(io.getHash());
    }
    readStat.setSuccess(true);
    return new Tuple<AccessStat, LinkedList<Object>>(readStat, ret);

  }

  public LinkedList<Object> read(ObjId objId) throws DataNotFound{
    return this.readWithStats(objId).getValue();
  }
  
  public Tuple<AccessStat, LinkedList<Object>> readWithStats(ObjId objId) throws DataNotFound{
    Tuple<AccessStat, LinkedList<Object>> retVal = null;
    //Env.logWrite(myId + " READ ISSUED with oid " +objId);
    System.out.println("READ issued with oid "+objId);

    final long startTime = System.nanoTime();
    final long startTimeMillis = System.currentTimeMillis();
    long t1=0,t2,t3;

    /**
     * has the read completed yet?
     */
    boolean success = false;

    AccessStat stats = null;

    LinkedList<Integer> servers = serverChooser.getServers(OpType.Read);
    IrisClientServerConnection sc = null;
    if(servers.size() > 0){
      sc = clientConnMgr.getNextConnection(servers, OpType.Read);
    }
    Tuple<AccessStat, LinkedList<IrisObject>> irisRetVal = null;
    stats = new AccessStat(0,0);
    
    while(!success && sc != null && sc.isConnected()){
      if(StorageConfig.S3Emulation){
        retVal = this.readS3(objId, sc);
        stats = retVal.getKey();
        success = true;
      }else{
        //perhaps at the end of this function?
        if(StorageConfig.useBeacon){
          if(beaconWatcher.checkBeacon()){

          } else {
            System.out.println("BEACON EXPIRED");
          }
        }

        /**
         * the set of hashes of current bodies for the specified key
         */
        LinkedList<IrisObject> localHashList = irisNode.efficientRead(objId);
        /**
         * the list for actual objects
         */
        LinkedList<IrisObject> localObjList = null;
        Env.logWrite(myId + " LATENCY-EFFICIENT_READ " + (System.currentTimeMillis() - startTimeMillis) );
        if(localHashList.size() <= 0){
          assert false ; // shouldn't happen in our experiments since we have preload
          SyncStatus ss = null;
          t1 = System.currentTimeMillis();

          // obtain updates from the server along with bodies
          do{
            ss = this.syncTo(sc.getServerId(), objId);
          } while( ss!= null && ss.status != SyncStatus.SyncSuccessful && ss.status != SyncStatus.NetworkError);
          t2 = System.currentTimeMillis();

          // read the recent values for objId
          localHashList = irisNode.read(objId);

          localObjList = localHashList; //because in this case we should have received bodies for all most recent updates to objId

          t3 = System.currentTimeMillis();
          Env.logWrite(myId + " Local Iris Read Op Latency " + (t3-t2) + " NO INITIAL HASH");
          Env.logWrite(myId + " Local Iris Read SLOW PATH Latency " + (t3-t1) + " NO INITIAL HASH");
        } else {
          Env.printDebug(System.currentTimeMillis() +"servers to contact " + servers);

          List<IrisHashObject> hashList = getHashObject(localHashList);

          long t0 = System.currentTimeMillis();
          Env.logWrite(myId + " TIME Spent before go to server " + (t0-startTimeMillis));

          irisRetVal = 
            getBodyFromServer(objId, hashList, sc, false);

          stats.addReadStat(irisRetVal.getKey());

          localObjList = irisRetVal.getValue();

          t1 = System.currentTimeMillis();
          Env.logWrite(myId + " GET BODY FROM SERVER LATENCY " + (t1-t0));
          if(SangminConfig.fineGrainedTracking)System.out.println(myId + " GET BODY FROM SERVER LATENCY " + (t1-t0));

          if(localObjList == null || localObjList.size() <=0){ 
            // this case implies that our local set of hashes didn't match
            // the server's most recent hashes and therefore the server returned null or empty list 
            // (as an optimization) instead of returning the matching subset of values
            // so we try to sync from the server, hoping that the server has newer updates that 
            // we will obtain through the sync
            SyncStatus ss = null;
            t1 = System.currentTimeMillis();
            do{
              ss = this.syncTo(sc.getServerId(), objId);
            } while( ss!= null && ss.status != SyncStatus.SyncSuccessful && ss.status != SyncStatus.NetworkError);

            t2 = System.currentTimeMillis();

            if(ss != null && ss.status == SyncStatus.SyncSuccessful){
              // now the read should return the objects---otherwise this server can't help us, move to other servers
              localObjList = irisNode.read(objId);
              localHashList = localObjList; //because in this case we should have received bodies for all most recent updates to objId
              assert localHashList.size() > 0;

              t3 = System.currentTimeMillis();
              Env.logWrite(myId + " Second Local Iris Read Op Latency " + (t3-t2));
              Env.logWrite(myId + " Second Local Iris Read SLOW PATH Latency " + (t3-t1));
            }
          } 
          else if(localObjList.size() != localHashList.size()){
            assert localObjList.size() == localHashList.size(): "localObjList:" + localObjList +"\nlocalHashList:" + localHashList;
          }
        }

        // check if the values we finally have match the verified hashes in our store for objId 
        success = compareObjHashList(localObjList, localHashList);

        if(!success){
          sc = clientConnMgr.getNextConnection(OpType.Read);
        }else{
          LinkedList<Object> ret = new LinkedList<Object>();
          for(IrisObject io : localObjList){
            ret.add(((IrisDataObject)io).getUnsafeBytes());
            stats.addHash(io.getHash());
          }
          retVal = new Tuple<AccessStat, LinkedList<Object>>(stats, ret);
        }

        // now print the debug message

        //      Env.logWrite(myId + " TIME SPENT BEFORE PREPAREING RET_VAL " + (System.currentTimeMillis()-t1));

        long endTime = System.nanoTime();
        Env.logWrite(myId + " READ LATENCY in " + (((double)(endTime -startTime))/1000000));
        Env.logWrite(myId + " TIME SPENT AFTER GETTING BODY " + (endTime-t1*1000000));
        System.out.println("READ completed with oid "+objId + " latency " + (((double)(endTime - startTime))/1000000));
      }
    }

    // start p2p mode if we failed to obtain bodies in the client-server mode
    if(retVal == null || retVal.getValue() == null || retVal.getValue().size() == 0){
      assert !StorageConfig.S3Emulation;
      this.serverChooser.notifyAllServersFailed();
      retVal = readPeer(objId);
      stats.addReadStat(retVal.getKey());
      success = true;
    }
    stats.setSuccess(success);
    assert stats != null;
    assert success;
    assert retVal != null;
    assert retVal.getKey().getHashString() != null;
    return retVal;
  }

  /**
   * check if the localObjList has object values corresponding to exactly the hashes present in the localHashList
   * @param localObjList
   * @param localHashList
   * @return
   */
  private static boolean compareObjHashList(LinkedList<IrisObject> localObjList, LinkedList<IrisObject> localHashList){
    // common case
    if((localObjList == null && localHashList !=null) || 
        (localObjList != null && localHashList == null)){
      return false;
    }
    else if(localObjList.size() != localHashList.size()){
      return false;
    }
    // check that all the values in localObjList are instances of IrisDataObject
    for(IrisObject io:localObjList){
      if(!(io instanceof IrisDataObject)){
        return false;
      }
    }

    // if succeeded so far then check that all the hashes in localHashList are present in the 
    // localObjList

    // first create a list of hashes of values in localObjList
    TreeSet<Hash> c1 = new TreeSet<Hash>();
    for(IrisObject io : localHashList){
      c1.add(io.getHash());
    }

    TreeSet<Hash> c2 = new TreeSet<Hash>();
    for(IrisObject io : localObjList){
      c2.add(io.getHash());
    }

    if(!c1.equals(c2)){
      return false;
    }

    return true;
  }

  
  public void shutdown(){
    this.shutdown = true;
    irisNode.close(); 
    this.sl.close();
    if(StorageConfig.useNetty){
      networkHandler.close();
    }
  }
  
  private class SyncThread extends Thread{

    public SyncThread(){
      this.setName("Client Node Sync Thread for node " + myId);
      this.setDaemon(true);
    }

    public void run(){
      if(StorageConfig.ClientServerSyncInterval < 0L || StorageConfig.S3Emulation){
        return;
      }

      try{
        Thread.sleep(5000);
      }catch(InterruptedException e1){
      }

      while(!shutdown){

        Collection<BranchID> syncServerIdSet = serverChooser.getSyncServers();
        syncServerIdSet.remove(irisNode.getBranchID());
        for(BranchID bid : syncServerIdSet){
          try{
            SyncStatus ss;
	    AcceptVV oldVV = irisNode.getCurrentVV();
            int count = 3;
            do{
              count--;
              ss = irisNode.exponentialBackoffSync(bid, irisNode.getCurrentVV());
            } while(ss.status == SyncStatus.NetworkError && count >= 0);

	    //	    if(SangminConfig.debugLevel==DebugLevel.Verbose)
		//		Env.printDebug("sync status " + ss + "\n oldVV: " + oldVV + "\nnewVV:"+irisNode.getCurrentVV());
          }catch(Exception e){
            e.printStackTrace();
          }
          try{
            sleep(StorageConfig.ClientServerSyncInterval);
          }catch(InterruptedException e){

          }
        }

      }

    }

  }


  public void addWork(byte[] msg){
  }

  /* (non-Javadoc)
   * @see java.lang.Object#toString()
   */
  @Override
  public String toString(){
    return "ClientNode [irisNode=" + irisNode + "]";
  }

  public void dumpLogs(){
    this.irisNode.dumpLogs();
  }

  

}


//class ForceManager {
//  private boolean forceNeeded = false;
//  private IrisNode irisNode;
//  private ObjId forceObjId;
//
//  ForceManager(IrisNode irisNode){
//    this.irisNode = irisNode;
//    forceObjId = new ObjId(NamespaceLayout.getForceObjID(irisNode.getBranchID()));
//  }
//
//  private long lastTimeForceWritten = 0L; // when the most recent "force" is written
//
//  synchronized boolean isForceNeeded(){
//    return forceNeeded;
//  }
//
//  synchronized long writeForce(){
//    SimPreciseInv spi = irisNode.write(forceObjId, new IrisDataObject("force"));
//    lastTimeForceWritten = System.currentTimeMillis();
//    forceNeeded = false;
//    return spi.getAcceptStamp().getLocalClock();
//  }
//
//  synchronized void requestForce(){
//    forceNeeded = true;
//  }
//
//  long getLastTimeForceWritten(){
//    return lastTimeForceWritten;
//  }
//
//} 
//
//
//class ForceThread extends Thread{
//
//  private ForceManager forceManager;
//  private long forceInterval;
//  //private IrisNode irisNode;
//
//  ForceThread(ForceManager forceManager, long forceInterval){
//    this.forceManager = forceManager;
//    this.forceInterval = forceInterval;
//    this.setDaemon(true);
//  }
//
//  public void run(){
//
//    long sleepDuration = forceInterval;
//
//    while(true){
//
//      try{
//        sleep(sleepDuration);
//      }catch(InterruptedException e){
//        // TODO Auto-generated catch block
//        e.printStackTrace();
//      }
//
//      long timeElapsedSinceLastForce = (System.currentTimeMillis() - forceManager.getLastTimeForceWritten());
//
//      if(forceManager.isForceNeeded() && timeElapsedSinceLastForce > forceInterval  ){
//        forceManager.writeForce();
//        sleepDuration = forceInterval;
//      } else {
//        if(forceManager.isForceNeeded()){
//          sleepDuration = forceInterval - timeElapsedSinceLastForce;
//        } else {
//          sleepDuration = forceInterval;
//        }
//      }
//
//    }
//  }    
//
//}
