package code.untrustedstorage.writeanyreadany.client;

import java.io.ByteArrayInputStream;
import java.io.ByteArrayOutputStream;
import java.io.File;
import java.io.IOException;
import java.io.InputStream;
import java.io.ObjectInputStream;
import java.io.ObjectOutputStream;
import java.io.Serializable;
import java.net.Socket;
import java.net.UnknownHostException;
import java.rmi.Naming;
import java.rmi.NotBoundException;
import java.rmi.RemoteException;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.HashMap;
import java.util.HashSet;
import java.util.LinkedList;
import java.util.List;
import java.util.Random;
import java.util.Set;
import java.util.TreeSet;
import java.util.Vector;

import code.lasr.db.DbException;
import code.lasr.db.DbTransaction;
import code.lasr.db.TxnFailedException;
import code.simulator.agreement.Tuple;
import code.simulator.irisnetty.IrisNetworkQueue;
import code.simulator.irisnetty.NetworkHandler;
import code.simulator.netty.NettyTCPReceiver;

import code.Config;
import code.Env;
import code.NodeId;
import code.ObjId;
import code.SubscriptionSet;
import code.branchDetecting.BranchID;
import code.security.SangminConfig;
import code.security.SangminConfig.DebugLevel;
import code.serialization.IrisInputStream;
import code.serialization.IrisObjectInputStream;
import code.serialization.IrisObjectOutputStream;
import code.serialization.IrisOutputStream;
import code.serialization.SerializationHelper;
import code.simulator.Hash;
import code.simulator.IrisDataObject;
import code.simulator.IrisHashObject;
import code.simulator.IrisNode;
import code.simulator.IrisObject;
import code.simulator.NamespaceWatch;
import code.simulator.NodeFactory;
import code.simulator.SimPreciseInv;
import code.simulator.SyncRequest;
import code.simulator.SyncStatus;
import code.simulator.netty.NettyTCPSender;
import code.simulator.netty.NetworkQueue;
import code.simulator.persistentLog.PersistentStore;
import code.simulator.store.BodyStore;
import code.simulator.store.BodyStoreInterface;
import code.simulator.store.PersistentBodyStore;
import code.simulator.store.StoreEntry;
import code.untrustedstorage.writeanyreadany.BodyRequestData;
import code.untrustedstorage.writeanyreadany.CommitCutFilter;
import code.untrustedstorage.writeanyreadany.NamespaceLayout;
import code.untrustedstorage.writeanyreadany.NewWriteRequestData;
import code.untrustedstorage.writeanyreadany.P2PBodyRequestData;
import code.untrustedstorage.writeanyreadany.Reply;
import code.untrustedstorage.writeanyreadany.Request;
import code.untrustedstorage.writeanyreadany.S3Inval;
import code.untrustedstorage.writeanyreadany.StorageConfig;
import code.untrustedstorage.writeanyreadany.WatcherForLatency;
import code.untrustedstorage.writeanyreadany.Request.RequestType;
import code.untrustedstorage.writeanyreadany.client.ServerChooserInterface.OpType;
import code.untrustedstorage.writeanyreadany.server.ServerListener;

/**
 * Class responsible for generating appropriate workload  
 * @author princem
 *
 */
public class ClientNodeWrapper{
  int numTotalWrites = 0;
  protected ClientNode clientNode;
  long myLogicalId;
  long myPhysicalId;
  // hack to avoid memory copy on hackedSig and hackedHash version
  public static IrisDataObject cachedIrisDataObject = null;
  public static byte[] byteArray = null;
  byte[] privateData;
  public static final boolean useNewSigEmulator = true;
  private PersistentStore pe = null;
  private BodyStoreInterface bodyStore = null;

  
  public ClientNodeWrapper(int myId, int configId, int primaryWriteServerId, int primaryReadServerId, 
      Set<Integer> syncServerIdSet, Set<Integer> clientIdSet, Set<Integer> serverIdSet, int size) 
  throws UnknownHostException, NotBoundException, IOException{
    this.myPhysicalId = configId;
    this.myLogicalId = myId;
    privateData = new byte[size];
    /*    if(SangminConfig.hackedHash){
          size += Hash.Length;
          }else if(SangminConfig.hackedSignature){
          size += 132;
          }*/
    
    if(SangminConfig.hackedStore){
      String dbPath = SangminConfig.configFile+myId;//Config.getLocalStore(branchId);
      System.out.println("dbPath for node " + myId + " is " + dbPath);
      try{
        pe = new PersistentStore(dbPath, SangminConfig.BDBCacheSize, true, SangminConfig.syncToDisk);
        bodyStore = new PersistentBodyStore(pe);
      }catch(DbException e){
        e.printStackTrace();
        assert false;
      } 
    }
    
    clientNode = new ClientNode(myId, configId, primaryWriteServerId, primaryReadServerId, syncServerIdSet, clientIdSet, serverIdSet);
    if(SangminConfig.hackedHash || SangminConfig.hackedSignature){
      ClientNodeWrapper.byteArray = privateData;
      int len = size;
      if(SangminConfig.hackedSignature && useNewSigEmulator){
        len += IrisDataObject.otherSigMetadataLen;
      }else if(SangminConfig.hackedHash && useNewSigEmulator){
        len+=Hash.Length;
      }
      try{
        ClientNodeWrapper.cachedIrisDataObject = IrisDataObject.createIrisDataObject(privateData, clientNode.getID().getIDint());
      }catch(IOException e){
        e.printStackTrace();
      }
      privateData = new byte[len];
    }
  }

  public LinkedList<Object> read(ObjId objId) throws DataNotFound{
    long startTime = System.nanoTime();
    long startTimeMillis = System.currentTimeMillis();
    Tuple<AccessStat, LinkedList<Object>> retVal = clientNode.readWithStats(objId);
    if(StorageConfig.S3Emulation){
      if(SangminConfig.hackedSignature || SangminConfig.hackedHash){
        for(Object o: retVal.getValue()){
          try{
            // the following if-else clause is used to emulate signature baseline impl. We sign and verify a cachedIrisObject
            if(useNewSigEmulator){
              IrisDataObject.verify(cachedIrisDataObject.getUnsafeBytes());
            }else{
              new IrisDataObject((byte[])o).verify();
            }
          }catch(IOException e){
            e.printStackTrace();
          }
        }
      }
    }

    long endTime = System.nanoTime();

    Env.logEvent(startTimeMillis + " GET oid: [" + objId + "] " + getVV() + " " +getVV() + 
        " sentBytes: " + retVal.getKey().sentBytes +  " receivedBytes: " + retVal.getKey().receivedBytes + " " + retVal.getKey().getHashString());

    Env.logEvent(startTimeMillis + " GET myNodeId: " + clientNode.getID() + " oid: [" + objId + "] " + getVV() + " " + 
        getVV() + " totalTime " + (((double)(endTime-startTime))/1000000) + " " +retVal.getKey().getHashString());

    Env.logEvent("GET status " + retVal.getKey().success + " value obtained " + retVal.getValue());
    return retVal.getValue();
  }
  
  public String getVV(){
    return clientNode.getVV();
  }
  
  public void write(ObjId objId, Object value){
    long startTime = System.nanoTime();
    long startTimeMillis = System.currentTimeMillis();
    assert value instanceof byte[];
    IrisDataObject ido = null;
    if(SangminConfig.hackedSignature || SangminConfig.hackedHash){
      try{
        if(useNewSigEmulator){
          // emulate the cost of signature by creating a padded buf etc...however the value array is
          // really the one that is present in the IrisDataObject
          ido = IrisDataObject.createIrisDataObject(byteArray, (byte[])value, clientNode.getID().getIDint());
        }else{
          // actually create an obj with signature etc attached---forces extra copy and 
          // therefore extra overhead. the new approach avoids this overhead.
          ido = IrisDataObject.createIrisDataObject((byte[])value, clientNode.getID().getIDint());
        }
      }catch(IOException e){
        e.printStackTrace();
        assert false;
      }                              
    }else{
      ido = new IrisDataObject((byte[])value);                              
    }
    if(SangminConfig.hackedStore){
      DbTransaction txn = pe.getDB().newTransaction();
      bodyStore.addBody(txn, ido);
      try{
        txn.commit();
      }catch(TxnFailedException e){
        e.printStackTrace();
        txn.abort();
      }
    }

    Tuple<Long, IrisDataObject> stat = clientNode.writeWithStat(objId, ido);
    long endTime = System.nanoTime();
    Env.logEvent(startTimeMillis + " PUT myNodeId: " + clientNode.getID() + " oid: [" + objId + "] " + getVV() + " " + stat.getValue().getHash() +
        " localCompletionTime: " +(((double)(stat.getKey() - startTime))/1000000) +" totalTime: " + (((double)(endTime-startTime))/1000000) );
  }
  
  public static IDPair initConfigs(String[] args){
    String USAGE = "Usage : java " + ClientNodeWrapper.class.getCanonicalName() +
    " <Iris Configurtaion File Path> <Storage Config File Path> <my iris id> [<reftime>]";

    if(args.length < 3){
      System.err.println(USAGE);
      System.exit(-1);
    }

    int myid = -1;
    int irisConfigIdx = -1;
    
    if(args[2].contains(".")){
        String[] idPair = args[2].split("\\.");
        myid = Integer.parseInt(idPair[0]);
        irisConfigIdx = Integer.parseInt(idPair[1]);
    } else {
    	myid = Integer.parseInt(args[2]);
    	irisConfigIdx = myid;
    }

    
//    int myId = Integer.parseInt(args[2]);
    String irisConfigFileName = args[0];
    String storageConfigFileName = args[1];
    if(!(new File(irisConfigFileName)).exists()){
      System.err.println("Iris Config file is not found : " + irisConfigFileName);
      System.err.println(USAGE);
      System.exit(-1);
    }
    if(!(new File(storageConfigFileName)).exists()){
      System.err.println("Storage Config file is not found : " + storageConfigFileName);
      System.err.println(USAGE);
      System.exit(-1);
    }

    Config.readConfig(irisConfigFileName);
    try{
      StorageConfig.readConfig(storageConfigFileName);
    }catch(IOException e2){
      e2.printStackTrace();
      System.exit(-1);
    }

    if(args.length > 3){
      StorageConfig.refTime = Long.parseLong(args[3]);
    }
    return new IDPair(myid, irisConfigIdx);
  }
  
  static class IDPair{
	  int id;
	  int irisConfigIdx;
	  public IDPair(int id, int configIdx){
		  this.id = id;
		  this.irisConfigIdx = configIdx;
	  }
  }
  
  // <wait for PRELOAD_TIME or for specified number of secs since the start of PRELOAD> <wait for sometime> <execute> <wait to be killed>
  //
  // Usage: ClientNodeWrapper <myIrisId> <Iris_ConfigFile> <StorageConfigFile>
  public static void main(String[] args) {

    IDPair idPair = initConfigs(args);
    int myId = idPair.id;
    int configIdx = idPair.irisConfigIdx;

    Tuple<Integer, Integer> writeRange;
    if(StorageConfig.clientWriteRange.containsKey(configIdx)){
      writeRange = StorageConfig.clientWriteRange.get(configIdx);
    }else{
      writeRange = new Tuple<Integer, Integer>(0, StorageConfig.numObjs);
    }

    ClientNodeWrapper cn = null;
    try{
      cn = new ClientNodeWrapper(myId, configIdx,
          StorageConfig.Client2WriteServerMap.get(myId),
          StorageConfig.Client2ReadServerMap.get(myId),
          StorageConfig.Client2SyncServerMap.get(myId),
          StorageConfig.getClientIdSet(), 
          StorageConfig.getServerIdSet(), 
          StorageConfig.objSize
          );
    }catch(Exception e2){
      e2.printStackTrace();
      System.exit(-1);
    }
    
    if(myId != configIdx){
      while(!cn.clientNode.getVVRaw().containsNodeId(cn.clientNode.getID())){
        try{
          Thread.sleep(1000);
        }catch(InterruptedException e){
        }
        cn.clientNode.syncTo(myId);
      }
    }

    cn.preload(writeRange);

    Env.logWrite("Begin normal execution.");

    cn.issueRequests(writeRange);

  }
  
  public BranchID getID(){
    return clientNode.getID();
  }

  public void issuePreloadOps(Tuple<Integer,Integer> writeRange){

    Vector<Integer> clients = new Vector<Integer>(StorageConfig.getClientIdSet());
    Collections.sort(clients);
    int numClients = StorageConfig.getClientIdSet().size();
    int numServers = StorageConfig.getServerIdSet().size();
    int myId = (int)clientNode.getID().getIDint();
    assert myId >= numServers; //client check
    ObjId oid;
    for(int i=writeRange.getKey(); i<writeRange.getValue() ; i++){
      // if the range partitioned writing strategy is in use then don't do the following check
      if(!StorageConfig.clientWriteRange.containsKey(myId) &&  clients.get(i%numClients) != myId){
        continue;
      }
      oid = new ObjId(getKeyString(i));

      byte[] data = getData();
      write(oid, data);
      try{
        Thread.sleep(250);
      }catch(InterruptedException e){
      }
    }
    Random r= new Random();
    for(int i = 0; i < 5; i++){
      int oidInt = r.nextInt(writeRange.getValue()-writeRange.getKey())+writeRange.getKey();
      oid = new ObjId(getKeyString(oidInt));
      try{
        read(oid);
      }catch(DataNotFound e){
        assert false;
        e.printStackTrace();
      }
    }
  }
  
  protected byte[] getData(){
    numTotalWrites++;
    int mod = (SangminConfig.usePersistentStore?256:10);
    privateData[0] = (byte)this.myPhysicalId;
    privateData[1] = (byte)(numTotalWrites%mod);
    privateData[2] = (byte)(Long.rotateRight(numTotalWrites, 8)%mod);    
    return privateData;
  }

  static int keysize =32;
  public static String getKeyString(int key){
    String ret = "/" + String.format("%" + ( keysize-1) +"d", key);
    ret = ret.replace(' ', '0');      
    return ret;
  }
  
  public void preload(Tuple<Integer,Integer> writeRange){
    //----- PRELOADING
    if(StorageConfig.usePreload != StorageConfig.NONE){

      long preloadStartTime = System.currentTimeMillis();
      this.issuePreloadOps(writeRange);
      long timeToSleep = StorageConfig.preloadTime;

      long t;
      if(StorageConfig.usePreload == StorageConfig.DELAY){ // DELAY mode
        while((t=System.currentTimeMillis() - preloadStartTime) < timeToSleep){      
          try{
            Thread.sleep(timeToSleep - t);
          }catch(InterruptedException e1){
          }
        }
      }else{ // TIMED mode
        long execStartTime = (StorageConfig.refTime + StorageConfig.preloadTime);
        if(System.currentTimeMillis() > execStartTime){
          // preload already passed
          System.err.println("TIMED mode PRELOAD---insufficient time; preload time already passed");
        }else{
          while((t=System.currentTimeMillis()) < execStartTime){      
            try{
              Thread.sleep(execStartTime-t);
            }catch(InterruptedException e1){
            }
          }
        }
      }
    }
    //--------- END OF PRELOADING

    Env.logEvent(System.currentTimeMillis() + " PRELOAD FINISHED");
    if(StorageConfig.waitAfterPreload > 0){
      try{
        Thread.sleep(StorageConfig.waitAfterPreload);
      }catch(InterruptedException e1){
      }     
    }
    Env.logEvent(System.currentTimeMillis() + " POST-PRELOAD OVER");
    /*    for(int i = 0; i < StorageConfig.numObjs; i++){
	int oidInt = i;
      ObjId oid = new ObjId(getKeyString(oidInt));
      try{
        read(oid);
      }catch(DataNotFound e){
        assert false;
        e.printStackTrace();
      }
      }*/
  }
  
  public void shutdown(){
    clientNode.shutdown();
  }
  protected void issueRequests(Tuple<Integer,Integer> writeRange){
    /**
     * the != condition below enables us to work in both the timed (where infinite requests are issued at specified freq) 
     * and num-reqs mode (where only a specified number of reqs are issued).
     */
    long t1, t2;
    int numRequests = 0;
    ObjId oid;
    Random r= new Random();
    float readRatio = StorageConfig.readRatio;
    long sleepDuration = StorageConfig.ClientWriteInterval;

    while(numRequests != StorageConfig.numRequests){

      t1 = System.currentTimeMillis();

      float f = r.nextFloat();
      if(f < readRatio){
        int oidInt = r.nextInt(StorageConfig.numObjs);
        oid = new ObjId(getKeyString(oidInt));
        try{
          read(oid);
        }catch(DataNotFound e){
          e.printStackTrace();
        }
      } else {
        int oidInt = r.nextInt(writeRange.getValue()-writeRange.getKey())+writeRange.getKey();
        oid = new ObjId(getKeyString(oidInt));
        byte []data = getData();
        write(oid, data);
      }

      numRequests++;

      t2 = System.currentTimeMillis();
      System.out.println(t2 + " " + numRequests + " requests have executed.");
      sleepDuration = StorageConfig.ClientWriteInterval - ( t2 - t1);
      if(sleepDuration>0){
        try{
          Thread.sleep(sleepDuration);
        }catch(InterruptedException e){
        }
      }
    }
    this.clientNode.dumpLogs();
    try{
      Thread.sleep(25000); // wait to ensure that someone kills us and the process statistics are appropriately recorded.
    }catch(InterruptedException e){
      e.printStackTrace();
    }
    this.clientNode.dumpLogs();

    try{
      Thread.sleep(100000); // wait to ensure that someone kills us and the process statistics are appropriately recorded.
    }catch(InterruptedException e){
      e.printStackTrace();
    }
  }

  /**
   * @return the clientNode
   */
  public ClientNode getClientNode(){
    return clientNode;
  }
  
}

