package code.untrustedstorage.writeanyreadany.server;

import java.io.File;
import java.io.IOException;
import java.rmi.RemoteException;
import java.util.HashSet;
import java.util.LinkedList;
import java.util.List;
import java.util.Random;
import java.util.Set;

import code.Config;
import code.Env;
import code.NodeId;
import code.ObjId;
import code.SubscriptionSet;
import code.security.SangminConfig;
import code.security.SangminConfig.DebugLevel;
import code.branchDetecting.BranchID;
import code.simulator.IrisDataObject;
import code.simulator.IrisHashObject;
import code.simulator.IrisNode;
import code.simulator.IrisObject;
import code.simulator.NodeFactory;
import code.simulator.SimPreciseInv;
import code.simulator.irisnetty.IrisNetworkQueue;
import code.simulator.irisnetty.NetworkHandler;
import code.simulator.netty.NettyTCPReceiver;
import code.simulator.netty.NettyTCPSender;
import code.simulator.netty.NetworkQueue;
import code.untrustedstorage.writeanyreadany.CommitCutFilter;
import code.untrustedstorage.writeanyreadany.NamespaceLayout;
import code.untrustedstorage.writeanyreadany.StorageConfig;
import code.untrustedstorage.writeanyreadany.WatcherForLatency;


public class ServerNode implements NetworkQueue{

  private IrisNode      irisNode;
  private BranchID             myId;

  private Set<BranchID> serverToSyncSet;
  private ServerListener serverListener;
  private CommitManager commitManager;

  private boolean shutdown = false;

  private NetworkHandler networkHandler = null; 

  public boolean ServerServerPartitionPhase = false;
  /**
   * Constructor of ServerNode
   * @param myID iris id of this server instance
   * @param syncServerIdSet a set of server IDes to which this server syncs periodically
   * @throws IOException
   */
  public ServerNode(BranchID myID, Set<Integer> syncServerIdSet) throws IOException{

    myId = myID;

    this.irisNode = (IrisNode) NodeFactory.createIrisNode(this.myId);
    serverToSyncSet = new HashSet<BranchID>();
    for(Integer id : syncServerIdSet){
      serverToSyncSet.add(NodeFactory.createNodeId(id));
    }
    serverToSyncSet.remove(this.myId);
    if(StorageConfig.useNetty){
      networkHandler = new NetworkHandler(irisNode.getBranchID().getIDint(), new IrisNetworkQueue(this)); 
    }

    serverListener = new ServerListener(irisNode, serverToSyncSet, networkHandler);
    serverListener.start();

    if(StorageConfig.useCommit){
      commitManager = new CommitManager(irisNode);
      CommitCutFilter cf = new CommitCutFilter(null);
      irisNode.addSyncFilter(cf);
      irisNode.addSyncFilter(new CommitFilter(commitManager));
      irisNode.addPOMFilter(cf);
    } else {
      commitManager = null;
    }    

    irisNode.addNamespaceWatch(new WatcherForLatency(this.myId, "RECEIVED WRITE"), SubscriptionSet.makeSubscriptionSet("/*"));

  }

  public void shutdown(){
    this.shutdown  =true;
    this.serverListener.close();
    irisNode.close();
    if(StorageConfig.useNetty){
      networkHandler.close();
    }
  }
  
  public void start(){
    (new SyncThread()).start();
    if(StorageConfig.useCommit){
      (new CommitThread(commitManager, StorageConfig.CommitInterval)).start();
    }

  }

  public void addWork(byte[] msg) {
    serverListener.addWork(msg);
  }

  private class SyncThread extends Thread{

    public SyncThread(){
      this.setDaemon(true);
      this.setName("Server Node Sync Thread of node " + myId);
    }

    public void run(){
      if(StorageConfig.InterServerSyncInterval < 0L){
        return;
      }
      int count = 0;

      try{
        Thread.sleep(5000);
      }catch(InterruptedException e2){
      }

      BranchID[] serversToSync = serverToSyncSet.toArray(new BranchID[0]);
      int numServer = serversToSync.length;
      Random r = new Random();
      while(!shutdown){

        try{
          sleep(StorageConfig.InterServerSyncInterval);
        }catch(InterruptedException e1){
        }
        count++;
        if(count %30==0){
          irisNode.dumpLogs();
        }

	if(numServer > 0){
	    BranchID bid = serversToSync[r.nextInt(numServer)];
	    
	    if(SangminConfig.debugLevel == DebugLevel.Verbose)Env.printDebug("Sync thread at " + myId.getIDint() + " going to sync with " + bid.getIDint() + " currentVV " + irisNode.getCurrentVV());
	    try{
		irisNode.exponentialBackoffSync(bid, irisNode.getCurrentVV());
	    }catch(Exception e){
		e.printStackTrace();
	    }
	}
      }
    }
  }

  public static void main(String args[]) throws IOException{

    String USAGE = "Usage : java " + ServerNode.class.getCanonicalName() +
    " <Iris Configurtaion File Path> <Storage Config File Path> <my iris id>";

    if(args.length < 3){
      System.err.println(USAGE);
      System.exit(-1);
    }

    String irisConfigFileName = args[0];
    String storageConfigFileName = args[1];

    if(!(new File(irisConfigFileName)).exists()){
      System.err.println("Iris Config file is not found : " + irisConfigFileName);
      System.err.println(USAGE);
      System.exit(-1);
    }
    if(!(new File(storageConfigFileName)).exists()){
      System.err.println("Storage Config file is not found : " + storageConfigFileName);
      System.err.println(USAGE);
      System.exit(-1);
    }



    int myId = Integer.parseInt(args[2]);

    Config.readConfig(irisConfigFileName);
    StorageConfig.readConfig(storageConfigFileName);

    ServerNode server = new ServerNode(new BranchID(myId), StorageConfig.serverMap.get(myId));
    server.start();
    long starttime = System.currentTimeMillis();
    long firstSSPartitionStart = StorageConfig.firstSSPartitionStart;
    long firstSSPartitionEnd = StorageConfig.firstSSPartitionEnd;
    while(true){
      try{
        Thread.sleep(1000);
      }catch(InterruptedException e){

      }
      if(StorageConfig.simulateSSPartition){
        long ctime = System.currentTimeMillis();
        if(ctime - starttime > firstSSPartitionStart ){
          //Go to Partition MODE
          Env.logEvent(System.currentTimeMillis() + " SERVER PARTITION BEGIN");
          server.ServerServerPartitionPhase = true;
          firstSSPartitionStart=Long.MAX_VALUE;
        }
        if(ctime - starttime > firstSSPartitionEnd){
          //Partition Restored
          Env.logEvent(System.currentTimeMillis() + " SERVER PARTITION END");
          server.ServerServerPartitionPhase = false;
          firstSSPartitionEnd=Long.MAX_VALUE;
        }
      }

    }

  }

  /* (non-Javadoc)
   * @see java.lang.Object#toString()
   */
  @Override
  public String toString(){
    return "ServerNode [irisNode=" + irisNode + "]";
  }




  //  public List<IrisObject> bodyRequestedFromClient(ObjId oid, List<IrisHashObject> hashObjList){
  //    LinkedList<IrisObject> ret = new LinkedList<IrisObject>();
  //    
  //    
  //    LinkedList<IrisObject> list = irisNode.read(oid);
  //    for(IrisHashObject iho : hashObjList){
  //      for(IrisObject io : list){
  //        if(io.getHash().equals(iho.getHash())){
  //          assert io instanceof IrisDataObject;
  //          ret.add((IrisDataObject)io);
  //          break;
  //        }
  //      }
  //    }    
  //    
  //    if(ret.size() == hashObjList.size()){
  //      return ret;
  //    }
  //    ret.clear();
  //    return ret;
  //  }

}
