package code.utils.junit;
import java.util.*;
import junit.framework.*;
import java.io.*;
import java.lang.reflect.*;
/**
 * Test suite for the entire Java implementation. Runs
 * all Java test cases in the source tree that extend TestCase and are of a specified type.
 * The following system properties must be set:
 * <ul> 
 * <li><code>class_root</code>: Directory containing classes. e.g /product/classes
 * <li><code>test_type</code>: Type of test case to load. All test cases with public static final String TEST_ALL_TEST_TYPE defined
 * to match the value of <code>test_type</code> will be loaded.
 * <li><code>package_root</code>: Where to start looking for packages within the package tree. e.g. com.company.app.sub1 will only run tests in package com.company.app.sub1 or below.
 * </ul>
 */
public class TestAll extends TestCase {
    /**
     * Iterates over the classes accessible via the iterator and adds them to the test suite.
     */
    private static int addAllTests(final TestSuite suite, final Iterator classIterator)
    throws java.io.IOException {
        int testClassCount = 0;
        while (classIterator.hasNext ()) {
            Class testCaseClass = (Class)classIterator.next ();
            
            try {
                Method suiteMethod = testCaseClass.getMethod("suite", new Class[0]);
                Test test= (Test)suiteMethod.invoke(null, new Class[0]); // static method
                suite.addTest (test);
            } catch (NoSuchMethodException e) {
                suite.addTest (new TestSuite (testCaseClass));
            } catch (Exception e) {
                System.err.println ("Failed to execute suite ()");
                e.printStackTrace ();
            }
            System.out.println ("Loaded test case: " + testCaseClass.getName ());
            testClassCount++;
        }
        return testClassCount;
    }

    /**
     * Dynamically create a test suite from a set of class files in a directory tree,
     */
    public static Test suite()
    throws Throwable {
        try {
            String classRootString = System.getProperty("class_root");
            if (classRootString == null) throw new IllegalArgumentException ("System property class_root must be set.");
            String testType = System.getProperty("test_type");
            if (testType == null) throw new IllegalArgumentException ("System property test_type must be set.");
            File classRoot = new File(classRootString);
            String packageRoot = System.getProperty ("package_root");
            if (packageRoot == null) {
                packageRoot = "";
            }
            //packageRoot = "code";
            ClassFinder classFinder = new ClassFinder (classRoot, packageRoot);
            TestCaseLoader testCaseLoader = new TestCaseLoader (testType);
            testCaseLoader.loadTestCases (classFinder.getClasses ());
            TestSuite suite = new TestSuite();
            int numberOfTests = addAllTests (suite, testCaseLoader.getClasses ());
            System.out.println("Number of test classes found: " + numberOfTests);
            return suite;
        } catch (Throwable t) {
            // This ensures we have extra information. Otherwise all we get is a "Could not invoke the suite method." message. 
            t.printStackTrace ();
            throw t;
        }
    }

    /**
     * Basic constructor - called by the test runners.
     */

    public TestAll(String s) {
        super(s);
    }
    
    
}
