#include <config.h>
// serial_aux.C

#include <assert.h>
#include <stdlib.h>
#include <stream.h>
#include <unistd.h>

#include <misc/general.h>
#include <exmodel/abstree.h>
#include <exmodel/exmodel.h>
#include <symtab/symtab.h>
#include <lib/UidTable.h>
#include <lib/MsgUI.h>
#include <treebuild/pass.h>
#include <misc/voidlist.h>
#include <misc/sfile.h>
#include <misc/fileutil.h>
#include <symtab/symtab.h>
#include <lib/Syntax.h>
#include <backends/serial/serial_aux.h>
#include <translate/translate.h>
#include <backends/serial/serial_arc.h>

int SerialWriteFiles(cProgram *ProgramRoot)
{
   cBlock *b;
   cCursor c;
   ofstream OutFile;
   cGraph *g;
   cObIterator i;
   cObject *o;
   char *CurrentWorkingDirectory;
   MString TranslationDirectory;

   CurrentWorkingDirectory = new char[512]; // cwd must not be longer than 510 bytes
   if ( getcwd(CurrentWorkingDirectory, 512) == NULL ) {
      FixMe("Could not cd to program dir for translate\n");
      return 0;
   }
   TranslationDirectory = MString(TestDirectory) + MString("/") + MString(ProgramRoot->Name) + MString(".serial");
   if ( chdir((char *)TranslationDirectory) ) {
      FixMe("Could not cd to program dir for translate\n");
      return 0;
   }

   // Makefile

   b = (cBlock *) c.RootBlock()->SubBlocks.PopFirst();

   if (b->ID != eMakefile)
     Die("Internal error: ID should have been eMakefile\n");

   OutFile.open("Makefile");

   if (!OutFile) {
      FixMe("Could not open Makefile\n");
      return 0;      
   }

   b->Output(OutFile);

   OutFile.close();

   // c2_main.h

   b = (cBlock *) c.RootBlock()->SubBlocks.PopFirst();

   if (b->ID != eMain_h)
     Die("Internal error: ID should have been eMain_h\n");

   OutFile.open("c2_main.h");

   if (!OutFile) {
      FixMe("Could not open c2_main.h file\n");
      return 0;
   }

   b->Output(OutFile);

   OutFile.close();

   // c2_main.c

   b = (cBlock *) c.RootBlock()->SubBlocks.PopFirst();

   if (b->ID != eMain_c)
     Die("Internal error: ID should have been eMain_c\n");

   OutFile.open("c2_main.c");

   if (!OutFile) {
      FixMe("Could not open c2_main.c file\n");
      return 0;      
   }

   b->Output(OutFile);

   OutFile.close();

   // c2_globtype.h

   b = (cBlock *) c.RootBlock()->SubBlocks.PopFirst();

   if (b->ID != eGlob_h)
     Die("Internal error: ID should have been eGlob_h\n");

   OutFile.open("c2_globtype.h");

   if (!OutFile) {
      FixMe("Could not open c2_globtype file\n");
      return 0;
   }

   b->Output(OutFile);

   OutFile.close();

   // graph.c for each graph

   for (o = i.Init(ProgramRoot->Graphs); i.MoreLeft(); o = i.Next()) {
      g = (cGraph *) o;
      b = (cBlock *) c.RootBlock()->SubBlocks.PopFirst();

      if (b->ID != eGraph_c)
	Die("Internal error: ID should have been eGraph_c\n");

      OutFile.open((char *) (g->Name + ".c"));

      if (!OutFile) {
	 FixMe("Could not open a graph c file\n");
	 return  0;      
      }

      b->Output(OutFile);

      OutFile.close();
   }

   if ( chdir(CurrentWorkingDirectory) ) {
      FixMe("Internal error: Could not cd to parent dir from translate\n");
      return 0;
   }

   return 1;
}



void SerialMakeFile(cProgram *p)
{
   cCursor c;
   cObIterator i;
   void *v;
   cGraph *g;
   MString LibFile;

   c.cd(eMakefile);
  c << "# Makefile for \"" << p->Name << "\" produced by CODE.\n"
    << "# Requires GNU Make.\n"
    << "# For more information on CODE, see <http://www.cs.utexas.edu/users/code>.\n\n";

  c << "# Define a default CODE_LIBRARY.\n"
    << "ifndef CODE_LIBRARY\n"
    << "define CODE_LIBRARY\n"
    << "/usr/local/lib\n"
    << "endef\n"
    << "endif\n\n";

   c.cd(eDecl);

   LibFile = "-L$(CODE_LIBRARY)/runtime/serial -lcode_serial";

   c << "CFLAGS = -g\n";
   c << "C2DIR = $(CODE_LIBRARY)/runtime/serial\n";

   c << "INC = -I$(C2DIR) -I$(CODE_LIBRARY)/runtime/stl\n"
     << "INDENT = indent\n"
     << "INDENTFLG = -i2\n";

   c.up();
   c.cd(eObj);
   if (AddDbgrCode) {
       c << "OBJS = c2_main.o \\\n      ";
    } else {
       c << "OBJS = c2_main.o \\\n      ";
    }

   c << FilesToLink << " \\\n      ";

   c.up();
   c.cd(eProgt);

  c << "SRCS = c2_main.C ";

  for (v = i.Init (p->Graphs); i.MoreLeft (); v = i.Next ()) {
    g = (cGraph *) v;
    c << g->Name << ".c ";
  }
  c << "\n\n";

  c << ".c.o:\n\t$(CC) $(CFLAGS) $(INC) -c $<\n\n";
  c << ".C.o:\n\t$(CXX) $(CCFLAGS) $(INC) -c $<\n\n";

  c << p->Name << ": $(OBJS)\n" << "\t$(CXX) $(OBJS) " << LibFile << "  -o " << p->Name << " -lm\n\n"
     << "clean:\n"
     << "\t\\rm -f *.o *.pretty core " << p->Name << "\n\n"
     << "pretty:\n"
     << "\t$(INDENT) c2_globtype.h c2_globtype.h.pretty $(INDENTFLG)\n"
     << "\t$(INDENT) c2_main.h c2_main.h.pretty $(INDENTFLG)\n"
     << "\t$(INDENT) c2_main.c c2_main.c.pretty $(INDENTFLG)\n";

   for (v = i.Init(p->Graphs); i.MoreLeft(); v = i.Next()) {
      g = (cGraph *) v;
      c << "\t$(INDENT) " << g->Name << ".c " << g->Name
        <<       ".c.pretty $(INDENTFLG)\n";
   }

   c << "\nc2_main.o: c2_main.c\n";

}



MString SerialGetName(MString& Ident, cAbsTree *Here)
{
   cSymTabEntry *t;

   // Find a symbol table that contains ident, checking parents up
   // the tree.

   while (Here != 0) {
      if (Here->SymbolTable != 0) {
	 t = Here->SymbolTable->Get(Ident);
	 if (t != 0) break;
      }
      Here = Here->Parent;     // Check the parent
   }

   if (t == 0)     // Ident is not found.
     Die("Internal error: Ident not found in SerialGetName\n");

   switch (t->IsA()) {
    case SEFuncArg:
      return Ident;
    case SEFuncName:
      return Ident;
    case SEVariable:
      if (Here->IsA() == UC || Here->IsA() == NSRel) {
	 if (Ident == MString("NodeIndex"))
	   return "(_c2_MyAddr->Index.Ind)";
	 else if (Ident == MString("GraphIndex"))
	   return "(_c2_MyAddr->MyGraph->Index.Ind)";
	 else
	   return "(_c2_l->" + Ident + ")";
      } else
	return Ident;
    case SETypeName:
      return Ident;
    case SEPortName:
      return Ident;
    case SEIntNode:
      return Ident;
    case SESharedVar:
      return "(_c2_l->" + Ident + ")";
    case SECreP:
      return "(_c2_g->" + Ident + ")";
    case SEGraph:
      return Ident;
    default:
      Die("Internal error: no case in SerialGetName\n");
      break;
   }

   return "";  // just to silence gcc

}


void SerialCreateMain()
{
   cCursor c;
   
   c.cd(eMain_c); c.cd(eMainProg);
   if (AddDbgrCode) {
      c << "main (argc, argv)\n"
	<< "int argc;\n"
	<< "char **argv;\n";
   } else {
      c << "main ()\n";
   }
   
   c << "{\n"
       << "  _c2_Index Index;\n"
       << "  _c2_NodeBase *StartNode;\n"
       << "  _c2_GraphBase *MainGraph;\n"
       << "  int ObjType;\n\n"
       << "  Index.NumInd = 0;\n\n"
       << "  _c2_AddrHashTabInit();\n\n"
       << "  MainGraph = (_c2_GraphBase *) _c2_MakeObj(" << StartGraphUID			
       <<        ", &Index, 0L, &ObjType);\n"
       << "  StartNode = (_c2_NodeBase *) _c2_GetAddr(MainGraph, "
       <<        StartNodeUID << ", &Index);\n\n";

   if (AddDbgrCode)
       c << "  _c2_RunDebugger(StartNode, argc, argv);\n";
   else
       c << "  _c2_RunWorkers(StartNode);\n";
   
   c << "}\n";
   
   
   if (AddDbgrCode) {   
      c.cd();
      c.cd(eMain_c);
      c.cd(eDbgSymEntry);
      c.cd(eDbgSEHead);
      c << "void _c2_dbgSymEntry()\n"
	  << "{\n";
      c.up();
      c.cd(eDbgSETail);
      c << "}\n";

      c.cd(); c.cd(eMain_c); c.cd(eDbgSymAddr);
      c.cd(eSAHead);
      c << "void *_c2_getSymAddr(UID, pUID, Data)\n"
	<< "int UID, pUID; \n"
        << "void *Data;\n" 
	<< "{\n"
        << " switch(pUID) {\n";
      c.up();       c.cd(eSATail);
      c << " }\n"
	<< "}\n";
   }
   
   
   // Create the case in MakeObj for the start graph.
   
   c.cd();
   c.cd(eMain_c);
   c.cd(eMakeObj);
   c.cd(eMOEntry, StartGraphUID);
   c.cd(eMOEntryCase);
   
   c << "case " << StartGraphUID << ":\n";
}



// Handle the function defs that are stored in a GUIDE structure pointed
// to by a void *

void SerialDoFuncDefs(MString &FuncDefs, cAbsTree *Here)
{
   cCursor c;
   cAbsTree *t, *t2;
 
   t = GetContaining(Here, UC, NSRel, Graph, Program, 0);

   switch (t->IsA()) {
    case Program:
      c.cd(eMain_c);
      c.cd(eGlobFunc);
      break;
    case Graph:
      c.cd(eGraph_c, t->UID);
      c.cd(eGraphFunc);
      break;
    case NSRel:
      t2 = GetContaining(t, Graph, 0);
      c.cd(eGraph_c, t2->UID);
      c.cd(eNSScope, t->UID);
      c.cd(eNSFunc);
      break;
    case UC:
      t2 = GetContaining(t, Graph, 0);
      c.cd(eGraph_c, t2->UID);
      c.cd(eUCScope, t->UID);
      c.cd(eUCFunc);
      break;
   default:
     assert (0); // We should never get here.
     break;
   }

   c << FuncDefs << "\n";
   //InitGuideTextLine(FuncDefs);
   //while ((s = ReturnGuideTextLine()) != 0)
   //  c << s << "\n";

   c << "\n";
}



int InitSerial(cProgram *p)
{
   MString TranslationDirectory;

   // Make a driectory to write source into.
   TranslationDirectory = MString(TestDirectory) + MString("/") + MString(p->Name) + MString(".serial");
   if ( !MakeSrcDir((char *)TranslationDirectory) ) {
      FixMe("Could not create source directory.\n");
      return 0;
   }

   InitGetName(SerialGetName);
   InitSizeSpec(SerialSizeSpec);
   InitCursor(eRoot);

   return 1;
}


void SerialMakeObj()
{
   cCursor c;

   c.cd(eMain_c);
   c.cd(eMakeObj);
   c.cd(eMOHead);

   c << "\nvoid *_c2_MakeObj(UID, Index, Graph, ObjType)\n"
     << "  int UID;\n"
     << "  _c2_Index *Index;\n"
     << "  _c2_GraphBase *Graph;\n"
     << "  int *ObjType;\n"
     << "{\n"
     << "  _c2_NodeBase *tmpN;\n"
     << "  _c2_NSRelBase *tmpR;\n"
     << "  _c2_GraphBase *tmpG;\n"
     << "  void *Data;\n\n"
     << "  switch (UID) {\n";

   c.up();
   c.cd(eMOTail);

   c << "   default:\n"
     << "     fprintf(stderr, \"Internal error: "
     <<             "No such UID %d in MakeObj\\n\\n\", UID);\n"
     << "     break;\n"
     << "  }\n"
     << "}\n\n";
}



  // Create skeletons of UC Comp and Init Procs

void SerialUCProc(cUC *u)
{
   cCursor c;
   cGraph *g;

   g = (cGraph *) u->Parent;

   // Do the header of the procedure

   c.cd(eGraph_c, g->UID);
   c.cd(eUCScope, u->UID);
   c.cd(eUCProc);
   c.cd(eUCProcHead);

   c << "void _c2_cp" << u->UID << "(_c2_MyAddr)    /* " << u->Name << " */\n"
     << "  _c2_NodeBase *_c2_MyAddr;\n"
     << "{\n";
   if (u->StartNode)
     c << "  int _c2_NotStarted = 1;\n";
   c << "  struct _c2_nv" << u->UID << " *_c2_l;\n"
     << "  struct _c2_gv" << g->UID << " *_c2_g;\n"
     << "  int _c2_Stat;\n"
     << "  int _c2_NullRule;\n"
     << "  _c2_Index _c2_PortIndex;\n"
     << "  _c2_Value _c2_TmpValue;\n\n";

   c << "_c2_l = (struct _c2_nv" << u->UID << " *) (_c2_MyAddr->LocalData);\n"
     << "_c2_g = (struct _c2_gv" <<g->UID << " *)" 
     <<       " (_c2_MyAddr->MyGraph->LocalData);\n\n";


   if (AddDbgrCode) {
      c << "   if (_c2_chkStoppedBpAct(_c2_MyAddr)) {\n";
      if (u->StartNode) 
          c << "      _c2_NotStarted = 0;\n";
      c << "       if (_c2_stoppedBefore(_c2_MyAddr)) \n"
	<< "          goto _c2_BefComp;\n"
        << "       else\n"
        << "          goto _c2_AftComp;\n"
        << "  }\n";
   }
   


   c << "if (_c2_MyAddr->NSState == _c2_NOTINIT) {\n"
     << "  _c2_MyAddr->HeadLock = 0;\n";

   c.up();
   c.cd(eUCProcBefFrule);

   c << "  _c2_MyAddr->NextLock = _c2_MyAddr->HeadLock;\n"
     << "  _c2_MyAddr->NSState = _c2_NOBIND;\n"
     << "}\n";

   if (AddDbgrCode)
       c << "if (_c2_MyAddr->NSState == _c2_INPROG) goto _c2_nsrel;\n\n";

   // Lock Qlock for firing rule testing

   c << "_c2_again:\n"; 
   if (AddDbgrCode) {
      c << "_c2_beginUcEvBpAct(_c2_MyAddr);\n";
   }

   if (u->StartNode) {    // let it fire once for free
      c << "if (_c2_NotStarted) {\n"
        << "  goto _c2_comp;\n"
        << "}\n\n";
   } 

   // Do the compute section

   c.up();
   c.cd(eUCProcComp);

   c << "/* No rule is true */\n";
   if (AddDbgrCode) {
      c << "_c2_waitFpBpAct(_c2_MyAddr);\n\n";
   }
   c << "return;\n\n"
     << "_c2_comp:\n"; 

   if (TraceLevel == tr_sel_nodes)
     c << "printf(\"UC " << u->Name << " (" << u->UID << ") fires\\n\");\n"
       << "fflush(stdout);\n";

    if (AddDbgrCode) {
      c << "_c2_nsrel:\n"   // interactive dbg makes this non-deterministic!
        << "if (_c2_GetAllLocks(_c2_MyAddr, &(_c2_MyAddr->NextLock)))\n"
	<< "  _c2_MyAddr->NSState = _c2_NOBIND;\n"
	<< "else {\n"
	<< "  _c2_MyAddr->NSState = _c2_INPROG;\n"
	<< "  return;\n"
	<< "}\n\n";

      c << " if (_c2_befCompBpAct( _c2_MyAddr)) {\n"
        << "        return;\n"
        << "}\n"
        << "_c2_BefComp:\n\n";
   } else {      
      c << "_c2_nsrel:\n"
	<< "_c2_GetAllLocks(_c2_MyAddr, &(_c2_MyAddr->NextLock));\n";
   }

   if (u->Comp != 0) c << u->Comp->Translate();

   // Start the clock after the first firing of the start node.

   if (u->StartNode) {
     c << "if (_c2_NotStarted) {\n"
       << "  _c2_NotStarted = 0;\n"
       << "  _c2_StartClock();\n"
       << "}\n\n";
   }

   if (AddDbgrCode) {

      c << " if (_c2_aftCompBpAct( _c2_MyAddr)) {\n"
        << "        return;\n"
        << "}\n"
        << "_c2_AftComp:\n\n";
   }

   if (u->TermNode)   { // Stop the computation
      c << "_c2_StopClock();\n";
      if (AddDbgrCode) {
	 c << "_c2_firesTermNode(_c2_MyAddr);\n";
      } else {
	 c << "_c2_StopAll();\n";
      }
   }

   // Do the proc tail

   c.up();
   c.cd(eUCProcTail);
   if (AddDbgrCode) {
      c << "_c2_RelAllLocks(_c2_MyAddr, &(_c2_MyAddr->HeadLock), &(_c2_MyAddr->NextLock));\n"
	<< "_c2_endUcEvBpAct(_c2_MyAddr);\n\n";

   } else {
      c << "_c2_RelAllLocks(&(_c2_MyAddr->HeadLock), &(_c2_MyAddr->NextLock));\n";
   }
   
   c << "if (!_c2_Terminated) goto _c2_again;\n"
     << "}\n\n";

   // Create signature in c2_main.c

   c.cd();
   c.cd(eMain_c);
   c.cd(eIntFuncSig);

   c << "void _c2_cp" << u->UID << "();  /* " << u->Name << " */\n";

   // Create InitProc

   c.cd();
   c.cd(eGraph_c, g->UID);
   c.cd(eUCScope, u->UID);
   c.cd(eUCIProc);
   c.cd(eUCIProcHead);

   c << "void _c2_ip" << u->UID << "(_c2_MyAddr)    /* " << u->Name << " */\n"
     << "  _c2_NodeBase *_c2_MyAddr;\n"
     << "{\n"
     << "  struct _c2_nv" << u->UID << " *_c2_l;\n"
     << "  struct _c2_gv" << g->UID << " *_c2_g;\n\n"
     << "_c2_l = (struct _c2_nv" << u->UID << " *) (_c2_MyAddr->LocalData);\n"
     << "_c2_g = (struct _c2_gv" <<g->UID << " *)" 
     <<       " (_c2_MyAddr->MyGraph->LocalData);\n\n";

   c.up();
   c.cd(eUCIProcComp);
   if (u->InitComp != 0) c << u->InitComp->Translate();

   c.up();
   c.cd(eUCIProcTail);
   c << "}\n\n";

   // Create signature
   c.cd();
   c.cd(eMain_c);
   c.cd(eIntFuncSig);

   c << "void _c2_ip" << u->UID << "();  /* " << u->Name << " */\n";
}

  // Create skeleton of NSRel Init Proc

void SerialNSProc(cNSRel *n)
{
   cCursor c;
   cGraph *g;


   g = (cGraph *) n->Parent;

   c.cd(eGraph_c, g->UID);
   c.cd(eNSScope, n->UID);
   c.cd(eNSIProc);
   c.cd(eNSIProcHead);

   c << "void _c2_ip" << n->UID << "(_c2_MyAddr)    /* " << n->Name << " */\n"
     << "  _c2_NSRelBase *_c2_MyAddr;\n"
     << "{\n"
     << "  struct _c2_nv" << n->UID << " *_c2_l;\n"
     << "  struct _c2_gv" << g->UID << " *_c2_g;\n\n"
     << "_c2_l = (struct _c2_nv" << n->UID << " *) (_c2_MyAddr->LocalData);\n"
     << "_c2_g = (struct _c2_gv" <<g->UID << " *)" 
     <<       " (_c2_MyAddr->MyGraph->LocalData);\n";

   c.up();
   c.cd(eNSIProcComp);

   if (n->InitComp != 0)
     c << n->InitComp->Translate();

   c.up();
   c.cd(eNSIProcTail);
   c << "}\n\n";

   // Create signature
   c.cd();
   c.cd(eMain_c);
   c.cd(eIntFuncSig);

   c << "void _c2_ip" << n->UID << "();  /* " << n->Name << " */\n";
}


void SerialUCMakeObj(cUC *u)
{
   cCursor c;

   // Do MakeObj entry
   c.cd();
   c.cd(eMain_c);
   c.cd(eMakeObj);
   c.cd(eMOEntry, u->UID);
   c.cd(eMOEntryHead);
   c << "case " << u->UID << ":\n"
     << "Data = (void *) _c2_shmalloc(sizeof(struct _c2_nv" << u->UID 
     <<     "));\n"
     << "tmpN = _c2_MakeNode(UID, Index, Graph, Data, _c2_cp"
     <<     u->UID << ", " << "_c2_ip" << u->UID
     << ", Graph->CrepsToGo == 0);\n";

   c.up();
   c.cd(eMOEntryTail);
   c << "*ObjType = _c2_UCNODE;\n"
     << "return (void *) tmpN\n;"
     << "break;\n";

}

void SerialGraphMakeObj(cGraph *g)
{
   cCursor c;

   c.cd();
   c.cd(eMain_c);
   c.cd(eMakeObj);
   c.cd(eMOEntry, g->UID);
   c.cd(eMOEntryHead);

   c << "Data = (void *) _c2_shmalloc(sizeof(struct _c2_gv"
     <<    g->UID << "));\n"
     << "tmpG = _c2_MakeGraph(UID, Index, Graph, Data);\n"
     << "*ObjType = _c2_CALLNODE;\n"
     << "tmpG->CrepsToGo = " << g->CrepNodes.Length() << ";\n";

   c.up();
   c.cd(eMOEntrySWHead);
   c << "switch (UID) {\n";

   c.up();
   c.cd(eMOEntryTail);
   c << "};\n"
     << "return (void *) tmpG;\n"
     << "break;\n";
}


void SerialArcAssign(cCursor &c, cExpr *VarExpr, cExpr *PortExpr,
		      cAbsTree *Here)
{
   MString PN;
   cSEPortName *t;
   cPortSType *p;

   PN = FindPortName(PortExpr);

   t = (cSEPortName *) GetSymbol(PN, Here, SEPortName, 0);
   p = (cPortSType *) (t->Type);

   switch (p->BaseType->IsA()) {
    case IntSType :
      c << VarExpr->Translate() << " = _c2_TmpValue.u.i;\n";
      break;
    case RealSType:
      c << VarExpr->Translate() << " = _c2_TmpValue.u.d;\n";
      break;
    case CharSType:
      c << VarExpr->Translate() << " = _c2_TmpValue.u.c;\n";
      break;
    case ArraySType:
      c << "_c2_free_" << VarExpr->StructuralType->TypeName << "("
	<<  VarExpr->Translate() << ");\n"
        <<  VarExpr->Translate()
	<<    " = _c2_TmpValue.u.array;\n";
        
      break;
    case StructSType:
      c << "_c2_free_" << VarExpr->StructuralType->TypeName << "("
	<<  VarExpr->Translate() << ");\n"
        <<  VarExpr->Translate()
	<< " = _c2_TmpValue.u.structure;\n";
      break;
    default:
      Die("Internal Error: Wrong type in SerialTypeOf\n");
      break;
   }
   
}


void SerialSharVarFunc(cNSPortSpec *nsp, cSESharedVar *sv)
{
   cGraph *g;
   cSType *pt;
   cCursor c;
   cUC *u;
   cArc *a;
   ArcKind k;
   cAbsTree *ToNode;
   
   u = (cUC *) GetContaining(nsp, UC, 0);
   g = (cGraph *) GetContaining(nsp, Graph, 0);

   pt = (cSType *) sv->Type;

   // Put func sig in place
   c.cd(eMain_h);
   c.cd(ePortFuncSig);

   c << "void " << sv->GenName << "();\n";

   // Create function itself

   c.cd();
   c.cd(eGraph_c, g->UID);
   c.cd(ePortFunc);

   c << "void " << sv->GenName << "(_c2_na, _c2_pi)  /* Port " 
     <<          nsp->PortName << " */\n"
     << "  _c2_NodeBase *_c2_na;\n"
     << "  _c2_Index *_c2_pi;\n"
     << "{\n"
     << "  struct _c2_gv" << g->UID << " *_c2_g;\n"
     << "  _c2_GraphBase *_c2_gr;\n"
     << "  _c2_NSRelBase *_c2_TmpNSRel;\n"
     << "  _c2_Index _c2_gi;\n"
     << "  _c2_Index _c2_ni;\n"
     << "  void *_c2_LocalAddr;\n"
     << "  int _c2_LocalReqType;\n"
     << "  _c2_LockSet *_c2_NewLock = (_c2_LockSet *) _c2_shmalloc("
     <<    "sizeof(_c2_LockSet));\n\n";

   c << "  _c2_gr = _c2_na->MyGraph;\n"
     << "  _c2_ni = _c2_na->Index;\n"
     << "  _c2_g = (struct _c2_gv" <<g->UID << " *) (_c2_gr->LocalData);\n"
     << "  _c2_LocalAddr = (void *) &((struct _c2_nv" << u->UID
     << " *) (_c2_na->LocalData))->" << nsp->PortName << ";\n"
     << "_c2_LocalReqType = ";

   if (nsp->UseRule == NSReader)
     c << "_c2_READER;\n\n";
   else
     c << "_c2_WRITER;\n\n";

   FindArc(nsp->PortName, u, &a, &k, &ToNode);
   SerialDoNextArc(c, a, k, ToNode);

   c << "}\n\n";
}


void SerialPortFunc(cPortSpec *p)
{
   cGraph *g;
   cSEPortName *ps;
   cPortSType *pt;
   cCursor c;
   cUC *u;
   cArc *a;
   ArcKind k;
   cAbsTree *ToNode;
   
   u = (cUC *) (p->Parent);
   g = (cGraph *) GetContaining(p, Graph, 0);
   ps = (cSEPortName *) GetSymbol(p->PortName, p, SEPortName, 0);
   if (ps->Type->IsA() == PortSType)
     pt = (cPortSType *) ps->Type;
   else
     Die("Internal error: port type wrong in SerialPortFunc.\n");

   // Put func sig in place
   c.cd(eMain_h);
   c.cd(ePortFuncSig);

   c << "void " << ps->GenName << "();\n";

   // Put in cache variable declaration.
   c.cd();
   c.cd(eMain_h);
   c.cd(eNodeVar, u->UID);
   c << "struct _c2_Cache " << ps->GenName << "Cache;\n";

   // Initialize port cache
   c.cd();
   c.cd(eGraph_c, g->UID);
   c.cd(eUCScope, u->UID);
   c.cd(eUCIProc);
   c.cd(eUCIProcInit);
   c << "_c2_CacheInit(&(_c2_l->" << ps->GenName << "Cache));\n";

   // Create function itself

   c.cd();
   c.cd(eGraph_c, g->UID);
   c.cd(ePortFunc);

   c << "void " << ps->GenName << "(_c2_na, _c2_pi, _c2_value)  /* Port "
     <<    p->PortName << " */\n"
     << "  _c2_NodeBase *_c2_na;\n"
     << "  _c2_Index *_c2_pi;\n";

   switch (pt->BaseType->IsA()) {
    case IntSType:
      c << "  int _c2_value;\n";
      break;
    case RealSType:
      c << "  double _c2_value;\n";
      break;
    case CharSType:
      c << "  char _c2_value;\n";
      break;
    case ArraySType:
    case StructSType:
      c << "  void *_c2_value;\n";
      break;
    default:
      Die("Internal error:  Illegal base type in SerialPortFuncHead\n");
   }
   
   c << "{\n"
     << "  struct _c2_gv" << g->UID << " *_c2_g;\n"
     << "  _c2_GraphBase *_c2_gr;\n"
     << "  _c2_NodeBase *_c2_TmpNode;\n"
     << "  _c2_Index _c2_origpi;\n"
     << "  _c2_Index _c2_gi;\n"
     << "  _c2_Index _c2_ni;\n"
     << "  _c2_SeqVar **_c2_QVar;\n"
     << "  int _c2_AddToCache = 0;\n"
     << "  _c2_Value _c2_TmpValue;\n\n"
     << "  _c2_gr = _c2_na->MyGraph;\n"
     << "  _c2_ni = _c2_na->Index;\n"
     << "  _c2_origpi = *_c2_pi;\n"
     << "  _c2_g = (struct _c2_gv" <<g->UID << " *) (_c2_gr->LocalData);\n\n";
   if (AddDbgrCode) {
      c << " _c2_appndFpBpAct(_c2_na, &(_c2_TmpValue.fplist), "
	<< "&(_c2_TmpValue.cmd));\n\n";
   }
   c << "if (!_c2_FindCache(&(((struct _c2_nv" << u->UID << " *)"
     <<   "(_c2_na->LocalData))->" << ps->GenName << "Cache), "
     <<   "&_c2_origpi, _c2_pi, &_c2_QVar, &_c2_TmpNode)) {\n"
     << "_c2_AddToCache = 1;\n";

   FindArc(p->PortName, u, &a, &k, &ToNode);
   SerialDoNextArc(c, a, k, ToNode);

   c << "if (_c2_AddToCache)\n"
     << "_c2_EnCache(&(((struct _c2_nv" << u->UID << " *)"
     <<   "(_c2_na->LocalData))->" << ps->GenName << "Cache), "
     <<   "&_c2_origpi, _c2_pi, _c2_QVar, _c2_TmpNode);\n";

   c << "}\n\n";
}



// Create all of the things needed for an arc route function.  These are
// created for all arcs that exit a Call node.

void SerialMakeArcRoute(cArc *a, cGraph *CalledGraph, cCallNode *cn)
{
   cCursor c;
   cObIterator i;
   cObject *p;
   cIntNode *in;
   cAbsTree *ToNode;
   cGraph *g;
   ArcKind k;

   g = (cGraph *) GetContaining(a, Graph, 0);

   // Create the Arc Route Function signature in c2_main.h

   c.cd(eMain_h);
   c.cd(eArcFuncSig);
   c << "void _c2_ar" << a->UID << "();\n";

   // Create the entry in MakeObj that sets a graph's return function

   c.cd();
   c.cd(eMain_c);
   c.cd(eMakeObj);
   c.cd(eMOEntry, CalledGraph->UID);
   c.cd(eMOEntrySWInit, cn->UID);
   c.cd(eMOEntrySWInBind);

   // What port in called graph is arc bound to?
   for (p = i.Init(CalledGraph->IntNodes); i.MoreLeft(); p = i.Next()) {
      in = (cIntNode *) p;
      if (!(in->InParam) && (char *) in->Name == a->FromPort) {
	 c << "((struct _c2_gv" << CalledGraph->UID << " *) Data)->_c2_prf"
	   <<   in->UID << " = _c2_ar" << a->UID << ";\n";
	 break;
      }
   }

   // Create the Arc Route Function itself in graph.c
   c.cd();
   c.cd(eGraph_c, g->UID);
   c.cd(eArcFunc);

  if (AddDbgrCode) {
      c << "void _c2_ar" << a->UID << "(_c2_fplist, _c2_cmd, "
	<< "_c2_gr, _c2_gi, _c2_ni, _c2_pi, "
        <<   "_c2_RetNode, _c2_RetQVar, _c2_AddToCache, _c2_value)\n";
      c << "  _c2_FpList _c2_fplist;\n"
	<< "  _c2_Cmd _c2_cmd;\n";
    } else {
      c << "void _c2_ar" << a->UID << "(_c2_gr, _c2_gi, _c2_ni, _c2_pi, "
        <<   "_c2_RetNode, _c2_RetQVar, _c2_AddToCache, _c2_value)\n";
   }
   c << "  _c2_GraphBase *_c2_gr;\n"
     << "  _c2_Index _c2_gi;\n"
     << "  _c2_Index _c2_ni;\n"
     << "  _c2_Index *_c2_pi;\n"
     << "  _c2_NodeBase **_c2_RetNode;\n"
     << "  _c2_SeqVar ***_c2_RetQVar;\n"
     << "  int _c2_AddToCache;\n";

   switch (a->BaseType->IsA()) {
    case IntSType:
      c << "  int _c2_value;\n";
      break;
    case RealSType:
      c << "  double _c2_value;\n";
      break;
    case CharSType:
      c << "  char _c2_value;\n";
      break;
    case ArraySType:
    case StructSType:
      c << "  void *_c2_value;\n";
      break;
    default:
      Die("Internal error:  Illegal base type in SerialMakeArcRoute\n");
   }
      
   c << "{\n"
     << "  struct _c2_gv" << g->UID << " *_c2_g;\n"
     << "  _c2_NodeBase *_c2_TmpNode;\n"
     << "  _c2_SeqVar **_c2_QVar;\n"
     << "  _c2_Value _c2_TmpValue;\n\n"
     << "  _c2_g = (struct _c2_gv" <<g->UID << " *) (_c2_gr->LocalData);\n\n"
     << "  _c2_TmpNode = *_c2_RetNode;\n"
     << "  _c2_QVar = *_c2_RetQVar;\n\n";

   if (AddDbgrCode) {
        c << "  _c2_TmpValue.fplist = _c2_fplist;\n"
          << "  _c2_TmpValue.cmd = _c2_cmd;\n\n";
   }

     c 	<< "if (_c2_AddToCache) {\n";

   FindArc(in->Name, cn, &a, &k, &ToNode);
   SerialDoNextArc(c, a, k, ToNode);

   c << "*_c2_RetNode = _c2_TmpNode;\n"
     << "*_c2_RetQVar = _c2_QVar;\n"
     << "}\n\n";
}


  // Create arc route function (for returns) for arc that carries
  // NSRel.

void SerialMakeNSArcRoute(cArc *a, cGraph *CalledGraph, cCallNode *cn)
{
   cCursor c;
   cObIterator i;
   cObject *p;
   cIntNode *in;
   cAbsTree *ToNode;
   cGraph *g;
   ArcKind k;

   g = (cGraph *) GetContaining(a, Graph, 0);

   // Create the Arc Route Function signature in c2_main.h

   c.cd(eMain_h);
   c.cd(eArcFuncSig);
   c << "void _c2_ar" << a->UID << "();\n";

   // Create the entry in MakeObj that sets a graph's return function

   c.cd();
   c.cd(eMain_c);
   c.cd(eMakeObj);
   c.cd(eMOEntry, CalledGraph->UID);
   c.cd(eMOEntrySWInit, cn->UID);
   c.cd(eMOEntrySWInBind);

   // What port in called graph is arc bound to?
   for (p = i.Init(CalledGraph->IntNodes); i.MoreLeft(); p = i.Next()) {
      in = (cIntNode *) p;
      if (!(in->InParam) && (char *) in->Name == a->FromPort) {
	 c << "((struct _c2_gv" << CalledGraph->UID << " *) Data)->_c2_prf"
	   <<   in->UID << " = _c2_ar" << a->UID << ";\n";
	 break;
      }
   }

   // Create the Arc Route Function itself in graph.c
   c.cd();
   c.cd(eGraph_c, g->UID);
   c.cd(eArcFunc);

   c << "void _c2_ar" << a->UID << "(_c2_gr, _c2_gi, _c2_ni, _c2_pi, _c2_na, "
     <<   "_c2_LocalAddr, _c2_LocalReqType, _c2_NewLock)\n"
     << "  _c2_GraphBase *_c2_gr;\n"
     << "  _c2_Index _c2_gi;\n"
     << "  _c2_Index _c2_ni;\n"
     << "  _c2_Index *_c2_pi;\n"
     << "  _c2_NodeBase *_c2_na;\n"
     << "  void *_c2_LocalAddr;\n"
     << "  int _c2_LocalReqType;\n"
     << "  _c2_LockSet *_c2_NewLock;\n";

   c << "{\n"
     << "  _c2_NSRelBase *_c2_TmpNSRel;\n"
     << "  struct _c2_gv" << g->UID << " *_c2_g;\n\n"
     << "  _c2_g = (struct _c2_gv" <<g->UID << " *) (_c2_gr->LocalData);\n\n";

   FindArc(in->Name, cn, &a, &k, &ToNode);
   SerialDoNextArc(c, a, k, ToNode);

   c << "}\n\n";
}


// add the function signature and definition for size spec allocation function

static int IndexCount;

static void TranslateSizeSpec(cCursor c, cSizeSpec *SizeSpec, MString TypeName, MString Name)
{
   cObIterator i, j;
   cObject *p, *q;
   cSETypeName *SETypeName, *NextSETypeName;
   MString NextTypeName, Bound, Index;
   int LastSizeSpec, LastTypeDimension, BraceCount, IndexCountIncrease;

   SETypeName = (cSETypeName *)GetSymbol(TypeName, SizeSpec, ::SETypeName, 0);
   NextTypeName = SETypeName->UserBaseTypeName;
   NextSETypeName = (cSETypeName *) GetSymbol(NextTypeName, SizeSpec, ::SETypeName, 0);

   BraceCount = IndexCountIncrease = 0;

   for (p = i.Init(SizeSpec->ArrayBounds); i.MoreLeft(); p = i.Next()) {
      cExpr *Expr = (cExpr *)p;
      // the number of ArrayBounds in SizeSpec <= the number of Array Dimension of TypeName
      LastSizeSpec = i.Last();

      while (SETypeName->Type == NextSETypeName->Type) { // skips the named types if any
         NextTypeName = NextSETypeName->UserBaseTypeName;
         NextSETypeName = (cSETypeName *) GetSymbol(NextTypeName, SizeSpec, ::SETypeName, 0);
      }
      LastTypeDimension = (NextSETypeName->Type->IsA() != ::ArraySType ? 1 : 0);


      Bound = MString("bound") + IndexCount;
      Index = MString("index") + IndexCount;
      c << "int " << Bound << " = " << Expr->Translate() << ";\n";
      c << Name << " = (" << TypeName << ")_c2_shmalloc(sizeof(" << NextTypeName << ") * "
        << Bound << ");\n";
      c << "_c2_AddrHashTabAdd((char *)" << Name << ", " << Bound << ");\n";

      switch (NextSETypeName->Type->IsA()) {
      case ::CharSType:
      case ::IntSType:
      case ::RealSType:
         break;
      case ::ArraySType:
      case ::StructSType:
         c << "{\n"
          << "int " << Index << ";\n"
           << "for (" << Index << "=0; " << Index << "<" << Bound << "; " 
           << Index << "++) {\n";
         BraceCount += 2;
         IndexCount++;
         IndexCountIncrease++;
         Name = MString(Name) + MString("[") + MString(Index) + MString("]");
         break;
      default:
         break;
      }

      SETypeName = NextSETypeName;
      TypeName = NextTypeName;

      NextTypeName = SETypeName->UserBaseTypeName;
      NextSETypeName = (cSETypeName *) GetSymbol(NextTypeName, SizeSpec, ::SETypeName, 0);

   }

   if (SETypeName->Type->IsA() == ::ArraySType) {
   
      // this array dimension will not be malloc-ed so it needs to be zeroed
      c << Name << " = (" << TypeName << ")0;\n";

   } else if (SETypeName->Type->IsA() == ::StructSType) {

      c << Name << " = (" << TypeName << ")_c2_shmalloc(sizeof(*(" << Name << ")));\n"
        << "_c2_AddrHashTabAdd((char *)" << Name << ", sizeof(*(" << Name << ")));\n";

      while (SETypeName->UserBaseTypeName != MString("")) // skips the named types if any
         SETypeName = (cSETypeName *)
           GetSymbol(SETypeName->UserBaseTypeName, SizeSpec, ::SETypeName, 0);

      cStructSType *StructSType = (cStructSType *)SETypeName->Type; 
      for (p = i.Init(StructSType->Members); i.MoreLeft(); p = i.Next()) {
         cStructMemSType *StructMemSType = (cStructMemSType *)p;

         switch (StructMemSType->Type->IsA()) {
         case ::CharSType:
         case ::IntSType:
         case ::RealSType:
            break;
         case ::ArraySType:
         case ::StructSType:
         {
            cMemSize *MemSize;
            MString MemberTypeName, ExpandedName;
            int Found = 0;
            for (q = j.Init(SizeSpec->MemberSizes); j.MoreLeft();q = j.Next()) {
               MemSize = (cMemSize *)q;
               if ((char *) MemSize->MemberName == StructMemSType->Ident) {
                  Found = 1;
                  break;
               }
            }
            if (Found == 1) {
               ExpandedName = MString(Name) + MString("->") + MString(StructMemSType->Ident);
               MemberTypeName = StructMemSType->Type->TypeName;
               c << "{\n";
               TranslateSizeSpec(c, MemSize->Sizes, MemberTypeName, ExpandedName);
               c << "}\n";
            } else {
               ExpandedName = MString(Name) + MString("->") + MString(StructMemSType->Ident);
               MemberTypeName = StructMemSType->Type->TypeName;
               c << ExpandedName << " = (" << MemberTypeName << ")0;\n";
            }
         }
            break;
         default:
            break;
         }
      }
   }

   for (; BraceCount >0; BraceCount--) {
      c << "}\n";
   }
   IndexCount -= IndexCountIncrease;
}

void SerialSizeSpec(cSizeSpec *SizeSpec, MString TypeName)
{
   if (SizeSpec != 0) {
      cCursor c;
      cGraph *g;
      cAbsTree *n;
      MString c2TypeName;
      MString ReturnVariableName;

      n = GetContaining(SizeSpec, UC, NSRel, 0);
      g = (cGraph *) GetContaining(n, Graph, 0);

      c2TypeName = SerialGetName(TypeName, SizeSpec);
      ReturnVariableName = MString("_o_") + MString(c2TypeName);

      c.cd(eMain_h);
      c.cd(eNewFuncSig);
      c << c2TypeName  << " _c2_new" << -(SizeSpec->UID) << "();\n" ;

      c.cd();
      c.cd(eMain_c);
      c.cd(eNewFunc);
      c << c2TypeName << " _c2_new" << -(SizeSpec->UID) 
        << "(_c2_g, _c2_l)\n"
        << "struct _c2_gv" << g->UID << " *_c2_g;\n"
        << "struct _c2_nv" << n->UID << " *_c2_l;\n"
        << "{\n"
        << c2TypeName << " " << ReturnVariableName << ";\n\n";

      IndexCount = 0;
      TranslateSizeSpec(c, SizeSpec, TypeName, ReturnVariableName);

      c << "\nreturn " << ReturnVariableName << ";\n";
      c << "}\n";
   }
}




void SerialCreateReplCond(cExpr *rc, cReplThing *Where)
{
   cCursor c;
   cGraph *g;
   cRule *r;
   cUC *u;
   cAbsTree *tmp;
   int FiringRule;

   // Yes, this is horrid.

   if (rc == 0) return;

   tmp = GetContaining(Where, Rule, Guard, ArcInput, Binding, ArcOutput, 0);
   r = (cRule *) GetContaining(tmp, Rule, 0);
   FiringRule = r->FiringRule;
   u = (cUC *) GetContaining(r, UC, 0);
   g = (cGraph *) GetContaining(u, Graph, 0);

   c.cd(eGraph_c, g->UID);
   c.cd(eUCScope, u->UID);
   c.cd(eUCProc);

   if (FiringRule)
     switch (tmp->IsA()) {
      case Rule:
	c.cd(eUCProcFrule, r->UID);
	c.cd(eUCProcFRHead);
	c << "if (" << rc->Translate()  << ") {\n";
	c.up();
	c.cd(eUCProcFRTail);
	c << "}\n";
	break;
      case Guard:
	c.cd(eUCProcFrule, r->UID);
	c.cd(eUCProcFRCheck, tmp->UID);
	c.cd(eUCProcFRCheckHead);
	c << "if (" << rc->Translate()  << ") {\n";
	c.up();
	c.cd(eUCProcFRCheckTail);
	c << "}\n";
	break;
      case Binding: 
	c.cd(eUCProcFrule, r->UID);
	c.cd(eUCProcFRBind, tmp->UID);
	c.cd(eUCProcFRBindHead);
	c << "if (" << rc->Translate()  << ") {\n";
	c.up();
	c.cd(eUCProcFRBindTail);
	c << "}\n";
	break;
      case ArcInput:
	c.cd(eUCProcFrule, r->UID);
	c.cd(eUCProcFRCheck, tmp->UID);
	c.cd(eUCProcFRCheckHead);
	c << "if (" << rc->Translate()  << ") {\n";
	c.up();
	c.cd(eUCProcFRCheckTail);
	c << "}\n";
	//  *** This also goes into a binding part!
        c.up();
	c.up();
	c.cd(eUCProcFRBind, tmp->UID);
	c.cd(eUCProcFRBindHead);
	c << "if (" << rc->Translate()  << ") {\n";
	c.up();
	c.cd(eUCProcFRBindTail);
	c << "}\n";
	break;
     default:
       assert (0); // We should never get here.
       break;
     }
   else
     switch (tmp->IsA()) {
      case Rule:
	c.cd(eUCProcRrule, r->UID);
	c.cd(eUCProcRRHead);
        c << "if (" << rc->Translate()  << ") {\n";
	c.up();
	c.cd(eUCProcRRTail);
	c << "}\n";
	break;
      case Guard:
	c.cd(eUCProcRrule, r->UID);
	c.cd(eUCProcRRCheck, tmp->UID);
	c.cd(eUCProcRRCheckHead);
        c << "if (" << rc->Translate()  << ") {\n";
	c.up();
	c.cd(eUCProcRRCheckTail);
	c << "}\n";
	break;
      case ArcOutput:
      case Binding:
	c.cd(eUCProcRrule, r->UID);
	c.cd(eUCProcRRBind, tmp->UID);
	c.cd(eUCProcRRBindHead);
        c << "if (" << rc->Translate()  << ") {\n";
	c.up();
	c.cd(eUCProcRRBindTail);
	c << "}\n";
	break;
      default:
	assert (0); // We should never get here.
	break;
     }
}

