/*
  unified_aux.C
  ------------------------------------------------------------------------
  Auxiliary routines for the Unified backend.
  ------------------------------------------------------------------------
  @(#) $Id: unified_aux.C,v 1.52 1998/11/06 06:41:31 emery Exp $
  ------------------------------------------------------------------------
  AUTHOR/CONTACT:

  Emery Berger                    | <http://www.cs.utexas.edu/users/emery>
  Parallel Programming Group      |  <http://www.cs.utexas.edu/users/code>
  Department of Computer Sciences |             <http://www.cs.utexas.edu>
  University of Texas at Austin   |                <http://www.utexas.edu>
  ========================================================================
*/

#include <config.h>

#include <assert.h>
#include <stream.h>
#include <unistd.h>
#include <stdlib.h>

// CODE includes

#include <misc/general.h>
#include <exmodel/abstree.h>
#include <exmodel/exmodel.h>
#include <symtab/symtab.h>
#include <lib/UidTable.h>
#include <lib/MsgUI.h>
#include <treebuild/pass.h>
#include <misc/voidlist.h>
#include <misc/sfile.h>
#include <misc/fileutil.h>
#include <symtab/symtab.h>
#include <lib/Syntax.h>
#include <translate/translate.h>

#include "unified_aux.h"
#include "unified_arc.h"

#undef DONT_BROADCAST_CREPS


// Keep track of the number of functions
// we've created. We'll pass that number
// as an index into a global function array
// when we do arc and port messages.

static int ArcRouteFunctions;
static int PortFunctions;


int UnifiedWriteFiles (cProgram * ProgramRoot)
{
  cBlock *b;
  cCursor c;
  ofstream OutFile;
  cGraph *g;
  cObIterator i;
  cObject *o;
  char *CurrentWorkingDirectory;
  MString TranslationDirectory;

  CurrentWorkingDirectory = new char[512];
  //cwd must not be longer than 510 bytes
  if (getcwd (CurrentWorkingDirectory, 512) == NULL) {
    FixMe ("Could not cd to program dir for translate\n");
    return 0;
  }
  switch (TranslatorT) {
  case MPI:
    TranslationDirectory = MString(TestDirectory) + MString ("/") + MString(ProgramRoot->Name) + MString (".mpi");
    break;

  case PVM:
    TranslationDirectory = MString(TestDirectory) + MString ("/") + MString(ProgramRoot->Name) + MString (".pvm");
    break;

  case UNIFIED:
    TranslationDirectory = MString(TestDirectory) + MString ("/") + MString(ProgramRoot->Name) + MString (".smp");
    break;

  default:
    assert (0); // We should never get here.
  }


  if (chdir ((char *) TranslationDirectory)) {
    FixMe ("Could not cd to program dir for translate\n");
    return 0;
  }
  //Makefile

  b = (cBlock *) c.RootBlock ()->SubBlocks.PopFirst ();

  if (b->ID != eMakefile)
    Die ("Internal error: ID should have been eMakefile\n");

  OutFile.open ("Makefile");

  if (!OutFile) {
    FixMe ("Could not open Makefile\n");
    return 0;
  }
  b->Output (OutFile);

  OutFile.close ();

  //c2_main.h

  b = (cBlock *) c.RootBlock ()->SubBlocks.PopFirst ();

  if (b->ID != eMain_h)
    Die ("Internal error: ID should have been eMain_h\n");

  OutFile.open ("c2_main.h");

  if (!OutFile) {
    FixMe ("Could not open c2_main.h file\n");
    return 0;
  }
  b->Output (OutFile);

  OutFile.close ();

  //c2_main.c

  b = (cBlock *) c.RootBlock ()->SubBlocks.PopFirst ();

  if (b->ID != eMain_c)
    Die ("Internal error: ID should have been eMain_c\n");

  OutFile.open ("c2_main.C");

  if (!OutFile) {
    FixMe ("Could not open c2_main.C file\n");
    return 0;
  }
  b->Output (OutFile);

  OutFile.close ();

  //c2_globtype.h

  b = (cBlock *) c.RootBlock ()->SubBlocks.PopFirst ();

  if (b->ID != eGlob_h)
    Die ("Internal error: ID should have been eGlob_h\n");

  OutFile.open ("c2_globtype.h");

  if (!OutFile) {
    FixMe ("Could not open c2_globtype file\n");
    return 0;
  }
  b->Output (OutFile);

  OutFile.close ();

  //graph.c for each graph

  for (o = i.Init (ProgramRoot->Graphs); i.MoreLeft (); o = i.Next ()) {
    g = (cGraph *) o;
    b = (cBlock *) c.RootBlock ()->SubBlocks.PopFirst ();

    if (b->ID != eGraph_c)
      Die ("Internal error: ID should have been eGraph_c\n");

    OutFile.open ((char *) (g->Name + ".C"));

    if (!OutFile) {
      FixMe ("Could not open a graph C file\n");
      return 0;
    }
    b->Output (OutFile);

    OutFile.close ();
  }

  if (chdir (CurrentWorkingDirectory)) {
    FixMe ("Internal error: Could not cd to parent dir from translate\n");
    return 0;
  }
  return 1;
}



void UnifiedMakefile (cProgram * p) 
{
  cCursor c;
  cObIterator i;
  void *v;
  cGraph *g;
  MString LibFile, LibFile_2, LibFile_3;

  c.cd (eMakefile);
  c << "# Makefile for \"" << p->Name << "\" produced by CODE.\n"
    << "# Requires GNU Make.\n"
    << "# For more information on CODE, see <http://www.cs.utexas.edu/users/code>.\n\n";

  switch (TranslatorT) {
  case PVM:
    c << "PVM = 1\n";
    break;
  case MPI:
    c << "MPI = 1\n";
    break;
  case UNIFIED:
    break;
  default:
    assert (0); // We should never get here.
    break;
  }


  c << "# Define a default CODE_LIBRARY.\n"
    << "ifndef CODE_LIBRARY\n"
    << "define CODE_LIBRARY\n"
    << "/usr/local/lib\n"
    << "endef\n"
    << "endif\n\n";

  c << "# Define a default MPIRUN_DEVICE.\n"
    << "ifndef MPIRUN_DEVICE\n"
    << "define MPIRUN_DEVICE\n"
    << "ch_p4\n"
    << "endef\n"
    << "endif\n\n";

  c.cd (eDecl);

  c << "ifdef MPI\n"
    << "MPIDEF = -DMPI\n"
    << "else\n"
    << "ifdef PVM\n"
    << "MPIDEF = -DPVM\n"
    << "else\n"
    << "MPIDEF = \n"
    << "endif\n"
    << "endif\n\n";

  c << "ifeq ($(CXX),g++)\n"
    << "OPTIMIZE =  # g++ 2.8.1 optimizer breaks on switch in c2_main.C!\n"
    << "else\n";
    
  c << "ifdef DEBUG\n"
    << "OPTIMIZE = -g\n"
    << "else\n"
    << "OPTIMIZE = -O\n"
    << "endif\n\n";

  c << "endif\n\n";

  c << "ifeq ($(OSTYPE),unicos)\n"

    << "CXX = CC\n"
    << "MPICXX = CC\n"
    << "CFLAGS = $(MPIDEF) $(OPTIMIZE)\n"
    << "CCFLAGS = $(MPIDEF) $(OPTIMIZE) -h instantiate=local\n"
    << "MPILIBS = -lmpi\n"

    << "else\n"

    << "ifeq ($(OSTYPE),solaris)\n"
    << "ifdef MPI\n\n"

    << "ifndef MPICXX\n"
    << "MPICXX = mpiCC # $(CXX) -I$(MPIHOME)/include\n"
    << "endif\n"
    << "ifdef HPC # Sun's high-performance MPI\n"
    << "MPILIBS = -L/opt/SUNWhpc/lib -lmpi_mt -lsocket -lnsl\n"
    << "else\n"
    << "MPILIBS = -lmpi++ -lpmpi -lmpi -lsocket -lnsl\n\n"
    << "endif # HPC\n\n"

    << "else # not MPI\n\n"

    << "ifdef PVM\n\n"
    << "MPICXX = $(CXX) -I$(PVM_ROOT)/include\n"
    << "MPILIBS = -lpvm3 -lsocket -lnsl\n\n"

    << "else\n"

    << "MPICXX = $(CXX)\n"

    << "endif # PVM\n"
    << "endif # not MPI\n"

    << "CFLAGS = $(MPIDEF) $(OPTIMIZE)\n"
    << "ifeq ($(CXX), CC)\n"
    << "CCFLAGS = $(MPIDEF) $(OPTIMIZE) +w2 -pto -mt\n"
    << "else # CXX != CC\n"
    << "CCFLAGS = $(MPIDEF) $(OPTIMIZE)\n"
    << "endif # CXX != CC\n"
    << "else # not Solaris\n\n"

    << "ifdef MPI\n"
    << "ifndef MPICXX\n"
    << "MPICXX = mpiCC\n"
    << "endif\n"
    << "MPILIBS = -lmpi\n"
    << "else # not MPI\n"
    << "MPICXX = $(CXX)\n"
    << "MPILIBS = \n"
    << "endif # not MPI\n"
    << "CFLAGS = $(MPIDEF) $(OPTIMIZE)\n"
    << "CCFLAGS = $(MPIDEF) $(OPTIMIZE)\n"
    << "endif # not Solaris\n"

    << "endif\n\n";


  LibFile = "-L$(CODE_LIBRARY)/runtime/unified -lcode_unified";
  LibFile_2 = "-L$(CODE_LIBRARY)/runtime/unified -lcode_unified_mpi";
  LibFile_3 = "-L$(CODE_LIBRARY)/runtime/unified -lcode_unified_pvm";

  c << "C2DIR = $(CODE_LIBRARY)/runtime/unified\n";

  c << "ifeq ($(OSTYPE),aix)\n"
    << "PTHREADLIB = pthreads\n"
    << "else\n"
    << "PTHREADLIB = pthread\n"
    << "endif\n\n";

  c << "ifeq ($(CXX),g++)\n"
    << "# CCFLAGS += -fguiding-decls\n"
    << "STLINCLUDE =\n"
    << "else\n"
    << "STLINCLUDE = -I$(CODE_LIBRARY)/runtime/stl\n"
    << "endif\n\n";

  c << "INC = -I$(C2DIR) $(STLINCLUDE) -I$(MPIHOME)/include\n"
    << "INDENT = indent\n"
    << "INDENTFLG = -i2\n";

  c.up ();
  c.cd (eObj);
  if (AddDbgrCode) {
    c << "OBJS = c2_main.o \\\n      ";
  } else {
    c << "OBJS = c2_main.o \\\n      ";
  }

  c << FilesToLink << " \\\n      ";

  c.up ();
  c.cd (eProgt);

  c << "SRCS = c2_main.C ";

  for (v = i.Init (p->Graphs); i.MoreLeft (); v = i.Next ()) {
    g = (cGraph *) v;
    c << g->Name << ".c ";
  }
  c << "\n\n";

  c << ".c.o:\n\t$(CXX) $(CCFLAGS) $(INC) -c $<\n\n";
  c << ".C.o:\n\t$(CXX) $(CCFLAGS) $(INC) -c $<\n\n";

  c << "ifdef MPI\n"
    << p->Name << ": $(OBJS)\n"
    << "\t$(MPICXX) $(CCFLAGS) $(OBJS) " << LibFile_2 << "\\\n"
    << " -L$(MPIHOME)/lib/$(MPIARCH)/$(MPIRUN_DEVICE) $(MPILIBS) -lm -l$(PTHREADLIB) -o " << p->Name << "\n"
    << "else\n";

  c << "ifdef PVM\n"
    << p->Name << ": $(OBJS)\n"
    << "\t$(MPICXX) $(CCFLAGS) $(OBJS) " << LibFile_3 << "\\\n"
    << " -L$(PVM_ROOT)/lib/$(PVM_ARCH) $(MPILIBS) -lm -l$(PTHREADLIB) -o " << p->Name << "\n"
    << "\tcp " << p->Name << " $(PVM_ROOT)/bin/$(PVM_ARCH)/\n"
    << "else\n"

    << p->Name << ": $(OBJS)\n"
    << "\t$(CXX) $(CCFLAGS) $(OBJS) " << LibFile << "\\\n"
    << " -lm -l$(PTHREADLIB) -o " << p->Name << "\n"
    << "endif\n"
    << "endif\n\n";

  c << "clean:\n"
    << "\t\\rm -f *.o *.pretty core " << p->Name << "\n\n"
    << "depend:\n"
    << "\tmakedepend $(INC) $(SRCS)\n\n"
    << "pretty:\n"
    << "\t$(INDENT) c2_globtype.h c2_globtype.h.pretty $(INDENTFLG)\n"
    << "\t$(INDENT) c2_main.h c2_main.h.pretty $(INDENTFLG)\n"
    << "\t$(INDENT) c2_main.C c2_main.C.pretty $(INDENTFLG)\n";

  for (v = i.Init (p->Graphs); i.MoreLeft (); v = i.Next ()) {
    g = (cGraph *) v;
    c << "\t$(INDENT) " << g->Name << ".C " << g->Name
      << ".C.pretty $(INDENTFLG)\n";
  }

  c << "\nc2_main.o: c2_main.C\n";

}


MString UnifiedGetName (MString & Ident, cAbsTree * Here) 
{
  cSymTabEntry *t;

  //Find a symbol table that contains ident, checking parents up
  // the tree.

  while (Here != 0) {
    if (Here->SymbolTable != 0) {
      t = Here->SymbolTable->Get (Ident);
      if (t != 0)
	break;
    }
    Here = Here->Parent;
    //Check the parent
  }

  if (t == 0)
    //Ident is not found.
    Die ("Internal error: Ident not found in UnifiedGetName\n");

  switch (t->IsA ()) {
  case SEFuncArg:
    return Ident;
  case SEFuncName:
    return Ident;
  case SEVariable:
    if (Here->IsA () == UC || Here->IsA () == NSRel) {
      if (Ident == MString("NodeIndex"))
	return "(_c2_MyAddr->MyIndex())";
      else if (Ident == MString("GraphIndex"))
	return "(_c2_MyAddr->MyGraph()->MyIndex())";
      else
	return "(_c2_l->" + Ident + ")";
    } else
      return Ident;
  case SETypeName:
    return Ident;
  case SEPortName:
    return Ident;
  case SEIntNode:
    return Ident;
  case SESharedVar:
    return "(_c2_l->" + Ident + ")";
  case SECreP:
    return "(_c2_g->" + Ident + ")";
  case SEGraph:
    return Ident;
  default:
    Die ("Internal error: no case in UnifiedGetName\n");
    break;
  }

  return "";
  //just to silence gcc

}


void UnifiedCreateMain (cProgram * ProgramRoot) 
{
  cCursor c;

  c.cd (eMain_c);
  c.cd (eMainProg);

  c << "int numProcessors;\n"
    << "int numWorkers;\n"
    << "Graph *MainGraph;\n\n";

  c << "#ifdef MPI\n"
    << "#include <MPImessenger.H>\n"
    << "#else\n"
    << "#ifdef PVM\n"
    << "#include <PVMmessenger.H>\n"
    << "#else\n"
    << "#include <Unmessenger.H>\n"
    << "#endif\n"
    << "#endif\n\n";

  c << "int main (int argc, char *argv[])\n"
    << "{\n"
    << "#ifdef MPI\n"
    << "typedef MPIMessenger * MPIMessengerPtr;\n"
    << "MPIMessengerPtr * _messenger = new MPIMessengerPtr[CODE_Machine::MAX_THREADS+1];\n"
    << "for (int i = 0; i < CODE_Machine::MAX_THREADS+1; i++) {\n"
    << "_messenger[i] = new MPIMessenger;\n"
    << "}\n"
    << "  _c2_startup(argc, argv, \"" << ProgramRoot->Name << "\", " << StartGraphUID << ", " << StartNodeUID << ", \"" << StartGraphUID << "/" << StartNodeUID << "\", (Messenger **) _messenger);\n"
    << "#else\n"
    << "#ifdef PVM\n"
    << "typedef PVMMessenger * PVMMessengerPtr;\n"
    << "PVMMessengerPtr * _messenger = new PVMMessengerPtr[CODE_Machine::MAX_THREADS+1];\n"
    << "for (int i = 0; i < CODE_Machine::MAX_THREADS+1; i++) {\n"
    << "_messenger[i] = new PVMMessenger;\n"
    << "}\n"
    << "int tids[Messenger::MAX_PROCESSORS];\n"
    << "int info, nhosts, narch;\n"
    << "struct pvmhostinfo * hostp;\n"
    << "info = pvm_config (&nhosts, &narch, &hostp);\n"
    << "if (pvm_parent() == PvmNoParent) {\n"
    << "  pvm_catchout (stdout);\n"
    << "  pvm_catchout (stderr);\n"
    << "  info = pvm_spawn (\"" << ProgramRoot->Name << "\", argv, PvmTaskDefault, \"\", nhosts, tids);\n"
    << "  pvm_initsend (PvmDataDefault);\n"
    << "  pvm_pkint (tids, nhosts, 1);\n"
    << "  pvm_mcast (tids, nhosts, 999);\n"
    << "  pvm_exit();\n"
    << "} else {\n"
    << "  pvm_recv (-1, 999);\n"
    << "  pvm_upkint (tids, nhosts, 1);\n"
    << "  _messenger[0]->PVMinitialize (nhosts, tids);\n"
    << "  _c2_startup(argc, argv, \"" << ProgramRoot->Name << "\", " << StartGraphUID << ", " << StartNodeUID << ", \"" << StartGraphUID << "/" << StartNodeUID << "\", (Messenger **) _messenger);\n"
    << "  pvm_exit();\n"
    << "}\n"
    << "#else\n"
    << "typedef UnMessenger * UnMessengerPtr;\n"
    << "UnMessengerPtr * _messenger = new UnMessengerPtr[CODE_Machine::MAX_THREADS+1];\n"
    << "for (int i = 0; i < CODE_Machine::MAX_THREADS+1; i++) {\n"
    << "_messenger[i] = new UnMessenger;\n"
    << "}\n"
    << "  _c2_startup(argc, argv, \"" << ProgramRoot->Name << "\", " << StartGraphUID << ", " << StartNodeUID << ", \"" << StartGraphUID << "/" << StartNodeUID << "\", (Messenger **) _messenger);\n"
    << "#endif\n"
    << "#endif\n\n";

  if (AddDbgrCode)
    c << "  // _c2_RunDebugger(StartNode, argc, argv);\n";
  c << "}\n";

  if (AddDbgrCode) {
    c.cd ();
    c.cd (eMain_c);
    c.cd (eDbgSymEntry);
    c.cd (eDbgSEHead);
    c << "void _c2_dbgSymEntry()\n"
      << "{\n";
    c.up ();
    c.cd (eDbgSETail);
    c << "}\n";

    c.cd ();
    c.cd (eMain_c);
    c.cd (eDbgSymAddr);
    c.cd (eSAHead);
    c << "void *_c2_getSymAddr(UID, pUID, Data)\n"
      << "int UID, pUID; \n"
      << "void *Data;\n"
      << "{\n"
      << " switch(pUID) {\n";
    c.up ();
    c.cd (eSATail);
    c << " }\n"
      << "}\n";
  }
  //Create the case in MakeObj for the start graph.

  c.cd ();
  c.cd (eMain_c);
  c.cd (eMakeObj);
  c.cd (eMOEntry, StartGraphUID);
  c.cd (eMOEntryCase);

  c << "case " << StartGraphUID << ":\n";
}



//Handle the function defs that are stored in a GUIDE structure pointed
// to by a void *

void UnifiedDoFuncDefs (MString & FuncDefs, cAbsTree * Here) 
{
  cCursor c;
  cAbsTree *t, *t2;

  t = GetContaining (Here, UC, NSRel, Graph, Program, 0);

  switch (t->IsA ()) {
  case Program:
    c.cd (eMain_c);
    c.cd (eGlobFunc);
    break;
  case Graph:
    c.cd (eGraph_c, t->UID);
    c.cd (eGraphFunc);
    break;
  case NSRel:
    t2 = GetContaining (t, Graph, 0);
    c.cd (eGraph_c, t2->UID);
    c.cd (eNSScope, t->UID);
    c.cd (eNSFunc);
    break;
  case UC:
    t2 = GetContaining (t, Graph, 0);
    c.cd (eGraph_c, t2->UID);
    c.cd (eUCScope, t->UID);
    c.cd (eUCFunc);
    break;
  default:
    assert (0); // We should never get here.
    break;
  }

#if 0
  c << "#ifdef __cplusplus\n"
    << "extern \"C\" {\n"
    << "#endif\n";
#endif

  c << FuncDefs << "\n";

#if 0
  c << "#ifdef __cplusplus\n"
    << "}\n"
    << "#endif\n";
#endif

  c << "\n";
}



int InitUnified (cProgram * p) 
{
  MString TranslationDirectory;

  //Make a directory to write source into.

  switch (TranslatorT) {
  case PVM:
    TranslationDirectory = MString(TestDirectory) + MString ("/") + MString(p->Name) + MString (".pvm");
    break;

  case MPI:
    TranslationDirectory = MString(TestDirectory) + MString ("/") + MString(p->Name) + MString (".mpi");
    break;

  case UNIFIED:
    TranslationDirectory = MString(TestDirectory) + MString ("/") + MString(p->Name) + MString (".smp");
    break;

  default:
    assert (0); // We should never get here.
  }

  if (!MakeSrcDir ((char *) TranslationDirectory)) {
    FixMe ("Could not create source directory.\n");
    return 0;
  }
  InitGetName (UnifiedGetName);
  InitSizeSpec (UnifiedSizeSpec);
  InitCursor (eRoot);

  return 1;
}


void UnifiedMakeObj (void)
{
  cCursor c;

  c.cd (eMain_c);
  c.cd (eMakeObj);
  c.cd (eMOHead);

  c << "\nvoid *_c2_MakeObj(int UID, Index *index, Graph *graph, int *ObjType, SerialMachine * TheMachine)\n"
    << "{\n"
    << "  CompNode *tmpN;\n"
    << "  NameSharingRelation *tmpR;\n"
    << "  Graph *tmpG;\n"
    << "  CodeDataBase *Data;\n\n"
    << "  switch (UID) {\n";

  c.up ();
  c.cd (eMOTail);

  c << "   default:\n"
    << "     fprintf(stderr, \"Internal error: "
    << "No such UID %d in MakeObj\\n\\n\", UID);\n"
    << "     return (void *) 0;\n"
    << "     break;\n"
    << "  }\n"
    << "}\n\n";
}


void UnifiedCopyUC ()
{
  cCursor c;

  c.cd (eMain_c);
  c.cd (eCopyUC);
  c.cd (eCopyUCHead);

  c << "\n"
    << "void CompNode::Copy (CompNode *_c2_MyAddr, int flag, Messenger& m)\n"
    << "{\n"
    << "  void *_c2_l;\n"
    << "  _c2_l = _c2_MyAddr->LocalData()->data();\n"
    << "  switch(_c2_MyAddr->ID())\n"
    << "  {\n";

  c.up ();

  c.cd (eCopyUCTail);

  c << "  default:\n"
    << "     WHEN(1, printf(\"CopyUC: UID = %d\\n\", _c2_MyAddr->ID()));\n"
    << "    _c2_KillComp(_c2_NOSUCHUID);\n"
    << "    break;\n"
    << "  }\n"
    << "}\n\n\n";

}


void UnifiedCopyGraph ()
{
  cCursor c;

  c.cd (eMain_c);
  c.cd (eCopyGraph);
  c.cd (eCopyGraphHead);

  c << "\n"
    << "void Graph::Copy (Graph *_c2_MyAddr, int flag, Messenger& m)\n"
    << "{\n"
    << "  void *_c2_l;\n"
    << "  _c2_l = _c2_MyAddr->LocalData()->data();\n"
    << "  switch(_c2_MyAddr->ID())\n"
    << "  {\n";

  c.up ();

  c.cd (eCopyGraphTail);

  c << "  default:\n"
    << "     WHEN(1, printf(\"Graph::Copy() : UID = %d\\n\", _c2_MyAddr->ID()));\n"
    << "    _c2_KillComp(_c2_NOSUCHUID);\n"
    << "    break;\n"
    << "  }\n"
    << "}\n\n\n";

}



void UnifiedCopyArc ()
{
  cCursor c;

  c.cd (eMain_c);
  c.cd (eCopyArc);
  c.cd (eCopyArcHead);

  c << "\n"
    << "void _c2_CopyArc (SerialMachine * SerMach,\n"
    << "                   _c2_MessageType mt,\n"
    << "                   int arc,\n"
    << "		   Graph * graph,\n"
    << "		   Index& graphIndex,\n"
    << "                   Index& nodeIndex,\n"
    << "		   Index& portIndex,\n"
    << "		   CompNode * node,\n"
    << "		   _c2_Value& value,\n"
    << "		   Messenger * messenger)\n"
    << "{\n"
    << "  CompNode * returnNode;\n"
    << "  _c2_SeqVarQ * returnQVar;\n"
    << "  switch(arc)\n"
    << "  {\n";

  c.up ();

  c.cd (eCopyArcTail);

  c << "  default:\n"
    << "     WHEN(1, printf(\"_c2_CopyArc : UID = %d\\n\", arc));\n"
    << "    _c2_KillComp(_c2_NOSUCHUID);\n"
    << "    break;\n"
    << "  }\n"
    << "}\n\n\n";

}


void UnifiedCopyNSRel (void)
{
  cCursor c;

#if 0
  c.cd ();
  c.cd (eMain_h);
  c.cd (eCopyNSRelSig);

  c << "void NameSharingRelation::Copy (int NSRelUID, int UID, void **LocalAddr, int direction, Messenger& messenger);\n";
#endif

  c.cd ();
  c.cd (eMain_c);
  c.cd (eCopyNSRel);

  c << "void NameSharingRelation::Copy (int NSRelUID, int UID, void **LocalAddr, int direction, Messenger& messenger)\n"
    << "{\n"
    << "  switch(NSRelUID)\n"
    << "  {\n";

  c.cd ();
  c.cd (eMain_c);
  c.cd (eCopyNSRelTail);

  c << "  default:\n"
    << "    _c2_KillComp(_c2_NOSUCHNSREL);\n"
    << "  }\n"
    << "}\n";
}


void UnifiedGetSharedVarInfo (void)
{
  cCursor c;

  c.cd ();
  c.cd (eMain_c);
  c.cd (eGetSharedVarInfo);

  c << "SharedVariableInfo& NameSharingRelation::GetSharedVarInfo (int UID)\n"
    << "{\n"
    << "  switch (UID)\n"
    << "  {\n";

  c.cd ();
  c.cd (eMain_c);
  c.cd (eGetSharedVarInfoTail);

  c << "  default:\n"
    << "    _c2_KillComp(_c2_NOSUCHNSREL);\n"
    << "  }\n"
    << "  // None was found. We return a dummy here to satisfy the compiler.\n"
    << "  SharedVariableInfo * dummy;\n"
    << "  assert (0);\n"
    << "  return * dummy;\n"
    << "}\n";
}


//Create skeletons of UC Comp and Init Procs

void UnifiedUCProc (cUC * u) 
{
  cCursor c;
  cGraph *g;

  g = (cGraph *) u->Parent;

  //Do the header of the procedure

  c.cd (eGraph_c, g->UID);
  c.cd (eUCScope, u->UID);
  c.cd (eUCProc);
  c.cd (eUCProcHead);

  c << "CompNode::ExecutionResult _c2_cp" << u->UID << "(CompNode *_c2_MyAddr)    /* " << u->Name << " */\n"
    << "{\n";
  if (u->StartNode)
    c << "  static int _c2_NotStarted = 1;\n";
  c << "  _c2_nv" << u->UID << " *_c2_l;\n"
    << "  _c2_gv" << g->UID << " *_c2_g;\n"
    << "  int _c2_Stat;\n"
    << "  int _c2_NullRule;\n"
    << "  Index _c2_PortIndex;\n"
    << "  _c2_Value _c2_TmpValue;\n\n";

  c << "/* Do not allow execution until state is returned by thief. */\n"
    << "assert (_c2_MyAddr->GetPhase() != CompNode::BLOCKED_AWAITING_THIEF);\n";

  c << "_c2_l = (_c2_nv" << u->UID << " *) (_c2_MyAddr->LocalData()->data());\n"
    << "_c2_g = (_c2_gv" << g->UID << " *)"
    << " (_c2_MyAddr->MyGraph()->LocalData()->data());\n\n";

  if (AddDbgrCode) {
    c << "   if (_c2_chkStoppedBpAct(_c2_MyAddr)) {\n";
    if (u->StartNode)
      c << "      _c2_NotStarted = 0;\n";
    c << "       goto _c2_BefComp;\n"
      << "}\n";
  }

  // Skip checking input queues for stolen nodes.
  c << "if (_c2_MyAddr->GetPhase() == CompNode::STOLEN_EXECUTION) {  /* Remote execution (after firing) for theft. */\n"
    << "  goto _c2_skiplocks;\n"
    << "}\n";

  if (u->Shared) {
    c << "if (!_c2_MyAddr->NSInitialized()) {\n";
  }

  c.up ();
  c.cd (eUCProcBefFrule);

  if (u->Shared) {
    c << "  _c2_MyAddr->NSInitialize ();\n"
      << "}\n"
      << "if (_c2_MyAddr->NSBusy()) goto _c2_nsrel;\n\n";
  }

  //Lock Qlock for firing rule testing


  if (AddDbgrCode) {
    c << "_c2_again:\n";
    c << "_c2_beginUcEvBpAct(_c2_MyAddr);\n";
  }
  if (u->StartNode) {
    //let it fire once for free
    c << "if (_c2_NotStarted) {\n"
      << "  goto _c2_comp;\n"
      << "}\n\n";
  }
  //Do the compute section

  c.up ();
  c.cd (eUCProcComp);

  c << "/* No rule is true */\n";

  if (AddDbgrCode) {
    c << "_c2_waitFpBpAct(_c2_MyAddr);\n\n";
  }
  c << "return CompNode::NO_FIRING_RULES_TRUE;\n\n"
    << "_c2_comp:\n";

  if (u->Shared) {
    c << "_c2_nsrel:\n";
    c << "if (!_c2_MyAddr->GetAllLocks()) {\n"
      << "  return CompNode::NSREL_BUSY;\n"
      << "}\n\n";
  }

  c << "  if (_c2_MyAddr->GetPhase() == CompNode::PRE_THEFT_FIRING) { /* Local pre-firing for theft. */\n"
    << "    return CompNode::SUCCESSFUL_PREFIRING;\n"
    << "  }\n";

  c << "_c2_skiplocks:\n\n";


  if (AddDbgrCode) {
    c << " if (_c2_befCompBpAct( _c2_MyAddr)) {\n"
      << "        return;\n"
      << "}\n"
      << "_c2_BefComp:\n\n";
  }
  if (TraceLevel == tr_sel_nodes)
    c << "printf(\"UC " << u->Name << " (" << u->UID << ") fires\\n\");\n"
      << "fflush(stdout);\n";

  if (u->Comp != 0) {
    //
    // c << "  cout << \"" << u->Name << ": executing computation.\" << endl;\n";
    //
    // c << "_c2_MyAddr->StartTimer ();\n";
    c << u->Comp->Translate ();
    // c << "_c2_MyAddr->StopTimer ();\n";

  }

  //Start the clock after the first firing of the start node.

  if (u->StartNode) {
    c << "if (_c2_NotStarted) {\n"
      << "  _c2_NotStarted = 0;\n"
      << "}\n\n";
  }
  if (u->TermNode) {
    //Stop the computation;
    if (AddDbgrCode) {
      c << "_c2_firesTermNode(_c2_MyAddr);\n";
    } else {
      c << "_c2_MyAddr->GetMachine().LocalStop ();\n";
    }
  }
  //Do the proc tail

  c.up ();
  c.cd (eUCProcTail);

  c << "// Stolen execution complete.\n" 
    << "if (_c2_MyAddr->GetPhase() == CompNode::STOLEN_EXECUTION) {\n";
  c << "  return CompNode::SUCCESSFUL_STOLEN;\n"
    << "} else {\n";

  if (AddDbgrCode) {
    c << "_c2_MyAddr->RelAllLocks ();\n"
      << "_c2_endUcEvBpAct(_c2_MyAddr);\n\n"
      << "goto _c2_again;\n";
  } else {
    c << "_c2_MyAddr->RelAllLocks ();\n";
  }

  c << "}\n";
  c << "return CompNode::SUCCESSFUL;\n";

  c << "}\n\n";

  //Create signature in c2_main.c

  c.cd ();
  c.cd (eMain_c);
  c.cd (eIntFuncSig);

  c << "CompNode::ExecutionResult _c2_cp" << u->UID << "(CompNode *_c2_MyAddr);  /* " << u->Name << " */\n";

  //Create InitProc

  c.cd ();
  c.cd (eGraph_c, g->UID);
  c.cd (eUCScope, u->UID);
  c.cd (eUCIProc);
  c.cd (eUCIProcHead);

  c << "void _c2_ip" << u->UID << "(CompNode *_c2_MyAddr)    /* " << u->Name << " */\n"
    << "{\n"
    << "  _c2_nv" << u->UID << " *_c2_l;\n"
    << "  _c2_gv" << g->UID << " *_c2_g;\n\n"
    << "_c2_l = (_c2_nv" << u->UID << " *) (_c2_MyAddr->LocalData()->data());\n"
    << "_c2_g = (_c2_gv" << g->UID << " *)"
    << " (_c2_MyAddr->MyGraph()->LocalData()->data());\n\n";

  c.up ();
  c.cd (eUCIProcComp);
  if (u->InitComp != 0)
    c << u->InitComp->Translate ();

  c.up ();
  c.cd (eUCIProcTail);
  c << "}\n\n";

  //Create signature
  c.cd ();
  c.cd (eMain_c);
  c.cd (eIntFuncSig);

  c << "void _c2_ip" << u->UID << "(CompNode *_c2_MyAddr);  /* " << u->Name << " */\n";
}

//Create skeleton of NSRel Init Proc

void UnifiedNSProc (cNSRel * n)
{
  cCursor c;
  cGraph *g;


  g = (cGraph *) n->Parent;

  c.cd (eGraph_c, g->UID);
  c.cd (eNSScope, n->UID);
  c.cd (eNSIProc);
  c.cd (eNSIProcHead);

  c << "void _c2_ip" << n->UID << "(NameSharingRelation *_c2_MyAddr)    /* " << n->Name << " */\n"
    << "{\n"
    << "  _c2_nv" << n->UID << " *_c2_l;\n"
    << "  _c2_gv" << g->UID << " *_c2_g;\n\n"
    << "_c2_l = (_c2_nv" << n->UID << " *) (_c2_MyAddr->LocalData()->data());\n"
    << "_c2_g = (_c2_gv" << g->UID << " *)"
    << " (_c2_MyAddr->MyGraph()->LocalData()->data());\n";

  c.up ();
  c.cd (eNSIProcComp);

  if (n->InitComp != 0)
    c << n->InitComp->Translate ();

  c.up ();
  c.cd (eNSIProcTail);
  c << "}\n\n";

  //Create signature
  c.cd ();
  c.cd (eMain_c);
  c.cd (eIntFuncSig);

  c << "void _c2_ip" << n->UID << "(NameSharingRelation *_c2_MyAddr);  /* " << n->Name << " */\n";
}


void UnifiedUCMakeObj (cUC * u)
{
  cCursor c;

  //Do MakeObj entry
  c.cd ();
  c.cd (eMain_c);
  c.cd (eMakeObj);
  c.cd (eMOEntry, u->UID);
  c.cd (eMOEntryHead);
  c << "case " << u->UID << ":\n"
    << "if (TheMachine) {\n"
    << "  Data = (CodeDataBase *) new (TheMachine) CodeData<_c2_nv" << u->UID << ">(TheMachine);\n"
    << "  tmpN = new (TheMachine) CompNode (UID, index, graph, Data, _c2_cp" << u->UID
    << ", " << "_c2_ip" << u->UID << ", TheMachine, " << u->Static << ", " << u->Shared << ");\n"
    << "} else {\n"
    << "  Data = (CodeDataBase *) new CodeData<_c2_nv" << u->UID << ">();\n"
    << "  tmpN = new CompNode (UID, index, graph, Data, _c2_cp" << u->UID
    << ", " << "_c2_ip" << u->UID << ", NULL, " << u->Static << ", " << u->Shared << ");\n"
    << "}\n";

  c.up ();
  c.cd (eMOEntryTail);
  c << "*ObjType = _c2_UCNODE;\n\n"
    << "return (void *) tmpN;\n"
    << "break;\n";

  c.cd ();
  c.cd (eMain_c);
  c.cd (eCopyUC);
  c.cd (eCopyUCHead);
  c.cd (eCopyUCCase, u->UID);
  c.cd (eCopyUCCaseHead);
  c << "case " << u->UID << ":\n";

  c.cd (eCopyUCCase1Head);
  c << "if ((flag & LOCAL) == TRANSMIT) {\n";
  c.up ();
  c.cd (eCopyUCCase1Tail);
  c << "}\n";
  c.up ();
  c.cd (eCopyUCCase2Head);
  c << "if ((flag & LOCAL) == RECEIVE) {\n";
  c.up ();
  c.cd (eCopyUCCase2Tail);
  c << "}\n"
    << "break;\n\n";
}

void UnifiedGraphMakeObj (cGraph * g)
{
  cCursor c;

  c.cd ();
  c.cd (eMain_c);
  c.cd (eMakeObj);
  c.cd (eMOEntry, g->UID);
  c.cd (eMOEntryHead);

  c << "if (TheMachine) {\n"
    << "Data = (CodeDataBase *) new (TheMachine) CodeData<_c2_gv" << g->UID << ">(TheMachine);\n"
    << "tmpG = new (TheMachine) Graph (UID, index, graph, Data, " << g->CrepNodes.Length () << ", TheMachine);\n"
    << "} else {\n"
    << "Data = (CodeDataBase *) new CodeData<_c2_gv" << g->UID << ">();\n"
    << "tmpG = new Graph (UID, index, graph, Data, " << g->CrepNodes.Length () << ", NULL);\n"
    << "}\n"
    << "*ObjType = _c2_CALLNODE;\n";

  c.up ();
  c.cd (eMOEntrySWHead);
  c << "switch (UID) {\n";

  c.up ();
  c.cd (eMOEntryTail);
  c << "};\n"
    << "return (void *) tmpG;\n";

  c.cd ();
  c.cd (eMain_c);
  c.cd (eCopyGraph);
  c.cd (eCopyGraphHead);
  c.cd (eCopyGraphCase, g->UID);
  c.cd (eCopyGraphCaseHead);
  c << "case " << g->UID << ":\n";

  c.cd (eCopyGraphCase1Head);
  c << "if ((flag & LOCAL) == TRANSMIT) {\n";
  c.up ();
  c.cd (eCopyGraphCase1Tail);
  c << "}\n";
  c.up ();
  c.cd (eCopyGraphCase2Head);
  c << "if ((flag & LOCAL) == RECEIVE) {\n";
  c.up ();
  c.cd (eCopyGraphCase2Tail);
  c << "}\n"
    << "break;\n\n";

}


void UnifiedArcAssign (cCursor & c, cExpr * VarExpr, cExpr * PortExpr,
		       cAbsTree * Here) 
{
  MString PN;
  cSEPortName *t;
  cPortSType *p;

  PN = FindPortName (PortExpr);

  t = (cSEPortName *) GetSymbol (PN, Here, SEPortName, 0);
  p = (cPortSType *) (t->Type);

  switch (p->BaseType->IsA ()) {
  case IntSType:
    c << VarExpr->Translate () << " = _c2_TmpValue.u.i;\n";
    break;
  case RealSType:
    c << VarExpr->Translate () << " = _c2_TmpValue.u.d;\n";
    break;
  case CharSType:
    c << VarExpr->Translate () << " = _c2_TmpValue.u.c;\n";
    break;
  case ArraySType:
    c << "_c2_free_" << VarExpr->StructuralType->TypeName << "("
      << VarExpr->Translate () << ");\n"
      << VarExpr->Translate ()
      << " = (" << VarExpr->StructuralType->TypeName << ") _c2_TmpValue.u.array;\n";

    break;
  case StructSType:
    c << "_c2_free_" << VarExpr->StructuralType->TypeName << "("
      << VarExpr->Translate () << ");\n"
      << VarExpr->Translate ()
      << " = (" << VarExpr->StructuralType->TypeName << ") _c2_TmpValue.u.structure;\n";
    break;
  default:
    Die ("Internal Error: Wrong type in UnifiedTypeOf\n");
    break;
  }

}


void UnifiedSharVarFunc (cNSPortSpec * nsp, cSESharedVar * sv)
{
  cGraph *g;
  cSType *pt;
  cCursor c;
  cUC *u;
  cArc *a;
  ArcKind k;
  cAbsTree *ToNode;

  u = (cUC *) GetContaining (nsp, UC, 0);
  g = (cGraph *) GetContaining (nsp, Graph, 0);

  pt = (cSType *) sv->Type;

  //Put func sig in place
  c.cd (eMain_h);
  c.cd (ePortFuncSig);

  c << "void " << sv->GenName << "(CompNode *_c2_na, Index *_c2_pi);\n";

  //Create function itself

  c.cd ();
  c.cd (eGraph_c, g->UID);
  c.cd (ePortFunc, nsp->UID);

  c << "void " << sv->GenName << "(CompNode *_c2_na, Index *_c2_pi)  /* Shared variable \"Port\" " << nsp->PortName << " */\n"
    << "{\n"
    << "  _c2_gv" << g->UID << " *_c2_g;\n"
    << "  Graph *_c2_gr;\n"
    << "  NameSharingRelation *_c2_TmpNSRel;\n"
    << "  Index _c2_gi;\n"
    << "  Index _c2_ni;\n"
    << "  void ** _c2_LocalAddr;\n"
    << "  _c2_RequestType _c2_LocalReqType;\n\n";

  c << "  _c2_gr = _c2_na->MyGraph();\n"
    << "  _c2_ni = _c2_na->MyIndex();\n"
    << "  _c2_g = (_c2_gv" << g->UID << " *) (_c2_gr->LocalData()->data());\n"
    << "  _c2_LocalAddr = (void **) &((_c2_nv" << u->UID
    << " *) (_c2_na->LocalData()->data()))->" << nsp->PortName << ";\n"
    << "_c2_LocalReqType = ";

  if (nsp->UseRule == NSReader)
    c << "_c2_READER;\n\n";
  else
    c << "_c2_WRITER;\n\n";

  FindArc (nsp->PortName, u, &a, &k, &ToNode);
  UnifiedDoNextArc (c, a, k, &ToNode, c);

  c << "}\n\n";
}


void UnifiedPortFunc (cPortSpec * p) 
{
  cGraph *g;
  cSEPortName *ps;
  cPortSType *pt;
  cCursor c, d;
  cUC *u;
  cArc *a;
  ArcKind k;
  cAbsTree *ToNode;


  cObIterator i;
  cObject *s;
  cCallNode *cn;
  int IsCrep;

  int arc;

  u = (cUC *) (p->Parent);
  g = (cGraph *) GetContaining (p, Graph, 0);
  ps = (cSEPortName *) GetSymbol (p->PortName, p, SEPortName, 0);
  if (ps->Type->IsA () == PortSType)
    pt = (cPortSType *) ps->Type;
  else
    Die ("Internal error: port type wrong in UnifiedPortFunc.\n");

  //Put func sig in place
  c.cd (eMain_h);
  c.cd (ePortFuncSig);

  c << "void " << ps->GenName << "_remote(CompNode *_c2_na, Index *_c2_pi, _c2_Value *_c2_value, Messenger& _Messenger);  /* Port " << p->PortName << " */\n";

  //Add a new Port function to the PortFunction table.
  c.cd ();
  c.cd (eMain_c);
  c.cd (ePortFunctions, ++PortFunctions);
  c << " " << ps->GenName << "_remote,";


  //Create function itself

  c.cd ();
  c.cd (eGraph_c, g->UID);
  c.cd (ePortFunc, p->UID);
  c.cd (ePortFuncHead);

  c << "void " << ps->GenName << "_remote(CompNode *_c2_na, Index *_c2_pi, _c2_Value *_c2_value, Messenger& _Messenger)  /* Port " << p->PortName << " */\n";

  c << "{\n"
    << "  _c2_gv" << g->UID << " *_c2_g;\n"
    << "  Graph *_c2_gr;\n"
    << "  Graph * destgraph;\n"
    << "  CompNode *_c2_TmpNode;\n"
    << "  Index _c2_origpi;\n"
    << "  Index _c2_gi;\n"
    << "  Index _c2_ni;\n"
    << "  _c2_SeqVarQ _c2_QVar = 0;\n"
    << "  int Id = 0;\n"
    << "  int _c2_me = _c2_na->GetMachine().WhichProcessor();\n"
    << "  _c2_Value _c2_TmpValue;\n\n";

  if (AddDbgrCode) {
    c << " _c2_FpList _c2_fplist;\n"
      << " _c2_Cmd _c2_cmd;\n\n";
  }

  {
    char temp[8];
    //Hack to get XXX out of _c2_gXXX
    int i, j;
    
    i = ps->GenName.length ();
    i -= 5;
    //subtract 5 from length to account for _c2_g
    for (j = 0; j < i; j++)
      temp[j] = ps->GenName[5 + j];
    
    temp[j] = '\0';
    
    arc = atoi (temp);
  }

  FindArc (p->PortName, u, &a, &k, &ToNode);

  if (a->ArcType != 1) {

    //Arc does not connect an interface node to a non - interface node.

    c.up ();
    c.cd (ePortFuncPvmBody);

    c << "  int j;\n"
      << "  _Messenger << _c2_gr->GetPathName().str();\n"
      << "  _Messenger << _c2_gi;\n"
      << "  _Messenger << _c2_ni;\n"
      << "  _Messenger << *_c2_pi;\n";

    c << "  j = " << arc << ";\n"
      << "  _Messenger << j;\n";

    if (AddDbgrCode) {
      c << "  _c2_packFPList(_c2_fplist);\n"
	<< "  _Messenger.pack (_c2_cmd);\n";
    }
    switch (a->BaseType->IsA ()) {
    case IntSType:
      c << "  _Messenger << (_c2_value->u.i);\n";
      break;
    case RealSType:
      c << "  _Messenger <<  (_c2_value->u.d);\n";
      break;
    case CharSType:
      c << "  _Messenger <<  (_c2_value->u.b);\n";
      break;
    case ArraySType:
      c << "  _c2_copy_" << a->BaseType->TypeName
	<< "((" << a->BaseType->TypeName << ") _c2_value->u.array, (" << a->BaseType->TypeName << ") 0, TRANSMIT, _Messenger);\n";
      break;
    case StructSType:
      c << "  _c2_copy_" << a->BaseType->TypeName
	<< "((" << a->BaseType->TypeName << ") _c2_value->u.structure, (" << a->BaseType->TypeName << ") 0, TRANSMIT, _Messenger);\n";
      break;
    default:
      Die ("Internal error:  Illegal base type in UnifiedPortFuncHead\n");
    }
  }
  d.cd (eMain_h);
  d.cd (eLocalPortFuncSig);

  d << "void " << ps->GenName << "_local(Graph *_c2_gr, Index _c2_gi, Index _c2_ni, Index *_c2_pi, CompNode **_c2_NodeAddr, _c2_Value * _c2_value";

  if (AddDbgrCode) {
    d << ", _c2_FpList _c2_fplist, _c2_Cmd _c2_cmd";
  }
  d << ", SerialMachine * SerMach);\n";


  d.cd ();
  d.cd (eGraph_c, g->UID);
  d.cd (eLocalPortFunc, p->UID);
  d.cd (eLocalPortFuncHead);
  d << "\n\nvoid " << ps->GenName << "_local(Graph *_c2_gr, Index _c2_gi, Index _c2_ni, Index *_c2_pi, CompNode **_c2_NodeAddr, _c2_Value * _c2_value";

  if (AddDbgrCode) {
    d << ", _c2_FpList _c2_fplist, _c2_Cmd _c2_cmd";
  }
  d << ", SerialMachine * SerMach)\n";

  d << "{\n"
    << "_c2_SeqVarQ *_c2_QVar = 0;\n"
    << "CompNode *_c2_TmpNode = 0;\n"
    << "_c2_Value _c2_TmpValue;\n\n";

  d.up ();
  d.cd (eLocalPortFuncTail);
  d << "}\n\n\n";

  d.up ();
  d.cd (eLocalPortFuncBody);


  c.up ();
  c.cd (ePortFuncLocalBody);

  c << "  _c2_gr = _c2_na->MyGraph();\n"
    << "  _c2_ni = _c2_na->MyIndex();\n"
    << "  _c2_origpi = *_c2_pi;\n"
    << "  _c2_g = (_c2_gv" << g->UID << " *) (_c2_gr->LocalData()->data());\n\n";

  if (AddDbgrCode) {
    c << " _c2_appndFpBpAct(_c2_na, &_c2_fplist, &_c2_cmd);\n\n";
  }

  UnifiedDoNextArc (c, a, k, &ToNode, d);

  IsCrep = 0;
  for (s = i.Init (g->CallNodes); i.MoreLeft (); s = i.Next ()) {
    cn = (cCallNode *) s;
    if (cn->UID == ToNode->UID) {
      IsCrep = 1;
      break;
    }
  }

  if (a->ArcType != 1) {
    c.up ();
    c.cd (ePortFuncPvmMap);

    if (IsCrep) {

#ifdef DONT_BROADCAST_CREPS
      c << "  destgraph = (Graph *) _c2_GetAddr(_c2_gr, " << a->ToUID << ", &_c2_gi, &_c2_na->GetMachine()); // foo3 \n";
      // c << "  cout << \"Destination graph (foo3) = \" << destgraph->GetPathName().str() << endl; \n";
      c << "Id = destgraph->Processor();\n"

	<< "if (Id != _c2_me) WHEN(1, printf(\"Sending crep to task %d\\n\", Id));\n"


	<< "if (Id != _c2_me)\n";
#endif

      c << "  { \n"
	<< "    Guard m (_Messenger.objectlock ());\n";

    } else {

      if (a->ToUID == ToNode->UID) {

	// Destination is in this graph.

	c << "  CompNode * destnode = (CompNode *) _c2_GetAddr(_c2_gr, " << ToNode->UID << ", &_c2_ni, &_c2_na->GetMachine()); // foo2.1 \n";

      } else {

	c << "  destgraph = (Graph *) _c2_GetAddr(_c2_gr, " << a->ToUID << ", &_c2_gi, &_c2_na->GetMachine());\n";
	// c << "  cout << \"Destination graph (foo2.2) = \" << destgraph->GetPathName().str() << endl; \n";
	c << "  CompNode * destnode = (CompNode *) _c2_GetAddr(destgraph, " << ToNode->UID << ", &_c2_ni, &_c2_na->GetMachine()); // foo2.2 \n";

      }

      c << "  Id = destnode->Processor ();\n";

      // c << "  cout << \"Arc firing (" << a->UID << ", Port " << p->PortName << ") from \" << *_c2_na << \" to \" << *destnode << endl;\n";

      c << "  _c2_na->AddTarget (destnode);\n";

      c << "  if (Id != _c2_me)\n "
	<< "  { \n"
	<< "    Guard m (_Messenger.objectlock ());\n";
    }

    c.up ();
    c.cd (ePortFuncPvmSendType);

    if (IsCrep) {

#ifdef DONT_BROADCAST_CREPS
      c << "  _Messenger.send (Id, CODE_Machine::REMOTE_ARC_DATA_TRANSFER);\n";
      c << " } else\n";
#else
      c << "  _Messenger.broadcast (0, _Messenger.numProcessors() - 1, CODE_Machine::REMOTE_ARC_DATA_TRANSFER);\n";
      c << " }\n";
#endif

      c << ps->GenName << "_local(_c2_gr, _c2_gi, _c2_ni, _c2_pi, &_c2_TmpNode, _c2_value";

      if (AddDbgrCode) {
	c << ", _c2_fplist, _c2_cmd);\n";
      }
      c << ", &_c2_na->GetMachine());\n";

    } else {

      c << "  _Messenger.send (Id, CODE_Machine::REMOTE_ARC_DATA_TRANSFER);\n"
	<< " }\n"
	<< " else \n"
	<< ps->GenName << "_local(_c2_gr, _c2_gi, _c2_ni, _c2_pi, &_c2_TmpNode, _c2_value";

      if (AddDbgrCode) {
	c << ", _c2_fplist, _c2_cmd);\n";
      }
      c << ", &_c2_na->GetMachine());\n";

    }

  }
  c.up ();
  c.cd (ePortFuncTail);

  c << "}\n\n";

  //Make a entry in the Message handler routine 's case structure

  c.cd ();
  c.cd (eMain_c);
  c.cd (eCopyArc);
  c.cd (eCopyArcCaseHead, arc);

  c << " case " << arc << ":\n";

  switch (a->BaseType->IsA ()) {
  case ::IntSType:
    c << " *messenger >> (value.u.i);\n";
    c << " value.type = _c2_IsAnInt;\n";
    break;
  case ::RealSType:
    c << " *messenger >> (value.u.d);\n";
    c << " value.type = _c2_IsADouble;\n";
    break;
  case ::CharSType:
    c << " *messenger >> (value.u.c);\n";
    c << " value.type = _c2_IsAChar;\n";
    break;
  case ::ArraySType:
    c << " value.u.array = (void *) _c2_copy_" << a->BaseType->TypeName
      << "((" << a->BaseType->TypeName << ") 0, (" << a->BaseType->TypeName << ") 0, mt, *messenger);\n";
    c << " value.type = _c2_IsAnArrayPtr;\n";
    break;
  case ::StructSType:
    c << " value.u.structure = (void *) _c2_copy_" << a->BaseType->TypeName
      << "((" << a->BaseType->TypeName << ") 0, (" << a->BaseType->TypeName << ") 0, mt, *messenger);\n";
    c << " value.type = _c2_IsAStructPtr;\n";
    break;
  default:
    break;
  }

  c << " " << ps->GenName << "_local(graph, graphIndex, nodeIndex, &portIndex, &node, &value, ";
  if (AddDbgrCode) {
    c << "_c2_fplist, _c2_cmd, ";
  }
  c << "SerMach);\n";

  c << " break;\n";
  
}



//Create all of the things needed for an arc route function.These are
// created for all arcs that exit a Call node.

void UnifiedMakeArcRoute (cArc * a, cGraph * CalledGraph, cCallNode * cn) 
{
  cCursor c, d;
  cObIterator i;
  cObject *p;
  cIntNode *in;
  cAbsTree *ToNode;
  cGraph *g;
  ArcKind k;

  int IsCrep;
  cCallNode *pn;

  g = (cGraph *) GetContaining (a, Graph, 0);

  //Create the Arc Route Function signature in c2_main.h

  c.cd (eMain_h);
  c.cd (eArcFuncSig);
  c << "void _c2_ar" << a->UID << "(Graph *_c2_gr, CompNode * source, Index _c2_gi, Index _c2_ni, Index *_c2_pi, CompNode **_c2_RetNode, _c2_SeqVarQ *_c2_RetQVar, _c2_Value * c2_value, Messenger& _Messenger);\n";

  //Create the entry in MakeObj that sets a graph 's return function

  c.cd ();
  c.cd (eMain_c);
  c.cd (eMakeObj);
  c.cd (eMOEntry, CalledGraph->UID);
  c.cd (eMOEntrySWInit, cn->UID);
  c.cd (eMOEntrySWInBind);

  //What port in called graph is arc bound to ?
  for (p = i.Init (CalledGraph->IntNodes); i.MoreLeft (); p = i.Next ()) {
    in = (cIntNode *) p;
    if (!(in->InParam) && (char *) in->Name == a->FromPort) {
      c << "((_c2_gv" << CalledGraph->UID << " *) Data->data())->_c2_prf"
	<< in->UID << " = _c2_ar" << a->UID << ";\n";
      break;
    }
  }


  c.cd ();
  c.cd (eMain_c);
  c.cd (eCopyArc);
  c.cd (eCopyArcCase, a->UID);

  c << " case " << a->UID << ":\n";

  switch (a->BaseType->IsA ()) {
  case IntSType:
    c << " *messenger >> (value.u.i);\n"
      << " value.type = _c2_IsAnInt;\n";
    break;
  case RealSType:
    c << " *messenger >> (value.u.d);\n"
      << " value.type = _c2_IsADouble;\n";
    break;
  case CharSType:
    c << " *messenger >> (value.u.c);\n"
      << " value.type = _c2_IsAChar;\n";
    break;
  case ArraySType:
    c << " value.u.array = (void *) _c2_copy_" << a->BaseType->TypeName
      << "((" << a->BaseType->TypeName << ") 0, (" << a->BaseType->TypeName << ") 0, mt, *messenger);\n"
      << " value.type = _c2_IsAnArrayPtr;\n";
    break;
  case StructSType:
    c << " value.u.structure = (void *) _c2_copy_" << a->BaseType->TypeName
      << "((" << a->BaseType->TypeName << ") 0, (" << a->BaseType->TypeName << ") 0, mt, *messenger);\n"
      << " value.type = _c2_IsAStructPtr;\n";
    break;
  default:
    Die ("Internal error:  Illegal base type in UnifiedMakeArcRoute\n");
  }

  c << " _c2_ar" << a->UID << "_local(graph, graphIndex, nodeIndex, "
    << "&portIndex, &returnNode, &returnQVar, &node, &value, ";

  if (AddDbgrCode) {
    c << "_c2_fplist, _c2_cmd, ";
  }
  c << "SerMach);\n";
  
  c << "break;\n";


  //Add a new Arc Route function to the ArcRouteFunction table.
  c.cd ();
  c.cd (eMain_c);
  c.cd (eArcRouteFunctions, ++ArcRouteFunctions);
  c << "  _c2_ar" << a->UID << ",";


  //Create the Arc Route Function itself in graph.c
  c.cd ();
  c.cd (eGraph_c, g->UID);
  c.cd (eArcFunc, a->UID);
  c.cd (eArcFuncHead);

  if (AddDbgrCode) {
    c << "void _c2_ar" << a->UID << "(_c2_fplist, _c2_cmd, "
      << "_c2_gr, _c2_gi, _c2_ni, _c2_pi, "
      << "_c2_RetNode, _c2_RetQVar, _c2_value)\n"
      << "  _c2_FpList _c2_fplist;\n"
      << "  _c2_Cmd _c2_cmd;\n";

    c << "  Graph *_c2_gr;\n"
      << "  Index _c2_gi;\n"
      << "  Index _c2_ni;\n"
      << "  Index *_c2_pi;\n"
      << "  CompNode **_c2_RetNode;\n"
      << "  _c2_SeqVarQ *_c2_RetQVar;\n";

    switch (a->BaseType->IsA ()) {
    case IntSType:
      c << "  int _c2_value;\n";
      break;
    case RealSType:
      c << "  double _c2_value;\n";
      break;
    case CharSType:
      c << "  char _c2_value;\n";
      break;
    case ArraySType:
    case StructSType:
      c << "  void *_c2_value;\n";
      break;
    default:
      Die ("Internal error:  Illegal base type in UnifiedMakeArcRoute\n");
    }

  } else {
    c << "void _c2_ar" << a->UID << "(Graph *_c2_gr, CompNode * source, Index _c2_gi, Index _c2_ni, Index *_c2_pi, CompNode **_c2_RetNode, _c2_SeqVarQ *_c2_RetQVar, _c2_Value * _c2_value, Messenger& _Messenger)\n";

  }

  c << "{\n"
    << "  int Id = 0;\n"
    << "  int _c2_me = _c2_gr->GetMachine().WhichProcessor();\n"
    << "  _c2_gv" << g->UID << " *_c2_g;\n"
    << "  CompNode *_c2_TmpNode;\n"
    << "  _c2_SeqVarQ _c2_QVar;\n"
    << "  _c2_Value _c2_TmpValue;\n\n"
    << "  _c2_g = (_c2_gv" << g->UID << " *) (_c2_gr->LocalData()->data());\n\n"
    << "  _c2_TmpNode = *_c2_RetNode;\n"
    << "  _c2_QVar = *_c2_RetQVar;\n";

  if (AddDbgrCode) {
    c << "  _c2_TmpValue.fplist = _c2_fplist;\n"
      << "  _c2_TmpValue.cmd  = _c2_cmd;\n\n";
  }


  c.up ();
  c.cd (eArcFuncPvmBody);

  c << " int j,k;\n\n"
    << " _Messenger <<  (_c2_gr->GetPathName().str());\n"
    << " _Messenger << _c2_gi;\n"
    << " _Messenger << _c2_ni;\n"
    << " _Messenger << * _c2_pi;\n";

  c << " j = " << a->UID << " ;\n"
    << " _Messenger <<  (j);\n";

  if (AddDbgrCode) {
    c << "  _c2_packFPList(_c2_fplist);\n"
      << "  _Messenger.pack (_c2_cmd);\n";
  }
  switch (a->BaseType->IsA ()) {
  case IntSType:
    c << " _Messenger <<  (_c2_value->u.i);\n";
    break;
  case RealSType:
    c << " _Messenger <<  (_c2_value->u.d);\n";
    break;
  case CharSType:
    c << " _Messenger <<  (_c2_value->u.c);\n";
    break;
  case ArraySType:
    c << " _c2_copy_" << a->BaseType->TypeName
      << "((" << a->BaseType->TypeName << ") _c2_value->u.array, (" << a->BaseType->TypeName << ") 0, TRANSMIT, _Messenger);\n";
    break;
  case StructSType:
    c << " _c2_copy_" << a->BaseType->TypeName
      << "((" << a->BaseType->TypeName << ") _c2_value->u.structure, (" << a->BaseType->TypeName << ") 0, TRANSMIT, _Messenger);\n";
    break;
  default:
    Die ("Internal error:  Illegal base type in UnifiedMakeArcRoute\n");
  }

  c.up ();
  c.cd (eArcFuncLocalBody);

  FindArc (in->Name, cn, &a, &k, &ToNode);

  d.cd (eMain_h);
  d.cd (eLocalArcFuncSig);

  if (AddDbgrCode) {
    d << "void _c2_ar" << a->UID << "_local(Graph *_c2_gr, Index _c2_gi, Index _c2_ni, Index *_c2_pi, CompNode **_c2_RetNode, _c2_SeqVarQ **_c2_RetQVar, CompNode **_c2_NodeAddr, _c2_Value *_c2_value";
  } else {
    d << "void _c2_ar" << a->UID << "_local(Graph *_c2_gr, Index _c2_gi, Index _c2_ni, Index *_c2_pi, CompNode **_c2_RetNode, _c2_SeqVarQ **_c2_RetQVar, CompNode **_c2_NodeAddr, _c2_Value *_c2_value";
  }

  if (AddDbgrCode) {
    d << ", _c2_FpList _c2_fplist, _c2_Cmd _c2_cmd";
  }
  d << ", SerialMachine * SerMach);\n";

  d.cd ();
  d.cd (eGraph_c, g->UID);
  d.cd (eLocalArcFunc, a->UID);
  d.cd (eLocalArcFuncHead);

  if (AddDbgrCode) {
    d << "void _c2_ar" << a->UID << "_local(Graph *_c2_gr, Index _c2_gi, Index _c2_ni, Index *_c2_pi, CompNode **_c2_RetNode, _c2_SeqVarQ **_c2_RetQVar, CompNode **_c2_NodeAddr, _c2_Value *_c2_value";
  } else {
    d << "void _c2_ar" << a->UID << "_local(Graph *_c2_gr, Index _c2_gi, Index _c2_ni, Index *_c2_pi, CompNode **_c2_RetNode, _c2_SeqVarQ **_c2_RetQVar, CompNode **_c2_NodeAddr, _c2_Value *_c2_value";
  }

  if (AddDbgrCode) {
    d << ", _c2_FpList _c2_fplist, _c2_Cmd _c2_cmd";
  }

  d << ", SerialMachine * SerMach)\n";

  d << "{\n"
    << "_c2_SeqVarQ *_c2_QVar = 0;\n"
    << "CompNode *_c2_TmpNode = 0;\n"
    << "_c2_Value _c2_TmpValue;\n\n";

  d.up ();
  d.cd (eLocalArcFuncTail);

  d << "*_c2_RetNode = _c2_TmpNode;\n"
    << "*_c2_RetQVar = _c2_QVar;\n"
    << "}\n";

  d.up ();
  d.cd (eLocalArcFuncBody);

  UnifiedDoNextArc (c, a, k, &ToNode, d);

  IsCrep = 0;
  for (p = i.Init (g->CallNodes); i.MoreLeft (); p = i.Next ()) {
    pn = (cCallNode *) p;
    if (pn->UID == ToNode->UID) {
      IsCrep = 1;
      break;
    }
  }

  c.up ();
  c.cd (eArcFuncPvmMap);

  if (IsCrep) {

#ifdef DONT_BROADCAST_CREPS
      c << "  destgraph = (Graph *) _c2_GetAddr(_c2_gr, " << a->ToUID << ", &_c2_gi, , &_c2_na->GetMachine()); // foo4 \n";
      // c << "  cout << \"Destination graph (foo4) = \" << destgraph->GetPathName().str() << endl; \n";

      c << "Id = destgraph->Processor();\n"
	<< "if (Id != _c2_me) WHEN(1, printf(\"Sending crep to task %d\\n\", Id));\n"
	<< "if (Id != _c2_me)\n";

#endif

    c << "  { \n"
      << "    Guard m (_Messenger.objectlock ());\n";

  } else {

    c << "// a->ToUID = " << a->ToUID << ", ToNode->UID = " << ToNode->UID << "\n";

    if (a->ToUID == ToNode->UID) {

      // Destination is in this graph.
      
      c << "  CompNode * destnode = (CompNode *) _c2_GetAddr(_c2_gr, " << ToNode->UID << ", &_c2_ni, &source->GetMachine()); // foo1 \n";

    } else {

      c << "  destgraph = (Graph *) _c2_GetAddr(_c2_gr, " << a->ToUID << ", &_c2_gi, &_c2_na->GetMachine()); // foo1 \n";
      
      // c << "  cout << \"Destination graph (foo1) = \" << destgraph->GetPathName().str() << endl; \n";
      
      c << "  CompNode * destnode = (CompNode *) _c2_GetAddr(destgraph, " << ToNode->UID << ", &_c2_ni, &_c2_na->GetMachine());\n";
      
    }

    c << "  Id = destnode->Processor ();\n";

    // c << "  cout << \"Arc firing (" << a->UID << ") from graph \" << *_c2_gr << \" to \" << *destnode << endl;\n";

    c << "  source->AddTarget (destnode);\n";

    c << " if (Id != _c2_me)\n "
      << " { \n"
      << "    Guard m (_Messenger.objectlock ());\n";
  }

  c.up ();
  c.cd (eArcFuncPvmSendType);

  if (IsCrep) {

#ifdef DONT_BROADCAST_CREPS
    c << "  _Messenger.send (Id, CODE_Machine::REMOTE_RETURN_FROM_GRAPH);\n";
    c << " } else\n";
#else
    c << "   _Messenger.broadcast (0, _Messenger.numProcessors() - 1, CODE_Machine::REMOTE_RETURN_FROM_GRAPH);\n";
    c << " }\n";
#endif

    c << "  _c2_ar" << a->UID << "_local(_c2_gr, _c2_gi, _c2_ni, _c2_pi"
      << ", _c2_RetNode, &_c2_RetQVar, &_c2_TmpNode, _c2_value, ";

    if (AddDbgrCode) {
      c << "_c2_fplist, _c2_cmd, ";
    }
    c << "SerMach);\n";

  } else {
    c << " _Messenger.send (Id, CODE_Machine::REMOTE_RETURN_FROM_GRAPH);\n"
      << "}\n"
      << " else \n"
      << "  _c2_ar" << a->UID << "_local(_c2_gr, _c2_gi, _c2_ni, _c2_pi"
      << ", _c2_RetNode, &_c2_RetQVar, &_c2_TmpNode, "
      << "  _c2_value";

    if (AddDbgrCode) {
      c << ", _c2_fplist, _c2_cmd";
    }
    c << ", &source->GetMachine());\n";

  }


  c.up ();
  c.cd (eArcFuncTail);

  //c << "*_c2_RetNode = _c2_TmpNode;\n"
  // <<"*_c2_RetQVar = _c2_QVar;\n"
  // <<"}\n"
  c << "}\n\n";
}


//Create arc route function (for returns) for arc that carries 
// NSRel.

void UnifiedMakeNSArcRoute (cArc * a, cGraph * CalledGraph, cCallNode * cn) 
{
  cCursor c;
  cObIterator i;
  cObject *p;
  cIntNode *in;
  cAbsTree *ToNode;
  cGraph *g;
  ArcKind k;

  g = (cGraph *) GetContaining (a, Graph, 0);

  //Create the Arc Route Function signature in c2_main.h

  c.cd (eMain_h);
  c.cd (eArcFuncSig);
  c << "void _c2_ar" << a->UID << "();\n";

  //Create the entry in MakeObj that sets a graph 's return function

  c.cd ();
  c.cd (eMain_c);
  c.cd (eMakeObj);
  c.cd (eMOEntry, CalledGraph->UID);
  c.cd (eMOEntrySWInit, cn->UID);
  c.cd (eMOEntrySWInBind);

  //What port in called graph is arc bound to ?
  for (p = i.Init (CalledGraph->IntNodes); i.MoreLeft (); p = i.Next ()) {
    in = (cIntNode *) p;
    if (!(in->InParam) && (char *) in->Name == a->FromPort) {
      c << "((_c2_gv" << CalledGraph->UID << " *) Data->data())->_c2_prf"
	<< in->UID << " = _c2_ar" << a->UID << ";\n";
      break;
    }
  }

  //Create the Arc Route Function itself in graph.c
  c.cd ();
  c.cd (eGraph_c, g->UID);
  c.cd (eArcFunc, a->UID);

  c << "void _c2_ar" << a->UID << "(Graph *_c2_gr, CompNode * source, Index _c2_gi, Index _c2_ni, Index *_c2_pi, CompNode *_c2_na, void *_c2_LocalAddr, int _c2_LocalReqType, NSLock *_c2_NewLock, Messenger& _Messenger)\n";

  c << "{\n"
    << "  NameSharingRelation *_c2_TmpNSRel;\n"
    << "  _c2_gv" << g->UID << " *_c2_g;\n\n"
    << "  _c2_g = (_c2_gv" << g->UID << " *) (_c2_gr->LocalData()->data());\n\n";

  FindArc (in->Name, cn, &a, &k, &ToNode);
  UnifiedDoNextArc (c, a, k, &ToNode, c);

  c << "}\n\n";
}


//add the function signature and definition for size spec allocation function

static int IndexCount;

static void TranslateSizeSpec (cCursor c, cSizeSpec * SizeSpec, MString TypeName, MString Name) 
{
  cObIterator i, j;
  cObject *p, *q;
  cSETypeName *SETypeName, *NextSETypeName;
  MString NextTypeName, Bound, Index;
  int LastSizeSpec, LastTypeDimension, BraceCount, IndexCountIncrease;

  SETypeName = (cSETypeName *) GetSymbol (TypeName, SizeSpec,::SETypeName, 0);
  NextTypeName = SETypeName->UserBaseTypeName;
  NextSETypeName = (cSETypeName *) GetSymbol (NextTypeName, SizeSpec,::SETypeName, 0);

  BraceCount = IndexCountIncrease = 0;

  for (p = i.Init (SizeSpec->ArrayBounds); i.MoreLeft (); p = i.Next ()) {
    cExpr *Expr = (cExpr *) p;
    //the number of ArrayBounds in SizeSpec <= the number of Array Dimension of TypeName
    LastSizeSpec = i.Last ();

    while (SETypeName->Type == NextSETypeName->Type) {
      //skips the named types if any
      NextTypeName = NextSETypeName->UserBaseTypeName;
      NextSETypeName = (cSETypeName *)
	GetSymbol (NextTypeName, SizeSpec,::SETypeName, 0);
    }
    LastTypeDimension = (NextSETypeName->Type->IsA () !=::ArraySType ? 1 : 0);


    Bound = MString ("bound") + IndexCount;
    Index = MString ("index") + IndexCount;
    c << "int " << Bound << " = " << Expr->Translate () << ";\n";
    c << Name << " = (" << TypeName << ") new Array<" << NextTypeName << "> (" << Bound << ");\n";
    c << "// _c2_AddrHashTabAdd((char *)" << Name << ", " << Bound << ");\n";

    switch (NextSETypeName->Type->IsA ()) {
    case ::CharSType:
    case ::IntSType:
    case ::RealSType:
      break;
    case ::ArraySType:
    case ::StructSType:
      c << "{\n"
	<< "int " << Index << ";\n"
	<< "for (" << Index << "=0; " << Index << "<" << Bound << "; "
	<< Index << "++) {\n";
      BraceCount += 2;
      IndexCount++;
      IndexCountIncrease++;
      Name = MString(Name) + MString("[") + MString(Index) + MString("]");
      break;
    default:
      break;
    }

    SETypeName = NextSETypeName;
    TypeName = NextTypeName;

    NextTypeName = SETypeName->UserBaseTypeName;
    NextSETypeName = (cSETypeName *) GetSymbol (NextTypeName, SizeSpec,::SETypeName, 0);

  }

  if (SETypeName->Type->IsA () ==::ArraySType) {

    //this array dimension will not be allocated so it needs to be zeroed
    c << Name << " = (" << TypeName << ")0;\n";

  } else if (SETypeName->Type->IsA () ==::StructSType) {

    c << Name << " = new (&_c2_MyAddr->GetMachine()) _" << TypeName << "_struct;\n"
      << "// _c2_AddrHashTabAdd((char *)" << Name << ", sizeof(*(" << Name << ")));\n";

    while (SETypeName->UserBaseTypeName != MString(""))
      //skips the named types if any
      SETypeName = (cSETypeName *)
	GetSymbol (SETypeName->UserBaseTypeName, SizeSpec,::SETypeName, 0);

    cStructSType *StructSType = (cStructSType *) SETypeName->Type;
    for (p = i.Init (StructSType->Members); i.MoreLeft (); p = i.Next ()) {
      cStructMemSType *StructMemSType = (cStructMemSType *) p;

      switch (StructMemSType->Type->IsA ()) {
      case ::CharSType:
      case ::IntSType:
      case ::RealSType:
	break;
      case ::ArraySType:
      case ::StructSType:
	{
	  cMemSize *MemSize;
	  MString MemberTypeName, ExpandedName;
	  int Found = 0;
	  for (q = j.Init (SizeSpec->MemberSizes); j.MoreLeft (); q = j.Next ()) {
	    MemSize = (cMemSize *) q;
	    if ((char *) MemSize->MemberName == StructMemSType->Ident) {
	      Found = 1;
	      break;
	    }
	  }
	  if (Found == 1) {
	    ExpandedName = MString(Name) + MString("->") + MString(StructMemSType->Ident);
	    MemberTypeName = StructMemSType->Type->TypeName;
	    c << "{\n";
	    TranslateSizeSpec (c, MemSize->Sizes, MemberTypeName, ExpandedName);
	    c << "}\n";
	  } else {
	    ExpandedName = MString(Name) + MString("->") + MString(StructMemSType->Ident);
	    MemberTypeName = StructMemSType->Type->TypeName;
	    c << ExpandedName << " = (" << MemberTypeName << ")0;\n";
	  }
	}
	break;
      default:
	break;
      }
    }
  }
  for (; BraceCount > 0; BraceCount--) {
    c << "}\n";
  }
  IndexCount -= IndexCountIncrease;
}

void UnifiedSizeSpec (cSizeSpec * SizeSpec, MString TypeName) {
  if (SizeSpec != 0) {
    cCursor c;
    cGraph *g;
    cAbsTree *n;
    MString c2TypeName;
    MString ReturnVariableName;

    n = GetContaining (SizeSpec, UC, NSRel, 0);
    g = (cGraph *) GetContaining (n, Graph, 0);

    c2TypeName = UnifiedGetName (TypeName, SizeSpec);
    ReturnVariableName = MString ("_o_") + MString(c2TypeName);

    c.cd (eMain_h);
    c.cd (eNewFuncSig);
    c << c2TypeName << " _c2_new" << -(SizeSpec->UID) << "(_c2_gv" << g->UID << " *_c2_g, _c2_nv" << n->UID << " *_c2_l, CompNode * _c2_MyAddr);\n";

    c.cd ();
    c.cd (eMain_c);
    c.cd (eNewFunc);
    c << c2TypeName << " _c2_new" << -(SizeSpec->UID)
      << "(_c2_gv" << g->UID << " *_c2_g, _c2_nv" << n->UID << " *_c2_l, CompNode * _c2_MyAddr)\n"
      << "{\n"
      << c2TypeName << " " << ReturnVariableName << ";\n\n";

    IndexCount = 0;
    TranslateSizeSpec (c, SizeSpec, TypeName, ReturnVariableName);

    c << "\nreturn " << ReturnVariableName << ";\n";
    c << "}\n";
  }
}




void UnifiedCreateReplCond (cExpr * rc, cReplThing * Where) {
  cCursor c;
  cGraph *g;
  cRule *r;
  cUC *u;
  cAbsTree *tmp;
  int FiringRule;

  //Yes, this is horrid.

  if (rc == 0)
    return;

  tmp = GetContaining (Where, Rule, Guard, ArcInput, Binding, ArcOutput, 0);
  r = (cRule *) GetContaining (tmp, Rule, 0);
  FiringRule = r->FiringRule;
  u = (cUC *) GetContaining (r, UC, 0);
  g = (cGraph *) GetContaining (u, Graph, 0);

  c.cd (eGraph_c, g->UID);
  c.cd (eUCScope, u->UID);
  c.cd (eUCProc);

  if (FiringRule)
    switch (tmp->IsA ()) {
    case Rule:
      c.cd (eUCProcFrule, r->UID);
      c.cd (eUCProcFRHead);
      c << "if (" << rc->Translate () << ") {\n";
      c.up ();
      c.cd (eUCProcFRTail);
      c << "}\n";
      break;
    case Guard:
      c.cd (eUCProcFrule, r->UID);
      c.cd (eUCProcFRCheck, tmp->UID);
      c.cd (eUCProcFRCheckHead);
      c << "if (" << rc->Translate () << ") {\n";
      c.up ();
      c.cd (eUCProcFRCheckTail);
      c << "}\n";
      break;
    case Binding:
      c.cd (eUCProcFrule, r->UID);
      c.cd (eUCProcFRBind, tmp->UID);
      c.cd (eUCProcFRBindHead);
      c << "if (" << rc->Translate () << ") {\n";
      c.up ();
      c.cd (eUCProcFRBindTail);
      c << "}\n";
      break;
    case ArcInput:
      c.cd (eUCProcFrule, r->UID);
      c.cd (eUCProcFRCheck, tmp->UID);
      c.cd (eUCProcFRCheckHead);
      c << "if (" << rc->Translate () << ") {\n";
      c.up ();
      c.cd (eUCProcFRCheckTail);
      c << "}\n";
      //***This also goes into a binding part !
      c.up ();
      c.up ();
      c.cd (eUCProcFRBind, tmp->UID);
      c.cd (eUCProcFRBindHead);
      c << "if (" << rc->Translate () << ") {\n";
      c.up ();
      c.cd (eUCProcFRBindTail);
      c << "}\n";
      break;
    default:
      assert (0); // We should never get here.
      break;
    }
  else
    switch (tmp->IsA ()) {
    case Rule:
      c.cd (eUCProcRrule, r->UID);
      c.cd (eUCProcRRHead);
      c << "if (" << rc->Translate () << ") {\n";
      c.up ();
      c.cd (eUCProcRRTail);
      c << "}\n";
      break;
    case Guard:
      c.cd (eUCProcRrule, r->UID);
      c.cd (eUCProcRRCheck, tmp->UID);
      c.cd (eUCProcRRCheckHead);
      c << "if (" << rc->Translate () << ") {\n";
      c.up ();
      c.cd (eUCProcRRCheckTail);
      c << "}\n";
      break;
    case ArcOutput:
    case Binding:
      c.cd (eUCProcRrule, r->UID);
      c.cd (eUCProcRRBind, tmp->UID);
      c.cd (eUCProcRRBindHead);
      c << "if (" << rc->Translate () << ") {\n";
      c.up ();
      c.cd (eUCProcRRBindTail);
      c << "}\n";
      break;
    default:
      assert (0); // We should never get here.
      break;
    }
}


int UnifiedFillIndVar (cCursor &c, MString Name, cExpr *e, int ic)
{
  int i;

  switch (e->IsA()) {
  case Ident:
    return ic;
  case ArrayIndOp:
    i = FillIndVar(c, Name, ((cArrayIndOp *) e)->Left, ic);
    c << Name << "[" << i << "] = "
      <<  ((cArrayIndOp *) e)->Right->Translate()
      << ";\n";
    return i+1;
  default:
    Die("Internal error: Wrong type in FillIndVar\n");
  }

  return 0;  // Quiet the compiler
}
