/*
   2-D Barnes-Hut particle simulator  (sequential)

   The main data structure is a quad tree of treenodes.  Leaf nodes represent
   lone particles and have no children.  The 4 pointers to children are
   arranged as follows.

          ---------
          | 0 | 1 |
          ---------
          | 3 | 2 |
          ---------

   There is also an array of particles.

*/

#include <stdio.h>
#include <math.h>

#include <stdio.h>
#include <limits.h>
#include <sys/times.h>
#include <time.h>

static float c2_SysSec;
static float c2_UserSec;
static clock_t start_time;
static clock_t end_time;


void StartClock() {
  struct tms before;

  start_time  = times(&before);

  c2_SysSec  = (float)before.tms_stime/(float)CLK_TCK;
  c2_UserSec = (float)before.tms_utime/(float)CLK_TCK;

}

void StopClock() {
  struct tms after;

  end_time    = times(&after);

  c2_SysSec  = (float)after.tms_stime/(float)CLK_TCK - c2_SysSec;
  c2_UserSec = (float)after.tms_utime/(float)CLK_TCK - c2_UserSec;
}


/* Signal a halt to the computation */

void StopAll()
{


  printf("Time between completion of node and firing of node:\n");
  printf("  Wall clock time = %f seconds.\n",(float)(end_time - start_time)/(float)CLK_TCK);
  printf("  System time: %f seconds.\n", c2_SysSec);
  printf("  User time  : %f seconds.\n", c2_UserSec);

  fflush(stdout);
  fflush(stderr);
}


#define pi 3.1415926536
#define nearness 1.0
#define G 0.0000000667
#define time_int 20.0

#define MAXPART 1000

#define INFINITY 10000000000.0
#define MAXMASS 500.0
#define MAXINITPOS 100.0
#define MAXINITVEL 2.0

typedef struct {
   double x;
   double y;
} point;


typedef struct {
   double mag;
   double angle;
} vector;


typedef struct {
   point LL;      /* lower left */
   point UR;      /* upper right */
} box;


typedef struct treenode_s {
   double mass;
   point pos;
   vector vel;
   box bound_box;
   struct treenode_s *child[4];
} treenode;


typedef struct {
   double mass;
   point pos;
   vector vel;
} partnode;


int particle_no;        /* Number of particles */
treenode *root;         /* Root of quad tree */
box space_box;          /* Bounding box of space */

int NodeCount;

partnode particles[MAXPART];


void compute_acc(part, tree, acc)
     partnode *part;
     treenode *tree;
     vector *acc;
{
   double dx, dy;

   dx = tree->pos.x - part->pos.x; 
   dy = tree->pos.y - part->pos.y;
   acc->angle = atan2(dy, dx);
   dy = dy*dy;
   dx = dx*dx;
   dx = dx+dy;
   acc->mag = G * tree->mass / dx;
}



int far(p1, p2, ll, ur)
     point *p1;
     point *p2;
     point *ll;
     point *ur;
{
   double cell_len, dist, dx, dy;

   dx = ur->x - ll->x; dx = dx*dx;
   dy = ur->y - ll->y; dy = dy*dy;

   cell_len = dx + dy;
   dx = p1->x - p2->x; dx = dx*dx;
   dy = p1->y - p2->y; dy = dy*dy;
   dist = dx + dy;

   return(cell_len/dist < nearness);
}


   /* Vector add:  a1 = a1 + a2 */

void vector_sum(a1, a2)
     vector *a1;
     vector *a2;
{
   double x_comp, y_comp;
vector tmp1, tmp2;

   tmp1 = *a1;
   tmp2 = *a2;

   x_comp = a1->mag * cos(a1->angle)  +  a2->mag * cos(a2->angle);
   y_comp = a1->mag * sin(a1->angle)  +  a2->mag * sin(a2->angle);
   a1->mag = sqrt(x_comp*x_comp + y_comp*y_comp);
   if (y_comp == 0.0 && x_comp == 0.0)
     a1->angle = 0.0;
   else
     a1->angle = atan2(y_comp,x_comp);
}


int leaf(tree)
     treenode *tree;
{
   int i;

   for (i = 0; i < 4; i++)
     if (tree->child[i] != NULL) return 0;
   return 1;
}


void acc(part, tree, a)
     partnode *part;
     treenode *tree;
     vector *a;
{
   int i;
   vector tmp;

   if (tree != NULL && (part->pos.x != tree->pos.x ||
       part->pos.y != tree->pos.y)) {
      if (leaf(tree))
	compute_acc(part, tree, a);
      else if (far(&(part->pos), &(tree->pos), &(tree->bound_box.LL),
		   &(tree->bound_box.UR)))
	compute_acc(part, tree, a);
      else {
	 tmp.mag = 0.0; tmp.angle = 0.0;
	 for (i = 0; i < 4; i++) {
	    if (tree->child[i] != NULL) {
	       acc(part, tree->child[i], &tmp);
	       vector_sum(a, &tmp);
	    }
	 }
      }

   } 
}



void compute_pos(part)
     partnode *part;
{
   vector delta;

   delta.mag = time_int * part->vel.mag;
   delta.angle = part->vel.angle;

   part->pos.x += delta.mag*cos(delta.angle);
   part->pos.y += delta.mag*sin(delta.angle);
}


void move()
{
   vector a;
   int count, i;

   space_box.LL.x = INFINITY;
   space_box.LL.y = INFINITY;
   space_box.UR.x = -INFINITY;
   space_box.UR.y = -INFINITY;;

   for (i = 0; i < particle_no; i++)
        if (particles[i].mass >= 0.0) {
	   a.mag = 0.0; a.angle = 0.0;
	   acc(&particles[i], root, &a);
	   compute_pos(&particles[i]);

	   if (particles[i].pos.x > space_box.UR.x)
	     space_box.UR.x = particles[i].pos.x;
	   if (particles[i].pos.x < space_box.LL.x)
	     space_box.LL.x = particles[i].pos.x;

	   if (particles[i].pos.y > space_box.UR.y) 
	     space_box.UR.y = particles[i].pos.y;
	   if (particles[i].pos.y < space_box.LL.y)
	     space_box.LL.y = particles[i].pos.y;

	   a.mag = time_int * a.mag;
	   vector_sum(&particles[i].vel, &a);
	}
}


void get_quad(tree, pos, quad_index, bb)
     treenode *tree;
     point *pos;
     int *quad_index;
     box *bb;
{
   point center;

   center.x = (tree->bound_box.LL.x + tree->bound_box.UR.x)/2.0;
   center.y = (tree->bound_box.LL.y + tree->bound_box.UR.y)/2.0;

   if (pos->x < center.x) {
      if (pos->y < center.y) {
         *quad_index = 3;
         bb->LL = tree->bound_box.LL;
         bb->UR = center;
      } else {
	*quad_index =  0;
	bb->LL.x = tree->bound_box.LL.x;
	bb->LL.y = center.y;
	bb->UR.x = center.x;
	bb->UR.y = tree->bound_box.UR.y;
     }
   } else if (pos->y < center.y) {
      *quad_index = 2;
	bb->LL.x = center.x;
	bb->LL.y = tree->bound_box.LL.y;
	bb->UR.x = tree->bound_box.UR.x;
	bb->UR.y = center.y;
   } else {
      *quad_index = 1;
      bb->LL = center;
      bb->UR = tree->bound_box.UR;
   }
   
}



void update(tree, part)
     treenode *tree;
     partnode *part;
{

   tree->pos.x = tree->pos.x*tree->mass + part->pos.x*part->mass;
   tree->pos.y = tree->pos.y*tree->mass + part->pos.y*part->mass;
   tree->mass = tree->mass + part->mass;

   tree->pos.x = tree->pos.x/tree->mass;
   tree->pos.y = tree->pos.y/tree->mass;
}


treenode *create_node(part, new_bb)
     partnode *part;
     box *new_bb;
{
   treenode *newnode;
   int i;

   newnode = (treenode *) malloc(sizeof(treenode));
   newnode->mass = part->mass;
   newnode->pos = part->pos;
   newnode->bound_box = *new_bb;
   for (i = 0; i < 4; i++)
     newnode->child[i] = NULL;

   return newnode;
}


void enter_particle(tree, newpart)
     treenode *tree;
     partnode *newpart;
{
   int quad_index1, quad_index2;
   box bb2, bb1;
   partnode part2;

   if (newpart->mass == 0.0)   /* Don't enter */
     return;

   if (tree == NULL) {

      fprintf(stderr, "error: enter_part called with null tree\n");

   } else if (leaf(tree)) {

      if (newpart->pos.x == tree->pos.x && newpart->pos.y == tree->pos.y) {
         /* This is wrong.  Really need to model a collision! */
	 newpart->mass = 0.0;
      } else {
	 get_quad(tree, &(tree->pos), &quad_index2, &bb2);
	 get_quad(tree, &(newpart->pos), &quad_index1, &bb1);
	 part2.mass = tree->mass;
         part2.pos = tree->pos;
         part2.vel = tree->vel;
	 update(tree, newpart);

	 if (tree->child[quad_index2] == NULL)
	   tree->child[quad_index2] = create_node(&part2, &bb2);
	 else
	   enter_particle(tree->child[quad_index2], &part2);

	 if (tree->child[quad_index1] == NULL)
	   tree->child[quad_index1] = create_node(newpart, &bb1);
	 else
	   enter_particle(tree->child[quad_index1], newpart);
      }

   } else {
      get_quad(tree, &(newpart->pos), &quad_index1, &bb1);
      update(tree, newpart);
      if (tree->child[quad_index1] == NULL)
	tree->child[quad_index1] = create_node(newpart, &bb1);
      else
	enter_particle(tree->child[quad_index1], newpart);
   }
}



void  print_tree(tree, level)
     treenode *tree;
     int level;
{
   if (tree != NULL) {
      NodeCount++;
/*      printf("quad_tree <%d> has node [%f, (%f, %f)]\n", level, tree->mass, 
	     tree->pos.x, tree->pos.y);
      printf("bounding box is (%f, %f, %f, %f)\n\n", tree->bound_box.LL.x,
	     tree->bound_box.LL.y, tree->bound_box.UR.x, 
	     tree->bound_box.UR.y);
*/
      print_tree(tree->child[0], level+1);
      print_tree(tree->child[1], level+1);
      print_tree(tree->child[2], level+1);
      print_tree(tree->child[3], level+1);
   }

}



  /* Generate a particle's initial state */

void InitializeParticle(part)
   partnode *part;
{
   double base = 0x7fffffff;

   part->mass = rand() / base * MAXMASS;

   part->pos.x = rand() / base * MAXINITPOS;
   if (part->pos.x > space_box.UR.x) space_box.UR.x = part->pos.x;
   if (part->pos.x < space_box.LL.x) space_box.LL.x = part->pos.x;

   part->pos.y = rand() / base * MAXINITPOS;
   if (part->pos.y > space_box.UR.y) space_box.UR.y = part->pos.y;
   if (part->pos.y < space_box.LL.y) space_box.LL.y = part->pos.y;

/*   part->vel.mag = rand() / base * MAXINITVEL; */
   part->vel.mag = 0.0;
   part->vel.angle = rand() / base * 2.0 * pi;
}



void print_particle(Part, Ind)
     partnode *Part;
     int Ind;
{
      printf("Particle [%d]\n", Ind);
      printf("  mass = %f\n", Part->mass);
      printf("  vel = (%f, %f)\n", Part->vel.mag,
	                           Part->vel.angle);
      printf("  pos = (%f, %f)\n", Part->pos.x,
	                           Part->pos.y);
}

void print_particles()
{
   int i;

   for (i = 0; i < particle_no; i++) {
      print_particle(&particles[i], i);
   }
}


void freetree(tree)
     treenode *tree;
{
   int i;

   for (i = 0; i< 4; i++)
     if (tree->child[i] != NULL) freetree(tree->child[i]);

   free(tree);
}


main (argc, argv)
     int argc;
     char **argv;
{
   int i, j, new_no, numit;

   srand(0);

   root = NULL;
   
   if (argc != 3) {
      fprintf(stderr, "usage: serial numpart numiterations\n");
      exit(1);
   }
   new_no = atoi(argv[1]);
   numit = atoi(argv[2]);

   printf("\nRunning with %d particles for %d iterations\n", new_no, numit);

   StartClock();

   space_box.LL.x = INFINITY;
   space_box.LL.y = INFINITY;
   space_box.UR.x = -INFINITY;
   space_box.UR.y = -INFINITY;

   particle_no = 0;

   root = NULL;

   for (j = 0; j < numit; j++) {

/*       printf("\n ******************** loop %d\n", j); */
      
      for (i = particle_no; i < particle_no+new_no; i++)
	InitializeParticle(&particles[i]);
      
      particle_no += new_no;
      
/*      print_particles();  */
/*      {
	 int i;
	 for (i = 0; i < 1; i++)
	   printf("%f %f\n", particles[i].pos.x, particles[i].pos.y);
      } */

      /* enter first particle */
      root = create_node(&particles[0], &space_box);
      
      for (i = 1; i < particle_no; i++)
	enter_particle(root, &particles[i]);

/*      NodeCount = 0;
      print_tree(root, 0); 
      printf("\nPrinted %d nodes\n", NodeCount); */

      move();

      new_no = 0;

      freetree(root);
      root = NULL;
   }

   StopClock();
   StopAll();

   print_particle(&particles[0], 0);
   
}
