
/*
 * "realfft.c", Pjotr '87
 */

/*
 * Bevat funkties realfft en realrft die resp. forward en reverse fast fourier
 * transform op reele samples doen.  Gebruikt pakket fft(3).
 */

#include	<math.h>
#include <stdlib.h>
#include	"header.h"

#define 	pi	3.1415926535897932384626434

void Fourier (COMPLEX *in,
	      unsigned n,
	      COMPLEX *out,
	      COMPLEX *W_factors,
	      unsigned Nfactors);

/*
 * Give smallest possible radix for n samples.
 * Determines (in a rude way) the smallest primefactor of n.
 */
static unsigned radix (unsigned n)
{
	unsigned r;

	if (n < 2)
		return 1;

	for (r = 2; r < n; r++)
		if (n % r == 0)
			break;
	return r;
}


/*
 * Sum the n / m parts of each m samples of in to n samples in out.
 * 		   r - 1
 * Out [j] becomes  sum  in [j % m] * W (j * k).  Here in is the k-th
 * 		   k = 0   k	       n		 k
 * part of in (indices k * m ... (k + 1) * m - 1), and r is the radix.
 * For k = 0, a complex multiplication with W (0) is avoided.
 */
static void join (COMPLEX *in,
		  register unsigned m,
		  register unsigned n,
		  COMPLEX *out,
		  COMPLEX *W_factors,
		  unsigned Nfactors)
{
	register unsigned i, j, jk, s;

	for (s = 0; s < m; s++)
		for (j = s; j < n; j += m) {
			out [j] = in [s];
			for (i = s + m, jk = j; i < n; i += m, jk += j)
				c_add_mul (out [j], in [i], W (n, jk));
		}
}


/*
 * W_init puts Wn ^ k (= e ^ (2pi * i * k / n)) in W_factors [k], 0 <= k < n.
 * If n is equal to Nfactors then nothing is done, so the same W_factors
 * array can used for several transforms of the same number of samples.
 * Notice the explicit calculation of sines and cosines, an iterative approach
 * introduces substantial errors.
 */
int W_init (unsigned n,
	    COMPLEX **W_factors,
	    unsigned *Nfactors)
{
	unsigned k;

	if (n == *Nfactors)
		return 0;
	if (*Nfactors != 0 && *W_factors != 0)
		free ((char *) *W_factors);
	if ((*Nfactors = n) == 0)
		return 0;
	if ((*W_factors = (COMPLEX *) malloc (n * sizeof (COMPLEX))) == 0)
		return -1;
	for (k = 0; k < n; k++) {
	    c_re ((*W_factors)[k]) = cos (2 * pi * k / n);
	    c_im ((*W_factors)[k]) = sin (2 * pi * k / n);
	}

	return 0;
}


/*
 * Split array in of r * m samples in r parts of each m samples,
 * such that in [i] goes to out [(i % r) * m + (i / r)].
 * Then call for each part of out Fourier, so the r recursively
 * transformed parts will go back to in.
 */
static void split (COMPLEX *in,
		   register unsigned r,
		   register unsigned m,
		   COMPLEX *out,
		   COMPLEX *W_factors,
		   unsigned Nfactors)
{
	register unsigned k, s, i, j;

	for (k = 0, j = 0; k < r; k++)
		for (s = 0, i = k; s < m; s++, i += r, j++)
			out [j] = in [i];

	for (k = 0; k < r; k++, out += m, in += m)
		Fourier (out, m, in, W_factors, Nfactors);
}


/*
 * "fourier.c", Pjotr '87.
 */

/*
 * Recursive (reverse) complex fast Fourier transform on the n
 * complex samples of array in, with the Cooley-Tukey method.
 * The result is placed in out.  The number of samples, n, is arbitrary.
 * The algorithm costs O (n * (r1 + .. + rk)), where k is the number
 * of factors in the prime-decomposition of n (also the maximum
 * depth of the recursion), and ri is the i-th primefactor.
 */
void Fourier (COMPLEX *in,
	      unsigned n,
	      COMPLEX *out,
	      COMPLEX *W_factors,
	      unsigned Nfactors)
{
	unsigned r;

	if ((r = radix (n)) < n)
		split (in, r, n / r, out, W_factors, Nfactors);
	join (in, n / r, n, out, W_factors, Nfactors);
}


/*
 * Forward Fast Fourier Transform on the n samples of complex array in.
 * The result is placed in out.  The number of samples, n, is arbitrary.
 * The W-factors are calculated in advance.
 */
int fft (COMPLEX *in,
	 unsigned n,
	 COMPLEX *out)
{
	unsigned i;
	COMPLEX *W_factors = 0;		/* array of W-factors */
	unsigned Nfactors = 0;		/* number of entries in W-factors */

	for (i = 0; i < n; i++)
		c_conj (in [i]);

	if (W_init (n, &W_factors, &Nfactors) == -1)
		return -1;

	Fourier (in, n, out, W_factors, Nfactors);

	for (i = 0; i < n; i++) {
		c_conj (out [i]);
		c_realdiv (out [i], n);
	}

	free (W_factors);

	return 0;
}


/*
 * Reele forward fast fourier transform van n samples van in naar
 * amplitudes van out.
 * De cosinus komponent van de dc komt in out [0], dan volgen in
 * out [2 * i - 1] en out [2 * i] steeds resp. de cosinus en sinus
 * komponenten van de i-de harmonische.  Bij een even aantal samples
 * bevat out [n - 1] de cosinus komponent van de Nyquist frequentie. 
 * Extraatje: Na afloop is in onaangetast.
 */
void realfft (double *in,
	      unsigned n,
	      double *out)
{
	COMPLEX *c_in, *c_out;
	unsigned i;

	if (n == 0 ||
	    (c_in = (COMPLEX *) malloc (n * sizeof (COMPLEX))) == 0 ||
	    (c_out = (COMPLEX *) malloc (n * sizeof (COMPLEX))) == 0)
		return;
	
	for (i = 0; i < n; i++) {
		c_re (c_in [i]) = in [i];
		c_im (c_in [i]) = 0;
	}

	fft (c_in, n, c_out);

	out [0] = c_re (c_out [0]);		/* cos van dc */
	for (i = 1; i < (n + 1) / 2; i++) {	/* cos/sin i-de harmonische */
		out [2 * i - 1] = c_re (c_out [i]) * 2;
		out [2 * i] = c_im (c_out [i]) * -2;
	}
	if (n % 2 == 0)				/* cos van Nyquist */
		out [n - 1] = c_re (c_out [n / 2]);

	free ((char *) c_in);
	free ((char *) c_out);
}

/*
 * Reverse Fast Fourier Transform on the n complex samples of array in.
 * The result is placed in out.  The number of samples, n, is arbitrary.
 * The W-factors are calculated in advance.
 */
int rft (COMPLEX *in,
	 unsigned n,
	 COMPLEX *out)
{

	COMPLEX *W_factors = 0;		/* array of W-factors */
	unsigned Nfactors = 0;		/* number of entries in W-factors */

	if (W_init (n, &W_factors, &Nfactors) == -1)
		return -1;

	Fourier (in, n, out, W_factors, Nfactors);

	return 0;
}


/*
 * Reele reverse fast fourier transform van amplitudes van in naar
 * n samples van out.
 * De cosinus komponent van de dc staat in in [0], dan volgen in
 * in [2 * i - 1] en in [2 * i] steeds resp. de cosinus en sinus
 * komponenten van de i-de harmonische.  Bij een even aantal samples
 * bevat in [n - 1] de cosinus komponent van de Nyquist frequentie. 
 * Extraatje: Na afloop is in onaangetast.
 */
void realrft (double *in,
	      unsigned n,
	      double *out)
{
	COMPLEX *c_in, *c_out;
	unsigned i;

	if (n == 0 ||
	    (c_in = (COMPLEX *) malloc (n * sizeof (COMPLEX))) == 0 ||
	    (c_out = (COMPLEX *) malloc (n * sizeof (COMPLEX))) == 0)
		return;
	
	c_re (c_in [0]) = in [0];		/* dc */
	c_im (c_in [0]) = 0;
	for (i = 1; i < (n + 1) / 2; i++) {	/* geconj. symm. harmonischen */
		c_re (c_in [i]) = in [2 * i - 1] / 2;
		c_im (c_in [i]) = in [2 * i] / -2;
		c_re (c_in [n - i]) = in [2 * i - 1] / 2;
		c_im (c_in [n - i]) = in [2 * i] / 2;
	}
	if (n % 2 == 0) {			/* Nyquist */
		c_re (c_in [n / 2]) = in [n - 1];
		c_im (c_in [n / 2]) = 0;
	}

	rft (c_in, n, c_out);

	for (i = 0; i < n; i++)
		out [i] = c_re (c_out [i]);

	free ((char *) c_in);
	free ((char *) c_out);
}


/*
 * "ft.c", Pjotr '87.
 */








/*
 * "w.c", Pjotr '87.
 */





/*
 * Test for realfft(3).
 */

/*
#define		N	8

void main ()
{
	unsigned i, j;

	double in [N], out [N];

	printf ("Example #1:\n");
	for (i = 0; i < N; i++)
		in [i] = i;
	printsamp (in, N);

	realfft (in, N, out);
	printf ("After a fast fft\n");
	printampl (out, N);

	printf ("A reverse slow ft gives:\n");
	srft (out, N, in);
	printsamp (in, N);

	printf ("And the reverse fast ft yields:\n");
	realrft (out, N, in);
	printsamp (in, N);

	printf ("\n\nExample #2\n");
	for (i = 0; i < N; i++) {
		in [i] = 0;
		for (j = 0; j <= N / 2; j++)
			in [i] += cos (2 * pi * i * j / N) +
				  sin (2 * pi * i * j / N);
	}
	printsamp (in, N);

	realfft (in, N, out);
	printf ("After a forward fast ft:\n");
	printampl (out, N);

	printf ("A reverse slow ft yields:\n");
	srft (out, N, in);
	printsamp (in, N);

	printf ("And a reverse fast ft gives:\n");
	realrft (out, N, in);
	printsamp (in, N);
}
*/

void printampl (double *ampl,
		unsigned n)
{
	unsigned i;

	printf ("Amplitudes\n");

	if (n == 0)
		return;

	printf ("%f (dc)\n", ampl [0]);
	for (i = 1; i < (n + 1) / 2; i++)
		printf ("%f, %f (%u-th harmonic)\n", ampl [2 * i - 1],
						     ampl [2 * i], i);
	if (n % 2 == 0)
		printf ("%f (Nyquist)\n", ampl [n - 1]);

	printf ("\n");
}

void printsamp (double *samp,
		unsigned n)
{
	unsigned i;
	printf ("Samples\n");

	for (i = 0; i < n; i++)
		printf ("%f\n", samp [i]);
	
	printf ("\n");
}

/*
 * Slow reverse fourier transform.  In [0] contains dc, in [n - 1] Nyquist.
 * This is just a gimmick to compare with realfft(3).
 */
void srft (double *in,
	   unsigned n,
	   double *out)
{
	unsigned i, j;

	for (i = 0; i < n; i++) {
		out [i] = in [0];			/* dc */
		for (j = 1; j < (n + 1) / 2; j++)	/* j-th harmonic */
			out [i] += in [2 * j - 1] * cos (2 * pi * i * j / n) +
				   in [2 * j] * sin (2 * pi * i * j / n);
		if (n % 2 == 0)				/* Nyquist */
			out [i] += in [n - 1] * cos (2 * pi * i * j / n);
	}
}


void fill_grid(double **in,
	       int R,
	       int C)
{

  int row, col;

  for (row = 0; row < R; row++)
   for (col = 0; col < C; col++)
     in[row][col] = row * R + col;

}
