/* -*- C++ -*- */
/*
  exmodel.h
  ------------------------------------------------------------------------
  Classes that define the CODE abstract syntax tree.
  ------------------------------------------------------------------------
  @(#) $Id: exmodel.h,v 1.44 1997/12/17 21:08:25 emery Exp $
  ------------------------------------------------------------------------
  AUTHOR/CONTACT:
 
  Emery Berger                    | <http://www.cs.utexas.edu/users/emery>
  Parallel Programming Group      |  <http://www.cs.utexas.edu/users/code>
  Department of Computer Sciences |             <http://www.cs.utexas.edu>
  University of Texas at Austin   |                <http://www.utexas.edu>
  ========================================================================
*/

// Every class must have a name (the class name w/o the c is good)
// defined in classid.h

#ifndef _exmodel_x
#define _exmodel_x

// exmodel.h requires
//  #include "misc/general.h"

#include <stream.h>
#include "abstree.h"
#include "symtab/stype.h"
#include "misc/estring.h"
#include "misc/sfile.h"


// used in gram.y to gather things in recursive rules into lists
// and never appears in the AST

class cGather : public cAbsTree {
 public:
   ~cGather() { L.SetEmpty(); }  // Don't delete the links.
   cObList L;  /* used to gather things into a list in recursive rules */

   ClassId IsA() { return Gather; }
   void PrintId() { cout << "Gather\n"; }
   void List() { Die("cGather::List() : This method should never be called."); }
};
 

  // Class for sizes of array (and array in struct) instantiations.

class cSizeSpec : public cAbsTree {
 public:
   cSizeSpec();
   ~cSizeSpec();

   cObList ArrayBounds;  // List of cExpr
   cObList MemberSizes;  // List of cMemSize;

   void Pass1();
   void Optim1();
   void Pass2();
   void TreeWalk();
   void Serial();
   void Unified();

   ClassId IsA() { return SizeSpec; }
   void PrintId() { cout << "SizeSpec\n"; }
   void List();
   virtual EString Translate(); 
};


class cMemSize : public cAbsTree { // Structure Member Size
 public:
   cMemSize(char *InitMemberName, cSizeSpec *InitSizes);
   ~cMemSize();

   MString MemberName;
   cSizeSpec *Sizes;

   void Pass1();
   void Optim1();
   void Pass2();
   void TreeWalk();
   void Serial();
   void Unified();
   ClassId IsA() { return MemSize; }
   void PrintId() { cout << "MemSize\n"; }
   void List();
   virtual EString Translate();
};


// Classes used in expressions.  All inherit from cExpr

  // Abstract class representing all operators includinding "zero-ary"
  // ones like constants

class cExpr : public cAbsTree {
 public:
   virtual int Precedence() = 0;
   virtual EString Translate() { 
     Die("Virtual function cExpr::Translate not defined."); return EString("BOGUS"); };
   int IsLHS;
   cSType *StructuralType;
};


  // Zero-ary operator (idents, constants) abstract class

class cZeroaryOp : public cExpr {
 public:
   int Precedence() {return 9999;}  // Never need parens
};


class cIdent : public cZeroaryOp {
 public:
   cIdent(char *InitPrintName);
   cIdent(MString &InitPrintName);
   ~cIdent();

   MString PrintName;              // The name is the user's program

   void Pass1();
   void Optim1();
   void Pass2();
   void TreeWalk();
   void Serial();
   void Unified();
   ClassId IsA() { return Ident; }
   void PrintId() { cout << "Ident\n"; }
   void List();
   EString Translate();
};


class cIntConst : public cZeroaryOp {
 public:
   cIntConst(int InitValue);
   ~cIntConst();



   int Value;

   void Pass1();
   void Optim1();
   void Pass2();
   void TreeWalk();
   void Serial();
   void Unified();
   ClassId IsA() { return IntConst; }
   void PrintId() { cout << "IntConst\n"; }
   void List();
   EString Translate();
};


class cRealConst : public cZeroaryOp {
 public:
   cRealConst(double InitValue);
   ~cRealConst();

   double Value;

   void Pass1();
   void Optim1();
   void Pass2();
   void TreeWalk();
   void Serial();
   void Unified();
   ClassId IsA() { return RealConst; }
   void PrintId() { cout << "RealConst\n"; }
   void List();
   EString Translate();
};



class cNewExpr : public cZeroaryOp {
 public:
   cNewExpr(char *InitTypeName);
   cNewExpr(char *InitTypeName, cSizeSpec *InitSizes);
   ~cNewExpr();

   MString TypeName;
   cSizeSpec *Sizes;

   void Pass1();
   void Optim1();
   void Pass2();
   void TreeWalk();
   void Serial();
   void Unified();
   ClassId IsA() { return NewExpr; }
   void PrintId() { cout << "NewExpr\n"; }
   void List();
   EString Translate();
};


class cFuncCallOp : public cZeroaryOp {
 public:
   cFuncCallOp(char *Fname);
   cFuncCallOp(char *Fname, cObList &tmp);
   ~cFuncCallOp();

   MString FuncName;
   cObList Args;                        // A list of expressions

   void Pass1();
   void Optim1();
   void Pass2();
   void TreeWalk();
   void Serial();
   void Unified();
   ClassId IsA() { return FuncCallOp; }
   void PrintId() { cout << "FuncCallOp\n"; }
   void List();
   EString Translate();
};


  // Unary operator abstract class

class cUnaryOp : public cExpr {
 public:
   cUnaryOp(cExpr *);
   ~cUnaryOp();

   cExpr *Left;
};


class cSelectOp : public cUnaryOp {
 public:
   cSelectOp(char *InitSelName, cExpr *l = 0);
   ~cSelectOp();

   MString SelName;              // Name of struct member selected

   int Precedence();
   void Pass1();
   void Optim1();
   void Pass2();
   void TreeWalk();
   void Serial();
   void Unified();
   ClassId IsA() { return SelectOp; }
   void PrintId() { cout << "SelectOp\n"; }
   void List();
   EString Translate();
};


class cNotOp : public cUnaryOp {
 public:
   cNotOp(cExpr *l = 0);
   ~cNotOp();

   int Precedence();
   void Pass1();
   void Optim1();
   void Pass2();
   void TreeWalk();
   void Serial();
   void Unified();
   ClassId IsA() { return NotOp; }
   void PrintId() { cout << "NotOp\n"; }
   void List();
   EString Translate();
};


class cUnaryMinusOp : public cUnaryOp {
 public:
   cUnaryMinusOp(cExpr *l = 0);
   ~cUnaryMinusOp();

   int Precedence();
   void Pass1();
   void Optim1();
   void Pass2();
   void TreeWalk();
   void Serial();
   void Unified();
   ClassId IsA() { return UnaryMinusOp; }
   void PrintId() { cout << "UnaryMinusOp\n"; }
   void List();
   EString Translate();
};


  // Binary operator (abstract class)

class cBinOp : public cExpr {
 public:
   cBinOp(cExpr *l, cExpr *r);
   ~cBinOp();

   cExpr *Left;
   cExpr *Right;

};


class cOrOp : public cBinOp {
 public:
   cOrOp(cExpr *l = 0, cExpr *r = 0);
   ~cOrOp();

   int Precedence();
   void Pass1();
   void Optim1();
   void Pass2();
   void TreeWalk();
   void Serial();
   void Unified();
   ClassId IsA() { return OrOp; }
   void PrintId() { cout << "OrOp\n"; }
   void List();
   EString Translate();
};


class cAndOp : public cBinOp {
 public:
   cAndOp(cExpr *l = 0, cExpr *r = 0);
   ~cAndOp();

   int Precedence();
   void Pass1();
   void Optim1();
   void Pass2();
   void TreeWalk();
   void Serial();
   void Unified();
   ClassId IsA() { return AndOp; }
   void PrintId() { cout << "AndOp\n"; }
   void List();
   EString Translate();
};


class cNEOp : public cBinOp {
 public:
   cNEOp(cExpr *l = 0, cExpr *r = 0);
   ~cNEOp();

   int Precedence();
   void Pass1();
   void Optim1();
   void Pass2();
   void TreeWalk();
   void Serial();
   void Unified();
   ClassId IsA() { return NEOp; }
   void PrintId() { cout << "NEOp\n"; }
   void List();
   EString Translate();
};


class cEqOp : public cBinOp {
 public:
   cEqOp(cExpr *l = 0, cExpr *r = 0);
   ~cEqOp();

   int Precedence();
   void Pass1();
   void Optim1();
   void Pass2();
   void TreeWalk();
   void Serial();
   void Unified();
   ClassId IsA() { return EqOp; }
   void PrintId() { cout << "EqOp\n"; }
   void List();
   EString Translate();
};


class cGEOp : public cBinOp {
 public:
   cGEOp(cExpr *l = 0, cExpr *r = 0);
   ~cGEOp();

   int Precedence();
   void Pass1();
   void Optim1();
   void Pass2();
   void TreeWalk();
   void Serial();
   void Unified();
   ClassId IsA() { return GEOp; }
   void PrintId() { cout << "GEOp\n"; }
   void List();
   EString Translate();
};


class cLEOp : public cBinOp {
 public:
   cLEOp(cExpr *l = 0, cExpr *r = 0);
   ~cLEOp();

   int Precedence();
   void Pass1();
   void Optim1();
   void Pass2();
   void TreeWalk();
   void Serial();
   void Unified();
   ClassId IsA() { return LEOp; }
   void PrintId() { cout << "LEOp\n"; }
   void List();
   EString Translate();
};


class cLTOp : public cBinOp {
 public:
   cLTOp(cExpr *l = 0, cExpr *r = 0);
   ~cLTOp();

   int Precedence();
   void Pass1();
   void Optim1();
   void Pass2();
   void TreeWalk();
   void Serial();
   void Unified();
   ClassId IsA() { return LTOp; }
   void PrintId() { cout << "LTOp\n"; }
   void List();
   EString Translate();
};


class cGTOp : public cBinOp {
 public:
   cGTOp(cExpr *l = 0, cExpr *r = 0);
   ~cGTOp();

   int Precedence();
   void Pass1();
   void Optim1();
   void Pass2();
   void TreeWalk();
   void Serial();
   void Unified();
   ClassId IsA() { return GTOp; }
   void PrintId() { cout << "GTOp\n"; }
   void List();
   EString Translate();
};


class cMinusOp : public cBinOp {
 public:
   cMinusOp(cExpr *l = 0, cExpr *r = 0);
   ~cMinusOp();

   int Precedence();
   void Pass1();
   void Optim1();
   void Pass2();
   void TreeWalk();
   void Serial();
   void Unified();
   ClassId IsA() { return MinusOp; }
   void PrintId() { cout << "MinusOp\n"; }
   void List();
   EString Translate();
};


class cPlusOp : public cBinOp {
 public:
   cPlusOp(cExpr *l = 0, cExpr *r = 0);
   ~cPlusOp();

   int Precedence();
   void Pass1();
   void Optim1();
   void Pass2();
   void TreeWalk();
   void Serial();
   void Unified();
   ClassId IsA() { return PlusOp; }
   void PrintId() { cout << "PlusOp\n"; }
   void List();
   EString Translate();
};


class cModOp : public cBinOp {
 public:
   cModOp(cExpr *l = 0, cExpr *r = 0);
   ~cModOp();

   int Precedence();
   void Pass1();
   void Optim1();
   void Pass2();
   void TreeWalk();
   void Serial();
   void Unified();
   ClassId IsA() { return ModOp; }
   void PrintId() { cout << "ModOp\n"; }
   void List();
   EString Translate();
};


class cDivOp : public cBinOp {
 public:
   cDivOp(cExpr *l = 0, cExpr *r = 0);
   ~cDivOp();

   int Precedence();
   void Pass1();
   void Optim1();
   void Pass2();
   void TreeWalk();
   void Serial();
   void Unified();
   ClassId IsA() { return DivOp; }
   void PrintId() { cout << "DivOp\n"; }
   void List();
   EString Translate();
};


class cMultOp : public cBinOp {
 public:
   cMultOp(cExpr *l = 0, cExpr *r = 0);
   ~cMultOp();

   int Precedence();
   void Pass1();
   void Optim1();
   void Pass2();
   void TreeWalk();
   void Serial();
   void Unified();
   ClassId IsA() { return MultOp; }
   void PrintId() { cout << "MultOp\n"; }
   void List();
   EString Translate();
};


class cArrayIndOp : public cBinOp {
 public:
   cArrayIndOp(cExpr *l = 0, cExpr *r = 0);
   ~cArrayIndOp();

   int Precedence();
   void Pass1();
   void Optim1();
   void Pass2();
   void TreeWalk();
   void Serial();
   void Unified();
   ClassId IsA() { return ArrayIndOp; }
   void PrintId() { cout << "ArrayIndOp\n"; }
   void List();
   EString Translate();
};


  // Function signatures

class cFuncArgSpec : public cAbsTree {
 public:
   cFuncArgSpec(char *InitTypeName, char *InitArgName, int Ref);
   ~cFuncArgSpec();

   MString TypeName;
   MString ArgName;
   int Reference;                // True if passed by reference

   void Pass1();
   void Optim1();
   void Pass2();
   void TreeWalk();
   void Serial();
   void Unified();
   ClassId IsA() { return FuncArgSpec; }
   void PrintId() { cout << "FuncArgSpec\n"; }
   void List();
};


class cFuncSig : public cAbsTree {
 public:
   cFuncSig(char *InitFuncName, char *InitReturnType, cObList &InitArgs);
   cFuncSig(char *InitFuncName, char *InitReturnType);
   cFuncSig(char *InitFuncName, cObList &InitArgs);
   cFuncSig(char *InitFuncName);
   ~cFuncSig();

   MString FuncName;
   MString ReturnType;
   int DontCheckArgs;      // for syntax foo(...)
   cObList Args;           // list of cFuncArgSpec 

   void Pass1();
   void Optim1();
   void MakeSTEntry(cSymTab *SymTab);
   void Pass2();
   void TreeWalk();
   void Serial();
   void Unified();
   ClassId IsA() { return FuncSig; }
   void PrintId() { cout << "FuncSig\n"; }
   void List();
};


  // Arcs

class cArc : public cAbsTree {
 public:

   cArc();
   ~cArc();

   MString Name;
   cSType *BaseType;
   int ConnectsNSRel; // 1 iff arc is an nsrel rather than dataflow arc.
   int ArcType;       // 1 is  X        (interface node to non-interface node)
                      // 2 is  . => .
                      // 3 is  . => ..
                      // 4 is  .. => .
                      // 5 is  .. => ..
                      // 6 is  empty    (interface node to interface node)

   int FromUID;
   MString FromPort;
   cObList CallVarList;              // List of cVarDecl
   cObList NodeVarList;              // List of cVarDecl
   cObList PortVarList;              // List of cVarDecl

   int ToUID;
   MString ToPort;
   cObList CallExprList;             // List of cExpr
   cObList NodeExprList;             // List of cExpr
   cObList PortExprList;             // List of cExpr

   void Pass1();
   void Optim1();
   void Pass2();
   void TreeWalk();
   void Serial();
   void Unified();
   ClassId IsA() { return Arc; }
   void PrintId() { cout << "Arc\n"; }
   void List();
};




class cPartArc : public cAbsTree {
 public:

   cPartArc();
   ~cPartArc();

   MString Array_orig;
   MString Array_part;

   cObList Part_Spec;
   cObList Size_Spec;
   cObList Overlap_Spec;

   void Pass1();
   void Optim1();
   void Pass2();
   void TreeWalk();
   void Serial();
   void Unified();
   ClassId IsA() { return PartArc; }
   void PrintId() { cout << "PartArc\n"; }
   void List();
};


class cMergeArc : public cAbsTree {
 public:

   cMergeArc();
   ~cMergeArc();

   MString Array_orig;
   MString Array_merged;

   cObList Merge_Spec;
   cObList Size_Spec;
   cObList Overlap_Spec;


   void Pass1();
   void Optim1();
   void Pass2();
   void TreeWalk();
   void Serial();
   void Unified();
   ClassId IsA() { return MergeArc; }
   void PrintId() { cout << "MergeArc\n"; }
   void List();
};





class cExpandArc : public cAbsTree {
 public:

   cExpandArc();
   ~cExpandArc();

   MString Array_orig;
   MString Array_expanded;

   int growth_factor;
   int exp_index; 


   void Pass1();
   void Optim1();
   void Pass2();
   void TreeWalk();
   void Serial();
   void Unified();
   ClassId IsA() { return ExpandArc; }
   void PrintId() { cout << "ExpandArc\n"; }
   void List();
};




class cContractArc : public cAbsTree {
 public:

   cContractArc();
   ~cContractArc();

   MString Array_orig;
   MString Array_contracted;

   int contract_factor;
   int con_index; 


   void Pass1();
   void Optim1();
   void Pass2();
   void TreeWalk();
   void Serial();
   void Unified();
   ClassId IsA() { return ContractArc; }
   void PrintId() { cout << "ContractArc\n"; }
   void List();
};





class cStrip_UpdateArc : public cAbsTree {
 public:

   cStrip_UpdateArc();
   ~cStrip_UpdateArc();

   MString Array_orig;
   MString Array_updated;

   cObList Update_Spec;
   cObList Size_Spec;
   cObList Overlap_Spec;


   void Pass1();
   void Optim1();
   void Pass2();
   void TreeWalk();
   void Serial();
   void Unified();
   ClassId IsA() { return Strip_UpdateArc; }
   void PrintId() { cout << "Strip_UpdateArc\n"; }
   void List();
};



class cAlignArc : public cAbsTree {
 public:

   cAlignArc();
   ~cAlignArc();

   MString Primary;
   cObList Secondary;
   cObList Indx_Stride_spec;

   void Pass1();
   void Optim1();
   void Pass2();
   void TreeWalk();
   void Serial();
   void Unified();
   ClassId IsA() { return AlignArc; }
   void PrintId() { cout << "AlignArc\n"; }
   void List();
};






  // Type related classes


class cNamedType : public cAbsTree {
 public:
   cNamedType(char *InitTypeName);
   ~cNamedType();

   MString TypeName;

   void Pass1();
   void Optim1();
   void Pass2();
   void TreeWalk();
   void Serial();
   void Unified();
   ClassId IsA() { return NamedType; }
   void PrintId() { cout << "NamedType\n"; }
   void List();
};


class cArrayType : public cAbsTree {
 public:
   cArrayType(char *InitBaseType);
   ~cArrayType();

   MString BaseType;

   void Pass1();
   void Optim1();
   void Pass2();
   void TreeWalk();
   void Serial();
   void Unified();
   ClassId IsA() { return ArrayType; }
   void PrintId() { cout << "ArrayType\n"; }
   void List();
};


class cStructMem : public cAbsTree {
 public:
   cStructMem(char *InitType, char *InitName);
   ~cStructMem();

   MString Type;
   MString Name;

   void Pass1();
   void Optim1();
   void Pass2();
   void TreeWalk();
   void Serial();
   void Unified();
   ClassId IsA() { return StructMem; }
   void PrintId() { cout << "StructMem\n"; }
   void List();
};


class cStructType : public cAbsTree {
 public:
   cStructType(cObList &L);
   ~cStructType();

   cObList Members;

   void Pass1();
   void Optim1();
   void Pass2();
   void TreeWalk();
   void Serial();
   void Unified();
   ClassId IsA() { return StructType; }
   void PrintId() { cout << "StructType\n"; }
   void List();
};


class cTypeDef : public cAbsTree {
 public:
   cTypeDef(char *InitName, cAbsTree *InitType);
   ~cTypeDef();

   MString Name;
   cAbsTree *Type; // cNamedType, cArrayType, or cStructType

   void Pass1();
   void Optim1();
   void MakeSTEntry(cSymTab *SymTab);
   void Pass2();
   void TreeWalk();
   void Serial();
   void Unified();
   ClassId IsA() { return TypeDef; }
   void PrintId() { cout << "TypeDef\n"; }
   void List();
};


  // Computation-oriented classes

  // cStmt is abstract

class cStmt : public cAbsTree {
 public:
   virtual EString Translate() { 
     Die("Virtual function cStmt::Translate not defined."); return EString("BOGUS"); };
};


class cCallStmt : public cStmt {
 public:
   cCallStmt(char *InitFuncName);
   cCallStmt(char *InitFuncName, cObList &L);
   ~cCallStmt();

   MString FuncName;
   cObList Args;

   void Pass1();
   void Optim1();
   void Pass2();
   void TreeWalk();
   void Serial();
   void Unified();
   EString Translate();
   ClassId IsA() { return CallStmt; }
   void PrintId() { cout << "CallStmt\n"; }
   void List();
};


class cWhileStmt : public cStmt {
 public:
   cWhileStmt(cExpr *InitCond, cStmt *InitBody);
   ~cWhileStmt();

   cExpr *Cond;
   cStmt *Body;

   void Pass1();
   void Optim1();
   void Pass2();
   void TreeWalk();
   void Serial();
   void Unified();
   EString Translate();
   ClassId IsA() { return WhileStmt; }
   void PrintId() { cout << "WhileStmt\n"; }
   void List();
};


class cBlockStmt : public cStmt {
 public:
   cBlockStmt(cObList &L);
   ~cBlockStmt();

   cObList StmtList;

   void Pass1();
   void Optim1();
   void Pass2();
   void TreeWalk();
   void Serial();
   void Unified();
   ClassId IsA() { return BlockStmt; }
   void PrintId() { cout << "BlockStmt\n"; }
   void List();
   EString Translate();
};


class cIfStmt : public cStmt {
 public:
   cIfStmt(cExpr *InitCond, cStmt *InitIfPart);
   cIfStmt(cExpr *InitCond, cStmt *InitIfPart, cStmt *InitElsePart);
   ~cIfStmt();

   cExpr *Cond;
   cStmt *IfPart;
   cStmt *ElsePart;

   void Pass1();
   void Optim1();
   void Pass2();
   void TreeWalk();
   void Serial();
   void Unified();
   ClassId IsA() { return IfStmt; }
   void PrintId() { cout << "IfStmt\n"; }
   void List();
   EString Translate();
};


class cAssignStmt : public cStmt {
 public:
   cAssignStmt(cExpr *InitLHS, cExpr *InitRHS);
   ~cAssignStmt();

   cExpr *LHS;
   cExpr *RHS;

   void Pass1();
   void Optim1();
   void Pass2();
   void TreeWalk();
   void Serial();
   void Unified();
   ClassId IsA() { return AssignStmt; }
   void PrintId() { cout << "AssignStmt\n"; }
   void List();
   EString Translate();
};


class cNullStmt : public cStmt {
 public:
   cNullStmt();
   ~cNullStmt();

   void Pass1();
   void Optim1();
   void Pass2();
   void TreeWalk();
   void Serial();
   void Unified();
   ClassId IsA() { return NullStmt; }
   void PrintId() { cout << "NullStmt\n"; }
   void List();
   EString Translate();
};


  // Classes for variable definitions

class cVarDecl : public cAbsTree {
 public:
   cVarDecl(char *InitVarName);                         // defaults to int
   cVarDecl(char *InitVarName, cSizeSpec *InitSizes);   // defaults to int
   ~cVarDecl();

   MString VarName;
   MString TypeName;
   cSizeSpec *Sizes;

   void Pass1();
   void Optim1();
   void MakeSTEntry(cSymTab *SymTab);
   void Pass2();
   void TreeWalk();
   void Serial();
   void Unified();
   ClassId IsA() { return VarDecl; }
   void PrintId() { cout << "VarDecl\n"; }
   void List();
   EString Translate();
};


  // Name Sharing Relations

class cNSRel : public cAbsTree {
 public:
   cNSRel(cObList &InitSharedVars, cObList &InitLocalVars,
	  cStmt *InitInitComp);
   ~cNSRel();

   MString Name;
   cObList FuncSigs;   // list of cFuncSig
   MString FuncDefs;

   cObList SharedVars; // list of cSharedVar
   cObList LocalVars;  // list of cVarDecl
   cStmt *InitComp;

   void Pass1();
   void Optim1();
   void Pass2();
   void TreeWalk();
   void Serial();
   void Unified();
   ClassId IsA() { return NSRel; }
   void PrintId() { cout << "NSRel\n"; }
   void List();
};


  // UC classes


class cReplSpec : public cAbsTree {
 public:
   cReplSpec(char *InitIndVar, cExpr *InitBound);
   ~cReplSpec();

   MString IndVar;
   cExpr *Bound;

   void Pass1();
   void Optim1();
   void MakeSTEntry(cSymTab *SymTab);
   void Pass2();
   void TreeWalk();
   void Serial();
   void Unified();
   ClassId IsA() { return ReplSpec; }
   void PrintId() { cout << "ReplSpec\n"; }
   void List();
};


class cReplThing : public cAbsTree {    // This class is abstract
 public:
   cReplThing() { ReplCond = 0; }

   cObList Repls;     // List of cReplSpec;
   cExpr *ReplCond;
};


class cGuard : public cReplThing {
 public:
   cGuard(cExpr *InitCond);
   ~cGuard();

   cExpr *Cond;

   void Pass1();
   void Optim1();
   void Pass2();
   void TreeWalk();
   void Serial();
   void Unified();
   virtual ClassId IsA() { return Guard; }
   virtual void PrintId() { cout << "Guard\n"; }
   virtual void List();
};


class cArcInput : public cReplThing {
 public:
   cArcInput(cExpr *InitPortExpr, cExpr *InitVarExpr);
   ~cArcInput();

   cExpr *PortExpr;
   cExpr *VarExpr;

   void Pass1();
   void Optim1();
   void Pass2();
   void TreeWalk();
   void Serial();
   void Unified();
   virtual ClassId IsA() { return ArcInput; }
   virtual void PrintId() { cout << "ArcInput\n"; }
   virtual void List();
};


class cBinding : public cReplThing {
 public:
   cBinding(cAssignStmt *InitBind);
   ~cBinding();

   cAssignStmt *Bind;

   void Pass1();
   void Optim1();
   void Pass2();
   void TreeWalk();
   void Serial();
   void Unified();
   virtual ClassId IsA() { return Binding; }
   virtual void PrintId() { cout << "Binding\n"; }
   virtual void List();
};


class cArcOutput : public cReplThing {
 public:
   cArcOutput(cExpr *InitPortExpr, cExpr *InitValueExpr);
   ~cArcOutput();

   cExpr *PortExpr;
   cExpr *ValueExpr;

   void Pass1();
   void Optim1();
   void Pass2();
   void TreeWalk();
   void Serial();
   void Unified();
   virtual ClassId IsA() { return ArcOutput; }
   virtual void PrintId() { cout << "ArcOutput\n"; }
   virtual void List();
};


class cRule : public cReplThing {
 public:
   cRule(cObList &InitGuards, cObList &InitBinds);
   ~cRule();

   int FiringRule;       // 1 => Firing Rule and 0 => Routing Rule
                         //         Routing Rule             Firing Rule
   cObList Guards;       // List of cGuard                   or cArcInput
   cObList Binds;        // List of (cArcOutput or cBinding) or cBinding

   void Pass1();
   void Optim1();
   void Pass2();
   void TreeWalk();
   void Serial();
   void Unified();
   ClassId IsA() { return Rule; }
   void PrintId() { cout << "Rule\n"; }
   void List();
};


class cPortSpec : public cAbsTree {
 public:
   cPortSpec(char *InitTypeName, char *InitPortName);
   ~cPortSpec();

   MString TypeName;
   MString PortName;

   int IsInputPort;

   void Pass1();
   void Optim1();
   void MakeSTEntry(cSymTab *SymTab);
   void Pass2();
   void TreeWalk();
   void Serial();
   void Unified();
   ClassId IsA() { return PortSpec; }
   void PrintId() { cout << "PortSpec\n"; }
   void List();
};


enum NSClass {NSReader, NSWriter};

class cNSPortSpec : public cAbsTree {
 public:
   cNSPortSpec(NSClass InitUseRule);
   ~cNSPortSpec();

   MString TypeName;
   MString PortName;
   NSClass UseRule;

   void Pass1();
   void Optim1();
   void MakeSTEntry(cSymTab *SymTab);
   void Pass2();
   void TreeWalk();
   void Serial();
   void Unified();
   ClassId IsA() { return NSPortSpec; }
   void PrintId() { cout << "NSPortSpec\n"; }
   void List();
};


class cUC : public cAbsTree {
 public:
   cUC();
   ~cUC();

   MString Name;
   int TermNode;
   int StartNode;
   int Static;          // True == we need to maintain the node's state.
   int Shared;           // True == this node accesses a shared variable.
   cObList FuncSigs;
   MString FuncDefs;

   cObList InputPorts;       /* List of cPortSpec   */
   cObList OutputPorts;      /* List of cPortSpec   */
   cObList NSPorts;          /* list of cNSPortSpec */

   cObList LocalVars;        /* List of cVarDecls   */

   cStmt *InitComp;          /* can be any kind of statement */

   cObList InRules;          /* List of cRule */

   cStmt *Comp;              /* can be any kind of statement */

   cObList OutRules;         /* List of cRule */

   void Pass1();
   void Optim1();
   void Pass2();
   void Pass3();
   void TreeWalk();
   void Serial();
   void Unified();
   ClassId IsA() { return UC; }
   void PrintId() { cout << "UC\n"; }
   void List();
};


class cIntNode : public cAbsTree {
 public:
   cIntNode();
   ~cIntNode();

   MString Name;
   MString Type;
   int InParam;        // TRUE if an in parameter; FALSE otherwise 

   void Pass1();
   void Optim1();
   void MakeSTEntry(cSymTab *SymTab);
   void Pass2();
   void TreeWalk();
   void Serial();
   void Unified();
   ClassId IsA() { return IntNode; }
   void PrintId() { cout << "IntNode\n"; }
   void List();
};


class cCrepNode : public cAbsTree {
 public:
   cCrepNode();
   ~cCrepNode();

   MString Name;
   MString Type;

   void Pass1();
   void Optim1();
   void MakeSTEntry(cSymTab *SymTab);
   void Pass2();
   void TreeWalk();
   void Serial();
   void Unified();
   ClassId IsA() { return CrepNode; }
   void PrintId() { cout << "CrepNode\n"; }
   void List();
};


class cCallNode : public cAbsTree {
 public:
   cCallNode();
   ~cCallNode();

   MString Name;
   MString CalledGraph;

   void Pass1();
   void Optim1();
   void Pass2();
   void TreeWalk();
   void Serial();
   void Unified();
   ClassId IsA() { return CallNode; }
   void PrintId() { cout << "CallNode\n"; }
   void List();
};


class cGraph : public cAbsTree {
 public:
   cGraph();
   ~cGraph();

   MString Name;
   cObList Types;       // list of cTypeDef
   cObList FuncSigs;    // list of cFuncSig
   MString FuncDefs;

   cObList IntNodes;    // list of cIntNode
   cObList CrepNodes;   // list of cCrepNode
   cObList UCNodes;     // list of cUC
   cObList CallNodes;   // list of cCallNode
   cObList NSRelNodes;  // list of cNSRel
   cObList Arcs;        // list of cArc

   void Pass1();
   void Optim1();
   void MakeSTEntry(cSymTab *SymTab);
   void Pass2();
   void Pass3();
   void TreeWalk();
   void Serial();
   void Unified();
   ClassId IsA() { return Graph; }
   void PrintId() { cout << "Graph\n"; }
   void List();
};


class cProgram : public cAbsTree {
 public:
   cProgram();
   ~cProgram();

   MString Name;
   cObList Types;  // list cTypeDef
   cObList FuncSigs; // list of cFuncSig
   MString FuncDefs;
   cObList Graphs;  // list cGraph

   void Pass1();
   void Optim1();
   void Pass2();
   void Pass3();
   void TreeWalk();
   void Serial();
   void Unified();
   ClassId IsA() { return Program; }
   void PrintId() { cout << "Program\n"; }
   void List();
};


  // Starting at "Here" in the AST return the first instance of type
  // given by ClassType up the tree.  The ... args must be of type
  // ClassId and the list must be ended with a 0.

  //   example: (void) GetContaining(this, UC, NSRel, 0);

cAbsTree *GetContaining(cAbsTree *Here, ...);

  // Like GetContaining but returns 0 in case of failure.

cAbsTree *CheckContaining(cAbsTree *Here, ...);


void InitGetName(MString (*TargetGetName)(MString &s,
					cAbsTree *Here));

void InitSizeSpec(void (*TargetSizeSpec)(cSizeSpec *SizeSpec, MString TypeName));

   // Translate an actual parameter list into C

EString TranslateArgList(MString &FuncName, cAbsTree *CallSite, cObList &Args);


   // Translate an actual parameter list for a polymorph function to C

EString TranslatePoly(MString &FuncName, cObList &Args);

#endif
