#include <config.h>
// This file contains methods for expression-related abstract syntax
// tree classes.

#include <stddef.h>
#include <stdio.h>
#include <string.h>
#include <stream.h>
#include <misc/general.h>
#include "classid.h"
#include "object.h"
#include "abstree.h"
#include "exmodel.h"
#include <symtab/symtab.h>

#include <lib/Syntax.h>

static MString (*GetName)(MString &s, cAbsTree *Here);

void InitGetName(MString (*TargetGetName)(MString &s,
				       cAbsTree *Here))
{
   GetName = TargetGetName;
}


static void (*TranslateSizeSpec)(cSizeSpec *SizeSpec, MString TypeName);

void InitSizeSpec(void (*TargetSizeSpec)(cSizeSpec *SizeSpec, MString TypeName))
{
   TranslateSizeSpec = TargetSizeSpec;
}


// Operator precedence information.  Here is what would have to be
// changes to translate to targets that differ from C.

enum {Or_Op, And_Op, NE_Op, Eq_Op, GE_Op, LE_Op, LT_Op, GT_Op,
	Minus_Op, Plus_Op, Mod_Op, Div_Op, Mult_Op, UnaryMinus_Op,
	Not_Op, FuncCall_Op, ArrayInd_Op, Select_Op, Illegal_Op};

static int DefPrecTable[] =
   { 4,  // Or_Op                 // Use no value greater than 999
     5,  // And_Op
     9,  // NE_Op
     9,  // Eq_Op
    10,  // GE_Op
    10,  // LE_Op
    10,  // LT_Op
    10,  // GT_Op
    12,  // Minus_Op
    12,  // Plus_Op
    13,  // Mod_Op
    13,  // Div_Op
    13,  // Mult_Op
    15,  // UnaryMinus_Op
    15,  // Not_Op
    16,  // FuncCall_Op
    16,  // ArrayInd_Op
    16,  // Select_Op
    0};  // Illegal


cIdent::cIdent(char *InitPrintName)
{

   PrintName = InitPrintName;

   if (DebugLevel >= 7) {
      cout << "cIdent instance created\n";
   }
}


cIdent::cIdent(MString &InitPrintName)
{

   PrintName = InitPrintName;

   if (DebugLevel >= 7) {
      cout << "cIdent instance created\n";
   }
}


cIdent::~cIdent()
{
   if (DebugLevel >= 7) {
      cout << "cIdent instance destroyed\n";
   }
}


void cIdent::List()
{
   cout << PrintName;
}


EString cIdent::Translate()
{
   EString E;
   E << GetName(PrintName, this);
   return E;
}


cIntConst::cIntConst(int InitValue)
{

   Value = InitValue;

   if (DebugLevel >= 7) {
      cout << "cIntConst instance created\n";
   }
}


cIntConst::~cIntConst()
{
   if (DebugLevel >= 7) {
      cout << "cIntConst instance destroyed\n";
   }
}


void cIntConst::List()
{
   cout << Value;
}


EString cIntConst::Translate()
{
   EString E;
   static char s[16];
   sprintf(s, "%d", Value);
   E << s;
   return E;
}


cRealConst::cRealConst(double InitValue)
{

   Value = InitValue;

   if (DebugLevel >= 7) {
      cout << "cRealConst instance created\n";
   }
}


cRealConst::~cRealConst()
{
   if (DebugLevel >= 7) {
      cout << "cRealConst instance destroyed\n";
   }
}


void cRealConst::List()
{
   cout << Value;
}


EString cRealConst::Translate()
{
   EString E;
   static char s[32];
   sprintf(s, "%.13e", Value);
   E << s;
   return E;
}


cNewExpr::cNewExpr(char *InitTypeName)
{
   TypeName = InitTypeName;
   Sizes = 0;
   if (DebugLevel >= 7) {
      cout << "cRealConst instance created\n";
   }
}


cNewExpr::cNewExpr(char *InitTypeName, cSizeSpec *InitSizes)
{
   TypeName = InitTypeName;
   Sizes = InitSizes;
   if (DebugLevel >= 7) {
      cout << "cRealConst instance created\n";
   }
}


cNewExpr::~cNewExpr()
{
   if (DebugLevel >= 7) {
      cout << "cRealConst instance created\n";
   }
}


void cNewExpr::List()
{

   cout << "new " << TypeName << " ";
   if (Sizes != 0) Sizes->List();
}


EString cNewExpr::Translate()
{
   EString E;
   
   if (Sizes != 0) {
     switch (TranslatorT) {
     case UNIFIED:
     case MPI:
     case PVM:
       E << "_c2_new" << -(Sizes->UID) << "(_c2_g, _c2_l, _c2_MyAddr)";
       break;
     default:
       E << "_c2_new" << -(Sizes->UID) << "(_c2_g, _c2_l)";
     }
     TranslateSizeSpec(Sizes, TypeName);
   }

   return E;
}


cFuncCallOp::cFuncCallOp(char *Fname)
{

   FuncName = Fname;

   if (DebugLevel >= 7) {
      cout << "cFuncCall instance created\n";
   }

}


cFuncCallOp::cFuncCallOp(char *Fname, cObList &tmp)
{

   FuncName = Fname;
   Args = tmp;

   if (DebugLevel >= 7) {
      cout << "cFuncCall instance created\n";
   }

}


cFuncCallOp::~cFuncCallOp()
{
   if (DebugLevel >= 7) {
      cout << "cFuncCallOp instance deleted\n";
   }

}



void cFuncCallOp::List()
{
   cObIterator i;
   cObject *tmp;
   int FirstInList = 1;

   cout << FuncName << "(";

   for (tmp = i.Init(Args); i.MoreLeft(); tmp = i.Next()) {
      if (FirstInList) FirstInList = 0;
      else cout << ", ";
      tmp->List();
   }

   cout << ")";

}


EString cFuncCallOp::Translate()
{
   EString E;
   cSEFuncName *SEFuncName;

   SEFuncName = (cSEFuncName *)GetSymbol(FuncName, this, ::SEFuncName, 0);
   if (!SEFuncName->IsPolyMorphic) {
      E << GetName(FuncName, this)
	<< "("
        <<    TranslateArgList(FuncName, this, Args) << ")";
   } else {
      if (FuncName == MString("size"))
         E << "asize( (char *)" << TranslateArgList(FuncName, this, Args) << " )";
      else if (FuncName == MString("copy"))
      {
         E <<"_c2_copy_" << StructuralType->TypeName 
           << "( " << TranslateArgList(FuncName, this, Args);

	 switch (TranslatorT) {
	   // E << ", (" << StructuralType->TypeName << ") 0, LOCAL)";

	 case PVM:
	 case MPI:
	 case UNIFIED:
	   E << ", (" << StructuralType->TypeName << ") 0, LOCAL, *(_c2_MyAddr->GetMessenger()), &_c2_MyAddr->GetMachine())";
	   break;

	 default:
           E << " )";
	   break;
	 }
      }
   }

   return E;
}


  // List a unary operator

static void UnaryList(char *Op, int OpPrec, cExpr *Left)
{
   cout << Op;

   if (Left == 0) Die("Left is NULL in UnaryList!\n");

   if (OpPrec > Left->Precedence()) {
      cout << "(";
      Left->List();
      cout << ")";
   } else
     Left->List();
}


static EString UnaryTranslate(char *Op, int OpPrec, cExpr *Left)
{
   EString E;

   E << Op;

   if (Left == 0) Die("Left is NULL in UnaryTranslate!\n");

   if (OpPrec > Left->Precedence()) {
      E << "(" << Left->Translate() << ")";
   } else
      E << Left->Translate();

   return E;
}


cUnaryOp::cUnaryOp(cExpr *l)
{
   Left =  l;
}


cUnaryOp::~cUnaryOp()
{
}


cSelectOp::cSelectOp(char *InitSelName, cExpr *l) : cUnaryOp(l)
{
   SelName = InitSelName;

   if (DebugLevel >= 7) {
      cout << "cSelectOp instance created\n";
   }

}


cSelectOp::~cSelectOp()
{
   // Left deletion inherited
   if (DebugLevel >= 7) {
      cout << "cSelectOp instance deleted\n";
   }

}


int cSelectOp::Precedence()
{
   return DefPrecTable[Select_Op];
}


void cSelectOp::List()
{
   if (Precedence() > Left->Precedence()) {
      cout << "(";
      Left->List();
      cout << ")";
   } else
     Left->List();

   cout << ".";
   cout << SelName;
}


EString cSelectOp::Translate()
{
   EString E;
   if (Precedence() > Left->Precedence()) {
      E << "(" << Left->Translate() << ")";
   } else
     E << Left->Translate();

   E << "->" << SelName;
   return E;
}


cNotOp::cNotOp(cExpr *l) : cUnaryOp(l)
{
   if (DebugLevel >= 7) {
      cout << "cOrOp instance created\n";
   }

}


cNotOp::~cNotOp()
{
   // Left Deletion inherited
   if (DebugLevel >= 7) {
      cout << "cNotOp instance deleted\n";
   }

}


int cNotOp::Precedence()
{
   return DefPrecTable[Not_Op];
}


void cNotOp::List()
{
   UnaryList("!", Precedence(), Left);
}


EString cNotOp::Translate()
{
   return UnaryTranslate("!", Precedence(), Left);
}


cUnaryMinusOp::cUnaryMinusOp(cExpr *l) : cUnaryOp(l)
{
   if (DebugLevel >= 7) {
      cout << "cUnaryMinusOp instance created\n";
   }

}


cUnaryMinusOp::~cUnaryMinusOp()
{
   // Left Deletion inherited
   if (DebugLevel >= 7) {
      cout << "cUnaryMinusOp instance deleted\n";
   }

}


int cUnaryMinusOp::Precedence()
{
   return DefPrecTable[UnaryMinus_Op];
}


void cUnaryMinusOp::List()
{
   UnaryList("-", Precedence(), Left);
}


EString cUnaryMinusOp::Translate()
{
   return UnaryTranslate("-", Precedence(), Left);
}


 // BinaryOp base

cBinOp::cBinOp(cExpr *l, cExpr *r)
{
   Left = l;
   Right = r;
}


cBinOp::~cBinOp()
{
}


  // List a binary operator

static void BinList(char *Op, int OpPrec, cExpr *Left, cExpr *Right)
{

   if (Left == 0) Die("Left is NULL in BinList!\n");
   if (Right == 0) Die("Right is NULL in BinList!\n");

   if (OpPrec > Left->Precedence()) {
      cout << "(";
      Left->List();
      cout << ")";
   } else
     Left->List();

   cout << Op;

   if (OpPrec > Right->Precedence()) {
      cout << "(";
      Right->List();
      cout << ")";
   } else
     Right->List();
}


static EString BinTranslate(char *Op, int OpPrec, cExpr *Left, cExpr *Right)
{
   EString E;

   if (Left == 0) Die("Left is NULL in BinTranslate!\n");
   if (Right == 0) Die("Right is NULL in BinTranslate!\n");

   E << "(" << Left->Translate() << ")"
     << Op
     << "(" << Right->Translate() << ")";

   // Changed by EDB 11/13/96 because
   // of incorrect translation of
   // a+b-(c+d) (translated to a+b-c+d !)

#if 0
   if (OpPrec > Left->Precedence()) {
      E << "(" << Left->Translate() << ")";
   } else
     E << Left->Translate();

   E << Op;

   if (OpPrec > Right->Precedence()) {
      E << "(" << Right->Translate() << ")";
   } else
     E << Right->Translate();
#endif

   return E;
}


cOrOp::cOrOp(cExpr *l, cExpr *r) : cBinOp(l, r)
{
   if (DebugLevel >= 7) {
      cout << "cOrOp instance created\n";
   }

}


cOrOp::~cOrOp()
{
   // Left & Right Deletion inherited
   if (DebugLevel >= 7) {
      cout << "cOrOp instance deleted\n";
   }

}


int cOrOp::Precedence()
{
   return DefPrecTable[Or_Op];
}


void cOrOp::List()
{
   BinList(" || ", Precedence(), Left, Right);
}


EString cOrOp::Translate()
{
   return BinTranslate(" || ", Precedence(), Left, Right);
}


cAndOp::cAndOp(cExpr *l, cExpr *r) : cBinOp(l, r)
{
   if (DebugLevel >= 7) {
      cout << "cAndOp instance created\n";
   }

}


cAndOp::~cAndOp()
{
   // Left & Right Deletion inherited
   if (DebugLevel >= 7) {
      cout << "cAndOp instance deleted\n";
   }

}


int cAndOp::Precedence()
{
   return DefPrecTable[And_Op];
}


void cAndOp::List()
{
   BinList(" && ", Precedence(), Left, Right);
}


EString cAndOp::Translate()
{
   return BinTranslate(" && ", Precedence(), Left, Right);
}


cNEOp::cNEOp(cExpr *l, cExpr *r) : cBinOp(l, r)
{
   if (DebugLevel >= 7) {
      cout << "cNEOp instance created\n";
   }

}


cNEOp::~cNEOp()
{
   // Left & Right Deletion inherited
   if (DebugLevel >= 7) {
      cout << "cNEOp instance deleted\n";
   }

}


int cNEOp::Precedence()
{
   return DefPrecTable[NE_Op];
}


void cNEOp::List()
{
   BinList(" != ", Precedence(), Left, Right);
}


EString cNEOp::Translate()
{
   return BinTranslate(" != ", Precedence(), Left, Right);
}


cEqOp::cEqOp(cExpr *l, cExpr *r) : cBinOp(l, r)
{
   if (DebugLevel >= 7) {
      cout << "cEqOp instance created\n";
   }

}


cEqOp::~cEqOp()
{
   // Left & Right Deletion inherited
   if (DebugLevel >= 7) {
      cout << "cEqOp instance deleted\n";
   }

}


int cEqOp::Precedence()
{
   return DefPrecTable[Eq_Op];
}


void cEqOp::List()
{
   BinList(" == ", Precedence(), Left, Right);
}


EString cEqOp::Translate()
{
   return BinTranslate(" == ", Precedence(), Left, Right);
}


cGEOp::cGEOp(cExpr *l, cExpr *r) : cBinOp(l, r)
{
   if (DebugLevel >= 7) {
      cout << "cGEOp instance created\n";
   }

}


cGEOp::~cGEOp()
{
   // Left & Right Deletion inherited
   if (DebugLevel >= 7) {
      cout << "cGEOp instance deleted\n";
   }

}


int cGEOp::Precedence()
{
   return DefPrecTable[GE_Op];
}


void cGEOp::List()
{
   BinList(" >= ", Precedence(), Left, Right);
}


EString cGEOp::Translate()
{
   return BinTranslate(" >= ", Precedence(), Left, Right);
}


cLEOp::cLEOp(cExpr *l, cExpr *r) : cBinOp(l, r)
{
   if (DebugLevel >= 7) {
      cout << "cLEOp instance created\n";
   }

}


cLEOp::~cLEOp()
{
   // Left & Right Deletion inherited
   if (DebugLevel >= 7) {
      cout << "cLEOp instance deleted\n";
   }

}


int cLEOp::Precedence()
{
   return DefPrecTable[LE_Op];
}


void cLEOp::List()
{
   BinList(" <= ", Precedence(), Left, Right);
}


EString cLEOp::Translate()
{
   return BinTranslate(" <= ", Precedence(), Left, Right);
}


cLTOp::cLTOp(cExpr *l, cExpr *r) : cBinOp(l, r)
{
   if (DebugLevel >= 7) {
      cout << "cLTOp instance created\n";
   }

}


cLTOp::~cLTOp()
{
   // Left & Right Deletion inherited
   if (DebugLevel >= 7) {
      cout << "cLTOp instance deleted\n";
   }

}


int cLTOp::Precedence()
{
   return DefPrecTable[LT_Op];
}


void cLTOp::List()
{
   BinList(" < ", Precedence(), Left, Right);
}


EString cLTOp::Translate()
{
   return BinTranslate(" < ", Precedence(), Left, Right);
}


cGTOp::cGTOp(cExpr *l, cExpr *r) : cBinOp(l, r)
{
   if (DebugLevel >= 7) {
      cout << "cGTOp instance created\n";
   }

}


cGTOp::~cGTOp()
{
   // Left & Right Deletion inherited
   if (DebugLevel >= 7) {
      cout << "cGTOp instance deleted\n";
   }

}


int cGTOp::Precedence()
{
   return DefPrecTable[GT_Op];
}


void cGTOp::List()
{
   BinList(" > ", Precedence(), Left, Right);
}


EString cGTOp::Translate()
{
   return BinTranslate(" > ", Precedence(), Left, Right);
}


cMinusOp::cMinusOp(cExpr *l, cExpr *r) : cBinOp(l, r)
{
   if (DebugLevel >= 7) {
      cout << "cMinusOp instance created\n";
   }

}


cMinusOp::~cMinusOp()
{
   // Left & Right Deletion inherited
   if (DebugLevel >= 7) {
      cout << "cMinusOp instance deleted\n";
   }

}


int cMinusOp::Precedence()
{
   return DefPrecTable[Minus_Op];
}


void cMinusOp::List()
{
   BinList("-", Precedence(), Left, Right);
}


EString cMinusOp::Translate()
{
   return BinTranslate("-", Precedence(), Left, Right);
}


cPlusOp::cPlusOp(cExpr *l, cExpr *r) : cBinOp(l, r)
{
   if (DebugLevel >= 7) {
      cout << "cPlusOp instance created\n";
   }

}


cPlusOp::~cPlusOp()
{
   // Left & Right Deletion inherited
   if (DebugLevel >= 7) {
      cout << "cPlusOp instance deleted\n";
   }

}


int cPlusOp::Precedence()
{
   return DefPrecTable[Plus_Op];
}


void cPlusOp::List()
{
   BinList("+", Precedence(), Left, Right);
}


EString cPlusOp::Translate()
{
   return BinTranslate("+", Precedence(), Left, Right);
}


cModOp::cModOp(cExpr *l, cExpr *r) : cBinOp(l, r)
{
   if (DebugLevel >= 7) {
      cout << "cModOp instance created\n";
   }

}


cModOp::~cModOp()
{
   // Left & Right Deletion inherited
   if (DebugLevel >= 7) {
      cout << "cModOp instance deleted\n";
   }

}


int cModOp::Precedence()
{
   return DefPrecTable[Mod_Op];
}


void cModOp::List()
{
   BinList("%", Precedence(), Left, Right);
}


EString cModOp::Translate()
{
   return BinTranslate("%", Precedence(), Left, Right);
}


cDivOp::cDivOp(cExpr *l, cExpr *r) : cBinOp(l, r)
{
   if (DebugLevel >= 7) {
      cout << "cDivOp instance created\n";
   }

}


cDivOp::~cDivOp()
{
   // Left & Right Deletion inherited
   if (DebugLevel >= 7) {
      cout << "cDivOp instance deleted\n";
   }

}


int cDivOp::Precedence()
{
   return DefPrecTable[Div_Op];
}


void cDivOp::List()
{
   BinList("/", Precedence(), Left, Right);
}


EString cDivOp::Translate()
{
   return BinTranslate("/", Precedence(), Left, Right);
}


cMultOp::cMultOp(cExpr *l, cExpr *r) : cBinOp(l, r)
{
   if (DebugLevel >= 7) {
      cout << "cMultOp instance created\n";
   }

}


cMultOp::~cMultOp()
{
   // Left & Right Deletion inherited
   if (DebugLevel >= 7) {
      cout << "cMultOp instance deleted\n";
   }

}


int cMultOp::Precedence()
{
   return DefPrecTable[Mult_Op];
}


void cMultOp::List()
{
   BinList("*", Precedence(), Left, Right);
}


EString cMultOp::Translate()
{
   return BinTranslate("*", Precedence(), Left, Right);
}


cArrayIndOp::cArrayIndOp(cExpr *l, cExpr *r) : cBinOp(l, r)
{
   if (DebugLevel >= 7) {
      cout << "cArrayIndOp instance created\n";
   }

}


cArrayIndOp::~cArrayIndOp()
{
   // Left & Right Deletion inherited
   if (DebugLevel >= 7) {
      cout << "cArrayIndOp instance deleted\n";
   }

}


int cArrayIndOp::Precedence()
{
   return DefPrecTable[ArrayInd_Op];
}


void cArrayIndOp::List()
{

   if (Left == 0) Die("Left is NULL in cArrayIndOp!\n");
   if (Right == 0) Die("Right is NULL in cArrayIndOp!\n");

   if (Precedence() > Left->Precedence()) {
      cout << "(";
      Left->List();
      cout << ")";
   } else
     Left->List();

   cout << "[";
   Right->List();
   cout << "]";

}


EString cArrayIndOp::Translate()
{
   EString E;

   if (Left == 0) Die("Left is NULL in cArrayIndOp::Translate!\n");
   if (Right == 0) Die("Right is NULL in cArrayIndO::Translatep!\n");

   if (Precedence() > Left->Precedence()) {
      E << "(" << Left->Translate() << ")";
   } else
     E << Left->Translate();

   E << "[" << Right->Translate() << "]";

   return E;
}
