#ifndef _object_x
#define _object_x

// **********************************************************************
// This file defines an abstract class called cObject.  All classes in
// the CODE model class hierarchy should be derived from it.   It's main
// purpose is to allow maintenance of lists of class instances.  For
// this purpose classes cObList and cObIterator are also defined here.

// The cObList operators are described below.  Pointers to nodes are
// stored in the list.

// void Insert (cObject *Val)
//   Insert node at front of list.

// void Append (cObject *val)
//   Append node at tail of list.

// int Length()
//   Return length of list.

// cObject *First();
//   Return first element in list.

// cObject *Last();
//   Return last element in list.

// cObject *Nth(int n);
//   Return Nth element in list (starting count at 0).

// int Empty()
//   Return non-zero Iff list is empty.

// int RemEq(cObject *Val)
//   Remove items that are eq in the Lisp sense (point to same object)
//   to Val.

// cObject *PopFirst()
//   Reomve and return first element in list.

// cObject *PopLast()
//   Reomve and return last element in list.

// void Map (cObjectFuncPtr)
//   Run function on each element of list.  The function should have
//   signature like void foo(cObject *p).

// void Destroy()
//   Delete all storage allocated by the list.  This does NOT delete
//   the Object nodes in the list.  Result is an empty list.

// void cObList::SetEmpty()
//    Set list to empty w/o freeing links.

// void ConCat(cObList &L)
//   Adds L's list members to the end of the list.  L becomes the empty list.

// See file testtree.C for usage examples.

// ************************************************************************

#include "../exmodel/classid.h"

class cObject {      // cObject is abstract
 public:
   virtual ~cObject();            // Lets ALL destructors be runtime chosen
   virtual ClassId IsA() = 0;     // See classid.h
   virtual void PrintId() {Die("Base PrintId Called"); };   // Print class id.
   virtual void List() {Die("Base List Called"); };  // Print node data.
};


// This beauty gives the name cObjectFuncPtr to the type "pointer to a
// function that returns nothing but takes a cObject * as an argument.

typedef void (*cObjectFuncPtr)(cObject *);

class cObList;
class cObIterator;

class cObLink {
   friend class cObList;
   friend class cObIterator;
 private:
   cObLink (cObject *v=NULL) { Val = v; Next = NULL; }
   cObLink *Next;
   cObject *Val;
};

class cObList {
   friend class cObIterator;
 private:
   int Count;
   cObLink *Start;
   cObLink *End;
 public:
   cObList () { Start = 0; End = 0; Count = 0; }
   ~cObList();    // Frees links but not Object nodes in list.
   void Insert (cObject *Val);
   void Append (cObject *val);
   int Length() { return Count; }
   cObject *First();
   cObject *Last();
   cObject *Nth(int n);
   int Empty() { return Start == 0; }
   int RemEq(cObject *Val);
   cObject *PopFirst();
   cObject *PopLast();
   void Map (cObjectFuncPtr);
   void Destroy();
   void SetEmpty();
   void ConCat(cObList &L);
};

class cObIterator {
  public:
   cObIterator() { Current = 0; }
   cObject *Init(cObList &x);
   int MoreLeft() { return Current != 0; }
   cObject *Next();
   int Last() { return (Current->Next == 0 ? 1 : 0); }
   cObLink *Current;
};


void DeleteNode(cObject *p);  // delete node pointed to by p

#endif
