/*
  stmts.C
  ------------------------------------------------------------------------
  This file contains methods for statement-related abstract syntax
  tree classes.
  ------------------------------------------------------------------------
  @(#) $Id: stmts.C,v 1.23 1998/05/28 22:51:21 emery Exp $
  ------------------------------------------------------------------------
  AUTHOR/CONTACT:
 
  Emery Berger                    | <http://www.cs.utexas.edu/users/emery>
  Parallel Programming Group      |  <http://www.cs.utexas.edu/users/code>
  Department of Computer Sciences |             <http://www.cs.utexas.edu>
  University of Texas at Austin   |                <http://www.utexas.edu>
  ========================================================================
*/

#include <config.h>

#include <stddef.h>
#include <stream.h>

// CODE includes
#include <misc/general.h>
#include "classid.h"
#include "object.h"
#include "abstree.h"
#include "exmodel.h"
#include <symtab/symtab.h>
#include <lib/Syntax.h>


cCallStmt::cCallStmt(char *InitFuncName)
{
   FuncName = InitFuncName;

   if (DebugLevel >= 7) {
      cout << "cCallStmt created\n";
   }
}


cCallStmt::cCallStmt(char *InitFuncName, cObList &L)
{
   FuncName = InitFuncName;
   Args = L;

   if (DebugLevel >= 7) {
      cout << "cCallStmt created\n";
   }
}


cCallStmt::~cCallStmt()
{
   if (DebugLevel >= 7) {
      cout << "cCallStmt destroyed\n";
   }
}


void cCallStmt::List()
{
   cObIterator i;
   cObject *tmp;
   int FirstInList = 1;

   cout << FuncName << "(";

   for (tmp = i.Init(Args); i.MoreLeft(); tmp = i.Next()) {
      if (FirstInList) FirstInList = 0;
      else cout << ", ";
      tmp->List();
   }

   cout << ");\n";

}


EString cCallStmt::Translate()
{
   EString E;
   cSEFuncName *SEFuncName;
   cExpr *e;

   SEFuncName = (cSEFuncName *)GetSymbol(FuncName, this, ::SEFuncName, 0);
   if (!SEFuncName->IsPolyMorphic) {
      E << FuncName << "(" << TranslateArgList(FuncName, this, Args) << " );\n";
   } else {
      if (FuncName == MString ("free")) { 
         e = (cExpr *) Args.First();    
         E << "_c2_free_" << e->StructuralType->TypeName 
           << "( " << TranslateArgList(FuncName, this, Args) << " );\n";
      }
   }

   return E;
}


cWhileStmt::cWhileStmt(cExpr *InitCond, cStmt *InitBody)
{
   Cond = InitCond;
   Body = InitBody;

   if (DebugLevel >= 7) {
      cout << "cWhileStmt created\n";
   }
}


cWhileStmt::~cWhileStmt()
{
   if (DebugLevel >= 7) {
      cout << "cWhileStmt destroyed\n";
   }
}


void cWhileStmt::List()
{
   cout << "while (";
   if (Cond == 0) Die("Cond is NULL in cWhileStmt!\n");
   Cond->List();
   cout << ")\n";
   if (Body == 0) Die("Body is NULL in cWhileStmt!\n");
   Body->List();
}


EString cWhileStmt::Translate()
{
   EString E;

   E << "while (" << Cond->Translate() << ")\n"
     << Body->Translate();
   return E;
}


cBlockStmt::cBlockStmt(cObList &L)
{
   StmtList = L;

   if (DebugLevel >= 7) {
      cout << "cBlockStmt created\n";
   }
}


cBlockStmt::~cBlockStmt()
{
   if (DebugLevel >= 7) {
      cout << "cBlockStmt destroyed\n";
   }
}


void cBlockStmt::List()
{
   cObIterator i;
   cObject *p;

   cout << "{\n";
   for (p = i.Init(StmtList); i.MoreLeft(); p = i.Next())
     p->List();
   cout << "}\n";
}


EString cBlockStmt::Translate()
{
   EString E;
   cObIterator i;
   cObject *p;

   E << "{\n";
   for (p = i.Init(StmtList); i.MoreLeft(); p = i.Next())
     E << ((cStmt *) p)->Translate();
   E << "}\n";
   
   return E;
}


cIfStmt::cIfStmt(cExpr *InitCond, cStmt *InitIfPart)
{
   Cond = InitCond;
   IfPart = InitIfPart;
   ElsePart = 0;

   if (DebugLevel >= 7) {
      cout << "cIfStmt created\n";
   }
}


cIfStmt::cIfStmt(cExpr *InitCond, cStmt *InitIfPart, cStmt *InitElsePart)
{
   Cond = InitCond;
   IfPart = InitIfPart;
   ElsePart = InitElsePart;

   if (DebugLevel >= 7) {
      cout << "cIfStmt created\n";
   }
}


cIfStmt::~cIfStmt()
{
   if (DebugLevel >= 7) {
      cout << "cIfStmt destroyed\n";
   }
}


void cIfStmt::List()
{
   cout << "if (";
   if (Cond == 0) Die("Cond is NULL in cIfStmt!\n");
   Cond->List();
   cout << ")\n";
   if (IfPart == 0) Die("IfPart is NULL in cIfStmt!\n");
   IfPart->List();
   if (ElsePart != 0) {
      cout << "else\n";
      ElsePart->List();
   }

}


EString cIfStmt::Translate()
{
   EString E;

   if (Cond == 0) Die("Cond is NULL in cIfStmt::Translate!\n");
   if (IfPart == 0) Die("IfPart is NULL in cIfStmt::Translate!\n");

   E << "if (" << Cond->Translate() << ")\n"
     << IfPart->Translate();

   if (ElsePart != 0) {
      E << "else\n" << ElsePart->Translate();
   }

   return E;
}


cAssignStmt::cAssignStmt(cExpr *InitLHS, cExpr *InitRHS)
{
   LHS = InitLHS;
   RHS = InitRHS;

   if (DebugLevel >= 7) {
      cout << "cAssignStmt created\n";
   }
}


cAssignStmt::~cAssignStmt()
{
   if (DebugLevel >= 7) {
      cout << "cAssignStmt destroyed\n";
   }
}


void cAssignStmt::List()
{
   if (LHS == 0) Die("LHS is NULL in cAssignStmt!\n");
   LHS->List();
   cout << " = ";
   if (RHS == 0) Die("RHS is NULL in cAssignStmt!\n");
   RHS->List();
   cout << ";\n";
}


EString cAssignStmt::Translate()
{
   EString E;

   if (LHS == 0) Die("LHS is NULL in cAssignStmt::Translate!\n");
   if (RHS == 0) Die("RHS is NULL in cAssignStmt::Translate!\n");

   switch (LHS->StructuralType->IsA()) {
   case ::CharSType:
   case ::IntSType:
   case ::RealSType:
      E << LHS->Translate() << " = " <<  RHS->Translate() << ";\n";
      break;
   case ::ArraySType:
   case ::StructSType:
   {
      E << "_c2_free_" << LHS->StructuralType->TypeName 
        << "( " << LHS->Translate() << " );\n";
      if (RHS->IsLHS == 0)
         E << LHS->Translate() << " = " <<  RHS->Translate() << ";\n";
      else
      {
         E << LHS->Translate() << " = " << "_c2_copy_" 
	   << LHS->StructuralType->TypeName 
           << "( " << RHS->Translate(); 

	 switch (TranslatorT) {
	   // E << ", (" << LHS->StructuralType->TypeName << ") 0, LOCAL);";
	   
	 case PVM:
	 case MPI:
	 case UNIFIED:
	   E << ", (" << LHS->StructuralType->TypeName << ") 0, LOCAL, *(_c2_MyAddr->GetMessenger()));";
	   break;
	   
	 default:
           E << " );";
	   break;
	 }
      }
   }
      break;
   default:
      break;
   }
   return E;
}


cNullStmt::cNullStmt()
{
   if (DebugLevel >= 7) {
      cout << "cNullStmt created\n";
   }
}


cNullStmt::~cNullStmt()
{
   if (DebugLevel >= 7) {
      cout << "cNullStmt destroyed\n";
   }
}


void cNullStmt::List()
{
   cout << ";\n";
}


EString cNullStmt::Translate()
{
   EString E;

   E << ";\n";
   return E;
}


cSizeSpec::cSizeSpec()
{

   if (DebugLevel >= 7) {
      cout << "cSizeSpec created\n";
   }
}


cSizeSpec::~cSizeSpec()
{
   if (DebugLevel >= 7) {
      cout << "cArraySize destroyed\n";
   }
}


void cSizeSpec::List()
{
   cObject *p;
   cObIterator i;

   for (p = i.Init(ArrayBounds); i.MoreLeft(); p = i.Next()) {
      cout << "[";
      p->List();
      cout << "]";
   }

   if (!MemberSizes.Empty()) cout << "{";
   for (p = i.Init(MemberSizes); i.MoreLeft(); p = i.Next()) {
      p->List();
   }
   if (!MemberSizes.Empty()) cout << "}";
}


EString cSizeSpec::Translate()
{
   EString E;
   cObject *p;
   cObIterator i;

   for (p = i.Init(MemberSizes); i.MoreLeft(); p = i.Next()) {
      cMemSize *MemSize = (cMemSize *)p;
      E << MemSize->Translate();  
   }

   for (p = i.Init(ArrayBounds); i.MoreLeft(); p = i.Next()) {
      cExpr *Expr = (cExpr *)p;
      E << Expr->Translate();
   }

   return E;
}


cMemSize::cMemSize(char *InitMemberName, cSizeSpec *InitSizes)
{
   MemberName = InitMemberName;
   Sizes = InitSizes;

   if (DebugLevel >= 7) {
      cout << "cMemSize created\n";
   }
}


cMemSize::~cMemSize()
{
   if (DebugLevel >= 7) {
      cout << "cArraySize destroyed\n";
   }
}


void cMemSize::List()
{
   cout << MemberName;
   Sizes->List();
   cout << "; ";
}


EString cMemSize::Translate()
{
   EString E;

   E << Sizes->Translate();
   return E;
}


cVarDecl::cVarDecl(char *InitVarName)
{
   VarName = InitVarName;
   Sizes = 0;
   TypeName = "int";                   // Gets changed later.

   if (DebugLevel >= 7) {
      cout << "cVarDecl created\n";
   }
}


cVarDecl::cVarDecl(char *InitVarName, cSizeSpec *InitSizes)
{
   VarName = InitVarName;
   Sizes = InitSizes;
   TypeName = "int";                   // Gets changed later.

   if (DebugLevel >= 7) {
      cout << "cVarDecl created\n";
   }
}


cVarDecl::~cVarDecl()
{
   if (DebugLevel >= 7) {
      cout << "cVarDecl destroyed\n";
   }
}


void cVarDecl::List()
{
   cout << TypeName << " " << VarName;
   if (Sizes != 0) Sizes->List();
   cout << ";\n";
}


//EString cVarDecl::Translate()
//{
//   EString E;
//
//   E << TypeName << " " << VarName;
//
//   if (Sizes != 0) 
//      E << Sizes->Translate();
//
//   E << ";\n";
//
//   return E;
//}


   // Translate an actual parameter list for a polymorph function to C

EString TranslatePoly(MString &FuncName, cObList &Args)
{
   EString E;
   cExpr *e;

   if (FuncName == MString ("size")) {
      // Do its one arg by value
      e = (cExpr *) Args.First();
      E << e->Translate();
   } else if (FuncName == MString ("copy")) {
      e = (cExpr *) Args.First();
      E << e->Translate();
   } else if (FuncName == MString ("free")) {
      e = (cExpr *) Args.First();
      E << e->Translate();
   } else
     Die("Internal Error: Polymorph function has no TranslatePoly entry.\n");

   return E;
}
      

   // Translate an actual parameter list into C

EString TranslateArgList(MString &FuncName, cAbsTree *CallSite, cObList &Args)
{
   EString E;
   cSEFuncName *FuncSTEntry;
   cSEFuncArg *FuncArgSTEntry;
   cObIterator i;
   cObIterator j;
   cObject *p;
   cExpr *e;
   int FirstInList = 1;

   FuncSTEntry = (cSEFuncName *) GetSymbol(FuncName, CallSite, SEFuncName, 0);

   // There are three cases:
   //   If args are not typechecked (...), pass all by value.
   //   If args are type checked, pass according to func sig.
   //   If func is polymorph, do on case-by-base basis.

   if (FuncSTEntry->IsPolyMorphic)

     E << TranslatePoly(FuncName, Args);

   else if (FuncSTEntry->DontCheckArgs) {

      // Do all by value
      for (p = i.Init(Args); i.MoreLeft(); p = i.Next()) {
	 e = (cExpr *) p;
	 if (FirstInList) FirstInList = 0;
	 else E << ", ";
	 E << e->Translate();
      }

   } else {

      // Do according to func sig-- arrays and structs are always by ref
      p = j.Init(FuncSTEntry->ArgList);
      FuncArgSTEntry = (cSEFuncArg *) p;
      for (p = i.Init(Args); i.MoreLeft(); p = i.Next()) {
	 e = (cExpr *) p;
	 if (FirstInList) FirstInList = 0;
	 else E << ", ";
	 if (FuncArgSTEntry->IsByValue ||
	     FuncArgSTEntry->Type->IsA() == ArraySType ||
	     FuncArgSTEntry->Type->IsA() == StructSType)
	   E << e->Translate();
	 else
	   E << "&(" << e->Translate() << ")";

	 p = j.Next();
	 FuncArgSTEntry = (cSEFuncArg *) p;
      }

   }  

   return E;
}
