/*
 * c2_general.h
 * ------------------------------------------------------------------------
 * General purpose types & defines.
 * ------------------------------------------------------------------------
 * @@(#) $Id: c2_general.h,v 1.3 1997/07/07 18:23:32 emery Exp $
 * ------------------------------------------------------------------------
 * AUTHOR/CONTACT:
 * 
 * Emery Berger                    | <http://www.cs.utexas.edu/users/emery>
 * Systems Analyst                 @@           <mailto:emery@@cs.utexas.edu>
 * Parallel Programming Group      |  <http://www.cs.utexas.edu/users/code>
 * Department of Computer Sciences |             <http://www.cs.utexas.edu>
 * University of Texas at Austin   |                <http://www.utexas.edu>
 * ========================================================================
 */

#ifndef _C2_GENERAL_H
#define _C2_GENERAL_H

#include <stdio.h>
#include <stdlib.h>

#define times times_
#include <time.h>
#undef times

#include "c2_forward.h"
#include "c2_misc.h"


/* DEFINES */

#define TRUE  1
#define FALSE 0

#define INVALID_POINTER 0xDeadBeef  /* Written into memory which should never be referenced. */

/*
 * A node instance is identified by a UID and up to MAXINDICES indices
 */

#define MAXINDICES 7

#define MAX_PORT_INDICES 7
#define MAX_NAME_LENGTH 30	/* The maximum length of a name, in
				 * characters. */
#define MAX_GRAPH_NESTING 255	/* The maximum nesting levels allowed. */


/* Defines for "copy" routines. */

/*
 * Binary representations:
 * 
 * LOCAL    =  11  (3), TRANSMIT =  01  (1), RECEIVE  =  10  (2)
 * 
 * The third bit signifies whether this comes from a thief:
 * 
 * TRANSMIT_FROM_THIEF = 101 (5), RECEIVE_FROM_THIEF  = 110 (6)
 * 
 */

#define LOCAL      3		/* a copy on this processor */
#define TRANSMIT   1		/* send all data to another processor */
#define RECEIVE    2		/* receive data from another processor */

#define TRANSMIT_FROM_THIEF 5	/* send all but reader-access shared vars */
#define RECEIVE_FROM_THIEF  6	/* receive all but reader-access shared vars */

#define _c2_CACHESIZE 16
#define _c2_PARENT 0		/* Use UID _c2_PARENT to get parent's address
				 * in _c2_GetAddr */

/*
 * Debugging routines (assertions and the like). Use WHEN for things like print
 * statements. ENSURE is the same as ASSERT. ASSERTMSG will print an
 * explanatory message after the assertion.
 * 
 */

#if DEBUG
#define WHEN(cond,stmts) if (cond) {stmts;}
#define IFDEBUG(stmts) {stmts;}
#define ASSERT(cond) \
    if (!(cond)) \
      {  \
        fprintf(stderr, "Assertion failed: %s line %d file %s\n", \
                #cond, __LINE__, __FILE__); \
        fflush(stderr); \
	abort(); \
      };

#define ENSURE(cond) ASSERT(cond)
#define ASSERTMSG(cond, msg) \
    if (!(cond)) \
      {  \
        fprintf(stderr, "Assertion failed: %s line %d file %s\n", \
                #cond, __LINE__, __FILE__); \
        fprintf(stderr, "%s\n", msg); \
        fflush(stderr); \
	abort(); \
      };

#define MESSAGE(msg) \
        fprintf(stderr, "Alert: line %d file %s\n", \
                __LINE__, __FILE__); \
        fprintf(stderr, "%s\n", msg); \
        fflush(stderr); \
        abort();

#else

#define WHEN(cond,stmts)
#define IFDEBUG(stmts)
#define ENSURE(cond)
#define ASSERT(cond)
#define ASSERTMSG(cond, msg)
#define MESSAGE(msg)
#endif				/* DEBUG */

/* Defines for address map functions. */

#define _c2_UCNODE 1
#define _c2_NSRELNODE 2
#define _c2_CALLNODE 3

/* NSRel requests */
#define _c2_READER 4
#define _c2_WRITER 5

/* NSRel states */
#define _c2_NOTINIT 6
#define _c2_NOBIND 7
#define _c2_INPROG 8



/* Node queue states */

#define _c2_IDLE 1		/* Node is not running, and does not need to. */
#define _c2_ONQUEUE 2		/* Node is queued to run, but has not
				 * started. */
#define _c2_RUNNING 3		/* Node is running. */
#define _c2_BLOCKED 4		/* Node cannot be executed right now. */
#define _c2_ONDBGQ 5		/* node is stopped and on the debug Q */
#define _c2_NEEDSRUN 6          /* DEPRECATED!! for old runtimes. */


/* Common data for all UC nodes, used by task manager */

enum _c2_ValueType {
    _c2_IsAChar, _c2_IsAnInt, _c2_IsADouble, _c2_IsAStructPtr, _c2_IsAnArrayPtr
};



/* TYPES */

/* Type to hold indices */
/*
 * A node instance is identified by a UID and up to MAXINDICES indices
 */

struct _c2_sIndex {
    int NumInd;
    int Ind[MAXINDICES];
};


/*
 * Provide the infrastructure for allowing object name lookup (for debugging,
 * etc.).
 */

typedef struct {
    int UID;
    char Name[MAX_NAME_LENGTH];
} _c2_NameTable;


/*
 * Table of comp node UID's paired with "is static" values.
 */

typedef struct {
  int UID;
  int Static;
} _c2_StaticTable;


/* A linked-list structure to keep track of node executions. */

typedef struct _c2_sExecution {
    _c2_NodeBase *Node;
    int Firing;
    float Walltime;
    struct _c2_sExecution *Next;
} _c2_Execution;


/* A linked-list structure to keep track of arc firings. */

typedef struct _c2_sFiring {
    _c2_NodeBase *SourceNode;
    int SourceFiring;
    char *TargetPath;
    int TargetFiring;
    struct _c2_sFiring *Next;
} _c2_Firing;


/* structure for links in ready Q-- actually LIFO */

typedef struct _c2_Link {
    _c2_NodeBase *Node;
    struct _c2_Link *Next, *Prev;
} _c2_RQLink;



typedef struct {
    enum _c2_ValueType type;
    union {
	char c;
	int i;
	double d;
	void *structure;
	void *array;
    } u;

#ifdef _C2_DBGR_
    _c2_FpList fplist;
    _c2_Cmd cmd;
#endif
} _c2_Value;


#if !(__cplusplus)

/* NSRel Link Set */

struct _c2_sNSLink {
    _c2_NodeBase *Node;
    struct _c2_sNSLink *Next;
};


struct _c2_sLockSet {
    struct _c2_sLockSet *Next;

    enum _c2_ValueType TypeTag;
    int UID;
    int ReqType;

    /*
     * All the items below are only valid if IsRemote == 0.
     */

    _c2_NSRelBase *NSRelAddr;
    int *RCount;
    _c2_NSLink **RQ;
    int *WCount;
    _c2_NSLink **WQ;
    void *SharedAddr;
    void *LocalAddr;

#ifdef _C2_DBGR_
    _c2_List *psplist;
#endif
};


typedef struct _c2_sSeqVar _c2_SeqVar;

typedef struct _c2_sSeqVar *_c2_SeqVarQ[MAX_PORT_INDICES + 1];


struct _c2_CacheEntry {
    _c2_Index PortIndex;
    _c2_Index MappedPortIndex;
    _c2_SeqVar **QVar;
    _c2_NodeBase *NodeAddr;
};


struct _c2_Cache {
    int Current;
    int Len;
    struct _c2_CacheEntry Cache[_c2_CACHESIZE];
};


struct _c2_sSeqVar {

    _c2_Value value;
    _c2_Index index;
    _c2_SeqVar *next;
};


/* A UC Node */

struct _c2_sNodeBase {
  /* FROM HERE TO Path, MUST BE THE SAME AS NSRelBase. */
    int UID;			/* Unique ID number */
    _c2_Index Index;		/* My index (e.g., foo[i][j]). */

    int QueueStatus;
    int NSState;
    _c2_LockSet * HeadLock;	/* List of NS locks needed to run */
    _c2_LockSet * NextLock;	/* Next NS lock needed */
    int Crepped;		/* Have all of my creps been filled in? */
    _c2_GraphBase * MyGraph;	/* A pointer to my enclosing graph. */

    int Attempts;		/* The number of attempts at enqueueing while
				 * the node was blocked. */
    int Firings;		/* The total number of firings of this node. */

    void (*InitProc) (_c2_NodeBase *);	/* Initial computation procedure. */
    int (*CompProc) (_c2_NodeBase *);	/* Every computation procedure. */

    void *LocalData;		/* Pointer to struct with local variables */

#ifdef _C2_DBGR_
    _c2_DbgUc dbg;
#endif

    /* Timing statistics (time spent in computations). */

    clock_t WallClockTime;
    clock_t AverageTime;
    float SystemTime;
    float UserTime;
    int RunCount;		/* Times run total. */
    int SuccessfulRunCount;	/* Times run when firing rules succeeded. */

};


/* Common data structure for all graphs. */

struct _c2_sGraphBase {
    int UID;
    _c2_Index Index;

    _c2_GraphBase *Parent;	/* Pointer to calling Graph */
    int CrepsToGo;		/* Count of unbound Creation Params */
    int Creps;			/* Total amount of creps -- used for mapping. */
    _c2_AddrMap *Map;

    void *LocalData;		/* Pointer to struct of creation params */

#ifdef _C2_DBGR_
    _c2_DbgGph dbg;
#endif
};


/* Common data for all NSRel nodes */

struct _c2_sNSRelBase {
  /* FROM HERE TO Path, MUST BE THE SAME AS NodeBase. */
    int UID;
    _c2_Index Index;

    int Crepped;
    _c2_GraphBase *MyGraph;

    void (*InitProc) (_c2_NSRelBase *);

    void *LocalData;		/* Pointer to struct with local variables */

    /* Timing statistics */

    clock_t WallClockTime;
    float SystemTime;
    float UserTime;
};



/* Implementation of Address Maps (but just use access functions) */

struct _c2_sAddrLink {
    int UID;			/* UID of entry */
    int ObjType;		/* _c2_UCNODE, _c2_NSRELNODE, _c2_CALLNODE,
				 * etc. */
    _c2_Index Index;		/* Index of Entry */
    void *Entry;
    _c2_AddrLink *Next;
};

struct _c2_sAddrMap {
    _c2_AddrLink *Head;
};


#endif /* not C++ or SMP. */

#endif				/* _C2_GENERAL_H */
