/*
  MPImessenger.H
  ------------------------------------------------------------------------
  MPI messenger class.
  ------------------------------------------------------------------------
  @(#) $Id: MPImessenger.H,v 1.15 1998/11/06 05:03:29 emery Exp $
  ------------------------------------------------------------------------
  AUTHOR/CONTACT:
 
  Emery Berger                    | <http://www.cs.utexas.edu/users/emery>
  Parallel Programming Group      |  <http://www.cs.utexas.edu/users/code>
  Department of Computer Sciences |             <http://www.cs.utexas.edu>
  University of Texas at Austin   |                <http://www.utexas.edu>
  ========================================================================
*/

#ifndef _MPIMESSENGER_H_
#define _MPIMESSENGER_H_

#include <assert.h>
#include <mpi.h>

#define times times_
#include <deque.h> // STL
#undef times

#include "atomic.H"
#include "messenger.H"
#include "unique.H"
#include "dynarray.H"
#include "parms.H"
#include "deq.h"


class MPIMessenger : public Messenger {
public:

  enum { BUFFSIZE = 1048576 }; // 1MB message buffer

  MPIMessenger (void);
  ~MPIMessenger (void);

  void initialize (int& argc, char **& argv);
  void finalize (void);

  int self (void);
  int numProcessors (void);

  void send (const int dest, const int tag);

  int receive (void);

  void broadcast (const int firstProcessor,
  		  const int lastProcessor,
  		  const int tag);

  void pack (const char ch) { pkbyte (&ch, 1); }
  void pack (const int i)   { pkint (&i, 1); }

  void pack (const int *data, const int length) { pkint (data, length); }
  void pack (const double *data, const int length) { pkdouble (data, length); }
  void pack (const float *data, const int length) { pkfloat (data, length); }
  void pack (const char *data, const int length) { pkbyte (data, length); }

  void pack (const long l)  { pklong (&l, 1); }
  void pack (const double d) { pkdouble (&d, 1); }
  void pack (const float f) { pkfloat (&f, 1); }
  void pack (const char *s) { pkstr (s); }
  void pack (char *s) { pkstr (s); }

  void unpack (char& ch) { upkbyte (&ch, 1); }
  void unpack (int& i)   { upkint (&i, 1); }
  void unpack (int* data, const int length)   { upkint (data, length); }
  void unpack (double * data, const int length) { upkdouble (data, length); }
  void unpack (float * data, const int length) { upkfloat (data, length); }
  void unpack (char * data, const int length = 0) {
    if (!length) {
      upkstr (data);
    } else {
      upkbyte (data, length);
    }
  }

  void unpack (long& l) { upklong (&l, 1); }
  void unpack (double& d) { upkdouble (&d, 1); }
  void unpack (float& f) { upkfloat (&f, 1); }

  int messageLength (void) { return _length; }
  int messageSender (void) { return _sender; }
  int messageTag (void) { return _tag; }

  int flush (void);
  int poll (void) { return 0; }

private:

  // Prevent copying and assignment.

  MPIMessenger (const MPIMessenger&);
  const MPIMessenger& operator= (const MPIMessenger&);

  // Packing routines, a la PVM.
  
  int pkint (const int *np, int nitem);
  int pklong (const long *np, int nitem);
  int pkdouble (const double *np, int nitem);
  int pkfloat (const float *np, int cnt);
  int pkbyte (const char *cp, int nitem);
  int pkstr (const char *cp);
  
  int upkint (int *np, int cnt);
  int upklong (long *np, int nitem);
  int upkdouble (double *np, int cnt);
  int upkfloat (float *np, int cnt);
  int upkbyte (char *cp, int cnt);
  int upkstr (char *cp);

  dynarray<char>	_inbuf;   // The input buffer.
  dynarray<char>	_outbuf;  // The output buffer.
  int    		_inpos;   // Current position in the input buffer.
  int    		_outpos;  // Current position in the output buffer.

  int			_sender;
  int			_tag;
  int			_length;

  // All messengers share one inbox.
  static deque<OutPacket *>		_inbox;
  static MutexRec			_inboxlock;

  // Each messenger has its own outbox.
  deque<OutPacket *>		_outbox;
  MutexRec			_outboxlock;

  // The MPI buffer.
  char * _buffer;
};



#endif // _MPIMESSENGER_H_
