/*
  PVMmessenger.C
  ------------------------------------------------------------------------
  PVM messenger class implementation.
  ------------------------------------------------------------------------
  @(#) $Id: PVMmessenger.C,v 1.2 1997/12/19 17:50:30 emery Exp $
  ------------------------------------------------------------------------
  AUTHOR/CONTACT:
 
  Emery Berger                    | <http://www.cs.utexas.edu/users/emery>
  Parallel Programming Group      |  <http://www.cs.utexas.edu/users/code>
  Department of Computer Sciences |             <http://www.cs.utexas.edu>
  University of Texas at Austin   |                <http://www.utexas.edu>
  ========================================================================
*/

#include <assert.h>
#include <iostream.h>
#include <stdio.h>
#include <pvm3.h>

#include "PVMmessenger.H"
#include "machine.H"

void PVMMessenger::initialize (int& argc, char **& argv)
{
}


void PVMMessenger::finalize (void)
{
}


void PVMMessenger::PVMinitialize (int nhosts, int * tids)
{
  _tids = new int[nhosts];
  memcpy (_tids, tids, nhosts * sizeof(int));
}


int PVMMessenger::self (void)
{
  Guard m1 (objectlock());
  static int _self = -1;
  if (_self == -1) {
    int tid;
    Guard m2 (classlock());
    if ((tid = pvm_mytid()) < 0) {
      pvm_perror("PVMMessenger: ");
    }
    for (int i = 0; i < numProcessors(); i++) {
      if (_tids[i] == tid) {
	_self = i;
      }
    }
    assert (_self != -1);
  }
  return _self;
}


int PVMMessenger::numProcessors (void)
{
  Guard m1 (objectlock());
  static int _numProcessors = -1;
  if (_numProcessors == -1) {
    struct pvmhostinfo *hostp;
    int narch;
    Guard m2 (classlock());
    if (pvm_config (&_numProcessors, &narch, &hostp)) {
      pvm_perror("PVMMessenger::numProcessors ");
    }
  }
  return _numProcessors;
}


void PVMMessenger::send (const int dest, const int tag)
{
  Guard m1 (objectlock());
  Guard m2 (classlock());

  assert (dest >= 0);
  assert (dest < numProcessors ());
  assert (tag >= 0);
  // printf ("[%d] Sending a '%s' message to %d (tid = %d).\n", self(), CODE_Machine::MessageTagString[tag], dest, _tids[dest]);

  if (pvm_send (_tids[dest], tag)) {
    pvm_perror("PVMMessenger::send ");
  }

  _startmsg = 1;

  // printf ("[%d] Performed a send.\n", self()); fflush (stdout);
  // Packing after a send resumes at the beginning of the buffer.
}


int PVMMessenger::receive (const int source,
			   const int tagval,
			   const Messenger::Protocol proto)
{
  assert (source >= -1);
  assert (source < numProcessors ());

  Guard m1 (objectlock());
  int src = source;
  int tag = tagval;

  // Use PVM defines here.
  if (src == Messenger::ANY_SOURCE) {
    src = PVM_ANY_SOURCE;
  } else {
    src = _tids[source];
  }

  if (tag == Messenger::ANY_TAG) {
    tag = PVM_ANY_TAG;
  }
  
  switch (proto) {

  case Messenger::BLOCKING:
    // BEWARE: This prevents ANY other PVM communications
    // while we wait for a message to arrive!
    {
      Guard m (classlock());

      if (pvm_recv (src, tag) < 0) {
	pvm_perror("PVMMessenger::receive ");
      }
      _bufid = pvm_probe (src, tag);
      if (_bufid < 0) {
	pvm_perror("PVMMessenger::receive ");
      }
      assert (_bufid != 0);
      // printf ("Received a '%s' message from %d.\n", CODE_Machine::MessageTagString[messageTag()], messageSender());
      return 1;
    }
  case Messenger::NONBLOCKING:
    {
      Guard m (classlock());
      _bufid = pvm_probe (src, tag);
      if (_bufid < 0) {
	pvm_perror("PVMMessenger::receive ");
      }
      if (_bufid > 0) {
	if (pvm_recv (src, tag) < 0) {
	  pvm_perror("PVMMessenger::receive ");
	}
	// printf ("Received a '%s' message from %d.\n", CODE_Machine::MessageTagString[messageTag()], messageSender()); fflush (stdout);
	return 1;
      } else {
	return 0;
      }
    }
  default:
    return 0;
  }
}


void PVMMessenger::broadcast (const int firstProcessor,
			      const int lastProcessor,
			      const int tag)
{
  assert (firstProcessor >= 0);
  assert (lastProcessor < numProcessors ());
  assert (firstProcessor <= lastProcessor);

  Guard m1 (objectlock());
  Guard m2 (classlock());

  int me = self();
  int i = 0;
  int tids[Messenger::MAX_PROCESSORS];

  // Pack up the tids array with the id's of destination processors.

  for (int dest = firstProcessor; dest <= lastProcessor; ++dest) {
    if (dest != me) {
      tids[i++] = _tids[dest];
    }
  }

  if (pvm_mcast (tids, i, tag) < 0) {
    pvm_perror("PVMMessenger::broadcast ");
  }
  _startmsg = 1;
}


int PVMMessenger::messageLength (void) 
{
  int nbytes;
  int source;
  int tag;
  Guard m1 (objectlock());
  Guard m2 (classlock());
  if (pvm_bufinfo( _bufid, &nbytes, &tag, &source ) < 0) {
    pvm_perror("PVMMessenger::messageLength ");
  }
  return nbytes;
}


int PVMMessenger::messageSender (void)
{
  int nbytes;
  int source;
  int tag;
  Guard m1 (objectlock());
  if (pvm_bufinfo( _bufid, &nbytes, &tag, &source) < 0) {
    pvm_perror("PVMMessenger::messageSender ");
  }
  // printf ("messageSender : source = %d\n", source); fflush (stdout);
  int src = -1;
  for (int i = 0; i < numProcessors(); i++) {
    if (_tids[i] == source) {
      src = i;
    }
  }
  assert (src != -1);
  return src;
}


int PVMMessenger::messageTag (void)
{
  int nbytes;
  int source;
  int tag;
  Guard m1 (objectlock());
  if (pvm_bufinfo( _bufid, &nbytes, &tag, &source ) < 0) {
    pvm_perror("PVMMessenger::messageTag ");
  }
  return tag;
}


int PVMMessenger::messageWaiting (void) 
{
  int flag;
  Guard m1 (objectlock());
  Guard m2 (classlock());
  flag = pvm_probe (PVM_ANY_SOURCE, PVM_ANY_TAG);
  if (flag == 0) {
    return 0;
  }
  if (flag > 0) {
    _bufid = flag;
    return 0;
  }
  // flag < 0
  pvm_perror ("PVMMessenger::messageWaiting ");
  return 0;
}
