/*
  PVMmessenger.H
  ------------------------------------------------------------------------
  PVM messenger class.
  ------------------------------------------------------------------------
  @(#) $Id: PVMmessenger.H,v 1.6 1997/12/17 17:05:05 emery Exp $
  ------------------------------------------------------------------------
  AUTHOR/CONTACT:
 
  Emery Berger                    | <http://www.cs.utexas.edu/users/emery>
  Parallel Programming Group      |  <http://www.cs.utexas.edu/users/code>
  Department of Computer Sciences |             <http://www.cs.utexas.edu>
  University of Texas at Austin   |                <http://www.utexas.edu>
  ========================================================================
*/

#ifndef _PVMMESSENGER_H_
#define _PVMMESSENGER_H_

#include <assert.h>
#include <stdio.h>
#include <string.h>
#include <pvm3.h>

#include "atomic.H"
#include "messenger.H"
#include "unique.H"
#include "dynarray.H"


class PVMMessenger : public Messenger {
public:

  PVMMessenger (void)
    : _startmsg (1),
      _tids (NULL)
    {}

  ~PVMMessenger (void) {
    delete _tids;
  }

  enum { PVM_ANY_SOURCE = -1,
	 PVM_ANY_TAG = -1 };

  void initialize (int& argc, char **& argv);

  void PVMinitialize (int nhosts, int * tids);

  void finalize (void);

  int self (void);
  int numProcessors (void);

  void send (const int dest, const int tag);

  int receive (const int src, const int tag, const Messenger::Protocol proto);

  void broadcast (const int firstProcessor,
  		  const int lastProcessor,
  		  const int tag);

  void pack (const char ch) { _check(); pkbyte (&ch, 1, 1); }
  void pack (const int i)   { _check(); pkint (&i, 1, 1); }

  void pack (const int *data, const int length) { _check(); pkint (data, length, 1); }
  void pack (const double *data, const int length) { _check(); pkdouble (data, length, 1); }
  void pack (const float *data, const int length) { _check(); pkfloat (data, length, 1); }
  void pack (const char *data, const int length) { _check(); pkbyte (data, length, 1); }

  void pack (const long l)  { _check(); pklong (&l, 1, 1); }
  void pack (const double d) { _check(); pkdouble (&d, 1, 1); }
  void pack (const float f) { _check(); pkfloat (&f, 1, 1); }
  void pack (const char *s) { _check(); pkstr (s); }
  void pack (char *s) { _check(); pkstr (s); }

  void unpack (char& ch) { upkbyte (&ch, 1, 1); }
  void unpack (int& i)   { upkint (&i, 1, 1); }
  void unpack (int* data, const int length)   { upkint (data, length, 1); }
  void unpack (double * data, const int length) { upkdouble (data, length, 1); }
  void unpack (float * data, const int length) { upkfloat (data, length, 1); }
  void unpack (char * data, const int length = 0) {
    if (!length) {
      upkstr (data);
    } else {
      upkbyte (data, length, 1);
    }
  }

  void unpack (long& l) { upklong (&l, 1, 1); }
  void unpack (double& d) { upkdouble (&d, 1, 1); }
  void unpack (float& f) { upkfloat (&f, 1, 1); }

  int messageLength (void);

  int messageSender (void);
  int messageTag (void);

  int messageWaiting (void);

private:

  // Prevent copying and assignment.

  PVMMessenger (const PVMMessenger&);
  const PVMMessenger& operator= (const PVMMessenger&);

  // Packing routines, a la PVM.
  
  int pkint (const int *np, int nitem, int stride) { return pvm_pkint (np, nitem, stride); }
  int pklong (const long *np, int nitem, int stride) { return pvm_pklong (np, nitem, stride); }
  int pkdouble (const double *np, int nitem, int stride) { return pvm_pkdouble (np, nitem, stride); }
  int pkfloat (const float *np, int cnt, int std) { return pvm_pkfloat (np, cnt, std); }
  int pkbyte (const char *cp, int nitem, int stride) { return pvm_pkbyte (cp, nitem, stride); }
  int pkstr (const char *cp) { return pvm_pkstr (cp); }
  
  int upkint (int *np, int cnt, int std) { return pvm_upkint (np, cnt, std); }
  int upklong (long *np, int nitem, int stride) { return pvm_upklong (np, nitem, stride); }
  int upkdouble (double *np, int cnt, int std) { return pvm_upkdouble (np, cnt, std); }
  int upkfloat (float *np, int cnt, int std) { return pvm_upkfloat (np, cnt, std); }
  int upkbyte (char *cp, int cnt, int std) { return pvm_upkbyte (cp, cnt, std); }
  int upkstr (char *cp) { return pvm_upkstr (cp); }

  void _check (void) {
    Guard m1 (objectlock());
    Guard m2 (classlock());
    if (_startmsg) {
      pvm_initsend (PvmDataDefault);
      _startmsg = 0;
    }
  }

  int	_bufid;
  int	_startmsg;
  int *	_tids;
};



#endif // _PVMMESSENGER_H_
