/*
  atomic.H
  ------------------------------------------------------------------------
  Implements a class for atomic operations on generic objects and
  numbers, based on ACE.
  ------------------------------------------------------------------------
  @(#) $Id: atomic.H,v 1.25 1998/04/07 04:48:25 emery Exp $
  ------------------------------------------------------------------------
  AUTHOR/CONTACT:
 
  Emery Berger                    | <http://www.cs.utexas.edu/users/emery>
  Parallel Programming Group      |  <http://www.cs.utexas.edu/users/code>
  Department of Computer Sciences |             <http://www.cs.utexas.edu>
  University of Texas at Austin   |                <http://www.utexas.edu>
  ========================================================================
*/


#ifndef _ATOMIC_H__
#define _ATOMIC_H__

#include <assert.h>
#include <pthread.h>
#include <string.h>

#include "header.H"

#include "fsynch.h"
#include "fthread.h"
#include "timer.H"
#include "unique.H"


// #define MEASURE_CONTENTION


// Define the interface for a Mutex.
struct MutexInterface {
  int acquire (void);
  int release (void);
};


// a wrapper for condition variables.
class Condition : public Unique {
public:

  Condition (void)
    {
      ConditionInit (&_cond);
      LockInit (&_condlock);
    }
  
  ~Condition (void) {
    this->remove();
  }

  int wait (Lock& condlock) {
    ConditionWait (&_cond, &condlock);
    return 0;
  }

  int wait (void) {
    ConditionWait (&_cond, &_condlock);
    return 0;
  }

  int timedwait (int sec) {
    return ConditionTimedWait (&_cond, &_condlock, sec);
  }

  int signal (void) {
    ConditionSignal (&_cond);
    return 0;
  }

  int broadcast (void) {
    ConditionBroadcast (&_cond);
    return 0;
  }

  void acquire (void) {
    LockAcquire (&_condlock);
  }

  void release (void) {
    LockRelease (&_condlock);
  }

  int remove (void) {
    int result = 0;
    
    do {
      ConditionBroadcast (&_cond);
    } while ((result = ConditionDestroy (&_cond)));
    
    return 0;
  }


private:
  Cond		_cond;
  Lock		_condlock;
};


// a wrapper for mutexes.

class Mutex : public MutexInterface, public Unique {

public:
  Mutex (char * name = NULL)
    : _name (name)
    {
      LockInit (&_lock);
    }
  
  ~Mutex (void) {
    LockRelease (&_lock);
#ifdef MEASURE_CONTENTION
    if ((double) _timer > 0.0) {
      cout << "Lock " << _name << ": contention time = " << (double) _timer << endl;
    }
#endif
  }

  int tryacquire (void) const {
    return LockTryAcquire (const_cast(Lock *, &_lock));
  }

  int acquire (void) const {
#ifdef MEASURE_CONTENTION
    if (tryacquire ()) {
      Timer t;
      t.start ();
      //cout << "Lock " << this << " waiting." << endl;
      LockAcquire (const_cast(Lock *, &_lock));
      //cout << "Lock " << this << " released." << endl;
      t.stop ();
      Timer& timer = (const_cast(Timer&, _timer));
      timer.set ((double) timer + (double) t);
    }
    return 0;
#else
    return LockAcquire (const_cast(Lock *, &_lock));
#endif
  }

  int release (void) const {
    return LockRelease (const_cast(Lock *, &_lock));
  }

  void wait (Condition& c) {
    c.wait (_lock);
  }

private:

  Mutex (const Mutex&) { assert (0); }
  Mutex& operator= (const Mutex&) { assert (0); return *this; }

  Lock _lock;
  char * _name;
#ifdef MEASURE_CONTENTION
  Timer _timer;
#endif

};


// Provide protection for a block using some mutex.

template <class TYPE>
class Protect : public Unique {
public:
  // acquire the lock on entry.
  Protect(TYPE& l) : _lock(l) { _lock.acquire(); }
  Protect(const TYPE& l) : _lock(const_cast(TYPE&,l)) { _lock.acquire(); }

  // release the lock when we leave the block.
  ~Protect(void)  { _lock.release(); }
private:
  mutable TYPE& _lock;
};


// implements a recursive mutex on top of Mutex.
class MutexRec : public MutexInterface, public Unique {
public:
  MutexRec (void)
    : 
    _name (NULL),
    _nesting (0),
    _thr_id (0)
  { }

  MutexRec (char * name)
    : 
    _name (name),
    _nesting (0),
    _thr_id (0)
  { }

  ~MutexRec (void)
  {
    if (_name) {
#ifdef MEASURE_CONTENTION
      if ((double) _timer > 0.0) {
	cout << "MutexRec " << this << " (" << _name << ") : contention time = " << (double) _timer << endl;
      }
#endif
    } else {
#ifdef MEASURE_CONTENTION
      if ((double) _timer > 0.0) {
	cout << "MutexRec " << this << ": contention time = " << (double) _timer << endl;
      }
#endif
    }
  }

  int acquire (void)
    {
      Protect<Mutex> m (_lock);

      pthread_t t_id = ThreadSelf();

      if (_nesting == 0) {
	// Lock is available.
	_thr_id = t_id;
      } else if (t_id != _thr_id) {
	// Somebody else holds the lock. Wait for it.
	while (_nesting > 0) {
	  //cout << "Lock " << _name << " waiting." << endl;
	  this->wait ();
	}
	//cout << "Lock " << _name << " released." << endl;
	_thr_id = t_id;
      }
      _nesting++;
      assert (_nesting > 0);
      return 0;
    }


  int release (void)
    {
      Protect<Mutex> m (_lock);
 
#if 0
      pthread_t t_id = ThreadSelf();
      if (_nesting == 0 || _thr_id != t_id) {
	return -1;
      }
#endif

      _nesting--;
      if (_nesting == 0) {
	// Free the lock.
	_thr_id = 0;  // an invalid thread ID.
	this->signal ();
      }
      return 0;
    }
  
  void wait (void) {
#ifdef MEASURE_CONTENTION
    Timer t;
    t.start ();
#endif
    _lock.wait (_lockfree);
#ifdef MEASURE_CONTENTION
    t.stop ();
    _timer.set ((double) _timer + (double) t);
#endif
  }

  void signal (void) {
    _lockfree.signal ();
  }

private:
  Mutex		_lock;
  Condition	_lockfree;
  volatile int		_nesting;
  volatile pthread_t	_thr_id;
  char *	_name;

  MutexRec (const MutexRec&) { assert (0); }
  MutexRec& operator= (const MutexRec&) { assert (0); return *this; }

#ifdef MEASURE_CONTENTION
  Timer _timer;
#endif

};


// put a Guard in the beginning
// of a block to protect the whole block
// from simultaneous access. This is
// better than lock acquisition since it
// is exception-safe (and cleaner).

class Guard : public Protect<MutexRec>
{
public:
  // acquire the lock on entry.
  Guard(MutexRec& l) : Protect<MutexRec> (l) { }
  Guard(const MutexRec& l) : Protect<MutexRec> (l) { }
};


// an Atomic<int> (or other numeric type)
// automatically guarantees atomic access
// to its value.

template <class TYPE>
class Atomic : public Unique {
public:

  Atomic (TYPE v)
    : _lock ("Atomic::lock"),
      value (v) { }

  TYPE operator++ (void) {
    Guard m (_lock);
    return ++value;
  }

  void operator+= (TYPE v) {
    Guard m (_lock);
    value += v;
  }

  TYPE operator-- (void) {
    Guard m (_lock);
    return --value;
  }

  int operator== (const TYPE v) {
    Guard m (_lock);
    return (value == v);
  }

  int operator> (const TYPE v) {
    Guard m (_lock);
    return (value > v);
  }

  int operator!= (const TYPE v) {
    Guard m (_lock);
    return (value != v);
  }

  int operator! (void) {
    Guard m (_lock);
    return !(value);
  }

  void operator= (const Atomic<TYPE>& ao) {
    if (this != &ao) {
      Guard m (_lock);
      value = ao.value;
    }
  }

  void operator= (const TYPE v) {
    Guard m (_lock);
    value = v;
  }

  operator const TYPE (void) const {
    Guard m (_lock);
    return value;
  }

  void lock (void) {
    _lock.acquire ();
  }

  void unlock (void) {
    _lock.release ();
  }


private:
  
  // Disallow non-const access.
  // operator TYPE (void);
  
  mutable MutexRec _lock;	// The lock on the structure.
  TYPE value;			// The value we are providing atomic access to.
};


#endif
