/*
  dynarray.H
  ------------------------------------------------------------------------
  A dynamic array class.
  ------------------------------------------------------------------------
  @(#) $Id: dynarray.H,v 1.7 1998/08/04 17:35:23 emery Exp $
  ------------------------------------------------------------------------
  AUTHOR/CONTACT:
 
  Emery Berger                    | <http://www.cs.utexas.edu/users/emery>
  Parallel Programming Group      |  <http://www.cs.utexas.edu/users/code>
  Department of Computer Sciences |             <http://www.cs.utexas.edu>
  University of Texas at Austin   |                <http://www.utexas.edu>
  ========================================================================
*/

#ifndef _DYNARRAY_H_
#define _DYNARRAY_H_

#include <assert.h>


template <class TYPE>
class dynarray {
public:

  dynarray (void)
    : _array (new TYPE [1]), // Start out with one element.
      _size (1),
      _perceived_size (0)
    {}

  ~dynarray (void) {
    delete _array;
  }


  TYPE * array (void) {
    return _array;
  }

  // Index-checking indexing operations.
  TYPE& operator[] (unsigned int n) {
    assert (_perceived_size <= _size);
    assert (n < _perceived_size);
    return _array[n];
  }

  // Index-checking indexing operations.
  const TYPE& operator[] (unsigned int n) const {
    assert (_perceived_size <= _size);
    assert (n < _perceived_size);
    return _array[n]; 
  }

  // Return the number of elements
  // available in the array.
  // NB: This is a lower bound on the actual size of the array.
  int size (void) {
    assert (_perceived_size <= _size);
    return _perceived_size;
  }

  // Adjust the array's size.
  void resize (unsigned int n) {

    assert (_perceived_size <= _size);

    // Whenever we exceed the size of the array,
    // we double the requested size.
    // When we drop below one-fourth of the array size,
    // we halve the size.
    // This yields O(1) amortized "insertion" and "deletion".

    if ((n > _size) /* || (n < _size / 4) */) {

      // Make a new array that is 2 * n in size,
      // copy in the existing (relevant) elements,
      // and delete the old array.

      _size = n * 2;
      TYPE * newarray = new TYPE[_size];

      if (newarray == NULL) {
	cerr << "Out of memory!" << endl;
	return;
      }

      memcpy (newarray, _array, min(n, _perceived_size) * sizeof(TYPE));

      // Use an auxiliary to provide exception-safety
      // (the destructor always tries to delete _array).

      TYPE * oldarray = _array;
      _array = newarray;
      delete oldarray;
    }

    assert (n <= _size);
    _perceived_size = n;

    assert (_perceived_size <= _size);
  }


private:

  int min (int x, int y) {
    return (x < y) ? x : y;
  }

  int		_size;
  int		_perceived_size;
  TYPE *	_array;
  
};

#endif // _DYNARRAY_H_
