/*
  graph.C
  ------------------------------------------------------------------------
  CODE Graph (container) class.
  ------------------------------------------------------------------------
  @(#) $Id: graph.C,v 1.30 1998/11/06 06:24:20 emery Exp $
  ------------------------------------------------------------------------
  AUTHOR/CONTACT:
 
  Emery Berger                    | <http://www.cs.utexas.edu/users/emery>
  Parallel Programming Group      |  <http://www.cs.utexas.edu/users/code>
  Department of Computer Sciences |             <http://www.cs.utexas.edu>
  University of Texas at Austin   |                <http://www.utexas.edu>
  ========================================================================
*/


#include "c2_misc.h"

#include "graph.H"
#include "index.H"
#include "machine.H"
#include "sermach.H"
#include "node.H"
#include "nsrel.H"

// Compute a hash value for an address link entry.

CodeObject * Graph::AddressMap::Find (int UID, Index * Index, SerialMachine * SerMach) {

  CodeObject * entry;

  // Look for the given UID and Index.

  AddressMap::iterator i;
  int hashval;
  if (Index->length() > 0) {
    hashval = (UID + (*Index)[Index->length()]) % Graph::AddressMap::ADDRESS_MAPS;
  } else {
    hashval = UID % Graph::AddressMap::ADDRESS_MAPS;
  }

  Guard m (_lock[hashval]);

#if 0
  const Identifier findme (UID, *Index);

  i = _Addresses.find (findme);
#else
  for (i = _Addresses[hashval].begin ();
       i != _Addresses[hashval].end ();
       ++i) {
    if ((*i)->ID () == UID) {
      if ((*i)->MyIndex() == *Index) {
	break;
      }
    }
  }
#endif

  if (i != _Addresses[hashval].end ()) {
    // We found it.
#if 0
    entry = (CodeObject *) (*i).second.Object;
#else
    entry = (*i);
#endif

    return entry;

  } else {

    // We need to create this object and add it to the map.
    
    int objtype;
    entry = (CodeObject *) _c2_MakeObj (UID, Index, _MyGraph, &objtype, SerMach);

#if 0
    Entry e (objtype, (void *) entry);
    Identifier id (UID, *Index);

    _Addresses[id] = e;
#else
    _Addresses[hashval].push_front (entry, SerMach);
#endif

    return entry;
  }
}


void Graph::AddressMap::InitializeMembers (void)
{
 
  int me = _MyGraph->GetMachine().WhichProcessor();

  // Go through every node and name-sharing relation
  // in our map.

  AddressMap::iterator i;

  CompNode *u;
  NameSharingRelation *n;

  for (int j = 0;
       j < ADDRESS_MAPS;
       j++) {
    Guard m (_lock[j]);
    for (i = _Addresses[j].begin ();
	 i != _Addresses[j].end ();
	 ++i)
      {
	switch ((*i)->GetType ()) {
	case CodeObject::UC:
	  u = (CompNode *) (*i);
	  ASSERT (u != NULL);
	  
	  // If this node belongs on this processor,
	  // queue it up.
	  // (it didn't execute because its crep was missing).
	  
	  if (me == u->Processor()) {
	    _MyGraph->GetMachine().GetReadyQ().Enqueue (u);
	  }
	  break;
	  
	case CodeObject::NSREL:
	  n = (NameSharingRelation *) (*i);
	  ASSERT (n != NULL);
	  // If this name-sharing relation belongs on this processor,
	  // initialize it.
	  if (me == n->Processor()) {
	    n->Initialize();
	  }
	  break;
	default:
	  break;
	}
      }
  }
}


void Graph::AddCrep (void) 
{
  
#ifdef _C2_DBGR_
  _c2_dbgGphCreatAction (g, fplist);
#endif
  
  if (--_CrepsToGo > 0) {
    return;
  } else {
    _Map.InitializeMembers();
  }
}


Graph::AddressMap::~AddressMap (void) 
{
#if 1

  AddressMap::iterator i;

  for (int j = 0;
       j < ADDRESS_MAPS;
       j++) {
    Guard m (_lock[j]);
    for (i = _Addresses[j].begin ();
	 i != _Addresses[j].end ();
	 ++i)
      {
	switch ((*i)->GetType()) {
	case CodeObject::UC:
	  delete ((CompNode *) (*i));
	  break;
	  
	case CodeObject::NSREL:
	  delete ((NameSharingRelation *) (*i));
	  break;
	  
	case CodeObject::GRAPH:
	  delete ((Graph *) (*i));
	  break;
	  
	default:
	  MESSAGE ("Invalid node tag.");
	  break;
	}
      }
  }
#endif
}


void Graph::AddressMap::ReportMemberStats (ostream& out) {
  AddressMap::iterator i;

  for (int j = 0;
       j < ADDRESS_MAPS;
       j++) {
    for (i = _Addresses[j].begin ();
	 i != _Addresses[j].end ();
	 ++i)
      {
	switch ((*i)->GetType()) {
	case CodeObject::UC:
	  out << *((CompNode *) (*i));
	  ((CompNode *) (*i))->ReportStats (out);
	  break;
	case CodeObject::NSREL:
	  // We don't keep track of initialization of NameSharingRelations.
	  break;
	case CodeObject::GRAPH:
	  ((Graph *) (*i))->ReportStats (out);
	  break;
	default:
	  MESSAGE ("Invalid node tag.");
	  break;
	}
      }
  }
}


void Graph::AddressMap::TransmitMemberStats (int processor,
					     Messenger& messenger)
{
  AddressMap::iterator i;

  for (int j = 0;
       j < ADDRESS_MAPS;
       j++) {
    for (i = _Addresses[j].begin ();
	 i != _Addresses[j].end ();
	 ++i)
      {
	switch ((*i)->GetType()) {
	case CodeObject::UC:
	  ((CompNode *) (*i))->TransmitStats (processor, messenger);
	  break;
	case CodeObject::NSREL:
	  // We don't keep track of initialization of NameSharingRelations.
	  break;
	case CodeObject::GRAPH:
	  ((Graph *) (*i))->TransmitStats (processor, messenger);
	  break;
	default:
	  MESSAGE ("Invalid node tag.");
	  break;
	}
      }
  }
}


double Graph::AddressMap::T1 (void)
{
  AddressMap::iterator i;

  double total = 0.0;
  double time = 0.0;

  for (int j = 0;
       j < ADDRESS_MAPS;
       j++) {
    for (i = _Addresses[j].begin ();
	 i != _Addresses[j].end ();
	 ++i)
      {
	switch ((*i)->GetType()) {
	case CodeObject::UC:
	  time = ((CompNode *) ((*i)))->GetRuntime ();
	  // cout << "Adding " << *((CompNode *) ((*i))) << "'s time (" << time << ")" << endl;
	  total += time;
	  // cout << "Total now = " << total << endl;
	  break;
	case CodeObject::NSREL:
	  // We don't keep track of initialization of NameSharingRelations.
	  break;
	case CodeObject::GRAPH:
	  time = ((Graph *) (*i))->T1 ();
	  // cout << "Adding " << *((Graph *) ((*i))) << "'s time (" << time << ")" << endl;
	  total += time;
	  // cout << "Total now = " << total << endl;
	  break;
	default:
	  MESSAGE ("Invalid node tag.");
	  break;
	}
      }
  }
  return total;
}


void Graph::ReportStats (ostream& out) {
  _Map.ReportMemberStats (out);
}


void Graph::TransmitStats (int processor, Messenger& messenger) {
  _Map.TransmitMemberStats (processor, messenger);
}


double Graph::T1 (void) {
  return _Map.T1 ();
}


bool Graph::GetTransmitted (const int dest) const
{
  Guard m (const_cast(Graph *, this)->_lock);
  // Find the bit position of the destination
  // processor and return TRUE if that bit is set.
  int index	= dest / (sizeof (int) * 8);
  int bitval	= 1 << (dest % (sizeof (int) * 8));
  return ((_Transmitted[index] & bitval) == 1);
}


void Graph::SetTransmitted (const int dest) const
{
  Guard m (const_cast(Graph *, this)->_lock);
  // Cast away const-ness for compilers without the "mutable" keyword.
  Graph * self = (Graph *) this;
  // Find the bit position of the destination
  // processor and OR in a 1 in that position.
  int index	= dest / (sizeof (int) * 8);
  int bitval	= 1 << (dest % (sizeof (int) * 8));
  self->_Transmitted[index] |= bitval;
}


int Graph::Processor (void) {
  // self->_Processor = 0;
  // return _Processor;

  if (_Processor != -1) {
    // Processor has been initialized.
    return _Processor;
  } else {
    Guard m (_lock);
    if (GetMachine().GetNumProcessors() == 1) {
      _Processor = 0;
    } else {
      _Processor = GetPathName().map (GetMachine().GetNumProcessors());
    }
    return _Processor;
  }
}


#if 0
void Graph::TransmitState (Messenger& messenger, int dest)
{
  // Disabled because we are broadcasting creation parameters
  // (cf. backends/unified/unified_aux.C)
  Guard m (_lock);
  if (!GetTransmitted (dest)) {
    Copy (this, TRANSMIT, messenger);
    SetTransmitted (dest);
  }
}
#else
void Graph::TransmitState (Messenger&, int)
{}
#endif


#if 0
void Graph::ReceiveState (Messenger& messenger)
{
  // Disabled because we are broadcasting creation parameters
  // (cf. backends/unified/unified_aux.C)
  Guard m (_lock);
  if (_CrepsToGo != 0) {
    Copy (this, RECEIVE, messenger);
    _CrepsToGo = 0;
  }
}
#else
void Graph::ReceiveState (Messenger&)
{}
#endif


ostream& operator<<(ostream& os,
		    Graph& graph)
{
  extern _c2_NameTable NameTable;
  _c2_NameTable * Table = &NameTable;
  int i;

  for (i = 0; Table[i].UID > 0; i++) {
    if (graph.ID() == Table[i].UID)
      break;
  }

  assert (graph.ID() == Table[i].UID);

  if (graph.MyGraph() != NULL) {
    os << *(graph.MyGraph()) << "/";
  }

  for (int index = 0; index < graph.MyIndex().length(); index++)
    os << "[" << graph.MyIndex()[index] << "]";

  os << Table[i].Name;

  return os;
}
