/*
  graph.H
  ------------------------------------------------------------------------
  CODE Graph (container) class.
  ------------------------------------------------------------------------
  @(#) $Id: graph.H,v 1.26 1998/11/06 06:24:20 emery Exp $
  ------------------------------------------------------------------------
  AUTHOR/CONTACT:
 
  Emery Berger                    | <http://www.cs.utexas.edu/users/emery>
  Parallel Programming Group      |  <http://www.cs.utexas.edu/users/code>
  Department of Computer Sciences |             <http://www.cs.utexas.edu>
  University of Texas at Austin   |                <http://www.utexas.edu>
  ========================================================================
*/


#ifndef _GRAPH_H_
#define _GRAPH_H_

#include "header.H"
#include "codeobject.H"
#include "messenger.H"
#include "path.H"
#include "unique.H"


class Identifier {
public:

  Identifier () {}

  Identifier (int uid, Index ind)
    : UID (uid),
      index (ind)
    {}

  int UID;
  Index index;
};


class Entry {
public:

  Entry () {}

  Entry (int objtype, void * entry)
    : ObjType ((_c2_ObjectType) objtype), Object (entry)
    {}

  _c2_ObjectType ObjType;
  void *Object;
};


#if 0
struct hash<Identifier>
{
  size_t operator()(const Identifier& a) const
    {
      int sum = a.UID * a.index.length();
      for (int i = 0; i < a.index.length(); i++) {
	sum += a.index[i];
      }
      return sum;
    }
};


struct equal_to<Identifier>
{
  int operator()(const Identifier& a, const Identifier& b) const
    {
      return (a.UID == b.UID && a.index == b.index);
    }
};

struct less<Identifier>
{
  int operator()(const Identifier& a, const Identifier& b) const
    {
      return (a.UID < b.UID
	      || ((a.UID == b.UID) && (a.index < b.index)));
    }
};

#endif


class Graph : public CodeObject {
  friend ostream& operator<<(ostream&, Graph&);
public:

  // Implementation of Address Maps.

  class AddressMap : public Unique {
  public:

    typedef deq <CodeObject *>::iterator iterator;
    typedef deq <CodeObject *>::const_iterator const_iterator;

    enum { ADDRESS_MAPS = 1 };

    AddressMap (Graph * graph)
      : //_lock ("AddressMap::lock"),
	_MyGraph (graph) { }
    
    ~AddressMap (void);
    
    // Find a given object in the map, creating it if necessary.
    CodeObject * Find (int UID, Index * Index, SerialMachine * SerMach);

    // Initialize all members of map.
    void InitializeMembers (void);
    
    // Print out stats for every member of map.
    void ReportMemberStats (ostream&);

    // Transmit stats for every member of map to a processor.
    void TransmitMemberStats (int, Messenger&);

    // Return the total amount of work (in seconds of computation time)
    // for this graph.
    double T1 (void);

  private:
    //map <Identifier, Entry, less<Identifier> > _Addresses;
    deq<CodeObject *> _Addresses[ADDRESS_MAPS];

    Graph * _MyGraph;

    double _pad[PAD_DBL]; // padding to avoid false sharing.

    mutable MutexRec _lock[ADDRESS_MAPS];
  };

public:
  Graph (int uid,
	 Index * index,
	 Graph * graph,
	 CodeDataBase * localdata,
	 int crepstogo,
	 SerialMachine * machine)
    : CodeObject(uid, index, localdata, graph, machine, CodeObject::GRAPH),
      _CrepsToGo(crepstogo),
      _Map (this)
    {

#ifdef _C2_DBGR_
      _c2_allocDbgGph (tmp);
#endif
    }

  ~Graph (void) {
  }

  inline void * operator new (size_t sz, SerialMachine * sm);
  void * operator new (size_t sz) { return (void *) ::new char[sz]; }
  void operator delete (void *) {}


  // Add a creation parameter to the graph,
  // initializing any nodes waiting once
  // CrepsToGo hits zero.
  void AddCrep (void);

  // Return a pointer to the object matching UID and Index,
  // creating such a node if one doesn't already exist
  // (uses the CODE backend-produced _c2_MakeObj procedure). 
  CodeObject * Find (int UID, Index * Index, SerialMachine * SerMach)
    {
      return _Map.Find (UID, Index, SerMach);
    }


  int Processor (void);

  // Print out stats of every item (recursively)
  // in this graph.
  void ReportStats (ostream& out);

  // Transmit the stats of every item (recursively)
  // in this graph.
  void TransmitStats (int, Messenger&);

  // ACCESSORS

  Graph * GetParent (void) {
    return MyGraph();
  }

  const Graph * GetParent (void) const {
    return MyGraph();
  }

  const int GetCrepsToGo (void) {
    Guard m (_lock);
    return _CrepsToGo;
  }

  const int GetCrepsToGo (void) const {
    Guard m (const_cast(Graph *, this)->_lock);
    return _CrepsToGo;
  }


  // MUTATORS.


  void TransmitState (Messenger&, int);
  void ReceiveState (Messenger&);
  void ReceiveStateFromThief (Messenger&) {}

  //// The following routine is provided
  //// by the CODE backend.

  void Copy (Graph *, int, Messenger&);

  // Return the total amount of work (in seconds of computation time)
  // for this graph.
  double T1 (void);

private:

  bool GetTransmitted (const int dest) const;
  void SetTransmitted (const int dest) const;

  // A bitstring indicating which processors have received my creps.
  mutable int	_Transmitted[Messenger::MAX_PROCESSOR_BYTES];

  int		_CrepsToGo;	// Count of unbound Creation Parameters ("creps").
  
#ifdef _C2_DBGR_
  _c2_DbgGph	_dbg;
#endif
  
  AddressMap	_Map;		// The map of all objects within this graph.

};


inline void * Graph::operator new (size_t sz, SerialMachine * sm)
{
  assert (sm);
  return sm->alloc (sz);
}



#endif // _GRAPH_H_ 

